// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#include <qpainter.h>
#include "ButtonRenderer.h"
#include "CheckBoxRenderer.h"
#include "DownloadRenderer.h"
#include "EntryRenderer.h"
#include "FormRenderer.h"
#include "ImageRenderer.h"
#include "ListBoxRenderer.h"
#include "OptionRenderer.h"
#include "Options.h"
#include "RadioRenderer.h"
#include "Style.h"
#include "StyleSheet.h"
#include "TextAreaRenderer.h"
#include "TextRenderer.h"

AnchorRenderer::AnchorRenderer( const QString& anchorName,
                                Canvas*        canvas,
                                int            clipWidth,
                                QObject*       parent,
                                const char*    name )
        : Renderer( canvas, clipWidth, parent, name ),
          _name( anchorName.copy() )
{
}

const QString& AnchorRenderer::name()
{
    return _name;
}

WordRenderer::WordRenderer( const QString& word,
                            bool           spaceBefore,
                            bool           spaceAfter,
                            bool           endOfLine,
                            Canvas*        canvas,
                            int            clipWidth,
                            QObject*       parent,
                            const char*    name )
        : Renderer( canvas, clipWidth, parent, name ),
          _word( word.copy() ),
          _spaceBefore( spaceBefore ),
          _spaceAfter( spaceAfter ),
          _endOfLine( endOfLine )
{
}

WordRenderer::~WordRenderer()
{
}

const QString& WordRenderer::word()
{
    return _word;
}

bool WordRenderer::spaceBefore()
{
    return _spaceBefore;
}

bool WordRenderer::spaceAfter()
{
    return _spaceAfter;
}

bool WordRenderer::endOfLine()
{
    return _endOfLine;
}

void WordRenderer::repaint( QPainter& p, const Rect& )
{
    p.drawText( x(), y() + p.fontMetrics().ascent(), _word );
}

StyleRenderer::StyleRenderer( const QFont&   font,
                              const QColor&  color,
                              const QString& url,
                              int            spaceWidth,
                              int            ascent,
                              Canvas*        canvas,
                              int            clipWidth,
                              QObject*       parent,
                              const char*    name )
        : Renderer( canvas, clipWidth, parent, name ),
          _font( font ),
          _color( color ),
          _url( url.copy() ),
          _spaceWidth( spaceWidth ),
          _ascent( ascent )
{
}

StyleRenderer::~StyleRenderer()
{
}

const QFont& StyleRenderer::font()
{
    return _font;
}

int StyleRenderer::spaceWidth()
{
    return _spaceWidth;
}

void StyleRenderer::repaint( QPainter& p, const Rect& )
{
    p.setFont( _font );
    p.setPen( _color );
}

TextRenderer::TextRenderer( Canvas*     canvas,
                            SgmlParser* parser,
                            int         clipWidth,
                            QObject*    parent,
                            const char* name )
        : SgmlRenderer( canvas, parser, clipWidth, parent, name ),
          _done( FALSE ),
          _startOfContent( TRUE ),
          _spaceBefore( FALSE ),
          _spaceAfter( FALSE ),
          _textArea( 0 ),
          _option( 0 ),
          _listBox( 0 ),
          _prefix( Style::None )
{
    resize( clipWidth, 0 );
}

TextRenderer::~TextRenderer()
{
    while ( _renderers.first() ) {
        delete _renderers.first();
        _renderers.removeFirst();
    }
    while ( _lines.first() ) {
        delete _lines.first();
        _lines.removeFirst();
    }
}

StyleRenderer* TextRenderer::makeStyle( Style* style, bool endTag )
{
    QString tmpStr;
    QColor color( "black" );
    if ( style->stringValue( "color", tmpStr ) ) {
        color.setNamedColor( tmpStr );
    }

    int tmpInt;
    QFont::Weight weight;
    if ( ( style->enumValue( "font-weight", tmpInt ) ) && ( tmpInt == Style::Bold ) ) {
        weight = QFont::Bold;
    } else {
        weight = QFont::Normal;
    }

    QString fontFamily = "helvetica";
    style->listValue( "font-family", fontFamily );

    int fontSize = 12;
    style->numberValue( "font-size", fontSize );

    int fontStyle = Style::Normal;
    style->enumValue( "font-style", fontStyle );
    
    QFont font( fontFamily, fontSize, weight, fontStyle == Style::Italic );

    int textDecoration = Style::Normal;
    style->enumValue( "text-decoration", textDecoration );
    
    font.setUnderline( textDecoration == Style::Underline );
    QFontMetrics fm( font );

    StyleRenderer* s = new StyleRenderer( font, color, findHyperlink( endTag ), fm.width( ' ' ), fm.ascent(), canvas(), 0, this );
    s->resize( 0, 0 );

    return s;
}

bool TextRenderer::findAnchor( const QString& name, int& x, int& y )
{
    Renderer* r;
    
    for ( r = _renderers.first(); r; r = _renderers.next() ) {
        if ( !strcmp( r->className(), "AnchorRenderer" ) ) {
            if ( ((AnchorRenderer*)r)->name() == name ) {
                x = r->x();
                y = r->y();
                return TRUE;
            }
        }
    }

    return FALSE;
}

const QString TextRenderer::findHyperlink( bool endTag )
{
    const QString* tmp = 0;
    QString        url;
    QString        tmpStr;
    QListIterator<STag>  i( tagStack() );
    QListIterator<Style> j( styleStack() );

    i.toLast();
    j.toLast();
    if ( endTag ) {
        --i;
        --j;
    }
    
    for ( ; i.current() && j.current(); --i, --j ) {
        if ( j.current()->displayType() == Style::Hyperlink ) {
            if ( ( j.current()->stringValue( "link-attr", tmpStr ) ) && ( tmp = i.current()->find( tmpStr ) ) ) {
                url = *tmp;
                break;
            }
        }
    }

    return url;
}

void TextRenderer::endOfData()
{
    // Need to flush any unprocessed content.
    _done = TRUE;
    content( "" );
    needRedraw();
}

void TextRenderer::endTag()
{
    _textArea = 0;

    if ( style()->displayType() == Style::FormOption ) {
        if ( _option ) {
            _option->addItem( _optionText, _optionSelected, _optionValue );
        } else if ( _listBox ) {
            _listBox->addItem( _optionText, _optionSelected, _optionValue );
        }
    } else if ( style()->displayType() == Style::FormOptionMenu ) {
        if ( _option ) {
            _option->redraw();
            if ( _option->width() > _minimumWidth ) {
                _minimumWidth = _option->width();
            }
            _maximumWidth += _option->width();
            _option = 0;
        }
    } else if ( style()->displayType() == Style::FormListBox ) {
        if ( _listBox ) {
            _listBox->redraw();
            if ( _listBox->width() > _minimumWidth ) {
                _minimumWidth = _listBox->width();
            }
            _maximumWidth += _listBox->width();
            _listBox = 0;
        }
    }
    
    // Need to flush any unprocessed content.
    _done = TRUE;
    content( "" );
    _done = FALSE;
    _spaceBefore = FALSE;
    _spaceAfter  = FALSE;

    QListIterator<Style> i( styleStack() );
    i.toLast();
    --i;
    _renderers.append( makeStyle( i.current(), TRUE ) );
}

void TextRenderer::startTag()
{
    _textArea = 0;
    
    // Need to flush any unprocessed content.
    _done = TRUE;
    content( "" );
    _done = FALSE;

    _spaceBefore = FALSE;
    _spaceAfter  = FALSE;

    _renderers.append( makeStyle( style() ) );

    QString tmp;
    const QString* attr1 = 0;
    const QString* attr2 = 0;
    const QString* attr3 = 0;
    
    if ( style()->displayType() == Style::Anchor ) {
        if ( ( style()->stringValue( "anchor-attr", tmp ) ) && ( attr1 = tag()->find( tmp ) ) ) {
            AnchorRenderer* anchor = new AnchorRenderer( *attr1, canvas(), 0, this );
            _renderers.append( anchor );
        }
    } else if ( style()->displayType() == Style::Image ) {
        if ( ( style()->stringValue( "source-attr", tmp ) ) && ( attr1 = tag()->find( tmp ) ) ) {
            int w = 40, h = 40;

            if ( ( style()->stringValue( "width-attr", tmp ) ) && ( attr2 = tag()->find( tmp ) ) ) {
                w = attr2->toInt();
            }
            if ( ( style()->stringValue( "height-attr", tmp ) ) && ( attr2 = tag()->find( tmp ) ) ) {
                h = attr2->toInt();
            }

            bool ismap = FALSE;

            if ( ( style()->stringValue( "is-map-attr", tmp ) ) && ( tag()->find( tmp ) ) ) {
                ismap = TRUE;
            }
            if ( ( style()->stringValue( "use-map-attr", tmp ) ) && ( tag()->find( tmp ) ) ) {
                ismap = TRUE;
            }

            ImageRenderer* image = new ImageRenderer( *attr1, w, h, ismap, FALSE, findHyperlink(), canvas(), 0, this );
            connect( image, SIGNAL( resized() ), this, SLOT( childSizeChanged() ) );
            if ( w > _minimumWidth ) {
                _minimumWidth = w;
            }
            _maximumWidth += w;
            _renderers.append( image );
        }
    } else if ( style()->displayType() == Style::FormButton ) {
        attr1 = attr2 = attr3 = 0;
        if ( style()->stringValue( "name-attr", tmp ) ) {
            attr1 = tag()->find( tmp );
        }
        if ( style()->stringValue( "value-attr", tmp ) ) {
            attr2 = tag()->find( tmp );
        }
        if ( style()->stringValue( "is-reset", tmp ) ) {
            attr3 = tag()->find( tmp );
        }
        ButtonRenderer* button;
        if ( attr1 ) {
            button = new ButtonRenderer( ( attr3 != NULL ), attr2, canvas(), 0, this, attr1->data() );
        } else {
            button = new ButtonRenderer( ( attr3 != NULL ), attr2, canvas(), 0, this );
        }
        if ( button->width() > _minimumWidth ) {
            _minimumWidth = button->width();
        }
        _maximumWidth += button->width();
        _renderers.append( button );
    } else if ( style()->displayType() == Style::FormCheckBox ) {
        attr1 = attr2 = attr3 = 0;
        if ( style()->stringValue( "name-attr", tmp ) ) {
            attr1 = tag()->find( tmp );
        }
        if ( style()->stringValue( "value-attr", tmp ) ) {
            attr2 = tag()->find( tmp );
        }
        if ( style()->stringValue( "checked-attr", tmp ) ) {
            attr3 = tag()->find( tmp );
        }
        CheckBoxRenderer* checkBox;
        if ( attr1 ) {
            checkBox = new CheckBoxRenderer( ( attr3 != NULL ), attr2, canvas(), 0, this, attr1->data() );
        } else {
            checkBox = new CheckBoxRenderer( ( attr3 != NULL ), attr2, canvas(), 0, this );
        }
        if ( checkBox->width() > _minimumWidth ) {
            _minimumWidth = checkBox->width();
        }
        _maximumWidth += checkBox->width();
        _renderers.append( checkBox );
    } else if ( style()->displayType() == Style::FormOptionMenu ) {
        attr1 = 0;
        if ( style()->stringValue( "name-attr", tmp ) ) {
            attr1 = tag()->find( tmp );
        }
        OptionRenderer* option;
        if ( attr1 ) {
            option = new OptionRenderer( canvas(), 0, this, attr1->data() );
        } else {
            option = new OptionRenderer( canvas(), 0, this );
        }
        _renderers.append( option );
        _option = option;
    } else if ( style()->displayType() == Style::FormEntry ) {
        attr1 = attr2 = 0;
        if ( style()->stringValue( "name-attr", tmp ) ) {
            attr1 = tag()->find( tmp );
        }
        if ( style()->stringValue( "value-attr", tmp ) ) {
            attr2 = tag()->find( tmp );
        }
        int maxLength = -1;
        if ( ( style()->stringValue( "max-length-attr", tmp ) ) && ( attr3 = tag()->find( tmp ) ) ) {
            maxLength = attr3->toInt();
        }
        int size = -1;
        if ( ( style()->stringValue( "size-attr", tmp ) ) && ( attr3 = tag()->find( tmp ) ) ) {
            size = attr3->toInt();
        }
        EntryRenderer* entry;
        if ( attr1 ) {
            entry = new EntryRenderer( maxLength, size, style()->flagValue( "is-password" ), attr2, canvas(), 0, this, attr1->data() );
        } else {
            entry = new EntryRenderer( maxLength, size, style()->flagValue( "is-password" ), attr2, canvas(), 0, this );
        }
        if ( entry->width() > _minimumWidth ) {
            _minimumWidth = entry->width();
        }
        _maximumWidth += entry->width();
        _renderers.append( entry );
    } else if ( style()->displayType() == Style::FormHidden ) {
        if ( ( style()->stringValue( "name-attr", tmp ) ) && ( attr1 = tag()->find( tmp ) ) ) {
            if ( ( style()->stringValue( "value-attr", tmp ) ) && ( attr2 = tag()->find( tmp ) ) ) {
                QObject* p;
                for ( p = parent(); p; p = p->parent() ) {
                    if ( !strcmp( p->className(), "FormRenderer" ) ) {
                        ((FormRenderer*)p)->addHidden( attr1->data(), attr2->data() );
                        break;
                    }
                }
            }
        }
    } else if ( style()->displayType() == Style::FormImage ) {
        if ( ( style()->stringValue( "source-attr", tmp ) ) && ( attr1 = tag()->find( tmp ) ) ) {
            int w = 40, h = 40;

            if ( ( style()->stringValue( "width-attr", tmp ) ) && ( attr2 = tag()->find( tmp ) ) ) {
                w = attr2->toInt();
            }
            if ( ( style()->stringValue( "height-attr", tmp ) ) && ( attr2 = tag()->find( tmp ) ) ) {
                h = attr2->toInt();
            }

            bool ismap = FALSE;

            if ( ( style()->stringValue( "is-map-attr", tmp ) ) && ( tag()->find( tmp ) ) ) {
                ismap = TRUE;
            }
            if ( ( style()->stringValue( "use-map-attr", tmp ) ) && ( tag()->find( tmp ) ) ) {
                ismap = TRUE;
            }
            
            attr2 = 0;
            if ( style()->stringValue( "name-attr", tmp ) ) {
                attr2 = tag()->find( tmp );
            }

            ImageRenderer* image;
            if ( attr2 ) {
                image = new ImageRenderer( *attr1, w, h, ismap, FALSE, findHyperlink(), canvas(), 0, this, attr2->data() );
            } else {
                image = new ImageRenderer( *attr1, w, h, ismap, FALSE, findHyperlink(), canvas(), 0, this );
            }
            connect( image, SIGNAL( resized() ), this, SLOT( childSizeChanged() ) );
            if ( w > _minimumWidth ) {
                _minimumWidth = w;
            }
            _maximumWidth += w;
            _renderers.append( image );
        }
    } else if ( style()->displayType() == Style::FormListBox ) {
        attr1 = 0;
        if ( style()->stringValue( "name-attr", tmp ) ) {
            attr1 = tag()->find( tmp );
        }
        int size = 1;
        if ( ( style()->stringValue( "size-attr", tmp ) ) && ( attr2 = tag()->find( tmp ) ) ) {
            size = attr2->toInt();
        }

        ListBoxRenderer* listBox;
        if ( attr1 ) {
            listBox = new ListBoxRenderer( size, canvas(), 0, this, attr1->data() );
        } else {
            listBox = new ListBoxRenderer( size, canvas(), 0, this );
        }
        _renderers.append( listBox );
        _listBox = listBox;
    } else if ( style()->displayType() == Style::FormMLE ) {
        attr1 = 0;
        if ( style()->stringValue( "name-attr", tmp ) ) {
            attr1 = tag()->find( tmp );
        }

        int rows = 10;
        if ( ( style()->stringValue( "rows-attr", tmp ) ) && ( attr2 = tag()->find( tmp ) ) ) {
            rows = attr2->toInt();
        }

        int cols = 20;
        if ( ( style()->stringValue( "cols-attr", tmp ) ) && ( attr2 = tag()->find( tmp ) ) ) {
            cols = attr2->toInt();
        }

        TextAreaRenderer* textArea;
        if ( attr1 ) {
            textArea = new TextAreaRenderer( rows, cols, canvas(), 0, this, attr1->data() );
        } else {
            textArea = new TextAreaRenderer( rows, cols, canvas(), 0, this );
        }
        if ( textArea->width() > _minimumWidth ) {
            _minimumWidth = textArea->width();
        }
        _maximumWidth += textArea->width();
        _renderers.append( textArea );

        _textArea = textArea;
    } else if ( style()->displayType() == Style::FormOption ) {
        attr1 = 0;
        if ( style()->stringValue( "value-attr", tmp ) ) {
            attr1 = tag()->find( tmp );
        }
        if ( attr1 ) {
            _optionValue = attr1->copy();
        } else {
            _optionValue = 0;
        }

        _optionSelected = ( ( style()->stringValue( "selected-attr", tmp ) ) && ( tag()->find( tmp ) ) );

        _optionText = "";
    } else if ( style()->displayType() == Style::FormRadio ) {
        attr1 = attr2 = attr3 = 0;
        if ( style()->stringValue( "name-attr", tmp ) ) {
            attr1 = tag()->find( tmp );
        }
        if ( style()->stringValue( "value-attr", tmp ) ) {
            attr2 = tag()->find( tmp );
        }
        if ( style()->stringValue( "checked-attr", tmp ) ) {
            attr3 = tag()->find( tmp );
        }
        RadioRenderer* radio;
        if ( attr1 ) {
            radio = new RadioRenderer( ( attr3 != NULL ), attr2, canvas(), 0, this, attr1->data() );
        } else {
            radio = new RadioRenderer( ( attr3 != NULL ), attr2, canvas(), 0, this );
        }
        if ( radio->width() > _minimumWidth ) {
            _minimumWidth = radio->width();
        }
        _maximumWidth += radio->width();
        _renderers.append( radio );
    }
}

void TextRenderer::repaint( QPainter& p, const Rect& r )
{
    p.translate( x(), y() );

    if ( _lines.first() ) {
        if ( _prefix == Style::Bullet ) {
            QBrush brush( black );
            p.setPen( black );
            p.setBrush( brush );
            int h = _lines.first()->height();
            p.drawEllipse( 0, (h-7)/2, 7, 7 );
        }
    }
    
    Line*     line;
    Renderer* i;
    for ( line = _lines.first(); line; line = _lines.next() ) {
        line->style->repaint( p, r );
        if ( line->intersects( r ) ) {
            for ( i = line->renderers.first(); i; i = line->renderers.next() ) {
                if ( !strcmp( i->className(), "StyleRenderer" ) ) {
                    i->repaint( p, r );
                } else if ( i->intersects( r ) ) {
                    Rect r1 = i->intersect( r );
                    r1.moveBy( -i->x(), -i->y() );
                    i->repaint( p, r1 );
                }
            }
        }
    }

    p.translate( -x(), -y() );
}
