/*\
||| Written By Fredrik Hbinette <hubbe@lysator.liu.se>
||| All rights reserved. No warrenties, use at your own risk.
\*/

#include <X11/X.h>
#include <X11/Xos.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <sysexits.h>
#include <signal.h>
#include <stdarg.h>
#include <npapi.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <sys/socket.h>

/* #define DEBUG */

#define H_LOOP 1
#define H_MANY 2
#define H_STREAM 4
#define H_NOISY 8
#define H_REPEATCOUNT 16
#define H_PRELOAD 32

#define BUFSIZE 1024*256
#define PRELOAD 40000

struct data
{
  Window window;
  int pid;

  char *mimetype;
  int repeats;

  int flags;
  char *command;

#ifdef STREAMER
  int fd;
  int peekfd;
  int waitfd;

  int buffering;
  int buffered;
  int offset;
  char *buffer;
#endif
};


#define THIS ((struct data *) instance->pdata)
#define FEND(X,Y) (strchr(X,Y)?strchr(X,Y):(X)+strlen(X))

#ifdef DEBUG
static FILE *getout()
{
  static FILE *foo=0;
  if(foo) return foo;
  foo=fopen("/tmp/ndebug","a+");
  fprintf(foo,"------------\n");
  return foo;
}
#define D(fmt, X...) \
  fprintf(getout(), "PID%4d:" fmt,getpid() ,##X),fflush(getout())
#else
#define D(fmt, X...)
#endif

#ifndef MIN
#define MIN(X,Y) ((X)<(Y)?(X):(Y))
#endif

#ifndef MAXINT
#define MAXINT 0x7fffffff
#endif


char* NPP_GetMIMEDescription(void)
{
  D("Getmimedescription\n");
  return 

#define BEGINHANDLE
#define MIME(TYPE,SUBTYPE,SUFFIXES,DESCRIPTION) \
   TYPE "/" SUBTYPE ":" SUFFIXES ":" DESCRIPTION ";" \
   TYPE "/x-" SUBTYPE ":" SUFFIXES ":" DESCRIPTION ";" 
#define HANDLERS
#define HANDLE(FLAGS, COMMAND)
#define INTERNAL_HANDLE(X)
#define ENDHANDLE

#include "mime.h"

#undef BEGINHANDLE
#undef MIME
#undef HANDLERS
#undef HANDLE
#undef ENDHANDLE
#undef INTERNAL_HANDLE

    ;
}

static int inpath(char *file)
{
  char tmp[16384];
  char *path=getenv("PATH");
  char *pos;
  if(!path) return 0;
  D("inpath(%s)\n",file);
  D("Hmm? PATH=%s\n",path);
  pos=path;
  while(1)
  {
    char *next;
    next=FEND(pos,':');
    if(next!=pos)
    {
      char *ptr=tmp;
      struct stat buf;

      memcpy(ptr=tmp,pos,next-pos);
      ptr+=next-pos;
      *(ptr++)='/';
      memcpy(ptr,
	     file,
	     FEND(file,' ')-file);
      ptr+=FEND(file,' ')-file;
      *ptr=0;
      D("stat(%s)\n",tmp);
      if(!stat(tmp, &buf)) return 1;
      D("nope\n");
    }
    if(!*next) return 0;
    pos=next+1;
  }
  D("GAAHHH!\n");
}

static int my_fork(NPP instance)
{
  int pid;
  sigset_t set,oset;
  D("forking\n");
  sigfillset(& set);
  sigprocmask(SIG_SETMASK,&set,&oset);
  pid=fork();
  if(pid==-1) return pid;
  if(!pid)
  {
    int signum;
    setsid();
    for(signum=0;signum<NSIG;signum++) signal(signum, SIG_DFL);
  }else{
#ifdef STREAMER
    if(THIS->peekfd>=0)
      close(THIS->peekfd);
#endif
    D("Child running\n");
  }
  sigprocmask(SIG_SETMASK,&oset,&set);
  return pid;
}


static void run(NPP instance, const char *file)
{
  while(THIS->repeats>0)
  {
    char *argv[256];
    char *s,*tmp,*pos,buffer[16384];
    int arg;
    int loops=1;

    if(file && (THIS->flags & H_STREAM))
    {
      int fd=open(file,O_RDONLY);
      dup2(fd,0);
      close(fd);
      D("Stream from file %s\n",file);
    }

    pos=buffer;
    s=THIS->command;
    while(1)
    {
      char *foo=FEND(s,'%');
      memcpy(pos,s,foo-s);
      pos+=foo-s;
      *pos=0;
      D("Partial command is: %s\n",buffer);
      if(!*foo) break;
      s=foo+1;
      
      switch(*(s++))
      {
      case '%': *(pos++)='%'; break;
	
      case 'f':
	if(THIS->flags & H_MANY)
	{
	  if(THIS->repeats)
	  {
	    int e;
	    e=MIN(THIS->repeats/loops,10);
	    loops*=e;
	    
	    while(--e>=0)
	    {
	      strcat(pos,file);
	      strcat(pos," ");
	      pos+=strlen(pos);
	    }
	    break;
	  }
	}else{
	  strcat(pos,file);
	  pos+=strlen(pos);
	  break;
	}
	
      case 'r':
	sprintf(pos,"%d",THIS->repeats);
	pos+=strlen(pos);
	loops=THIS->repeats;
	break;

      case 'w':
	sprintf(pos,"%ld",(long)THIS->window);
	pos+=strlen(pos);
	break;
	
      case 'L':
	tmp=FEND(s, ' ');
	if(THIS->repeats >= 0x7fffffff)
	{
	  memcpy(pos,s,tmp-s);
	  pos+=tmp-s;
	}
	s=tmp;
	loops=0x7fffffff;
	break;
	
      case 'l':
	tmp=FEND(s, ' ');
	if(THIS->repeats < 0x7ffffff)
	{
	  memcpy(pos,s,tmp-s);
	  pos+=tmp-s;
	}
	s=tmp;
	loops=0x7fffffff;
	break;
      
      }
    }

    D("Final command is: %s\n",buffer);

    s=buffer;
    arg=0;
    while(1)
    {
      char *foo;
      while(*s==' ') s++;
      foo=FEND(s,' ');
      if(foo!=s) argv[arg++]=s;
      if(!*foo) break;
      *foo=0;
      s=foo+1;
    }
    argv[arg++]=0;

#ifdef DEBUG
	{
	  int q;
	  for(q=0;argv[q];q++)
	    D("Argv[%d]=%s\n",q,argv[q]);
	}
#endif
    D("Execing\n");
    if(THIS->repeats > loops)
    {
      while(THIS->repeats > loops)
      {
	int pid;
	D("Running %s\n",buffer);
	pid=fork();
	if(pid==-1) exit(10);
	
	if(!pid)
	{

	  if(THIS->flags & H_NOISY)
	  {
	    int nl=open("/dev/null", O_RDONLY);
	    D("Redirecting stdout and stderr\n");
	    dup2(nl,1);
	    dup2(nl,2);
	    close(nl);
	  }

	  execvp(argv[0],argv);
	  D("Execvp failed..%d\n",errno);
	  exit(EX_UNAVAILABLE);
	}else{
	  int status;
	  D("waiting for (%d)\n",pid);
	  waitpid(pid,&status,0);
	  D("wait done\n");
	  if(!WIFEXITED(status)) exit(10);
	  if(WEXITSTATUS(status)) exit(1);
	  D("exited ok!\n");
	}
	if(THIS->repeats < MAXINT)
	  THIS->repeats-=loops;
      }
    }else{
      if(THIS->flags & H_NOISY)
      {
	int nl=open("/dev/null", O_RDONLY);
	D("Redirecting stdout and stderr\n");
	dup2(nl,1);
	dup2(nl,2);
	close(nl);
      }
      D("Execing!!!\n");
      execvp(argv[0],argv);
      D("Execvp failed!!!\n");
      exit(EX_UNAVAILABLE);
    }
  }
  exit(0);
}

static int check_command(NPP instance,
			 int streaming,
			 char *command,
			 int flags)
{
  if(!inpath(command)) return 0;
  if( (flags & H_LOOP)  && THIS->repeats != MAXINT) return 0;
  if( (!!streaming) != (!!(flags & H_STREAM))) return 0;
  THIS->command=command;
  THIS->flags=flags;
  return 1;
}

static int find_command(NPP instance,
			int streaming)
{
#define BEGINHANDLE if(
#define MIME(TYPE,SUBTYPE,SUFFIXES,DESCRIPTION) \
   !strcasecmp(TYPE "/" SUBTYPE ,THIS->mimetype) || \
   !strcasecmp(TYPE "/x-" SUBTYPE , THIS->mimetype) ||
#define HANDLERS 0) {
#define HANDLE(FLAGS, COMMAND) \
   if(check_command(instance, streaming, COMMAND, FLAGS)) return 1;
#define INTERNAL_HANDLE(X) THIS->command=X; return 1;
#define ENDHANDLE }

#include "mime.h"

#undef BEGINHANDLE
#undef MIME
#undef HANDLERS
#undef HANDLE
#undef INTERNAL_HANDLE
#undef ENDHANDLE

  return 0;
}

NPError NPP_GetValue(void *future, NPPVariable variable, void *value)
{
  NPError err = NPERR_NO_ERROR;

  D("Getvalue %d\n",variable);
  
  switch (variable)
  {
  case NPPVpluginNameString:
    *((char **)value) = "Plugger "
      VERSION
#ifdef STREAMER
      " (streaming)"
#else
      " (nostreams)"
#endif
      ;

  break;

  case NPPVpluginDescriptionString:
    *((char **)value) =
      "<a href=http://www.infovav.se/~hubbe/plugger.html>Plugger</a> "
      "version "
      VERSION
      ", written by "
      "<a href=http://www.lysator.liu.se/~hubbe/>Fredrik Hbinette</a> "
      "<a href=mailto:hubbe@lysator.liu.se>&lt;hubbe@lysator.liu.se&gt</a>"
      ;
  break;

  default:
    err = NPERR_GENERIC_ERROR;
  }
  return err;
}

NPError  NPP_New(NPMIMEType pluginType,
		 NPP instance,
		 uint16 mode,
		 int16 argc,
		 char* argn[],
		 char* argv[],
		 NPSavedData* saved)
{
  int e;
  NPError tmp;

  D("NEW (%s)\n",pluginType);

  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;

  THIS = NPN_MemAlloc(sizeof(struct data));
  if(!THIS) return NPERR_OUT_OF_MEMORY_ERROR;
  memset((char *)THIS, 0, sizeof(*THIS));

  THIS->repeats=MAXINT;
  THIS->pid=-1;
#ifdef STREAMER
  THIS->fd=-1;
  THIS->waitfd=-1;
  THIS->peekfd=-1;
#endif

  if(!pluginType) return NPERR_GENERIC_ERROR;

  for(e=0;e<argc;e++)
  {
    if(!strcasecmp("loop",argn[e]))
    {
      switch(argv[e][0])
      {
      case 't': case 'T': case 'y': case 'Y':
	THIS->repeats=MAXINT;
	break;
      case 'f': case 'F': case 'n': case 'N':
	THIS->repeats=1;
	break;
      case '0': case '1': case '2': case '3': case '4':
      case '5': case '6': case '7': case '8': case '9':
	THIS->repeats=atoi(argv[e]);
      }
    }
  }

  return NPERR_NO_ERROR;
}

NPError NPP_Destroy(NPP instance, NPSavedData** save)
{
  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;

  D("Destroy\n");

  if (THIS)
  {
    if(THIS->pid > 0)
      {
	D("killing %d\n",-THIS->pid);
	/* Die Die Die!!! */
	if(!kill(-THIS->pid, SIGTERM))
	{
	  if(!kill(-THIS->pid, SIGTERM))
	  {
	    usleep(20);
	    if(!kill(-THIS->pid, SIGTERM))
	    {
	      kill(-THIS->pid, SIGKILL);
	    }
	  }
	}
	THIS->pid=-1;
      }

    D("Freeing memory %p\n",THIS->mimetype);
    
    if(THIS->mimetype)
    {
      NPN_MemFree(THIS->mimetype);
      THIS->mimetype=0;
    }

#ifdef STREAMER
    D("Closing fds\n");
    if(THIS->fd >= 0)
    {
      close(THIS->fd);
      THIS->fd=-1;
    }
    
    if(THIS->peekfd>=0)
    {
      close(THIS->peekfd);
      THIS->peekfd=-1;
    }
    
    if(THIS->waitfd >= 0)
    {
      close(THIS->waitfd);
      THIS->waitfd=-1;
    }
    
    if(THIS->buffer)
    {
      NPN_MemFree(THIS->buffer);
      THIS->buffer=0;
    }

    NPN_MemFree(THIS);
    THIS = 0;
#endif
  }
  D("Destroy finished\n");
  
  return NPERR_NO_ERROR;
}


NPError NPP_NewStream(NPP instance,
		      NPMIMEType type,
		      NPStream *stream, 
		      NPBool seekable,
		      uint16 *stype)
{
  int wantstream;
  D("Newstream ... \n");

  if (instance == NULL)
    return NPERR_INVALID_INSTANCE_ERROR;

  if(!strncasecmp("image/",type,6) ||
     !strncasecmp("x-image/",type,6))
    THIS->repeats=1;

  D("Mime type %s\n",type);

  if(THIS->mimetype)
  {
    NPN_MemFree(THIS->mimetype);
    THIS->mimetype=0;
  }
  THIS->mimetype = NPN_MemAlloc(strlen(type)+1);
  if(!THIS->mimetype) return NPERR_OUT_OF_MEMORY_ERROR;
  strcpy(THIS->mimetype, type);

  D("Url is %s (seekable=%d)\n",stream->url,seekable);
#ifdef STREAMER
  wantstream=!(seekable && !strncasecmp(stream->url,"file:",5));
#else
  wantstream=0;
#endif
  
  if(!find_command(instance, wantstream))
  {
    if(!find_command(instance, !wantstream))
    {
      NPN_Status(instance, "No approperiate application found!");
      return NPERR_GENERIC_ERROR;
    }
  }

#ifdef STREAMER
  if((THIS->flags & H_STREAM) && strncasecmp(stream->url,"file:",5))
  {
    int foo[2],bar[2];

    if(THIS->repeats == 1 || 
      (THIS->flags & H_LOOP) ||
      (THIS->flags & H_REPEATCOUNT))
      *stype=NP_NORMAL;
    else
      *stype=NP_ASFILE;

    if(socketpair(AF_UNIX, SOCK_STREAM, 0, foo)<0 ||
       pipe(bar) < 0)
    {
      NPN_Status(instance, "Streamer: Failed to create a pipe!");
      return NPERR_GENERIC_ERROR;
    }

    THIS->pid=my_fork(instance);
    if(THIS->pid==-1)
    {
      NPN_Status(instance, "Streamer: My_Fork failed!");
      return;
    }
    
    if(!THIS->pid)
    {
      D("Streaming child running\n");
      close(bar[0]);

      close(foo[1]);
      dup2(foo[0],0);
      close(foo[0]);

      THIS->repeats=1;
      run(instance, 0);
    }else{
      THIS->buffer=NPN_MemAlloc(BUFSIZE);
      if(!THIS->buffer)
	return NPERR_OUT_OF_MEMORY_ERROR;
      
      if(THIS->repeats < MAXINT)
	THIS->repeats--;
      THIS->fd=foo[1];
      D("FD to parent = %d\n",THIS->fd);
      fcntl(THIS->fd, F_SETFL, FNDELAY);

      THIS->waitfd=bar[0];
      close(bar[1]);
      if(THIS->flags & H_PRELOAD)
      {
	THIS->buffering=1;
	THIS->peekfd=foo[0];
      }else{
	close(foo[0]);
      }
    }
    
    D("Ok\n");
  }else
#endif
    *stype = NP_ASFILEONLY;
  
  return NPERR_NO_ERROR;
}


#ifdef STREAMER

static int data_available(int fd)
{
  fd_set tmp;
  struct timeval timeout;

  do {
    timeout.tv_sec=0;
    timeout.tv_usec=0;
    FD_ZERO(&tmp);
    FD_SET(fd, &tmp);
  } while(select(fd+1, &tmp, 0, 0, &timeout)<0 && errno==EINTR);
  return FD_ISSET(fd, &tmp);
}

static int trywrite(NPP instance)
{
  D("trywrite (%d bytes buffered) fd=%d\n",THIS->buffered,THIS->fd);

  if(THIS->buffering)
  {
    if(THIS->buffered < PRELOAD)
    {
      char b[256];
      sprintf(b,"Buffering ... %2d%%",THIS->buffered * 100 / PRELOAD);
      D("%s\n",b);
      NPN_Status(instance, b);
      return 1;
    }
    THIS->buffering=0;
  }

  if(THIS->peekfd>=0)
  {
    D("Checking waitfd\n");
    if(data_available(THIS->waitfd))
    {
      close(THIS->peekfd);
      THIS->peekfd=-1;
    }
  }

  while(THIS->buffered)
  {
    int i;

    do {
      D("trywrite %d bytes (offset = %d)\n",
	MIN(THIS->buffered, BUFSIZE-THIS->offset),
	THIS->offset);
      i=write(THIS->fd,
	      THIS->buffer+THIS->offset,
	      MIN(THIS->buffered, BUFSIZE-THIS->offset) );
      D("Wrote %d bytes (errno = %d)\n",i, errno);
    } while(i<0 && errno==EINTR);

    if(i<0)
    {
      switch(errno)
      {
      case EALREADY:
      case EAGAIN:
	return 1;
      default:
	D("trywrite: Errno = %d\n",errno);
	return 0;
      }
    }

    if(!i) return 1;

    THIS->offset+=i;
    THIS->buffered-=i;
    if(THIS->offset == BUFSIZE) THIS->offset=0;
  }

  D("Checking preload\n");

  if((THIS->flags & H_PRELOAD) &&
     THIS->peekfd>=0 &&
     !data_available(THIS->peekfd))
  {
    D("(Re)-starting preload\n");
    THIS->buffering=1;
  }
  D("trywrite-exit: errno = %d\n",errno);
  return 1;
}
#endif

int32 NPP_WriteReady(NPP instance, NPStream *stream)
{
#ifdef STREAMER
  D("Writeready\n");
  if(!instance) return 0;

  trywrite(instance);

  D("Writeready returns: %d\n", BUFSIZE - THIS->buffered);

  /* Kluge to make netscape behave! */
  if(!BUFSIZE - THIS->buffered)
    usleep(10);
  return BUFSIZE - THIS->buffered;
#else
  return 0x7fffffff;
#endif
}

int32 NPP_Write(NPP instance,
		NPStream *stream,
		int32 offset,
		int32 len,
		void *buffer)
{
#ifdef STREAMER
  int32 origlen=len;
  int i;
  D("Write(len=%d, offset=%d)\n",len,offset);
  if (!instance) return 0;
		
  if(!trywrite(instance))
    return -1;

  D("Write: THIS->buffered=%d\n",THIS->buffered);
  if(!THIS->buffered && !THIS->buffering)
  {
    D("Attempting direct write\n");
    do {
      i=write(THIS->fd, buffer, len);
    } while(i<0 && errno==EINTR);

    if(i<0)
    {
      switch(errno)
      {
      case EALREADY:
      case EAGAIN:
	D("Nothing written\n");
	break;

      default:
	D("Errno: %d\n",errno);
	return -1;
      }
    }else{
      D("Wrote %d bytes directly\n",i);
      buffer+=i;
      len-=i;
    }
  }

  while(len>0 && THIS->buffered<BUFSIZE)
  {
    int end=THIS->offset + THIS->buffered;
    end%=BUFSIZE;
    i=MIN(len, BUFSIZE-end);

    i=MIN(i, BUFSIZE-THIS->buffered);

    memcpy(THIS->buffer+end, buffer, i);
    len-=i;
    buffer+=i;
    THIS->buffered+=i;
  }

  D("Write returns %d\n",origlen-len);
  return origlen-len;
#else
  return len;
#endif
}

NPError NPP_DestroyStream(NPP instance, NPStream *stream, NPError reason)
{
  D("Destroystream\n");
#ifdef STREAMER
  if(THIS->flags & H_STREAM)
  {
    THIS->buffering=0;

    if(THIS->peekfd>=0)
    {
      close(THIS->peekfd);
      THIS->peekfd=-1;
    }
    if(trywrite(instance))
    {
      if(THIS->buffered)
      {
	int pid=my_fork(instance);
	if(pid==-1) return NPERR_GENERIC_ERROR;
	if(!pid)
	{
	  fcntl(THIS->fd, F_SETFL, 0);
	  while(THIS->buffered && trywrite(instance));
	  D("Buffer-cleanup done\n");
	  exit(0);
	}
      }
    }
    
    close(THIS->fd);
    THIS->fd=-1;
  }
  D("Destroystream done\n");
#endif
  return NPERR_NO_ERROR;
}

void NPP_StreamAsFile(NPP instance,
		      NPStream *stream,
		      const char* fname)
{
  int pid;
  D("Streamasfile\n");
  if(!fname) return;
  if (instance == NULL) return;

  NPN_Status (instance, "Running helper ...");

  if(!strcmp(THIS->command, "internal:url"))
  {
    int fd;
    char *url=NPN_MemAlloc(stream->end+1);
    fd=open(fname,O_RDONLY);
    if(fd<0)
    {
      NPN_Status(instance,"Plugger: Hey, where did the file go?\n");
    } else {
      if(read(fd, url, stream->end) == stream->end)
      {
	url[stream->end]=0;
	FEND(url,'\n')[0]=0;
	NPN_GetURL(instance, url, 0);
      }
      close(fd);
    }
    NPN_MemFree(url);
  }else{
    THIS->pid=my_fork(instance);

    if(THIS->pid==-1) return;
    
    if(!THIS->pid)
    {
#ifdef STREAMER
      if(THIS->flags & H_STREAM)
      {
	char foo[1];
	if(THIS->fd > -1) close(THIS->fd);
	D("Waiting for streaming child to exit.\n");
	while(read(THIS->waitfd, foo, 1) < 0 && errno==EINTR);
	if(THIS->repeats < MAXINT)
	  THIS->repeats--;
      }
#endif

      if(!find_command(instance, 0))
	if(!find_command(instance, 1))
	  exit(EX_UNAVAILABLE);

      run(instance, fname);
    }
  }
}
  
NPError NPP_Initialize(void)
{
  D("init\n");
  return NPERR_NO_ERROR;
}
jref NPP_GetJavaClass()
{
  D("Getjavaclass\n");
  return NULL;
}
void NPP_Shutdown(void)
{
  D("Shutdown\n");
}

NPError NPP_SetWindow(NPP instance, NPWindow* window)
{
  D("SETWINDOW\n");

  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;
  
  if (!window)
    return NPERR_NO_ERROR;
  
  if (!window->window)
    return (NPERR_NO_ERROR);
  
  if(THIS->window == (Window) window->window)
  {
    /* Window size change.... */
  }else{
    /* New window */
    THIS->window = (Window) window->window;
    D("Received window %x\n",THIS->window);
  }

  return NPERR_NO_ERROR;
}

void NPP_Print(NPP instance, NPPrint* printInfo)
{
  D("PRINT\n");
  if(printInfo == NULL)
    return;
  
  if (instance != NULL)
  {
    if (printInfo->mode == NP_FULL) {
      void* platformPrint =
	printInfo->print.fullPrint.platformPrint;
      NPBool printOne =
	printInfo->print.fullPrint.printOne;
      
      /* Do the default*/
      printInfo->print.fullPrint.pluginPrinted = FALSE;
    }
    else
    {	/* If not fullscreen, we must be embedded */
      NPWindow* printWindow =
	&(printInfo->print.embedPrint.window);
      void* platformPrint =
	printInfo->print.embedPrint.platformPrint;
    }
  }
}


