// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  MLLoadScene.C

  Stephane Rehel
  April 22 1997
*/

#include <string.h>
#include <errno.h>

#include "tools/ListFiles.h"

#include "kw/KWDialog.h"

#include "interface/MLIOProgressDialog.h"

#include "model/ModelModule.h"

#include "scene/file/MLDatabasesList.h"
#include "scene/file/MLDatabase.h"
#include "scene/file/MLSceneIO.h"

#include "MLLoadScene.h"
#include "LoadSceneDlg.h"
#include "ReadingProgressDlg.h"

/////////////////////////////////////////////////////////////////////////////

static const char* errorString= "*ERROR*";

/////////////////////////////////////////////////////////////////////////////

MLLoadScene::MLLoadScene( ModelModule* _model ):
  MLGlobalDialog(_model)
{
  model= _model;
  dblist= 0;

  files= new ListFiles;
}

/////////////////////////////////////////////////////////////////////////////

MLLoadScene::~MLLoadScene()
{
  delete dblist;
  dblist= 0;

  delete files;
  files= 0;
}

/////////////////////////////////////////////////////////////////////////////

// return the abstract object read
// return 0 if cancelled or error
IBOOL MLLoadScene::run()
{
  delete dblist;
  dblist= new MLDatabasesList;
  dblist->build();

  if( dblist->get().empty() )
    {
    KWError( "Error",
               OString("No database available!\nCheck: ")
             + MLDatabasesList::getDatabasesResourceFilename() );
    return IFALSE;
    }

  if( ! MLGlobalDialog::create(LoadSceneDlg::data) )
    return IFALSE;

  SIListIterator<MLDatabase> li(dblist->get());
  while( ! li.eol() )
    {
    MLDatabase* db= li++;
    kwDialog->appendListText( LoadSceneDlg::databases,
                              db->getBasePath() );
    }
  kwDialog->setInteger( LoadSceneDlg::databases,
                        dblist->getCurrentDatabasePosition());

  OString dbpath= kwDialog->getText(LoadSceneDlg::databases);
  currentDatabase= dblist->getDatabase(dbpath);

  updateFilesList();

  kwDialog->focus(LoadSceneDlg::filename);

  int outcome= kwDialog->run();

  OString filename= kwDialog->getText(LoadSceneDlg::filename);
  if( filename == errorString )
    filename= "";

  if( outcome == KWDialog::CANCEL )
    {
    MLGlobalDialog::close();
    return IFALSE;
    }

  IBOOL ok= load(filename);

  MLGlobalDialog::close();

  return ok;
}

/////////////////////////////////////////////////////////////////////////////

IBOOL MLLoadScene::load( const OString& filename )
{
  if( currentDatabase == 0 || filename.length() == 0 )
    return IFALSE;

  model->stopLightingProcess();

  MLIOProgressDialog* progress= new MLIOProgressDialog(model);
  progress->create( ReadingProgressDlg::data,
                    ReadingProgressDlg::progress,
                    ReadingProgressDlg::label );
  progress->flush();

  model->pushWaitCursor();

  IBOOL outcome= MLSceneIO::read( currentDatabase,
                                  model->getScene(),
                                  filename,
                                  progress );

  delete progress;
  progress= 0;

  model->popWaitCursor();

  if( ! outcome )
    {
    KWError( "Error",
               OString("Unable to load:\n")
             + filename + OString("\n")
             + OString(strerror(errno)) );
    return IFALSE;
    }

  return ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

void MLLoadScene::updateFilesList()
{
  if( kwDialog == 0 )
    return;

  kwDialog->clearList(LoadSceneDlg::list);

  MLDatabase* database= currentDatabase;
  if( database == 0 )
    return;

  MLDatabasesList::setCurrent( currentDatabase->getBasePath() );

  OString path= database->getScenePath();
  OString pattern= OString("*") + OString(MLDatabase::sceneExtension);

  if( files->list(path,pattern,IFALSE) != ListFiles::SUCCESS )
    {
    kwDialog->appendListText( LoadSceneDlg::list, errorString );
    return;
    }

  IListIterator<ListFiles::FileItem> li(files->getFiles());
  while( ! li.eol() )
    {
    ListFiles::FileItem* item= li.next();
    kwDialog->appendListText( LoadSceneDlg::list, item->name );
    }
}

/////////////////////////////////////////////////////////////////////////////

IBOOL MLLoadScene::kwCallback( KWEvent& event )
{
  if( event.type == KWEvent::DOUBLE_SELECT &&
      event.widget == LoadSceneDlg::list )
    {
    kwDialog->quit(KWDialog::VALIDATE);
    return ITRUE;
    }

  if( event.type == KWEvent::SELECT &&
      event.widget == LoadSceneDlg::list )
    {
    OString filename= kwDialog->getText(LoadSceneDlg::list);
    if( filename == OString(errorString) )
      filename= "";

    kwDialog->setText( LoadSceneDlg::filename, filename );
    kwDialog->focus(LoadSceneDlg::filename);
    return ITRUE;
    }

  if( event.type == KWEvent::SELECT &&
      event.widget == LoadSceneDlg::databases )
    {
    OString dbpath= kwDialog->getText(LoadSceneDlg::databases);
    currentDatabase= dblist->getDatabase(dbpath);
    updateFilesList();
    return ITRUE;
    }

  return MLGlobalDialog::kwCallback(event);
}

/////////////////////////////////////////////////////////////////////////////

