;;; Interval timers for GNU Emacs
;;; Copyright (C) 1988, 1991, 1993 Kyle E. Jones
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 1, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; A copy of the GNU General Public License can be obtained from this
;;; program's author (send electronic mail to kyle@uunet.uu.net) or from
;;; the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA
;;; 02139, USA.
;;;
;;; Send bug reports to kyle@uunet.uu.net.

(provide 'itimer)

;; `itimer' feature means Emacs-Lisp programers get:
;;    itimerp, itimer-value, itimer-restart, itimer-function,
;;    set-itimer-value, set-itimer-restart, set-itimer-function
;;    get-itimer, start-itimer, read-itimer, delete-itimer
;;
;; Interactive users get these commands:
;;    edit-itimers, list-itimers, start-itimer
;;
;; See the doc strings of these functions for more information.

(defvar itimer-version "1.00"
  "Version number of the itimer package.")

(defvar itimer-list nil
  "List of all active itimers.")

(defvar itimer-process nil
  "Process that drives all itimers.")

;; This value is maintained internally; it does not determine itimer
;; granularity.  Itimer granularity is 1 second, plus delays due to
;; system and Emacs internal activity that delay dealing with process
;; output.
(defvar itimer-process-next-wakeup 1
  "Itimer process will wakeup to service running itimers within this
many seconds.")

(defvar itimer-edit-map nil
  "Keymap used when in Itimer Edit mode.")

(if itimer-edit-map
    ()
  (setq itimer-edit-map (make-sparse-keymap))
  (define-key itimer-edit-map "s" 'itimer-edit-set-field)
  (define-key itimer-edit-map "d" 'itimer-edit-delete-itimer)
  (define-key itimer-edit-map "q" 'itimer-edit-quit)
  (define-key itimer-edit-map "\t" 'itimer-edit-next-field)
  (define-key itimer-edit-map " " 'next-line)
  (define-key itimer-edit-map "n" 'next-line)
  (define-key itimer-edit-map "p" 'previous-line)
  (define-key itimer-edit-map "\C-?" 'itimer-edit-previous-field)
  (define-key itimer-edit-map "x" 'start-itimer)
  (define-key itimer-edit-map "?" 'itimer-edit-help))
  
(defvar itimer-edit-start-marker nil)

;; macros must come first... or byte-compile'd code will throw back its
;; head and scream.

(defmacro decrement (variable)
  (list 'setq variable (list '1- variable)))

(defmacro increment (variable)
  (list 'setq variable (list '1+ variable)))

(defmacro itimer-signum (n)
  (list 'if (list '> n 0) 1
    (list 'if (list 'zerop n) 0 -1)))

;; Itimer access functions should behave as if they were subrs.  These
;; macros are used to check the arguments to the itimer functions and
;; signal errors appropriately if the arguments are not valid.

(defmacro check-itimer (var)
  "If VAR is not bound to an itimer, signal wrong-type-argument.
This is a macro."
  (list 'setq var
	(list 'if (list 'itimerp var) var
	      (list 'signal ''wrong-type-argument
		    (list 'list ''itimerp var)))))

(defmacro check-itimer-coerce-string (var)
  "If VAR is not bound to a string, look up the itimer that it names and
bind VAR to it.  Otherwise if VAR is not bound to an itimer, signal
wrong-type-argument.  This is a macro."
  (list 'setq var
	(list 'cond
	      (list (list 'itimerp var) var)
	      (list (list 'stringp var) (list 'get-itimer var))
	      (list t (list 'signal ''wrong-type-argument
			    (list 'list ''string-or-itimer-p var))))))

(defmacro check-natnumber (var)
  "If VAR is not bound to a non-negative number, signal wrong-type-argument.
This is a macro."
  (list 'setq var
	(list 'if (list 'natnump var) var
	      (list 'signal ''wrong-type-argument
		    (list 'list ''natnump var)))))

(defmacro check-string (var)
  "If VAR is not bound to a string, signal wrong-type-argument.
This is a macro."
  (list 'setq var
	(list 'if (list 'stringp var) var
	      (list 'signal ''wrong-type-argument
		    (list 'list ''stringp var)))))

;; Functions to access and modify itimer attributes.

(defun itimerp (obj)
  "Returns non-nil iff OBJ is an itimer."
  (and (consp obj) (stringp (car obj)) (eq (length obj) 4)))

(defun itimer-name (itimer)
  "Returns the name of ITIMER."
  (check-itimer itimer)
  (car itimer))

(defun itimer-value (itimer)
  "Returns the number of seconds until ITIMER expires."
  (check-itimer itimer)
  (nth 1 itimer))

(defun itimer-restart (itimer)
  "Returns the value to which ITIMER will be set at restart.
nil is returned if this itimer doesn't restart."
  (check-itimer itimer)
  (nth 2 itimer))

(defun itimer-function (itimer)
  "Returns the function of ITIMER.
This function is called each time ITIMER expires."
  (check-itimer itimer)
  (nth 3 itimer))

(defun set-itimer-value (itimer value &optional nowakeup)
  "Set the timeout value of ITIMER to be VALUE.
Itimer will expire is this many seconds.
Returns VALUE."
;; Optional third arg NOWAKEUP non-nil means do not wakeup the itimer
;; process to recompute a correct wakeup time, even if it means this
;; itimer will expire late.  itimer-process-filter uses this option.
;; This is not meant for ordinary usage, which is why it is not
;; mentioned in the doc string.
  (check-itimer itimer)
  (check-natnumber value)
  (let ((inhibit-quit t))
    ;; If we're allowed to wakeup the itimer process,
    ;; and the itimer process's next wakeup needs to be recomputed,
    ;; and the itimer is running, then we wakeup the itimer process.
    (or (and (not nowakeup) (< value itimer-process-next-wakeup)
	     (get-itimer (itimer-name itimer))
	     (progn (itimer-process-wakeup)
		    (setcar (cdr itimer) value)
		    (itimer-process-wakeup)))
	(setcar (cdr itimer) value))
    value))

(defun set-itimer-restart (itimer restart)
  "Set the restart value of ITIMER to be RESTART.
If RESTART is nil, ITIMER will not restart when it expires.
Returns RESTART."
  (check-itimer itimer)
  (if restart (check-natnumber restart))
  (and restart (< restart 1) (signal 'args-out-of-range (list restart)))
  (setcar (cdr (cdr itimer)) restart))

(defun set-itimer-function (itimer function)
  "Set the function of ITIMER to be FUNCTION.
FUNCTION will be called when itimer expires.
Returns FUNCTION."
  (check-itimer itimer)
  (setcar (cdr (cdr (cdr itimer))) function))

(defun get-itimer (name)
  "Return itimer named NAME, or nil if there is none."
  (check-string name)
  (assoc name itimer-list))

(defun read-itimer (prompt &optional initial-input)
  "Read the name of an itimer from the minibuffer and return the itimer
associated with that name.  The user is prompted with PROMPT.
Optional second arg INITIAL-INPUT non-nil is inserted into the
  minibuffer as initial user input."
  (get-itimer (completing-read prompt itimer-list nil 'confirm initial-input)))

(defun delete-itimer (itimer)
  "Deletes ITIMER.  ITIMER may be an itimer or the name of one."
  (check-itimer-coerce-string itimer)
  (setq itimer-list (delq itimer itimer-list)))

(defun start-itimer (name function value &optional restart)
  "Start an itimer.
Args are NAME, FUNCTION, VALUE &optional RESTART.
NAME is an identifier for the itimer.  It must be a string.  If an itimer
  already exists with this name, NAME will be modified slightly to until
  it is unique.
FUNCTION should be a function (or symbol naming one) of no arguments.  It
  will be called each time the itimer expires.  The function can access
  itimer that invoked it through the variable `current-itimer'.
VALUE is the number of seconds until this itimer expires.
Optional fourth arg RESTART non-nil means that this itimer should be
  restarted automatically after its function is called.  Normally an itimer
  is deleted at expiration after its function has returned. 
  If non-nil RESTART should be a number indicating the value at which the
  itimer should be set at restart time.
Returns the newly created itimer."
  (interactive
   (list (completing-read "Start itimer: " itimer-list)
	 (read (completing-read "Itimer function: " obarray 'fboundp))
	 (let (value)
	   (while (not (natnump value))
	     (setq value (read-from-minibuffer "Itimer value: " nil nil t)))
	   value)
	 (let ((restart t))
	   (while (and restart (not (natnump restart)))
	     (setq restart (read-from-minibuffer "Itimer restart: " nil nil t)))
	   restart)))
  (check-string name)
  (check-natnumber value)
  (if restart (check-natnumber restart))
  ;; Make proposed itimer name unique if it's not already.
  (let ((oname name)
	(num 2))
    (while (get-itimer name)
      (setq name (concat oname "<" num ">"))
      (increment num)))
  ;; If there's no itimer process, start one now.
  ;; Otherwise wake up the itimer process so that seconds slept before
  ;; the new itimer is created won't be counted against it.
  (if itimer-process
      (itimer-process-wakeup)
    (itimer-process-start))
  (let ((inhibit-quit t))
    ;; add the itimer to the global list
    (setq itimer-list
	  (cons (list name value restart function)
		itimer-list))
    ;; If the itimer process is scheduled to wake up too late for the itimer
    ;; we wake it up to calculate a correct wakeup value giving consideration
    ;; to the newly added itimer.
    (if (< value itimer-process-next-wakeup)
	(itimer-process-wakeup)))
  (car itimer-list))

;; User level functions to list and modify existing itimers.
;; Itimer Edit major mode, and the editing commands thereof.

(defun list-itimers ()
  "Pop up a buffer containing a list of all itimers.
The major mode of the buffer is Itimer Edit mode.  This major mode provides
commands to manipulate itimers; see the documentation for
`itimer-edit-mode' for more information."
  (interactive)
  (let* ((buf (get-buffer-create "*Itimer List*"))
	 (opoint (point))
	 (standard-output buf)
	 (itimers (reverse itimer-list)))
    (set-buffer buf)
    (itimer-edit-mode)
    (setq buffer-read-only nil)
    (erase-buffer)
    (insert "Name                  Value     Restart   Function\n"
	    "----                  -----     -------   --------")
    (if (null itimer-edit-start-marker)
	(setq itimer-edit-start-marker (point)))
    (while itimers
      (newline 1)
      (prin1 (itimer-name (car itimers)))
      (tab-to-tab-stop)
      (prin1 (itimer-value (car itimers)))
      (tab-to-tab-stop)
      (prin1 (itimer-restart (car itimers)))
      (tab-to-tab-stop)
      (prin1 (itimer-function (car itimers)))
      (setq itimers (cdr itimers)))
    ;; restore point
    (goto-char opoint)
    (if (< (point) itimer-edit-start-marker)
	(goto-char itimer-edit-start-marker))
    (setq buffer-read-only t)
    (display-buffer buf)))

(defun edit-itimers ()
  "Display a list of all itimers and select it for editing.
The major mode of the buffer containing the listing is Itimer Edit mode.
This major mode provides commands to manipulate itimers; see the documentation
for `itimer-edit-mode' for more information."
  (interactive)
  ;; since user is editing, make sure displayed data is reasonably up-to-date
  (if itimer-process
      (itimer-process-wakeup))
  (list-itimers)
  (select-window (get-buffer-window "*Itimer List*"))
  (goto-char itimer-edit-start-marker)
  (if itimer-list
      (progn
	(forward-sexp 2)
	(backward-sexp)))
  (message "type q to quit, ? for help"))

;; no point in making this interactive.
(defun itimer-edit-mode ()
  "Major mode for manipulating itimers.
Atrributes of running itimers are changed by moving the cursor to the
desired field and typing `s' to set that field.  The field will then be
set to the value read from the minibuffer.

Commands:
TAB    move forward a field
DEL    move backward a field
s      set a field
d      delete the selected itimer
x      start a new itimer
?      help"
  (kill-all-local-variables)
  (make-local-variable 'tab-stop-list)
  (setq major-mode 'itimer-edit-mode
	mode-name "Itimer Edit"
	truncate-lines t
	tab-stop-list '(22 32 42))
  (abbrev-mode 0)
  (auto-fill-mode 0)
  (buffer-flush-undo (current-buffer))
  (use-local-map itimer-edit-map)
  (set-syntax-table emacs-lisp-mode-syntax-table))

(put 'itimer-edit-mode 'mode-class 'special)

(defun itimer-edit-help ()
  "Help function for Itimer Edit."
  (interactive)
  (if (eq last-command 'itimer-edit-help)
      (describe-mode)
    (message "TAB, DEL select fields, (s)et field, (d)elete itimer   (type ? for more help)")))

(defun itimer-edit-quit ()
  "End Itimer Edit."
  (interactive)
  (bury-buffer (current-buffer))
  (if (one-window-p t)
      (switch-to-buffer (other-buffer (current-buffer)))
    (delete-window)))

(defun itimer-edit-set-field ()
  (interactive)
  ;; First two lines in list buffer are headers.
  ;; Cry out against the luser who attempts to change a field there.
  (if (<= (point) itimer-edit-start-marker)
      (error ""))
  ;; field-value must be initialized to be something other than a
  ;; number, symbol, or list.
  (let (itimer field (field-value ""))
    (setq itimer (save-excursion
		  ;; read the name of the itimer from the beginning of
		  ;; the current line.
		  (beginning-of-line)
		  (get-itimer (read (current-buffer))))
	  field (save-excursion
		  (itimer-edit-beginning-of-field)
		  (let ((opoint (point))
			(n 0))
		    ;; count the number of sexprs until we reach the cursor
		    ;; and use this info to determine which field the user
		    ;; wants to modify.
		    (beginning-of-line)
		    (while (and (>= opoint (point)) (< n 4))
		      (forward-sexp 2)
		      (backward-sexp)
		      (increment n))
		    (cond ((eq n 1) (error "Cannot change itimer name."))
			  ((eq n 2) 'value)
			  ((eq n 3) 'restart)
			  ((eq n 4) 'function)))))
    (cond ((eq field 'value)
	   (let ((prompt "Set itimer value: "))
	     (while (not (natnump field-value))
	       (setq field-value (read-from-minibuffer prompt nil nil t)))))
	  ((eq field 'restart)
	   (let ((prompt "Set itimer restart: "))
	     (while (and field-value (not (natnump field-value)))
	       (setq field-value (read-from-minibuffer prompt nil nil t)))))
	  ((eq field 'function)
	   (let ((prompt "Set itimer function: "))
	     (while (not (or (and (symbolp field-value) (fboundp field-value))
			     (and (consp field-value)
				  (memq (car field-value) '(lambda macro)))))
	       (setq field-value
		     (read (completing-read prompt obarray 'fboundp nil)))))))
    ;; set the itimer field
    (funcall (intern (concat "set-itimer-" (symbol-name field)))
	     itimer field-value)
    ;; move to beginning of field to be changed
    (itimer-edit-beginning-of-field)
    ;; modify the list buffer to reflect the change.
    (let (buffer-read-only kill-ring)
      (kill-sexp 1)
      (kill-region (point) (progn (skip-chars-forward " \t") (point)))
      (prin1 field-value (current-buffer))
      (if (not (eolp))
	  (tab-to-tab-stop))
      (backward-sexp))))

(defun itimer-edit-delete-itimer ()
  (interactive)
  ;; First two lines in list buffer are headers.
  ;; Cry out against the luser who attempts to change a field there.
  (if (<= (point) itimer-edit-start-marker)
      (error ""))
  (delete-itimer
   (read-itimer "Delete itimer: "
	       (save-excursion (beginning-of-line) (read (current-buffer)))))
  ;; update list information
  (list-itimers))

(defun itimer-edit-next-field (count)
  (interactive "p")
  (itimer-edit-beginning-of-field)
  (cond ((> (itimer-signum count) 0)
	 (while (not (zerop count))
	   (forward-sexp)
	   ;; wrap from eob to itimer-edit-start-marker
	   (if (eobp)
	       (progn
		 (goto-char itimer-edit-start-marker)
		 (forward-sexp)))
	   (forward-sexp)
	   (backward-sexp)
	   ;; treat fields at beginning of line as if they weren't there.
	   (if (bolp)
	       (progn
		 (forward-sexp 2)
		 (backward-sexp)))
	   (decrement count)))
	((< (itimer-signum count) 0)
	 (while (not (zerop count))
	   (backward-sexp)
	   ;; treat fields at beginning of line as if they weren't there.
	   (if (bolp)
	       (backward-sexp))
	   ;; wrap from itimer-edit-start-marker to field at eob.
	   (if (<= (point) itimer-edit-start-marker)
	       (progn
		 (goto-char (point-max))
		 (backward-sexp)))
	   (increment count)))))

(defun itimer-edit-previous-field (count)
  (interactive "p")
  (itimer-edit-next-field (- count)))

(defun itimer-edit-beginning-of-field ()
  (let ((forw-back (save-excursion (forward-sexp) (backward-sexp) (point)))
	(back (save-excursion (backward-sexp) (point))))
    (cond ((eq forw-back back) (backward-sexp))
	  ((eq forw-back (point)) t)
	  (t (backward-sexp)))))


;; internals of the itimer implementation.

(defun itimer-process-filter (process string)
  ;; If the itimer process dies and generates output while doing
  ;; so, we may be called before the process-sentinel.  Sanity
  ;; check the output just in case...
  (if (not (string-match "^[0-9]" string))
      (progn (message "itimer process gave odd output: %s" string)
	     ;; it may be still alive and waiting for input
	     (process-send-string itimer-process "3\n"))
    ;; if there are no active itimers, return quickly.
    (if itimer-list
	(let ((time-elapsed (string-to-int string))
	      (itimers itimer-list)
	      (itimer)
	      ;; process filters can be hit by stray C-g's from the user,
	      ;; so we must protect this stuff appropriately.
	      ;; Quit's are allowed from within itimer functions, but we
	      ;; catch them.
	      (inhibit-quit t))
	  (setq itimer-process-next-wakeup 600)
	  (while itimers
	    (setq itimer (car itimers))
	    (set-itimer-value itimer (max 0 (- (itimer-value itimer) time-elapsed)) t)
	    (if (> (itimer-value itimer) 0)
		(setq itimer-process-next-wakeup
		      (min itimer-process-next-wakeup (itimer-value itimer)))
	      ;; itimer has expired, we must call its function.
	      ;; protect our local vars from the itimer function.
	      ;; allow keyboard quit to occur, but catch and report it.
	      ;; provide the variable `current-itimer' in case the function
	      ;; is interested.
	      (condition-case condition-data
		  (let* ((current-itimer itimer)
			 itimer itimers time-elapsed
			 quit-flag inhibit-quit)
		    (funcall (itimer-function current-itimer)))
		(error (message "itimer \"%s\" signaled: %s" (itimer-name itimer)
				(prin1-to-string condition-data)))
		(quit (message "itimer \"%s\" quit" (itimer-name itimer))))
	      ;; restart the itimer if we should, otherwise delete it.
	      (if (null (itimer-restart itimer))
		  (delete-itimer itimer)
		(set-itimer-value itimer (itimer-restart itimer) t)
		(setq itimer-process-next-wakeup
		      (min itimer-process-next-wakeup (itimer-value itimer)))))
	    (setq itimers (cdr itimers)))
	  ;; if user is editing itimers, update displayed info
	  (if (eq major-mode 'itimer-edit-mode)
	      (list-itimers)))
      (setq itimer-process-next-wakeup 600))
    ;; tell itimer-process when to wakeup again
    (process-send-string itimer-process
			 (concat (int-to-string itimer-process-next-wakeup)
				 "\n"))))

(defun itimer-process-sentinel (process message)
  (let ((inhibit-quit t))
    (if (eq (process-status process) 'stop)
	(continue-process process)
      ;; not stopped, so it must have died.
      ;; cleanup first...
      (delete-process process)
      (setq itimer-process nil)
      ;; now, if there are any active itimers then we need to immediately
      ;; start another itimer process, otherwise we can wait until the next
      ;; start-itimer call,  which will start one automatically.
      (if (null itimer-list)
	  ()
	;; there may have been an error message in the echo area;
	;; give the user at least a little time to read it.
	(sit-for 2)
	(message "itimer process %s... respawning." (substring message 0 -1))
	(itimer-process-start)))))

(defun itimer-process-start ()
  (let ((inhibit-quit t)
	(process-connection-type nil))
    (setq itimer-process (start-process "itimer" nil "itimer"))
    (process-kill-without-query itimer-process)
    (set-process-filter itimer-process 'itimer-process-filter)
    (set-process-sentinel itimer-process 'itimer-process-sentinel)
    ;; Tell itimer process to wake up quickly, so that a correct wakeup
    ;; time can be computed.  Zero instead of one here loses because of
    ;; underlying itimer implementations that use 0 to mean `disable the
    ;; itimer'.
    (setq itimer-process-next-wakeup 1)
    (process-send-string itimer-process "1\n")))

(defun itimer-process-wakeup ()
  (interrupt-process itimer-process)
  (accept-process-output))
