/* AX.25 Utilities: Attach an interface.
 * Alan Cox, 1993, 1994
 * modified by Bruce Perens, November 1994
 *
 * Copyright 1994 Alan Cox.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <linux/sockios.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <fcntl.h>
#include <signal.h>
#include <ctype.h>

#include "global.h"

#define	TRUE		1
#define	FALSE		0

static void		do_command(const char *, const char *,
			 const char *, const char *, int speed);
static int		setspeed(int fd, int speed);
static int		setifcall(int fd, char *name);
static void		usage();

static const char *	myname;

int
main(int argc, char *argv[])
{
	int  fd;
	int  disc = N_SLIP;
	char buf[64];
	int  v = 4;
	int  speed = 0;
	char * command = 0;
	
	myname = argv[0];

	while ( argc > 1 &&  argv[1][0] == '-' )
	{
		switch (argv[1][1]) {
		case 's':
			if(sscanf(argv[2],"%d",&speed) != 1)
			{
				fprintf(stderr,"%s: Invalid speed '%s'\n",argv[0],argv[2]);
				exit(1);
			}
			argc--;
			argv++;
			break;
		case 'c':
			command = argv[2];
			argc--;
			argv++;
			break;

		case '\0':
		default:
			usage();
		}	
		argc--;
		argv++;
	}
	
	if (argc != 3)
		usage();
	
	if ((fd = open(argv[1], O_RDONLY | O_NDELAY)) == -1)
	{
		perror(argv[1]);
		return(1);
	}

	if (speed!=0 && !setspeed(fd, speed))
		return(1);
	
	if (ioctl(fd, TIOCSETD, &disc) == -1)
	{
		perror("TIOCSETD");
		return(1);
	}
	
	if (ioctl(fd, SIOCGIFNAME, buf) == -1)
	{
		perror("SIOCGIFNAME");
		return(1);
	}
	
	if (!setifcall(fd, argv[2]))
		return(1);
	
	/* Now set the encapsulation */
	if (ioctl(fd, SIOCSIFENCAP, &v) == -1)
	{
		perror("SIOCSIFENCAP");
		return(1);
	}
		
	printf("Interface %s bound to device %s with callsign %s.\n",
		buf, argv[1], argv[2]);	

	if ( command != 0 )
		do_command(command, buf, argv[1], argv[2], speed);
		
	signal(SIGHUP, SIG_IGN);
	
	while (1)
		sleep(10000);
		
	/* NOT REACHED */
	return(0);
}

static void
do_command(const char * prototype, const char * interface,
	const char * device, const char * callsign, int speed)
{
	char		command[2048];
	char *		s = command;
	const char *	p = prototype;

	while ( *p != '\0' ) {
		if ( *p == '%' ) {

			*s = '\0';

			switch ( *++p ) {
			case 'i':	/* %i = Interface */
				p++;
				strcpy(s, interface);
				break;
			case 'd':	/* %t = Device */
				p++;
				strcpy(s, device);
				break;
			case 'c':	/* %c = Callsign */
				p++;
				strcpy(s, callsign);
				break;
			case 's':	/* %s = Speed */
				p++;
				sprintf(s, "%d", speed);
				break;
			case '\0':
			default:
				*s++ = '%';
				break;
			}
			while ( *s != '\0' )	
				s++;
		}
		else
			*s++ = *p++;
	}
	*s = '\0';
	system(command);
}

static int
setifcall(int fd, char *name)
{
	char call[7];
	int  ct = 0;
	int  ssid = 0;
	char *p = name;
	
	while (ct < 6)
	{
		if (islower(*p)) *p = toupper(*p);

		call[ct] = (*p << 1);
		p++;
		ct++;

		if (*p == '-' || *p == 0) break;
	}
	
	while (ct < 6)
	{
		call[ct] = ' ' << 1;
		ct++;
	}
	
	if (*p != 0)
	{
		if (sscanf(p + 1, "%d", &ssid) != 1 || ssid < 0 || ssid > 15)
		{
			fprintf(stderr, "SSID must follow '-' and be numeric in the range 0-15.\n");
			return(FALSE);
		}
	}

	call[6] = ((ssid + '0') << 1) & 0x1E;
		
	if (ioctl(fd, SIOCSIFHWADDR, call) != 0)
	{
		close(fd);
		perror("SIOCSIFHWADDR");
		return(FALSE);
	}

	return(TRUE);
}
	
extern int termios_speed_encoding(int);

static int
setspeed(int fd, int speed)
{
	struct termios	term;
	int		termios_speed = termios_speed_encoding(speed);
	
	if ( speed <= 0 || termios_speed < 0 )
	{
		fprintf(stderr, "Invalid speed %d\n", speed);
		return(FALSE);
	}
	
	if (tcgetattr(fd, &term) == -1)
	{
		perror("tcgetattr");
		return(FALSE);
	}
	
	cfsetispeed(&term, termios_speed);
	cfsetospeed(&term, termios_speed);
	
	if (tcsetattr(fd, TCSANOW, &term) == -1)
	{
		perror("tcsetattr");
		return(FALSE);
	}
	
	return(TRUE);
}

static void
usage()
{
	static const char	message[] =
	"\n"
	"Usage: %s [-c <command>] [-s <speed>] <device> <callsign>.\n"
	"\n"
	"\t-c <command>: <command> will be executed by the shell after the\n"
	"\t\tinterface is initialized. Interface parameters can be\n"
	"\t\tpassed in the command by using these \"%%\" patterns:\n"
	"\t\t\t%%i\tNetwork interface name.\n"
	"\t\t\t%%d\tDevice name.\n"
	"\t\t\t%%c\tCallsign.\n"
	"\t\t\t%%s\tSpeed.\n"
	"\t\t\t%%%%\tA single %% character.\n"
	"\n"
	"\t-s <speed>: Set the Baud rate for serial interfaces (default 0).\n"
	"\n";

	fprintf(stderr, message, myname);
	exit(1);
}
