/* `mfile.c' -- Routines to do i/o via mmap() whenever possible, or use
   read()/write() otherwise.
   For descriptions of the routines, see the comments in mmap.h.

   Copyright (C) 1996 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Written by Fung-Chai Lim (fclim@acm.org).
   Most of the codes in here was snarfed from GNU grep-2.0,
   written by Mike Haertel.

   This file is part of GNU ecc.  */

#ifdef HAVE_CONFIG_H
#if defined (CONFIG_BROKETS)
/* We use <config.h> instead of "config.h" so that a compilation
   using -I. -I$srcdir will use ./config.h rather than $srcdir/config.h
   (which it would do because it found this file in $srcdir).  */
#include <config.h>
#else
#include "config.h"
#endif
#endif

#include <stdio.h>
#if STDC_HEADERS
#include <stdlib.h>
#else
extern char *malloc ();
#endif
#include <sys/types.h>
#include <sys/stat.h>
#if HAVE_MMAP
#include <sys/mman.h>
#endif
#include "getpagesize.h"

#include <errno.h>
#ifndef errno
extern int errno;
#endif

#if __STDC__ && defined (HAVE_VPRINTF)
void error (int, int, char const *, ...);
#else
void error ();
#endif

#include "mfile.h"

extern char *program_name;

#ifndef MAX
#define MAX(a, b)	(((a) > (b)) ? (a) : (b))
#endif

     struct mfile *
mopen (fname, fd, mode)
     const char *const fname;
     int fd;
     enum MTYPE mode;
{
  struct mfile *file;
  int pagesize, save_size, buffer_size;

  file = (struct mfile *) malloc (sizeof *file);
  if (file == NULL)
    {
      error (0, errno, "virtual memory exhausted");
      close (fd);
      return NULL;
    }

  if (fstat (fd, &(file->st)) < 0)
    {
      close (fd);
      error (0, errno, "can't stat %s", fname);
      return NULL;
    }

  file->save_size = MAX (8192, getpagesize ());
  file->buffer_size = 4 * file->save_size;
  file->buffer = valloc (file->buffer_size + file->save_size);
  if (file->buffer == NULL)
    {
      close (fd);
      free (file);
      error (0, errno, "virtual memory exhausted");
      return NULL;
    }

  file->fd = fd;
  file->fname = fname;
  file->mode = mode;
  if (mode == MREAD)
    {
      file->beg = file->lim = file->buffer + file->save_size;
#if HAVE_MMAP
      file->prot = PROT_READ;
      file->flags = MAP_SHARED | MAP_FIXED;
#endif
    }
  else
    {
      file->beg = file->save = file->buffer + file->buffer_size;
#if HAVE_MMAP
      file->prot = PROT_WRITE | PROT_READ;
      file->flags = MAP_SHARED | MAP_FIXED;
#endif
    }

#if HAVE_MMAP
  if (S_ISREG (file->st.st_mode) == 0)
    {
      file->is_mapped = 0;
      if (file->mode == MWRITE)
	file->beg = file->buffer;
    }
  else
    {
      file->is_mapped = 1;
      file->offset = 0;
    }
#ifndef HAVE_FTRUNCATE
  if (file->mode == MWRITE)
    file->is_mapped = 0;
#endif
#else
  file->is_mapped = 0;
#endif
  return file;
}

     int
mclose (file)
     struct mfile *file;
{
#if HAVE_MMAP
  if (file->is_mapped)
    {
      if (file->mode == MWRITE)
	(void) mflush (file);
      munmap (file->buffer, file->buffer_size);
      if (file->mode == MWRITE && ftruncate (file->fd, file->wbytes) < 0)
	error (1, errno, "can't truncate %s", file->fname);
    }
#endif

  close (file->fd);
  free (file->buffer);
  free (file);
}

     int
mfill (file, more)
     struct mfile *file;
     int more;
{
  int pagesize;
  char *nbuffer, *dp, *sp;
  int cc;
  int save, nsave;
#if HAVE_MMAP
  caddr_t maddr;
#endif

  if (file->mode != MREAD)
    error (1, 0, "Internal error: mfill on file opened for writing.");

  save = nsave = file->lim - file->beg;
  sp = file->beg;
  dp = file->buffer + file->save_size - save;
  file->beg = dp;
  while (save--)
    *dp++ = *sp++;

#if HAVE_MMAP
  if (file->is_mapped && file->st.st_size - file->offset >= file->buffer_size)
    {
      maddr = file->buffer + file->save_size;
      maddr = mmap (maddr, file->buffer_size, file->prot,
		    file->flags, file->fd, file->offset);
      if (maddr == (caddr_t) -1)
	{
	  error (0, errno, "warning: unable to map %s", file->fname);
	  goto tryread;
	}
#if notdef
      /* You might think this (or MADV_WILLNEED) would help,
	 but it doesn't, at least not on a Sun running 4.1.
	 In fact, it actually slows us down about 30%!  */
      madvise (maddr, file->bsize - file->ssize, MADV_SEQUENTIAL);
#endif
      cc = file->buffer_size;
      file->offset += file->buffer_size;
    }
  else
    {
    tryread:
      /* We come here when we're not going to use mmap() any more.
         Note that we need to synchronize the file offset the
	 first time through.  */
      if (file->is_mapped)
	{
	  file->is_mapped = 0;
	  munmap (file->buffer + file->save_size, file->buffer_size);
	  lseek (file->fd, file->offset, 0);
	}
      cc = read (file->fd, file->buffer + file->save_size, file->buffer_size);
    }
#else
  cc = read (file->fd, file->buffer + file->save_size, file->buffer_size);
#endif
  file->lim = file->buffer + file->save_size;
  if (cc > 0)
    file->lim += cc;
  cc += nsave;
  return cc;
}

     int
mflush (file)
     struct mfile *file;
{
  char *p, *q;
#if HAVE_MMAP
  caddr_t maddr;
#endif

  if (file->mode != MWRITE)
    error (1, 0, "Internal error: mflush on file opened for reading.");

  if (!file->is_mapped)
    {
      int n = write (file->fd, file->buffer, file->beg - file->buffer);
      file->beg = file->buffer;
      return n;
    }

#if HAVE_MMAP
  if (file->beg < file->save)
    return 0;

  if (ftruncate (file->fd, file->offset + file->buffer_size) < 0)
    error (1, errno, "can't grow %s.", file->fname);

  maddr = file->buffer;
  maddr = mmap (maddr, file->buffer_size, file->prot,
		file->flags, file->fd, file->offset);
  if (maddr == (caddr_t) -1)
    {
      error (0, errno, "warning: unable to map %s", file->fname);
      if (file->is_mapped)
	{
          file->is_mapped = 0;
          munmap (file->buffer + file->save_size, file->buffer_size);
	  if (ftruncate (file->fd, file->wbytes) < 0)
	    error (0, errno, "can't truncate %s", file->fname);
          lseek (file->fd, file->offset, 0);
	}
    }

  p = (char *) file->save;
  q = (char *) file->buffer;
  while (p < (char *) file->beg)
    *q++ = *p++;
  file->beg = q;
  file->offset += file->buffer_size;

  return file->buffer_size;
#endif
}
