/* Fax file input processing
   Copyright (C) 1990, 1995  Frank D. Cringle.

This file is part of viewfax - g3/g4 fax processing software.
     
viewfax is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.
     
This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.
     
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include "faxexpand.h"

#define	FAXMAGIC	"\000PC Research, Inc\000\000\000\000\000\000"

/* Enter an argument in the linked list of pages */
struct pagenode *
notefile(char *name)
{
    struct pagenode *new = (struct pagenode *) xmalloc(sizeof *new);

    *new = defaultpage;
    if (firstpage == NULL)
	firstpage = new;
    new->prev = lastpage;
    new->next = NULL;
    if (lastpage != NULL)
	lastpage->next = new;
    lastpage = new;
    new->pathname = name;
    if ((new->name = strrchr(new->pathname, '/')) != NULL)
	new->name++;
    else
	new->name = new->pathname;

    if (new->width == 0)
	new->width = 1728;
    if (new->vres < 0)
	new->vres = !(new->name[0] == 'f' && new->name[1] == 'n');
    new->extra = NULL;
    return new;
}

#ifdef WITH_TIFF
static t32bits
get4(unsigned char *p, int endian)
{
    return endian ? (p[0]<<24)|(p[1]<<16)|(p[2]<<8)|p[3] :
	p[0]|(p[1]<<8)|(p[2]<<16)|(p[3]<<24);
}

static int
get2(unsigned char *p, int endian)
{
    return endian ? (p[0]<<8)|p[1] : p[0]|(p[1]<<8);
}

/* generate pagenodes for the images in a tiff file */
void
notetiff(char *name)
{
    FILE *tf;
    unsigned char header[8];
    char littleTIFF[4] = "\x49\x49\x2a\x00";
    char bigTIFF[4] = "\x4d\x4d\x00\x2a";
    int endian;
    t32bits IFDoff;
    struct pagenode *pn = NULL;

    if ((tf = fopen(name, "r")) == NULL) {
	perror(name);
	return;
    }

    if (fread(header, 8, 1, tf) == 0) {
    nottiff:
	fclose(tf);
	(void) notefile(name);
	return;
    }
    if (memcmp(header, &littleTIFF, 4) == 0)
	endian = 0;
    else if (memcmp(header, &bigTIFF, 4) == 0)
	endian = 1;
    else
	goto nottiff;
    IFDoff = get4(header+4, endian);
    if (IFDoff & 1)
	goto nottiff;
    do {			/* for each page */
	unsigned char buf[12];
	int ndirent;
	pixnum iwidth = defaultpage.width ? defaultpage.width : 1728;
	pixnum iheight = defaultpage.height ? defaultpage.height : 2339;
	int inverse = defaultpage.inverse;
	int lsbfirst = 0;
	int t4opt = 0, comp = 0;
	int orient = defaultpage.orient;
	off_t dataoffs = 0, here;
	size_t bytecount = 0;
	double yres = 196.0;

	if (fseek(tf, IFDoff, SEEK_SET) < 0) {
	bad:
	    /* have we recorded any valid pages yet? */
	    if (pn == NULL)
		goto nottiff;
	    else {
		fprintf(stderr, "%s:%s: invalid tiff file\n", ProgName, name);
		fclose(tf);
		return;
	    }
	}
	if (fread(buf, 2, 1, tf) == 0)
	    goto bad;
	ndirent = get2(buf, endian);
	while (ndirent--) {	/* for each directory entry */
	    int tag, ftype;
	    t32bits count, value = 0;
	    if (fread(buf, 12, 1, tf) == 0)
		goto bad;
	    tag = get2(buf, endian);
	    ftype = get2(buf+2, endian);
	    count = get4(buf+4, endian);
	    if (count == 1)
		switch(ftype) {
		case 3:		/* short */
		    value = get2(buf+8, endian);
		    break;
		case 4:		/* long */
		    value = get4(buf+8, endian);
		    break;
		case 5:		/* offset to rational */
		    value = get4(buf+8, endian);
		    break;
		}
	    switch(tag) {
	    case 256:		/* ImageWidth */
		iwidth = value;
		break;
	    case 257:		/* ImageLength */
		iheight = value;
		break;
	    case 259:		/* Compression */
		comp = value;
		break;
	    case 262:		/* PhotometricInterpretation */
		inverse ^= (value == 1);
		break;
	    case 266:		/* FillOrder */
		lsbfirst = (value == 2);
		break;
	    case 273:		/* StripOffsets */
		dataoffs = value;
		break;
	    case 274:		/* Orientation */
		switch(value) {
		/* mirroring is not supported */
		case 3:
		    orient = TURN_U;
		    break;
		case 6:
		    orient = TURN_L;
		    break;
		case 8:
		    orient = TURN_U|TURN_L;
		    break;
		}
		break;
	    case 278:		/* RowsPerStrip */
		if (value < iheight) {
		    fprintf(stderr, "%s:%s: can't handle multi-strip tiff images\n",
			    ProgName, name);
		    goto bad;
		}
		break;
	    case 279:		/* StripByteCounts */
		bytecount = value;
		break;
	    case 283:		/* YResolution */
		here = ftell(tf);
		if (fseek(tf, value, SEEK_SET) < 0 ||
		    fread(buf, 8, 1, tf) == 0)
		    goto bad;
		yres = get4(buf, endian) / get4(buf+4, endian);
		if (fseek(tf, here, SEEK_SET) < 0)
		    goto bad;		
		break;
	    case 292:		/* T4Options */
		t4opt = value;
		break;
	    case 293:		/* T6Options */
		/* later */
		break;
	    case 296:		/* ResolutionUnit */
		if (value == 3)
		    yres *= 2.54;
		break;
	    }
	}
	if (comp != 3 && comp != 4) {
	    fprintf(stderr, "%s:%s: this version only handles fax files\n",
		ProgName, name);
	    goto bad;
	}
	pn = notefile(name);
	pn->offset = dataoffs;
	pn->length = bytecount;
	pn->width = iwidth;
	pn->height = iheight;
	pn->inverse = inverse;
	pn->lsbfirst = lsbfirst;
	pn->orient = orient;
	pn->vres = (yres > 150);
	if (comp == 3)
	    pn->expander = (t4opt & 1) ? g32expand : g31expand;
	else
	    pn->expander = g4expand;
	if (fread(buf, 4, 1, tf) == 0)
	    goto bad;
	IFDoff = get4(buf, endian);
    } while (IFDoff);
    fclose(tf);
}
#endif

/* report error and remove bad file from the list */
static void
badfile(struct pagenode *pn)
{
    struct pagenode *p;

    if (errno)
	perror(pn->pathname);
    if (pn == firstpage) {
	if (pn->next == NULL)
	    exit(1);
	firstpage = thispage = firstpage->next;
	firstpage->prev = NULL;
    }
    else
	for (p = firstpage; p; p = p->next)
	    if (p->next == pn) {
		thispage = p;
		p->next = pn->next;
		if (pn->next)
		    pn->next->prev = p;
		break;
	    }
    if (pn) free(pn);
}

/* rearrange input bits into t16bits lsb-first chunks */
static void
normalize(struct pagenode *pn, int revbits, int swapbytes, size_t length)
{
    t32bits *p = (t32bits *) pn->data;

    switch ((revbits<<1)|swapbytes) {
    case 0:
	break;
    case 1:
	for ( ; length; length -= 4) {
	    t32bits t = *p;
	    *p++ = ((t & 0xff00ff00) >> 8) | ((t & 0x00ff00ff) << 8);
	}
	break;
    case 2:
	for ( ; length; length -= 4) {
	    t32bits t = *p;
	    t = ((t & 0xf0f0f0f0) >> 4) | ((t & 0x0f0f0f0f) << 4);
	    t = ((t & 0xcccccccc) >> 2) | ((t & 0x33333333) << 2);
	    *p++ = ((t & 0xaaaaaaaa) >> 1) | ((t & 0x55555555) << 1);
	}
	break;
    case 3:
	for ( ; length; length -= 4) {
	    t32bits t = *p;
	    t = ((t & 0xff00ff00) >> 8) | ((t & 0x00ff00ff) << 8);
	    t = ((t & 0xf0f0f0f0) >> 4) | ((t & 0x0f0f0f0f) << 4);
	    t = ((t & 0xcccccccc) >> 2) | ((t & 0x33333333) << 2);
	    *p++ = ((t & 0xaaaaaaaa) >> 1) | ((t & 0x55555555) << 1);
	}
    }
}


/* get and expand a fax page */
unsigned char *
getpage(struct pagenode *pn)
{
    int fd;
    size_t roundup;
    struct stat sbuf;
    unsigned char *Data;
    union { t16bits s; unsigned char b[2]; } so;
#define ShortOrder	so.b[1]

    so.s = 1;
    if ((fd = open(pn->pathname, O_RDONLY, 0)) < 0) {
	badfile(pn);
	return NULL;
    }

    if (pn->length) {
	/* either we have seen this page before or it is in a tiff file */
	if (lseek(fd, pn->offset, SEEK_SET) < 0) {
	    close(fd);
	    badfile(pn);
	    return NULL;
	}
	roundup = pn->length;
    }
    else {
	if (fstat(fd, &sbuf) != 0) {
	    close(fd);
	    badfile(pn);
	    return NULL;
	}
	pn->length = roundup = sbuf.st_size;
    }

    /* round size to full boundary plus t32bits */
    roundup = (roundup + 7) & ~3;

    Data = (unsigned char *) xmalloc(roundup);
    /* clear the last 2 t32bits, to force the expander to terminate
       even if the file ends in the middle of a fax line  */
    *((t32bits *) Data + roundup/4 - 8) = 0;
    *((t32bits *) Data + roundup/4 - 4) = 0;
    /* we expect to get it in one gulp... */
    if (read(fd, (char *)Data, (unsigned)pn->length) != (int)pn->length) {
	badfile(pn);
	free(Data);
	close(fd);
	return NULL;
    }
    close(fd);

    pn->data = (t16bits *) Data;
    if (pn->offset == 0 && memcmp(Data, FAXMAGIC, sizeof(FAXMAGIC)) == 0) {
	/* handle ghostscript / PC Research fax file */
	if (Data[24] != 1 || Data[25] != 0)
	    printf("%s: only first page of multipage file %s will be shown\n",
		   ProgName, pn->pathname);
	pn->offset = 64;
	pn->length -= 64;
	pn->vres = Data[29];
	pn->data += 32;
	roundup -= 64;
    }

    normalize(pn, !pn->lsbfirst, ShortOrder, roundup);
    if (pn->height == 0)
	pn->height = G3count(pn, pn->expander == g32expand);
    if (pn->height == 0) {
	fprintf(stderr, "%s: no fax found in file %s\n", ProgName,
		pn->pathname);
	errno = 0;
	badfile(pn);
	free(Data);
	return NULL;
    }
    if (verbose)
	printf("%s:\n\twidth = %d\n\theight = %d\n\tresolution = %s\n",
	       pn->name, pn->width, pn->height, pn->vres ?
	       "fine" : "normal");
    return Data;
}
