/*
 * ProFTPD - FTP server daemon
 * Copyright (c) 1997, Public Flood Software
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * House initialization and main program loop
 * $Id: main.c,v 1.15 1997/12/29 20:22:03 flood Exp $
 */

/*
 * History Log:
 *
 * 9/21/97 current: 0.99.0pl6, next: 0.99.0pl7
 *   Removed the -o (--core) option, because some kernels won't
 *   produce a core dump after the euid/egid have changed.  If
 *   anyone knows a way around this, please let me know.
 * 4/28/97 current: 0.99.0pl1, next: 0.99.0pl2
 *   Added checking for <Limit LOGIN> in fork_server(), in order
 *   to disconnect any connections which can never be authorized.
 * 4/24/97 current: 0.99.0pl1, next: 0.99.0pl2
 *   Removed include/proftpd_conf.h; unnecessary header file
 */

#include "conf.h"

#include <signal.h>
#include <sys/resource.h>

#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif

#include "privs.h"

typedef struct _pidrec {
  struct _pidrec *next,*prev;

  pid_t pid;
} pidrec_t;

extern xaset_t *servers;

session_t session;
int master = TRUE;			/* Master daemon in standalone mode */
pid_t mpid = 0;				/* Master pid */
int rehash = 0;

time_t shut = (time_t)0,deny = (time_t)0, disc = (time_t)0;
char shutmsg[81] = "";

xaset_t *children = NULL;

static int (*auth_check)(cmd_rec*) = NULL;
static char sbuf[1024];
static char _ml_numeric[4];
static char **Argv = NULL;
static char *LastArgv = NULL;

static int shutdownp = 0;
static int abort_core = 0;
static RETSIGTYPE sig_disconnect(int);
char *config_filename = CONFIG_FILE_PATH;

void set_proc_title(char *fmt,...)
{
  va_list msg;
  char *p;
  int i,maxlen = (LastArgv - Argv[0])+1;

  va_start(msg,fmt);
  vsnprintf(sbuf,sizeof(sbuf),fmt,msg);
  va_end(msg);

  sbuf[1023] = '\0';
  i = strlen(sbuf);

  strncpy(Argv[0], sbuf, maxlen);
  p = &Argv[0][i];
  while(p < LastArgv)
    *p++ = ' ';

  Argv[1] = NULL;
}
  
void main_set_idle()
{
  time_t now;

  time(&now);

  log_add_run(mpid,&now,session.user,"proftpd: %s - %s: IDLE",
              session.user,session.proc_prefix);
  set_proc_title("proftpd: %s - %s: IDLE",
              session.user,session.proc_prefix);
}

void send_response_raw(const char *fmt, ...)
{
  va_list msg;

  va_start(msg,fmt);
  vsnprintf(sbuf,sizeof(sbuf),fmt,msg);
  va_end(msg);

  sbuf[1023] = '\0';
  io_printf(session.c->outf,"%s\r\n",sbuf);
}

void send_response_async(const char *resp_numeric, const char *fmt, ...)
{
  char buf[1023],*cp = buf;
  va_list msg;
  int maxlen;

  strcpy(cp,resp_numeric);
  cp += strlen(resp_numeric);
  *cp++ = ' ';

  maxlen = sizeof(buf) - strlen(resp_numeric) - 1;

  va_start(msg,fmt);
  vsnprintf(cp,maxlen,fmt,msg);
  va_end(msg);

  buf[1022] = '\0';
  cp = buf + strlen(buf);
  *cp++ = '\r'; *cp++ = '\n'; *cp++ = '\0';

  io_write_async(session.c->outf,buf,strlen(buf),NULL,NULL,NULL);
}

void send_response(const char *resp_numeric, const char *fmt, ...)
{
  va_list msg;

  va_start(msg,fmt);
  vsnprintf(sbuf,sizeof(sbuf),fmt,msg);
  va_end(msg);

  sbuf[1023] = '\0';
  io_printf(session.c->outf,"%s %s\r\n",resp_numeric,sbuf);
}

void send_response_ml_start(const char *resp_numeric, const char *fmt, ...)
{
  va_list msg;

  va_start(msg,fmt);
  vsnprintf(sbuf,sizeof(sbuf),fmt,msg);
  va_end(msg);

  sbuf[1023] = '\0';
  strncpy(_ml_numeric,resp_numeric,3); _ml_numeric[3] = '\0';

  io_printf(session.c->outf,"%s-%s\r\n",_ml_numeric,sbuf);
}

void send_response_ml(const char *fmt, ...)
{
  va_list msg;

  va_start(msg,fmt);
  vsnprintf(sbuf,sizeof(sbuf),fmt,msg);
  va_end(msg);

  sbuf[1023] = '\0';

  io_printf(session.c->outf,"   %s\r\n",sbuf);
}

void send_response_ml_end(const char *fmt, ...)
{
  va_list msg;
 
  va_start(msg,fmt);
  vsnprintf(sbuf,sizeof(sbuf),fmt,msg);
  va_end(msg);

  sbuf[1023] = '\0';

  io_printf(session.c->outf,"%s %s\r\n",_ml_numeric,sbuf);
}

void set_auth_check(int (*ck)(cmd_rec*))
{
  auth_check = ck;
}

void end_login_noexit()
{
  /* Run all the exit handlers */
  run_exit_handlers();

  /* If session.user is set, we have a valid login */
  if(session.user) {
#if (defined(BSD) && (BSD >= 199103))
    sprintf(sbuf,"ftp%ld",(long)getpid());
#else
    sprintf(sbuf,"ftpd%d",(int)getpid());
#endif
    log_wtmp(sbuf,"",
      (session.c && session.c->remote_name ? session.c->remote_name : ""),
      (session.c && session.c->remote_ipaddr ? session.c->remote_ipaddr : NULL));
    log_add_run(mpid,NULL,NULL,NULL);
    log_close_run();
  }
}

/* Finish any cleaning up, mark utmp as closed and exit
 * without flushing buffers
 */

void end_login(int exitcode)
{
  end_login_noexit();
  _exit(exitcode);
}

void main_exit(void *pv, void *lv, void *ev, void *dummy)
{
  int pri = (int)pv;
  char *log = (char*)lv;
  int exitcode = (int)ev;

  log_pri(pri,log);
  end_login(exitcode);
}

void shutdown_exit(void *d1, void *d2, void *d3, void *d4)
{
  char *msg;

  if(check_shutmsg(&shut,&deny,&disc,shutmsg,sizeof(shutmsg)) == 1) {
    char *user;
    time_t now;

    time(&now);
    if(get_param_int(main_server->conf,"authenticated",FALSE) == 1)
      user = get_param_ptr(main_server->conf,"USER",FALSE);
    else
      user = "NONE";

    msg = sreplace(permanent_pool,shutmsg,
                   "%s",pstrdup(permanent_pool,fmt_time(shut)),
                   "%r",pstrdup(permanent_pool,fmt_time(deny)),
                   "%d",pstrdup(permanent_pool,fmt_time(disc)),
		   "%C",(session.cwd[0] ? session.cwd : "(none)"),
		   "%L",main_server->ServerAddress,
		   "%R",(session.c && session.c->remote_name ?
                         session.c->remote_name : "(unknown)"),
		   "%T",pstrdup(permanent_pool,fmt_time(now)),
		   "%U",user,
                   NULL );

    send_response_async(R_421,"FTP server shutting down - %s",msg);

    main_exit((void*)LOG_NOTICE,msg,(void*)0,NULL);
  }

  signal(SIGUSR1,sig_disconnect);
}

static void _dispatch_cmd(cmd_rec *cmd)
{
  char *cp,*resmsg,*argstr;
  cmdtable *c;

  cmd->server = main_server;

  for(cp = cmd->argv[0]; *cp; cp++)
    *cp = toupper(*cp);

  /* debug_print_dispatch(cmd); */

  for(c = m_cmdtable; c->command; c++)
    if(!strcasecmp(c->command,cmd->argv[0])) {
      if(c->group)
        cmd->group = pstrdup(cmd->pool,c->group);

      if(c->requires_auth && auth_check &&
         !auth_check(cmd))
        break;

      if(!c->interrupt_xfer && (session.flags & SF_XFER)) {
        send_response(R_451,"Cannot accept command, transfer is in progress.");
        break;
      }

      cmd->tmp_pool = make_sub_pool(cmd->pool);

      argstr = make_arg_str(cmd->tmp_pool,cmd->argc,cmd->argv);

      if(session.user && (session.flags & SF_XFER) == 0) {
        log_add_run(mpid,NULL,session.user,"proftpd: %s - %s: %s",
                    session.user,session.proc_prefix,
                    make_arg_str(cmd->tmp_pool,cmd->argc,cmd->argv));
        set_proc_title("proftpd: %s - %s: %s",
                       session.user,session.proc_prefix,
                       argstr);
      }

      /* Hack to hide passwords */

      if(!strcasecmp(cmd->argv[0],"PASS"))
        log_debug(DEBUG4,"received: PASS (hidden)");
      else
        log_debug(DEBUG4,"received: %s",argstr);
      
      if((resmsg = (char*)call_module_cmd(c->m,c->handler,cmd)) != NULL)
        send_response_raw("%s",resmsg);

      if(session.user && (session.flags & SF_XFER) == 0)
        main_set_idle();

      destroy_pool(cmd->tmp_pool);
      break;
    }
  
  if(!c->command)
    send_response("500", "%s not understood.", cmd->argv[0]);
}

cmd_rec *make_cmd(pool *p, char *buf)
{
  char *cp = buf, *wrd;
  cmd_rec *newcmd;
  pool *newpool;
  array_header *tarr;

  newpool = make_sub_pool(p);
  newcmd = (cmd_rec*)pcalloc(newpool,sizeof(cmd_rec));
  newcmd->pool = newpool;
  tarr = make_array(newpool,2,sizeof(char**));

  if((wrd = get_word(&cp)) != NULL) {
    *((char**)push_array(tarr)) = pstrdup(newpool,wrd);
    newcmd->argc++;
    newcmd->arg = pstrdup(newpool,cp);
 
    while((wrd = get_word(&cp)) != NULL) {
      *((char**)push_array(tarr)) = pstrdup(newpool,wrd);
      newcmd->argc++;
    }
  }

  *((char**)push_array(tarr)) = NULL;

  newcmd->argv = (char**)tarr->elts;

  return newcmd;
}

static int _idle_timeout(CALLBACK_FRAME)
{
  send_response_async(R_421,"Idle Timeout (%d seconds): closing control connection.", 
                      TimeoutIdle);

  schedule(main_exit,0,(void*)LOG_NOTICE,
           "FTP session idle timeout, disconnected.",
           (void*)0,NULL);

  remove_timer(TIMER_LOGIN,ANY_MODULE);
  remove_timer(TIMER_NOXFER,ANY_MODULE);
  return 0;
}

void cmd_loop(server_rec *server, conn_t *c)
{
  char buf[1024];
  char *cp;
  int i;

  /* Setup the main idle timer */
  if(TimeoutIdle)
    add_timer(TimeoutIdle,TIMER_IDLE,NULL,_idle_timeout);

  if(get_param_int(server->conf,"DeferWelcome",FALSE) == 1)
    send_response("220", "ProFTPD " VERSION " Server ready.");
  else
    send_response("220", "ProFTPD " VERSION " Server (%s) [%s]",
           server->ServerName,server->ServerAddress);

  while(1) {
    if(io_gets(buf,sizeof(buf)-1,session.c->inf) == NULL) {
      if(errno == EINTR)
	continue;		/* Simple interrupted syscall */
      
      /* Otherwise, EOF */
      log_pri(LOG_NOTICE,"FTP session closed.");
      end_login(0);
    }

    /* Data received, reset idle timer */
    if(TimeoutIdle)
      reset_timer(TIMER_IDLE,NULL);

    buf[1023] = '\0';
    i = strlen(buf);

    if(i && (buf[i-1] == '\n' || buf[i-1] == '\n')) {
      buf[i-1] = '\0'; i--;
      if(i && (buf[i-1] == '\n' || buf[i-1] =='\r'))
        buf[i-1] = '\0';
    }

    cp = buf;
    if(*cp == '\r') cp++;

    if(*cp) {
      cmd_rec *cmd;

      cmd = make_cmd(permanent_pool,cp);
      if(cmd) {
        _dispatch_cmd(cmd);
        destroy_pool(cmd->pool);
      }
    }
  }
}

static void _server_conn_cleanup(void *connp)
{
  *((conn_t**)connp) = NULL;
}

void main_rehash(void *d1,void *d2,void *d3,void *d4)
{
  server_rec *s,*snext,*old_main;
  xaset_t *old_servers;

  rehash++;

  old_servers = servers;
  old_main = main_server;

  if(master && mpid) {
    log_pri(LOG_NOTICE,"received SIGHUP -- master server rehashing configuration file.");

    init_config();
    init_conf_stacks();

    PRIVS_ROOT
    if(parse_config_file(config_filename) == -1) {
      PRIVS_RELINQUISH
      log_pri(LOG_ERR,"Fatal: unable to read configuration file '%s'.",
              config_filename);
      end_login(1);
    }
    PRIVS_RELINQUISH
    free_conf_stacks();

    fixup_servers();

    /* Free old configuration completely */

    for(s = (server_rec*)old_servers->xas_list; s; s=snext) {
      snext = s->next;
      destroy_pool(s->pool);
    }

    destroy_pool(old_servers->mempool);

    /* Recreate the listen connection */
    main_server->listen =
      inet_create_connection(main_server->pool,servers,-1,
                             (SocketBindTight ? main_server->ipaddr : NULL),
		   	     main_server->ServerPort);
    
    for(s = main_server->next; s; s=s->next) {
      if(s->ServerPort != main_server->ServerPort || SocketBindTight)
        s->listen = inet_create_connection(
                      s->pool,servers,-1,
                      (SocketBindTight ? s->ipaddr : NULL),
                      s->ServerPort);
      else {
        s->listen = main_server->listen;
        register_cleanup(s->listen->pool,&s->listen,
                         _server_conn_cleanup,
                         _server_conn_cleanup);
      }
    }
  } else
    /* Child process -- cannot rehash, log error */
    log_pri(LOG_ERR,"received SIGHUP, cannot rehash child process");
}

void fork_server(int fd,conn_t *l,int nofork)
{
  server_rec *s,*serv = NULL;
  conn_t *conn;
  pid_t pid;
  sigset_t sigset;
  pool *p;
  int i;
  in_addr_t loopback,loopmask,tmp;

#ifndef DEBUG_NOFORK
  if(!nofork) {
    pidrec_t *cpid;

    /* We block SIGCHLD to prevent a race condition if the child
     * dies before we can record it's pid.  Also block SIGTERM to
     * prevent sig_terminate() from examining the child list
     */

    sigemptyset(&sigset);
    sigaddset(&sigset,SIGTERM);
    sigaddset(&sigset,SIGCHLD);
    sigaddset(&sigset,SIGUSR1);

    sigprocmask(SIG_BLOCK,&sigset,NULL);

    switch((pid = fork())) {
    case 0: /* child */
      master = FALSE;		/* We aren't the master anymore */
      sigprocmask(SIG_UNBLOCK,&sigset,NULL);
      break;
    case -1:
      sigprocmask(SIG_UNBLOCK,&sigset,NULL);
      log_pri(LOG_ERR,"fork(): %s",strerror(errno));
      return;
    default: /* parent */
      /* The parent doesn't need the socket open */
      close(fd);

      if(!children) {
        p = make_sub_pool(permanent_pool);
        children = xaset_create(p,NULL);
      } else
        p = children->mempool;

      cpid = (pidrec_t*)pcalloc(p,sizeof(pidrec_t));
      cpid->pid = pid;
      xaset_insert(children,(xasetmember_t*)cpid);

      /* Unblock the signals now as sig_child() will catch
       * an "immediate" death and remove the pid from the children list
       */
      sigprocmask(SIG_UNBLOCK,&sigset,NULL);
      return;
    }
  }

#ifdef SETPGRP_VOID
  setpgrp();
#else
  setpgrp(0,getpid());
#endif

#endif /* DEBUG_NOFORK */


  /* Child is running here */
  signal(SIGUSR1,sig_disconnect);
  signal(SIGCHLD,SIG_DFL);
  signal(SIGHUP,SIG_IGN);

  /* Inform all the modules that we have forked */

  init_child_modules();

  /* From this point on, syslog stays open */
  /* We close it first so that the logger will pick up our
   * new pid.
   */
  log_closesyslog();
  log_opensyslog();

  /* It's safe to call inet_openrw now (it might block),
   * because the parent is off answering new connections
   */

  conn = inet_openrw(permanent_pool,l,NULL,fd,
                     STDIN_FILENO,STDOUT_FILENO,TRUE);

  if(!conn) {
    log_pri(LOG_ERR,"Fatal: unable to open incoming connection: %s",
                   strerror(errno));
    exit(1);
  }

  /* If a connection comes in on the loopback, assume it's for the
   * main server.
   */

#ifdef HAVE_INET_ATON
  inet_aton(LOOPBACK_NET,&loopback);
  inet_aton(LOOPBACK_MASK,&loopmask);
#else
  loopback.s_addr = inet_addr(LOOPBACK_NET);
  loopmask.s_addr = inet_addr(LOOPBACK_MASK);
#endif

  loopback.s_addr = ntohl(loopback.s_addr);
  loopmask.s_addr = ntohl(loopmask.s_addr);
  tmp.s_addr = ntohl(conn->local_ipaddr->s_addr);

  if((tmp.s_addr & loopmask.s_addr) == loopback.s_addr &&
     main_server->listen == l)
    serv = main_server;
  else {
    /* Determine which server this connection is */
    for(s = main_server; s; s=s->next)
      if(s->listen == l &&
         s->ipaddr->s_addr == conn->local_ipaddr->s_addr) {
            	serv = s;
                break;
         }
  }

  /* If no server is configured to specifically handle the destination
   * address, search for the first server with DefaultServer set.
   */

  if(!serv) {
    for(s = main_server; s; s=s->next)
      if(s->listen == l && 
         get_param_int(s->conf,"DefaultServer",FALSE) == 1) {
        serv = s;
        break;
      }
  }

  /* To conserve memory, free all other servers and associated
   * configurations
   */
  for(s = main_server; s; s=s->next)
    if(s != serv) {
      if(s->listen && s->listen != l) {
	/* If our former listen socket was stdin or stdout (0 or 1),
         * inet_close() will attempt to close it, and in the process
         * close our read/write sockets for this connection.
         */
        if(s->listen->listen_fd == conn->rfd ||
           s->listen->listen_fd == conn->wfd)
          s->listen->listen_fd = -1;
        else
          inet_close(s->pool,s->listen);
      }

      if(s->listen->listen_fd == conn->rfd ||
         s->listen->listen_fd == conn->wfd)
           s->listen->listen_fd = -1;

      xaset_remove(servers,(xasetmember_t*)s);
      destroy_pool(s->pool);
    }

  main_server = serv;
    
  session.pool = permanent_pool;
  session.c = conn;
  session.data_port = conn->remote_port - 1;

  /* Check and see if we are shutdown */
  if(shutdownp) {
    time_t now;

    time(&now);
    if(!deny || deny <= now) {
      char *reason =
          sreplace(permanent_pool,shutmsg,
                   "%s",pstrdup(permanent_pool,fmt_time(shut)),
                   "%r",pstrdup(permanent_pool,fmt_time(deny)),
                   "%d",pstrdup(permanent_pool,fmt_time(disc)),
		   "%C",(session.cwd[0] ? session.cwd : "(none)"),
		   "%L",main_server->ServerAddress,
		   "%R",(session.c && session.c->remote_name ?
                         session.c->remote_name : "(unknown)"),
		   "%T",pstrdup(permanent_pool,fmt_time(now)),
		   "%U","NONE",
                   NULL );

      log_auth(LOG_NOTICE,"connection refused (%s) from %s [%s]",
               reason,session.c->remote_name,
               inet_ntoa(*session.c->remote_ipaddr));

      printf("500 FTP server shut down (%s) -- please try again later.\r\n",
             reason); 
      fflush(stdout);
      exit(0);
    }
  }

  /* If no server is configured to handle the addr the user is
   * connected to, drop them.
   */

  if(!serv) {
    printf("500 Sorry, no server available to handle request on %s.\r\n",
           inet_getname(conn->pool,conn->local_ipaddr));
    fflush(stdout);
    exit(0);
  }

  if(serv->listen) {
    if(serv->listen->listen_fd == conn->rfd ||
        serv->listen->listen_fd == conn->wfd)
          serv->listen->listen_fd = -1;

    destroy_pool(serv->listen->pool);
    serv->listen = NULL;
  }

  /* Check config tree for <Limit LOGIN> directives */
  if(!login_check_limits(serv->conf,TRUE,FALSE,&i)) {
    log_pri(LOG_NOTICE,"Connection from %s [%s] denied.",
            session.c->remote_name,inet_ntoa(*session.c->remote_ipaddr));
    exit(0);
  }

  log_debug(DEBUG4,"connected - local  : %s:%d",
                    inet_ntoa(*session.c->local_ipaddr),
                    session.c->local_port);
  log_debug(DEBUG4,"connected - remote : %s:%d",
  		    inet_ntoa(*session.c->remote_ipaddr),
  		    session.c->remote_port);

  /* Use the ident protocol (RFC1413) to try to get remote ident_user
   */

  session.ident_user = get_ident(session.pool,conn);
  /* xfer_set_data_port(conn->local_ipaddr,conn->local_port-1); */
  cmd_loop(serv,conn);
}

void disc_children()
{
  sigset_t sigset;
  pidrec_t *cp;

  if(disc && disc <= time(NULL) && children) {
    sigemptyset(&sigset);
    sigaddset(&sigset,SIGTERM);
    sigaddset(&sigset,SIGCHLD);
    sigaddset(&sigset,SIGUSR1);

    sigprocmask(SIG_BLOCK,&sigset,NULL);

    PRIVS_ROOT
    for(cp = (pidrec_t*)children->xas_list; cp; cp=cp->next)
      kill(cp->pid,SIGUSR1);
    PRIVS_RELINQUISH

    sigprocmask(SIG_UNBLOCK,&sigset,NULL);
  }
}

void server_loop()
{
  fd_set rfd;
  server_rec *s;
  int fd;
  int i,err_count = 0;
  time_t last_error;
  struct timeval tv;

  set_proc_title("proftpd (accepting connections)");

  time(&last_error);

  while(1) {
    run_schedule();

    FD_ZERO(&rfd);
    for(s = main_server; s; s=s->next) {
      if(s->listen->mode == CM_NONE)
        inet_listen(s->listen->pool,s->listen,tcpBackLog);
 
      if(s->listen->mode == CM_ACCEPT)
        inet_resetlisten(s->listen->pool,s->listen);
      if(s->listen->mode == CM_LISTEN)
        FD_SET(s->listen->listen_fd,&rfd);
    }

    /* Check for ftp shutdown message file */
    switch(check_shutmsg(&shut,&deny,&disc,shutmsg,sizeof(shutmsg))) {
    case 1: if(!shutdownp) disc_children(); shutdownp = 1; break;
    case 0: shutdownp = 0; deny = disc = (time_t)0; break;
    }

    if(shutdownp) {
      tv.tv_usec = 0L;
      tv.tv_sec = 5L;
    } else {
      tv.tv_usec = 0L;
      tv.tv_sec = 30L;
    }

    i = select(NFDBITS,&rfd,NULL,NULL,&tv);

    if(i == -1) {
      time_t this_error;

      if(errno == EINTR)
        continue;

      time(&this_error);
      if((this_error - last_error) <= 5 && err_count++ > 10) {
        log_pri(LOG_ERR,"Fatal: select() failing repeatedly, shutting down.");
        exit(1);
      } else if((this_error - last_error) > 5) {
        last_error = this_error;
        err_count = 0;
      }

      log_pri(LOG_NOTICE,"select() failed in server_loop(): %s",
              strerror(errno));
    }

    if(i == 0)
      continue;

    /* fork off servers to handle each connection
     * our job is to get back to answering connections asap,
     * so leave the work of determining which server the connection
     * is for to our child.
     */
    for(s = main_server; s; s=s->next)
      if(FD_ISSET(s->listen->listen_fd,&rfd) &&
         s->listen->mode == CM_LISTEN) {
        /* accept the connection without blocking, the
         * top loop will reset all connections in accept
         * mode to listen mode.
         */
        fd = inet_accept_nowait(s->listen->pool,s->listen);
        if(fd != -1)
          fork_server(fd,s->listen,FALSE);
      }
  }
}

/* sig_rehash occurs in the master daemon when manually "kill -HUP"
 * in order to re-read configuration files, and is sent to all
 * children by the master.
 */

static RETSIGTYPE sig_rehash(int sig)
{
  schedule(main_rehash,0,NULL,NULL,NULL,NULL);

  signal(SIGHUP,sig_rehash);
}

/* sig_disconnect is called in children when the parent daemon
 * detects that shutmsg has been created and ftp sessions should
 * be destroyed.  If a file transfer is underway, the process simply
 * dies, otherwise a function is scheduled to attempt to display
 * the shutdown reason.
 */

static RETSIGTYPE sig_disconnect(int sig)
{
  if((session.flags & SF_ANON) || (session.flags & SF_XFER))
    schedule(main_exit,0,(void*)LOG_NOTICE,
             "Parent process requested shutdown",
             (void*)0,NULL);
  else
    schedule(shutdown_exit,0,NULL,NULL,NULL,NULL);

  signal(SIGUSR1,SIG_IGN);
}

static RETSIGTYPE sig_child(int sig)
{
  sigset_t sigset;
  pid_t cpid;
  pidrec_t *cp,*cpnext;

  sigemptyset(&sigset);
  sigaddset(&sigset,SIGTERM);

  block_alarms();
  sigprocmask(SIG_BLOCK,&sigset,NULL);

  /* block SIGTERM in here, so we don't create screw with the
   * child list while modifying it.
   */

  while((cpid = waitpid(-1,NULL,WNOHANG)) > 0) {
    if(children) {
      for(cp = (pidrec_t*)children->xas_list; cp; cp=cpnext) {
        cpnext = cp->next;
        if(cp->pid == cpid)
          xaset_remove(children,(xasetmember_t*)cp);
      }
      /* Don't need the pool anymore */
      if(!children->xas_list) {
        destroy_pool(children->mempool);
        children = NULL;
      }
    }
  }

  sigprocmask(SIG_UNBLOCK,&sigset,NULL);
  unblock_alarms();
  signal(SIGCHLD,sig_child);
}

static char *_prepare_core()
{
  static char dir[256];

#if (defined(BSD) && (BSD >= 199103))
  sprintf(dir,"%s/proftpd-core-%ld",CORE_PATH,(long)getpid());
#else
  sprintf(dir,"%s/proftpd-core-%d",CORE_PATH,(int)getpid());
#endif

  if(mkdir(dir,0700) != -1)
    chdir(dir);

  return dir;
}

static RETSIGTYPE sig_abort(int sig)
{
  if(abort_core)
    log_pri(LOG_NOTICE,"ProFTPD received SIGABRT signal, generating core file in %s",_prepare_core());
  else
    log_pri(LOG_NOTICE,"ProFTPD received SIGABRT signal, no core dump.");
  
  signal(SIGABRT,SIG_DFL);
  end_login_noexit();
  abort();
}  

static void _internal_abort()
{
  if(abort_core) {
    log_pri(LOG_NOTICE,"core file dumped to %s",_prepare_core());

    signal(SIGABRT,SIG_DFL);
    end_login_noexit();
    abort();
  }
}

static RETSIGTYPE sig_terminate(int sig)
{
  pidrec_t *pid;

  if(sig == SIGTERM) {
    /* Don't log if we are a child that has been terminated */
    if(master) {
      /* Send a SIGKILL to all our children */
      if(children) {
        PRIVS_ROOT
        for(pid = (pidrec_t*)children->xas_list; pid; pid=pid->next)
          kill(pid->pid,SIGTERM);
        PRIVS_RELINQUISH
      }

      log_pri(LOG_NOTICE,"ProFTPD killed (signal %d)",sig);
    }
  } else
    log_pri(LOG_ERR,"ProFTPD terminating (signal %d)",sig);

  if(master && mpid == getpid()) {
    PRIVS_ROOT
    log_close_run();
    log_rm_run();
    PRIVS_RELINQUISH
  }

  _internal_abort();  
  end_login(1);
}

static void install_signal_handlers()
{
  sigset_t sigset;

  /* Should the master server (only applicable in standalone mode)
   * kill off children if we receive a signal that causes termination?
   * hmmmm... Maybe this needs to be rethought, but I've done it in
   * such a way as to only kill off our children if we receive a SIGTERM,
   * meaning that the admin wants us dead (and prolly our kids too).
   */

  /* The sub-pool for the child list is created the first time we fork
   * off a child.  To conserve memory, the pool and list is destroyed
   * when our last child dies (to prevent the list from eating more and
   * more memory on long uptimes)
   */

  sigemptyset(&sigset);
  sigaddset(&sigset,SIGCHLD);
  sigaddset(&sigset,SIGINT);
  sigaddset(&sigset,SIGQUIT);
  sigaddset(&sigset,SIGILL);
  sigaddset(&sigset,SIGABRT);
  sigaddset(&sigset,SIGFPE);
  sigaddset(&sigset,SIGSEGV);
  sigaddset(&sigset,SIGALRM);
  sigaddset(&sigset,SIGTERM);
#ifdef SIGSTKFLT
  sigaddset(&sigset,SIGSTKFLT);
#endif
  sigaddset(&sigset,SIGIO);
#ifdef SIGBUS
  sigaddset(&sigset,SIGBUS);
#endif
  sigaddset(&sigset,SIGHUP);
  
  signal(SIGCHLD,sig_child);
  signal(SIGHUP,sig_rehash);

#ifndef DEBUG_NOSIG
  signal(SIGINT,sig_terminate);
  signal(SIGQUIT,sig_terminate);
  signal(SIGILL,sig_terminate);
  signal(SIGABRT,sig_abort);
  signal(SIGFPE,sig_terminate);
  signal(SIGSEGV,sig_terminate);
  signal(SIGTERM,sig_terminate);
#ifdef SIGSTKFLT
  signal(SIGSTKFLT,sig_terminate);
#endif /* SIGSTKFLT */
  signal(SIGIO,sig_terminate);
#ifdef SIGBUS
  signal(SIGBUS,sig_terminate);
#endif /* SIGBUS */
#endif /* DEBUG_NOSIG */

  /* In case our parent left signals blocked (as happens under some
   * poor inetd implementations)
   */
  sigprocmask(SIG_UNBLOCK,&sigset,NULL);
}

static void set_rlimits()
{
  struct rlimit rlim;

  if(getrlimit(RLIMIT_CORE,&rlim) == -1)
    log_pri(LOG_ERR,"getrlimit(): %s",strerror(errno));
  else {
    if(abort_core)
      rlim.rlim_cur = rlim.rlim_max = RLIM_INFINITY;
    else
      rlim.rlim_cur = rlim.rlim_max = 0;

    PRIVS_ROOT
    if(setrlimit(RLIMIT_CORE,&rlim) == -1) {
      PRIVS_RELINQUISH
      log_pri(LOG_ERR,"setrlimit(): %s",strerror(errno));
      return;
    }
    PRIVS_RELINQUISH
  }
}

void start_daemon()
{
#ifndef HAVE_SETSID
  int ttyfd;
#endif

  /* Fork off and have parent exit */
  switch(fork()) {
    case -1: perror("fork"); exit(1);
    case 0: break;
    default: exit(0);
  }

#ifdef HAVE_SETSID
  /* setsid() is the preferred way to disassociate from the 
   * controlling terminal
   */
  setsid();
#else
  /* Open /dev/tty to access our controlling tty (if any) */
  if( (ttyfd = open("/dev/tty",O_RDWR)) != -1)
  {
    if(ioctl(ttyfd,TIOCNOTTY,NULL) == -1) {
      perror("ioctl"); exit(1);
    }

    close(ttyfd);
  }
#endif /* HAVE_SETSID */

  /* Close the three big boys */
  close(fileno(stdin));
  close(fileno(stdout));
  close(fileno(stderr));

  /* Portable way to prevent re-acquiring a tty in the future */
#ifdef SETPGRP_VOID
  setpgrp();
#else
  setpgrp(0,getpid());
#endif

  chdir("/");
}

void inetd_main()
{
  server_rec *s;

  main_server->listen = 
    inet_create_connection(main_server->pool,servers,STDIN_FILENO,
                           NULL,INPORT_ANY);

  /* Fill in all the important connection info */
  if(inet_get_conn_info(main_server->listen,STDIN_FILENO) == -1) {
    log_pri(LOG_ERR,"Fatal: %s",strerror(errno));
    exit(1);
  }

  /* Now attach the faked connection to all virtual servers */
  for(s = main_server->next; s; s=s->next) {
      /* Because this server is sharing the connection with the
       * main server, we need a cleanup handler to remove
       * the server's reference when the original connection's
       * pool is destroyed.
       */
      s->listen = main_server->listen;
      register_cleanup(s->listen->pool,&s->listen,
                       _server_conn_cleanup,
                       _server_conn_cleanup);
  }

  /* Check our shutdown status */
  if(check_shutmsg(&shut,&deny,&disc,shutmsg,sizeof(shutmsg)) == 1)
    shutdownp = 1;

  /* Finally, call right into fork_server() to start servicing the
   * connection immediately
   */
  fork_server(STDIN_FILENO,main_server->listen,TRUE);
}

void standalone_main(int nodaemon)
{
  server_rec *s;

  if(nodaemon) {
    log_stderr(TRUE);
    close(fileno(stdin));
    close(fileno(stdout));
  }
  else {
    log_stderr(FALSE);
    start_daemon();
  }

  mpid = getpid();

  PRIVS_ROOT
  log_open_run(mpid,TRUE);
  log_close_run();
  PRIVS_RELINQUISH

  main_server->listen =
    inet_create_connection(main_server->pool,servers,-1,
                           (SocketBindTight ? main_server->ipaddr : NULL),
                           main_server->ServerPort);

  for(s = main_server->next; s; s=s->next)
    if(s->ServerPort != main_server->ServerPort || SocketBindTight)
      s->listen = inet_create_connection(
                    s->pool,servers,-1,
		    (SocketBindTight ? s->ipaddr : NULL),
                    s->ServerPort);
    else {
      /* Because this server is sharing the connection with the
       * main server, we need a cleanup handler to remove
       * the server's reference when the original connection's
       * pool is destroyed.
       */
      s->listen = main_server->listen;
      register_cleanup(s->listen->pool,&s->listen,
                       _server_conn_cleanup,
                       _server_conn_cleanup);
    }

  server_loop();
}

extern char *optarg;
extern int optind,opterr,optopt;

struct option opts[] = {
  { "nodaemon",	0, NULL, 'n' },
  { "debug",	1, NULL, 'd' },
  { "config",	1, NULL, 'c' },
  { "version",  0, NULL, 'v' },
/*
  { "core",     0, NULL, 'o' },
*/
  { "help",	0, NULL, 'h' },
  { NULL,	0, NULL,  0  }
};

struct option_help {
  char *long_opt,*short_opt,*desc;
} opts_help[] = {
  { "--help","-h","display proftpd usage"},
  { "--nodaemon","-n","disable daemon mode (all output goes to tty, instead of syslog)" },
  { "--debug","-d [level]","set debugging level (0-5, 5 == most debugging)" },
  { "--config","-c [config-file]","specify alternate configuration file" },
/*
  { "--core","-o","enable core dump for profiling/debugging on serious errors"},
*/
  { "--version","-v","print version number and exit" },
  { NULL,NULL,NULL }
};


void show_usage(int exit_code)
{
  struct option_help *h;

  printf("usage: proftpd [options]\n");
  for(h = opts_help; h->short_opt; h++) {
    printf("  %s,%s\n",h->long_opt,h->short_opt);
    printf("    %s\n",h->desc);
  }
  exit(exit_code);
}

int main(int argc, char **argv, char **envp)
{
  int daemon_uid,daemon_gid,socketp;
  int _umask = 0,nodaemon = 0,c;
  struct sockaddr peer;


  bzero(&session,sizeof(session));

  /* Save argument/environment globals for use by set_proc_title */

  Argv = argv;
  while(*envp)
    envp++;

  LastArgv = envp[-1] + strlen(envp[-1]);

  /* getpeername() fails if the fd isn't a socket */
  socketp = sizeof(peer);
  if(getpeername(fileno(stdin),&peer,&socketp) != -1) {
    log_stderr(FALSE);
    socketp = TRUE;
  } else
    socketp = FALSE;

  /* Open the syslog */
  log_opensyslog();

  /* Command line options supported:
   * -n,--nodaemon	standalone server doesn't background itself,
   *                    all logging dumped to stderr
   *
   * -d n,--debug n	set debug level
   *
   * -c, --config path  set the configuration path
   *
   * -v, --version      report version number
   */

  opterr = 0;
  while((c = getopt_long(argc,argv,"nd:c:hv",opts,NULL)) != -1) {
    switch(c) {
    case 'n': 
      nodaemon++; break;
    case 'd': 
      if(!optarg) {
        log_pri(LOG_ERR,"Fatal: -d requires debugging level argument.");
        exit(1);
      }
      log_setdebuglevel(atoi(optarg));
      break;
    case 'c':
      if(!optarg) {
        log_pri(LOG_ERR,"Fatal: -c requires configuration path argument.");
        exit(1);
      }
      config_filename = strdup(optarg);
      break;
    /*
    case 'o':
      abort_core++;
      break;
    */
    case 'v':
      log_pri(LOG_NOTICE,"ProFTPD Version " VERSION);
      exit(0);
    case 'h':
      show_usage(0);
    case '?':
      log_pri(LOG_ERR,"Unknown option: %c",(char)optopt);
      show_usage(1);
    }
  }
 
  /* Initialize sub-systems */
  init_log();
  init_alloc();
  init_inet();
  init_io();
  init_config();
  init_modules();

  init_conf_stacks();
  if(parse_config_file(config_filename) == -1) {
    log_pri(LOG_ERR,"Fatal: unable to read configuration file '%s'.",
            config_filename);
    exit(1);
  }

  free_conf_stacks();

  fixup_servers();

  /* After configuration is complete, make sure that passwd, group
   * aren't held open (unnecessary fds for master daemon)
   */

  endpwent();
  endgrent();

  /* Security */
  daemon_uid = get_param_int(main_server->conf,"User",FALSE);
  if(daemon_uid == -1)
    daemon_uid = 0;
  daemon_gid = get_param_int(main_server->conf,"Group",FALSE);
  if(daemon_gid == -1)
    daemon_gid = 0;

  if(daemon_uid)
    initgroups((const char*)get_param_ptr(main_server->conf,"UserName",
                  FALSE),daemon_gid);
  
   if((_umask = get_param_int(main_server->conf,"Umask",FALSE)) == -1)
    _umask = 0022;

  umask(_umask);

  log_closesyslog();

  /* Give up root and save our uid/gid for later use (if supported)
   * If we aren't currently root, PRIVS_SETUP will get rid of setuid
   * granted root and prevent further uid switching from being attempted.
   */

  PRIVS_SETUP(daemon_uid,daemon_gid)

  /* Install a signal handlers/abort handler */
  install_signal_handlers();
  set_rlimits();

  switch(ServerType) {
  case SERVER_STANDALONE: standalone_main(nodaemon);
  case SERVER_INETD:      inetd_main();
  }

  return 0;
}
