/*
 * ProFTPD - FTP server daemon
 * Copyright (c) 1997, Public Flood Software
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * ProFTPD logging support
 * $Id: log.c,v 1.11 1997/12/29 20:22:02 flood Exp $
 */

/* History Log:
 *
 * 4/24/97 0.99.0pl1
 *   Added log_debug() and log_setdebuglevel() in order to facilitate
 *   altering the amount of debugging info printed or syslogged.
 *   Also added a command line argument (-d,--debug) to alter the
 *   debug level at runtime.  See main.c.
 */

#include "conf.h"

#include <signal.h>

static int syslog_open = FALSE;
static int logstderr = TRUE;
static int debug_level = DEBUG0;	/* Default is no debug logging */
static int facility = LOG_DAEMON;
static int runfd = -1;
static char *runfn = NULL;
static int xferfd = -1;

char *fmt_time(time_t t)
{
  static char buf[30];
  static char *mons[] =
  { "Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec" };
  static char *days[] =
  { "Sun","Mon","Tue","Wed","Thu","Fri","Sat" };
  struct tm *tr;

  if((tr = localtime(&t)) != NULL) {
    sprintf(buf,"%s %s %2d %02d:%02d:%02d %d",
            days[tr->tm_wday],
            mons[tr->tm_mon],
            tr->tm_mday,
            tr->tm_hour,
            tr->tm_min,
            tr->tm_sec,
            tr->tm_year + 1900);
  } else
    buf[0] = '\0';
  
  return buf;
}

void log_close_xfer()
{
  if(xferfd != -1)
    close(xferfd);
  xferfd = -1;
}

int log_open_xfer(const char *fn)
{
  if(xferfd == -1)
    xferfd = open(fn,O_WRONLY|O_APPEND|O_CREAT,0644);

  return xferfd;
}

int log_xfer(int xfertime,char *remhost,unsigned long fsize,
             char *fname,char xfertype,char direction,
             char access,char *user)
{
  char buf[1024];

  sprintf(buf,"%s %d %s %lu %s %c _ %c %c %s ftp 0 *\n",
          fmt_time(time(NULL)),xfertime,remhost,fsize,
          fname,xfertype,direction,access,user);

  return(write(xferfd,buf,strlen(buf)));
}

void log_rm_run()
{
  if(runfd > -1)
    close(runfd);
  if(runfn)
    unlink(runfn);
}

int log_close_run()
{
  if(runfd == -1)
    return 0;

  close(runfd);
  runfd = -1;
  return 0;
}

int log_open_run(pid_t mpid, int trunc)
{
  char fname[256];

  if(runfd > -1)
    return 0;

  if(!mpid)
    sprintf(fname,"%s/proftpd-inetd",RUN_DIR);
  else
    sprintf(fname,"%s/proftpd-%d",RUN_DIR,(int)mpid);

  runfn = pstrdup(permanent_pool,fname);
  if((runfd = open(runfn,O_RDWR|O_CREAT|(trunc ? O_TRUNC : 0),
                   0644)) == -1)
    return -1;

  return runfd;
}
 
static int _pid_exists(pid_t pid)
{
#ifdef LINUX
  char procfn[20];
  struct stat sbuf;
#endif
  int res;

  res = kill(pid,SIGCONT);
#ifdef LINUX
  sprintf(procfn,"/proc/%d",pid);    
  if( (res == -1 && errno == EPERM) || !res ||
    stat(procfn,&sbuf) != -1)
#else
  if( (res == -1 && errno == EPERM) || !res )
#endif
    return 1;
  return 0;
}

static int _read_run(int fd, logrun_t *ent)
{
  while(read(fd,(char*)ent,sizeof(logrun_t)) == sizeof(logrun_t))
    if(ent->pid) {
      /* Try to determine if the process still exists */

      return _pid_exists(ent->pid);
    }

  return -1;
}

logrun_t *log_read_run(pid_t *mpid)
{
  static DIR *dir = NULL;
  static struct dirent *dent = NULL;
  static int fd = -1;
  static logrun_t ent;
  char *cp,buf[256];

  errno = 0;
  if(!dir) {
    dir = opendir(RUN_DIR);
    if(!dir)
      return NULL;
  }

  while(fd != -1) {
    switch(_read_run(fd,&ent)) {
    case 1:
      errno = 0;
      return &ent;
    case -1:
      close(fd); fd = -1; break;
    }
  }

  while((dent = readdir(dir)) != NULL)
    if(strncmp(dent->d_name,"proftpd",7) == 0) {
      cp = rindex(dent->d_name,'-');
      if(cp) {
        cp++;
        if(mpid)
          *mpid = (pid_t)atoi(cp);
        sprintf(buf,"%s/%s",RUN_DIR,dent->d_name);
        fd = open(buf,O_RDONLY,0644);

        if(fd != -1) {

          while(fd != -1) {
            switch(_read_run(fd,&ent)) {
            case 1: errno = 0; return &ent;
            case -1: close(fd); fd = -1; errno = 0;
            }
          }
        } else
          return NULL;
      }
    }

  closedir(dir);
  dir = NULL;

  return NULL;
}

/* log_add_run() logs the current process and connection information to
 * the RUN_DIR/proftpd-[master_daemon_pid] file.  If an existing record
 * for the current pid is found, it is overwritten.  Passing user ==
 * NULL clears the entry.
 */

int log_add_run(pid_t mpid, time_t *idle_since, char *user, char *op, ...)
{
  logrun_t ent,fent;
  int res = 0,c = 0,first = -1;
  va_list msg;
  char buf[1500] = "";

#ifndef HAVE_FLOCK
  struct flock arg;
#endif

  if(op) {
    va_start(msg,op);
    vsnprintf(buf,sizeof(buf),op,msg);
    va_end(msg);
    buf[sizeof(buf)-1] = '\0';
  }

  if(runfd == -1)
    log_open_run(mpid,FALSE);

  if(runfd == -1)
    return -1;

  bzero(&ent,sizeof(ent));
  ent.pid = getpid();
  ent.uid = geteuid();
  ent.gid = getegid();
  if(idle_since)
    ent.idle_since = *idle_since;
  if(user)
    strncpy(ent.user,user,sizeof(ent.user)-1);
  if(buf[0])
    strncpy(ent.op,buf,sizeof(ent.op)-1);

#ifdef HAVE_FLOCK
  flock(runfd,LOCK_EX);
#else
  arg.l_type = F_WRLCK; arg.l_whence = arg.l_start = arg.l_len = 0;
  fcntl(runfd, F_SETLKW, &arg);
#endif

  if(lseek(runfd,0,SEEK_SET) != -1) {
    while(read(runfd,(char*)&fent,sizeof(fent)) == sizeof(fent) &&
          fent.pid != ent.pid) {
      if((!fent.pid || !_pid_exists(fent.pid)) && first == -1)
        first = c;
      c += sizeof(fent);
    }

    if(fent.pid == ent.pid) {
      first = -1;
      lseek(runfd,c,SEEK_SET);
    } else
      lseek(runfd,0,SEEK_END);
  }

  if(!user) {
    if(fent.pid == ent.pid) {
      bzero(&ent,sizeof(ent));
      res = write(runfd,(char*)&ent,sizeof(ent));
    }
  } else {
    if(first != -1)
      lseek(runfd,first,SEEK_SET);
    res = write(runfd,(char*)&ent,sizeof(ent));
  }

  /* 11/16/97 - fsync() causes the kernel to flush related file buffers to
   * disk, not necessary here.
   */

  /* fsync(runfd); */

#ifdef HAVE_FLOCK
  flock(runfd,LOCK_UN);
#else
  arg.l_type = F_UNLCK; arg.l_whence = arg.l_start = arg.l_len = 0;
  fcntl(runfd, F_SETLKW, &arg);
#endif

  return res;
}

/* This next function logs an entry to wtmp, it MUST be called as
 * root BEFORE a chroot occurs.
 * Note: This has some portability ifdefs in it.  They *should* work,
 * but I haven't been able to test them.
 */

int log_wtmp(char *line, char *name, char *host, in_addr_t *ip)
{
  struct stat buf;
  struct utmp ut;
  int res = 0;
  static int fd = -1;

#ifdef SVR4
#if !(defined(LINUX) || defined(__hpux) || defined (_AIX))
  /* This "auxilliary" utmp doesn't exist under linux. */
  struct utmpx utx;
  static int fdx = -1;

  if(fdx < 0 && (fdx = open(WTMPX_FILE, O_WRONLY | O_APPEND, 0)) < 0) {
    log_pri(LOG_WARNING,"wtmpx %s: %s",WTMPX_FILE,strerror(errno));
    return -1;
  }

  if(fstat(fdx,&buf) == 0) {
    memset(&utx,0,sizeof(utx));
    strncpy(utx.ut_user,name,sizeof(utx.ut_user));
    strncpy(utx.ut_id,"ftp",sizeof(utx.ut_user));
    strncpy(utx.ut_line,line,sizeof(utx.ut_line));
    strncpy(utx.ut_host,host,sizeof(utx.ut_host));
    utx.ut_syslen = strlen(utx.ut_host)+1;
    utx.ut_pid = getpid();
    time(&utx.ut_tv.tv_sec);
    if(*name)
      utx.ut_type = USER_PROCESS;
    else
      utx.ut_type = DEAD_PROCESS;
    utx.ut_exit.e_termination = 0;
    utx.ut_exit.e_exit = 0;
    if(write(fdx,(char*)&utx,sizeof(utx)) != sizeof(utx))
      ftruncate(fdx, buf.st_size);
  } else {
    log_debug(DEBUG0,"%s fstat(): %s",WTMPX_FILE,strerror(errno));
    res = -1;
  }

#endif
#endif /* SVR4 */

  if(fd < 0 && (fd = open(WTMP_FILE,O_WRONLY|O_APPEND,0)) < 0) {
    log_pri(LOG_WARNING,"wtmp %s: %s",WTMP_FILE,strerror(errno));
    return -1;
  }
 
  if(fstat(fd,&buf) == 0) {
    memset(&ut,0,sizeof(ut));
#ifdef HAVE_UTMAXTYPE
#ifdef LINUX
    if(ip)
      memcpy(&ut.ut_addr,ip,sizeof(ut.ut_addr));
#else
    strncpy(ut.ut_id,"ftp",sizeof(ut.ut_id));
    ut.ut_exit.e_termination = 0;
    ut.ut_exit.e_exit = 0;
#endif
    strncpy(ut.ut_line,line,sizeof(ut.ut_line));
    if(name && *name)
      strncpy(ut.ut_user,name,sizeof(ut.ut_user));
    ut.ut_pid = getpid();
    if(name && *name)
      ut.ut_type = USER_PROCESS;
    else
      ut.ut_type = DEAD_PROCESS;
#else  /* !HAVE_UTMAXTYPE */
    strncpy(ut.ut_line,line,sizeof(ut.ut_line));
    if(name && *name)
      strncpy(ut.ut_name,name,sizeof(ut.ut_name));
#endif /* HAVE_UTMAXTYPE */

#ifdef HAVE_UT_UT_HOST
    if(host && *host)
      strncpy(ut.ut_host,host,sizeof(ut.ut_host));
#endif /* HAVE_UT_UT_HOST */

    time(&ut.ut_time);
    if(write(fd,(char*)&ut,sizeof(ut)) != sizeof(ut))
      ftruncate(fd,buf.st_size);
  } else {
    log_debug(DEBUG0,"%s fstat(): %s",WTMP_FILE,strerror(errno));
    res = -1;
  }

  return res;
}

void log_opensyslog()
{
  openlog("proftpd",LOG_NDELAY|LOG_PID,facility);
  syslog_open = TRUE;
}

void log_closesyslog()
{
  closelog();
  syslog_open = FALSE;
}

void log(int priority, int f, char *s)
{
  if(logstderr) {
    fprintf(stderr,"%s\n",s);
    return;
  }

  if(f != facility || !syslog_open)
    openlog("proftpd",LOG_NDELAY|LOG_PID,f);

  syslog(priority,"%s\n",s);

  if(!syslog_open)
    closelog();
  else if(f != facility)
    openlog("proftpd",LOG_NDELAY|LOG_PID,facility);
}

void log_pri(int priority,char *fmt,...)
{
  char buf[1024];
  va_list msg;

  va_start(msg,fmt);
  vsnprintf(buf,sizeof(buf),fmt,msg);
  va_end(msg);

  buf[1023] = '\0';

  log(priority,facility,buf);
}

/* Like log_pri(), but sends the log entry in the LOG_AUTHPRIV
 * facility (presumable it doesn't need to be seen by everyone
 */

void log_auth(int priority, char *fmt, ...)
{
  char buf[1024];
  va_list msg;

  va_start(msg,fmt);
  vsnprintf(buf,sizeof(buf),fmt,msg);
  va_end(msg);

  buf[1023] = '\0';

  log(priority,LOG_AUTHPRIV,buf);
}

/* Disable logging to stderr, should be done right before forking
 * or disassociation from controlling tty.  After disabling stderr
 * logging, all messages go to syslog.
 */

void log_stderr(int bool)
{
  logstderr = bool;
}

/* Set the debug logging level, see log.h for constants.  Higher
 * numbers mean print more, DEBUG0 (0) == print no debugging log
 * (default)
 */

int log_setdebuglevel(int level)
{
  int old_level = debug_level;
  debug_level = level;
  return old_level;
}

void log_debug(int level,char *str,...)
{
  char buf[1024];
  va_list msg;

  if(debug_level < level)
    return;

  va_start(msg,str);
  vsnprintf(buf,sizeof(buf),str,msg);
  va_end(msg);

  buf[1023] = '\0';

  log(LOG_DEBUG,facility,buf);
}

void init_log()
{
}
