/* Copyright Abandoned 1996 TCX DataKonsult AB & Monty Program KB & Detron HB
   This file is public domain and comes with NO WARRANTY of any kind */

/* read and write locks for posix & solaris threads. All tread must acquire
** all locks it neads through thr_multi_lock() to avoid dead-locks.
** A lock consists of a master lock (THR_LOCK), and lock instances
** (THR_LOCK_DATA).
** Any thread can have any number of lock instances (read and write:s) on
** any lock. All lock instances must be freed.
** Write locks are prefered over read-locks. Locks are sheduled through a FIFO.
*/

#if !defined(MAIN) && !defined(DBUG_OFF) && !defined(EXTRA_DEBUG)
#define DBUG_OFF
#endif

#include "mysys_priv.h"
#include <m_string.h>
#include <thr_lock.h>
#include <errno.h>


static LIST *thread_list;			/* List of threads in use */
static int thr_lock_inited=0;
static pthread_key(pthread_cond_t,THR_suspend);

#ifndef __WIN32__
static void free_cond(pthread_cond_t* cond)
{
  if (cond)
  {
#if !defined(__bsdi__) && !defined(HAVE_mit_thread) /* bsdi dumps core here */
    pthread_cond_destroy(cond);
#endif
    free((char*) cond);
  }
}

static pthread_cond_t *get_cond(void)
{
  pthread_cond_t *cond;
  if (!thr_lock_inited)
  {
    thr_lock_inited=1;
    pthread_key_create(&THR_suspend,(void (*)(void *)) free_cond);
  }
  if (!(cond=my_pthread_getspecific(pthread_cond_t*, THR_suspend)))
  {
    if ((cond=(pthread_cond_t*) malloc(sizeof(*cond))))
    {
      pthread_cond_init( cond, NULL );
      pthread_setspecific(THR_suspend,(gptr) cond);
    }
    DBUG_PRINT("info",("Creating cond: %lx",cond));
  }
  return cond;
}

/*
** Alloc memory so we don't have to check this later
*/

my_bool init_thr_lock()
{
  return !get_cond();
}

#else /* __WIN32__ */

__declspec(thread) pthread_cond_t THR_suspend;
#define get_cond() &THR_suspend

my_bool init_thr_lock()
{
  return 0;
}

#endif /* __WIN32__ */


	/* Initialize a lock */

void thr_lock_init(THR_LOCK *lock)
{
  DBUG_ENTER("thr_lock_init");
  bzero(lock,sizeof(*lock));
  VOID(pthread_mutex_init(&lock->mutex,NULL));
  lock->read.last= &lock->read.data;
  lock->read_wait.last= &lock->read_wait.data;
  lock->write_wait.last= &lock->write_wait.data;
  lock->write.last= &lock->write.data;

  pthread_mutex_lock(&THR_LOCK_lock);		/* Add to locks in use */
  lock->list.data=(void*) lock;
  thread_list=list_add(thread_list,&lock->list);
  pthread_mutex_unlock(&THR_LOCK_lock);
  DBUG_VOID_RETURN;
}


void thr_lock_delete(THR_LOCK *lock)
{
  DBUG_ENTER("thr_lock_delete");
  VOID(pthread_mutex_destroy(&lock->mutex));
  pthread_mutex_lock(&THR_LOCK_lock);
  thread_list=list_delete(thread_list,&lock->list);
  pthread_mutex_unlock(&THR_LOCK_lock);
  DBUG_VOID_RETURN;
}

	/* Initialize a lock instance */

void thr_lock_data_init(THR_LOCK *lock,THR_LOCK_DATA *data)
{
  data->lock=lock;
  data->thread=pthread_self();			/* Only needed for debug */
}


static inline my_bool have_old_read_lock(THR_LOCK_DATA *data,pthread_t thread)
{
  for ( ; data ; data=data->next)
  {
    if ((pthread_equal(data->thread,thread)))
      return 1;					/* Already locked by thread */
  }
  return 0;
}

int thr_lock(THR_LOCK_DATA *data,int lock_type)
{
  THR_LOCK *lock=data->lock;
  DBUG_ENTER("thr_lock");

  data->next=0;
  data->type=lock_type;
  data->thread=pthread_self();			/* Must be reset ! */
  VOID(pthread_mutex_lock(&lock->mutex));
  DBUG_PRINT("enter",("data: %lx  thread: %lx  lock: %lx  type: %d",
		      data,data->thread,lock,lock_type));
  if (lock_type == F_RDLCK)
  {
    if (lock->write.data)
    {
      DBUG_PRINT("lock",("write locked by thread: %lx",
			 lock->write.data->thread));
      if (pthread_equal(data->thread,lock->write.data->thread))
      {						/* Already got a write lock */
	(*lock->read.last)=data;		/* Add to running FIFO */
	data->prev=lock->read.last;
	lock->read.last= &data->next;
	goto end;
      }
    }
    else if (!lock->write_wait.data ||
	     have_old_read_lock(lock->read.data,data->thread))
    {						/* No write-locks */
      (*lock->read.last)=data;			/* Add to running FIFO */
      data->prev=lock->read.last;
      lock->read.last= &data->next;
      goto end;
    }
    /* Can't get lock yet */
    (*lock->read_wait.last)=data;		/* Wait for write-lock */
    data->prev=lock->read_wait.last;
    lock->read_wait.last= &data->next;
    {
      pthread_cond_t *cond=get_cond();
      data->cond=cond;
      do
      {
	pthread_cond_wait(cond,&lock->mutex);
      } while (data->cond == cond);
    }
  }
  else						/* lock for writing */
  {
    if (lock->write.data)
    {
      if (pthread_equal(data->thread,lock->write.data->thread))
      {						/* Already got a write lock */
	(*lock->write.last)=data;		/* Add to running fifo */
	data->prev=lock->write.last;
	lock->write.last= &data->next;
	goto end;
      }
      DBUG_PRINT("lock",("write locked by thread: %lx",
			  lock->write.data->thread));
    }
    else
    {
      if (!lock->write_wait.data && !lock->read.data)
      {
	(*lock->write.last)=data;		/* Add as current write lock */
	data->prev=lock->write.last;
	lock->write.last= &data->next;
	goto end;
      }
      DBUG_PRINT("lock",("read locked by thread: %lx",
			  lock->read.data->thread));
    }
    (*lock->write_wait.last)=data;		/* Put in wait list */
    data->prev=lock->write_wait.last;
    lock->write_wait.last= &data->next;
    {						/* Wait for lock */
      pthread_cond_t *cond=get_cond();
      data->cond=cond;
      do
      {
	pthread_cond_wait(cond,&lock->mutex);
      } while (data->cond == cond);
    }
  }
end:
  pthread_mutex_unlock(&lock->mutex);
  DBUG_RETURN(0);
}


	/* Unlock lock and free next thread on same lock */

void thr_unlock(THR_LOCK_DATA *data)
{
  THR_LOCK *lock=data->lock;
  DBUG_ENTER("thr_unlock");
  pthread_mutex_lock(&lock->mutex);
  DBUG_PRINT("enter",("data: %lx  thread: %lx  lock: %lx",
		      data,data->thread,lock));

  if ((*data->prev)=data->next)		/* remove from lock-list */
    data->next->prev= data->prev;
  else if (data->type == F_RDLCK)
    lock->read.last=data->prev;
  else
    lock->write.last=data->prev;
  data->type=F_UNLCK;			/* Mark unlocked */

  if (!lock->write.data && !lock->read.data) /* If no more locks in use */
  {
    if ((data=lock->write_wait.data))	/* Release write-locks first */
    {
      if ((*data->prev)=data->next)	/* remove from wait-list */
	data->next->prev= data->prev;
      else
	lock->write_wait.last=data->prev;

      (*lock->write.last)=data;		/* Put in execute list */
      data->prev=lock->write.last;
      data->next=0;
      lock->write.last= &data->next;
      DBUG_PRINT("lock",("giving write lock to thread: %lx",
			 data->thread));
      {
	pthread_cond_t *cond=data->cond;
	data->cond=0;				/* Mark thread free */
	VOID(pthread_cond_signal(cond));	/* Start waiting thread */
      }
    }
    else if ((data=lock->read_wait.data))
    {
      (*lock->read.last)=data;		/* move locks from waiting list */
      data->prev=lock->read.last;
      lock->read.last=lock->read_wait.last;

      lock->read_wait.data=0;
      lock->read_wait.last= &lock->read_wait.data;

      do
      {
	pthread_cond_t *cond=data->cond;
	DBUG_PRINT("lock",("giving read lock to thread: %lx",
			   data->thread));
	data->cond=0;				/* Mark thread free */
	VOID(pthread_cond_signal(cond));
      } while ((data=data->next));
    }
    else
    {
      DBUG_PRINT("lock",("No locks to free"));
    }
  }
  pthread_mutex_unlock(&lock->mutex);
  DBUG_VOID_RETURN;
}


/*
** Get all locks in a specific order to avoid dead-locks
** Sort acording to lock position and put write_locks before read_locks if
** lock on same lock.
*/


#define LOCK_CMP(A,B) ((byte*) (A->lock)+test(A->type == F_RDLCK) < (byte*) (B->lock)+ test(B->type == F_RDLCK))

static void sort_locks(THR_LOCK_DATA **data,uint count)
{
  THR_LOCK_DATA **pos,**end,**prev,*tmp;

  /* Sort locks with insertion sort (fast because almost always few locks) */

  for (pos=data+1,end=data+count; pos < end ; pos++)
  {
    tmp= *pos;
    if (LOCK_CMP(tmp,pos[-1]))
    {
      prev=pos;
      do {
	prev[0]=prev[-1];
      } while (--prev != data && LOCK_CMP(tmp,prev[-1]));
      prev[0]=tmp;
    }
  }
}


void thr_multi_lock(THR_LOCK_DATA **data,uint count)
{
  THR_LOCK_DATA **pos,**end;
  DBUG_ENTER("thr_multi_lock");
  DBUG_PRINT("enter",("data: %lx  count: %d",data,count));
  if (count > 1)
    sort_locks(data,count);
  /* lock everything */
  for (pos=data,end=data+count; pos < end ; pos++)
  {
    VOID(thr_lock(*pos,(*pos)->type));
#ifdef MAIN
    printf("Thread: %lx  Got lock: %lx  type: %d\n",pthread_self_ptr(),
	   pos[0]->lock, pos[0]->type); fflush(stdout);
#endif
  }
  DBUG_VOID_RETURN;
}

  /* free all locks */

void thr_multi_unlock(THR_LOCK_DATA **data,uint count)
{
  THR_LOCK_DATA **pos,**end;
  DBUG_ENTER("thr_multi_unlock");
  DBUG_PRINT("enter",("data: %lx  count: %d",data,count));

  for (pos=data,end=data+count; pos < end ; pos++)
  {
#ifdef MAIN
    printf("Thread: %lx  Rel lock: %lx  type: %d\n",
	   pthread_self_ptr(), pos[0]->lock,
	   pos[0]->type); fflush(stdout);
#endif
    if ((*pos)->type != F_UNLCK)
      thr_unlock(*pos);
  }
  DBUG_VOID_RETURN;
}

#include <my_sys.h>

void thr_print_lock(string name,struct st_lock_list *list)
{
  THR_LOCK_DATA *data,**prev;
  if (list->data)
  {
    printf("%-10s: ",name);
    prev= &list->data;
    for (data=list->data; data ; data=data->next)
    {
      printf("%lx (%lx); ",data,data->thread);
      if (data->prev != prev)
	printf("\nWarning: prev didn't point at previous lock\n");
      prev= &data->next;
    }
    puts("");
    if (prev != list->last)
      printf("Warning: last didn't point at last lock\n");
  }
}

void thr_print_locks(void)
{
  LIST *list;

  pthread_mutex_lock(&THR_LOCK_lock);
  puts("Current locks:");
  for (list=thread_list ; list ; list=rest(list))
  {
    THR_LOCK *lock=(THR_LOCK*) list->data;
    VOID(pthread_mutex_lock(&lock->mutex));
    printf("lock: %lx:",lock);
    if (lock->write.data)
      printf(" write");
    if (lock->write_wait.data)
      printf(" write_wait");
    if (lock->read.data)
      printf(" read");
    if (lock->read_wait.data)
      printf(" read_wait");
    puts("");
    thr_print_lock("write",&lock->write);
    thr_print_lock("write_wait",&lock->write_wait);
    thr_print_lock("read",&lock->read);
    thr_print_lock("read_wait",&lock->read_wait);
    VOID(pthread_mutex_unlock(&lock->mutex));
    puts("");
  }
  fflush(stdout);
  pthread_mutex_unlock(&THR_LOCK_lock);
}


#ifdef MAIN

struct st_test {
  uint lock_nr,lock_type;
};

THR_LOCK locks[5];			/* 4 locks */

struct st_test test_0[] = {{0,0}};	/* One lock */
struct st_test test_1[] = {{0,0},{0,1}}; /* Read and write lock of lock 0 */
struct st_test test_2[] = {{1,1},{0,0},{2,0}};
struct st_test test_3[] = {{2,1},{1,0},{0,0}}; /* Deadlock with test_2 ? */
struct st_test test_4[] = {{0,1},{0,0},{0,1},{0,0}};
struct st_test test_5[] = {{0,0},{1,0},{2,0},{3,0}}; /* Many reads */
struct st_test test_6[] = {{0,1},{1,1},{2,1},{3,1}}; /* Many writes */
struct st_test test_7[] = {{3,0}};
struct st_test test_8[] = {{1,0},{2,0},{3,0}};	/* Should be quick */
struct st_test test_9[] = {{4,0}};
struct st_test test_10[] ={{4,1}};


struct st_test *tests[] = {test_0,test_1,test_2,test_3,test_4,test_5,test_6,
			   test_7,test_8,test_9,test_10};
int lock_counts[]= {sizeof(test_0)/sizeof(struct st_test),
		    sizeof(test_1)/sizeof(struct st_test),
		    sizeof(test_2)/sizeof(struct st_test),
		    sizeof(test_3)/sizeof(struct st_test),
		    sizeof(test_4)/sizeof(struct st_test),
		    sizeof(test_5)/sizeof(struct st_test),
		    sizeof(test_6)/sizeof(struct st_test),
		    sizeof(test_7)/sizeof(struct st_test),
		    sizeof(test_8)/sizeof(struct st_test),
		    sizeof(test_9)/sizeof(struct st_test),
		    sizeof(test_10)/sizeof(struct st_test)
};


static pthread_cond_t COND_thread_count;
static pthread_mutex_t LOCK_thread_count;
static uint thread_count;
static ulong sum=0;

#define MAX_LOCK_COUNT 8

static void *test_thread(void *arg)
{
  int i,j,param=*((int*) arg);
  THR_LOCK_DATA data[MAX_LOCK_COUNT];
  THR_LOCK_DATA *multi_locks[MAX_LOCK_COUNT];

  printf("Thread %lx (%d) started\n",pthread_self_ptr(),param); fflush(stdout);

  for (i=0; i < lock_counts[param] ; i++)
    thr_lock_data_init(locks+tests[param][i].lock_nr,data+i);
  for (j=1 ; j < 10 ; j++)		/* try locking 10 times */
  {
    for (i=0; i < lock_counts[param] ; i++)
    {					/* Init multi locks */
      multi_locks[i]= &data[i];
      data[i].type= tests[param][i].lock_type == 0 ? F_RDLCK : F_WRLCK;
    }
    thr_multi_lock(multi_locks,lock_counts[param]);
    pthread_mutex_lock(&LOCK_thread_count);
    {
      int tmp=rand() & 7;			/* Do something from 0-2 sec */
      if (tmp == 0)
	sleep(1);
      else if (tmp == 1)
	sleep(2);
      else
      {
	ulong k;
	for (k=0 ; k < (ulong) (tmp-2)*100000L ; k++)
	  sum+=k;
      }
    }
    pthread_mutex_unlock(&LOCK_thread_count);
    thr_multi_unlock(multi_locks,lock_counts[param]);
  }

  printf("Tread %lx (%d) ended\n",pthread_self_ptr(),param); fflush(stdout);
  thr_print_locks();
  pthread_mutex_lock(&LOCK_thread_count);
  thread_count--;
  VOID(pthread_cond_signal(&COND_thread_count)); /* Tell main we are ready */
  pthread_mutex_unlock(&LOCK_thread_count);
  free((gptr) arg);
  return 0;
}


int main(int argc __attribute__((unused)),char **argv __attribute__((unused)))
{
  pthread_t tid;
  pthread_attr_t thr_attr;
  int i,*param,error;
  MY_INIT(argv[0]);
  if (argc > 1 && argv[1][0] == '-' && argv[1][1] == '#')
    DBUG_PUSH(argv[1]+2);

  printf("Main thread: %lx\n",pthread_self_ptr());

  if ((error=pthread_cond_init(&COND_thread_count,NULL)))
  {
    fprintf(stderr,"Got error: %d from pthread_cond_init (errno: %d)",
	    error,errno);
    exit(1);
  }
  if ((error=pthread_mutex_init(&LOCK_thread_count,NULL)))
  {
    fprintf(stderr,"Got error: %d from pthread_cond_init (errno: %d)",
	    error,errno);
    exit(1);
  }

  for (i=0 ; i < (int) array_elements(locks) ; i++)
    thr_lock_init(locks+i);

  if ((error=pthread_attr_init(&thr_attr)))
  {
    fprintf(stderr,"Got error: %d from pthread_attr_init (errno: %d)",
	    error,errno);
    exit(1);
  }
  if ((error=pthread_attr_setdetachstate(&thr_attr,PTHREAD_CREATE_DETACHED)))
  {
    fprintf(stderr,
	    "Got error: %d from pthread_attr_setdetachstate (errno: %d)",
	    error,errno);
    exit(1);
  }
#ifndef pthread_attr_setstacksize		/* void return value */
  if ((error=pthread_attr_setstacksize(&thr_attr,65536L)))
  {
    fprintf(stderr,"Got error: %d from pthread_attr_setstacksize (errno: %d)",
	    error,errno);
    exit(1);
  }
#endif
#ifdef HAVE_THR_SETCONCURRENCY
  VOID(thr_setconcurrency(2));
#endif
  for (i=0 ; i < (int) array_elements(lock_counts) ; i++)
  {
    param=(int*) malloc(sizeof(int));
    *param=i;

    if ((error=pthread_mutex_lock(&LOCK_thread_count)))
    {
      fprintf(stderr,"Got error: %d from pthread_mutex_lock (errno: %d)",
	      error,errno);
      exit(1);
    }
    if ((error=pthread_create(&tid,&thr_attr,test_thread,(void*) param)))
    {
      fprintf(stderr,"Got error: %d from pthread_create (errno: %d)\n",
	      error,errno);
      pthread_mutex_unlock(&LOCK_thread_count);
      exit(1);
    }
    thread_count++;
    pthread_mutex_unlock(&LOCK_thread_count);
  }

  pthread_attr_destroy(&thr_attr);
  if ((error=pthread_mutex_lock(&LOCK_thread_count)))
    fprintf(stderr,"Got error: %d from pthread_mutex_lock\n",error);
  while (thread_count)
  {
    if ((error=pthread_cond_wait(&COND_thread_count,&LOCK_thread_count)))
      fprintf(stderr,"Got error: %d from pthread_cond_wait\n",error);
  }
  if ((error=pthread_mutex_unlock(&LOCK_thread_count)))
    fprintf(stderr,"Got error: %d from pthread_mutex_unlock\n",error);
  for (i=0 ; i < (int) array_elements(locks) ; i++)
    thr_lock_delete(locks+i);
  printf("end\n");
  return 0;
}
#endif
