/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

/* 
 * rfc1524 defines a format for the Multimedia Mail Configuration, which
 * is the standard mailcap file format under Unix which specifies what 
 * external programs should be used to view/compose/edit multimedia files
 * based on content type.
 *
 * This file contains various functions for implementing a fair subset of 
 * rfc1524.
 */

#include "mutt.h"
#include "rfc1524.h"

#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>
#include <string.h>

/* The command semantics include the following:
 * %s is the filename that contains the mail body data
 * %t is the content type, like text/plain
 * \% is %
 * Unsupported rfc1524 parameters: these would probably require some doing
 * by mutt, and can probably just be done by piping the message to metamail
 * %{parameter} is replaced by the parameter value from the content-type field
 * %n is the integer number of sub-parts in the multipart
 * %F is "content-type filename" repeated for each sub-part
 *
 * In addition, this function returns a 0 if the command works on a file,
 * and 1 if the command works on a pipe.
 */
int rfc1524_expand_command (BODY *a, char *filename, char *type, 
    char *command, int clen)
{
  int x=0,y=0;
  int needspipe = TRUE;
  char buf[LONG_STRING];

  while (command[x] && x<clen && y<sizeof(buf)) {
    if (command[x] == '\\') {
      x++;
      buf[y++] = command[x++];
    }
    else if (command[x] == '%') {
      x++;
      if (command[x] == '{') {
	char param[STRING];
	int z = 0;
	char *ret = NULL;

	x++;
	while (command[x] && command[x] != '}' && z<sizeof(param))
	  param[z++] = command[x++];
	param[z] = '\0';
	dprint(2,(debugfile,"Parameter: %s  Returns: %s\n",param,ret));
	ret = mutt_get_parameter(param,a->parameter);
	dprint(2,(debugfile,"Parameter: %s  Returns: %s\n",param,ret));
	z = 0;
	while (ret && ret[z] && y<sizeof(buf))
	  buf[y++] = ret[z++];
      }
      else if (command[x] == 's' && filename != NULL)
      {
	char *fn = filename;

	while (*fn && y < sizeof (buf))
	  buf[y++] = *fn++;
	needspipe = FALSE;
      }
      else if (command[x] == 't')
      {
	while (*type && y < sizeof (buf))
	  buf[y++] = *type++;
      }
      x++;
    }
    else
      buf[y++] = command[x++];
  }
  buf[y] = '\0';
  strfcpy (command, buf, clen);

  return needspipe;
}

int rfc1524_mailcap_parse (BODY *a,
			   char *filename,
			   char *type, 
			   rfc1524_entry *entry,
			   int opt)
{
  FILE *fp = NULL;
  char buf[LONG_STRING];
  char basetype[STRING];
  char *ch;
  int found = FALSE;
  int match = TRUE;
  int wrap = FALSE;
  int copiousoutput = FALSE;
  int composecommand = FALSE;
  int editcommand = FALSE;
  int printcommand = FALSE;
  int x = 0, y = 0;
  int btlen = 0;

  /* rfc1524 mailcap file is of the format:
   * base/type; command; extradefs
   * type can be * for matching all
   * base with no /type is an implicit wild
   * command contains a %s for the filename to pass, default to pipe on stdin
   * extradefs are of the form:
   *  def1="definition"; def2="define \;";
   * line wraps with a \ at the end of the line
   * # for comments
   */
  if ((fp = fopen (filename, "r")) != NULL)
  {
    /* copy base type into basetype */
    x = 0;
    while (type[x] && x < sizeof (basetype) && type[x] != '/')
    {
      basetype[x] = type[x];
      x++;
    }
    basetype[x] = '\0';
    dprint (2, (debugfile, "mailcap basetype: %s\n", basetype));
    btlen = x;
    while (!found && fgets (buf, sizeof (buf), fp))
    {
      /* Strip comments (ie, everything after #, but not \# */
      if ((ch = strchr (buf, '#')))
        if ((ch == buf) || (*(ch-1) != '\\'))
	  *ch = '\0';
  
      /* Strip trailing white space */
      y = strlen (buf);
      y = (y ? y - 1 : 0);
      while (y && (buf[y] == ' ' || buf[y] == '\t' || buf[y] == '\n'))
	buf[y--] = '\0';
  
      /* handle line wrap */
      if (buf[y] == '\\')
	wrap = TRUE;
      else
	wrap = FALSE;

      while (wrap && fgets (buf + strlen (buf) - 1, sizeof (buf) - strlen (buf),fp)) 
      {
	if ((ch = strchr (buf,'#')))
	{
	  if (*(ch-1) != '\\')
	    *ch = '\0';
	}
	y = strlen (buf);
	y = (y ? y - 1 : 0);
	while (buf[y] == ' ' || buf[y] == '\t' || buf[y] == '\n') 
	  buf[y--] = '\0';
	if (buf[y] == '\\')
	  wrap = TRUE; 
	else
	  wrap = FALSE; 
      }
      dprint (2, (debugfile, "mailcap entry: %s\n", buf));

      if (!strncasecmp (buf, type, strlen (type)) && (buf[strlen (type)] == ';'))
	found = TRUE;
      else
      {
	/* Check * case */
	if (strncmp (buf, basetype, btlen-1) == 0)
	{
	  /* check * wildcard */ 
	  if (buf[btlen] && buf[btlen+1] && buf[btlen] == '/' && 
	      buf[btlen+1] == '*') 
	    found = TRUE;
	  /* check implicit wild */
	  else if (buf[btlen] && buf[btlen] == ';')
	    found = TRUE;
	}
      }

      /* Parse the found line for the information we need */
      if (found)
      {
	char *tok;
	int z = 0, i = 0;
	char token[STRING];

	/* first field is content type, already checked, so skip */
	while (z < sizeof (buf) && buf[z] && buf[z] != ';')
	  z++;
	z++;

	/* next field is the command */
	while (z < sizeof (buf) && i < sizeof (token) && buf[z] && 
	      !(buf[z] == ';' && buf[z-1] != '\\'))
	  token[i++] = buf[z++];
	token[i] = '\0';
	z++;
	tok = skip_whitespace (token);
	if (entry)
	  entry->command = safe_strdup (tok);

	/* parse the optional fields */
	/* next field is the command */
	i = 0; 
	while (z < sizeof (buf) && i < sizeof (token) && buf[z] && 
	      !(buf[z] == ';' && buf[z-1] != '\\')) 
	  token[i++] = buf[z++];
	token[i] = '\0';
	z++;
	while (token[0])
	{
	  tok = skip_whitespace (token);
	  dprint (2, (debugfile, "token: %s\n", tok));
	  if (!strcasecmp (tok, "needsterminal"))
	  {
	    if (entry)
	      entry->needsterminal = TRUE;
	  }
	  else if (!strcasecmp (tok, "copiousoutput"))
	  {
	    copiousoutput = TRUE;
	    if (entry)
	      entry->copiousoutput = TRUE;
	  }
	  else if (!strncasecmp (tok, "composetyped", 12))
	  {
	    /* this compare most occur before compose to match correctly */
	    /* this isn't particularly well specified in the rfc, so skip it */
	  }
	  else if (!strncasecmp (tok, "compose", 7))
	  {
	    tok += 7;
	    tok = skip_whitespace (tok);
	    if (*tok == '=')
	    {
	      composecommand = TRUE;
	      if (entry)
	      {
		tok = skip_whitespace (++tok);
		safe_free ((void **) &entry->composecommand);
		entry->composecommand = safe_strdup (tok);
	      }
	    }
	    else 
	      mutt_error ("Improperly formated mailcap entry for type %s", type);
	  } 
	  else if (!strncasecmp (tok, "print", 5))
	  {
	    tok+=5;
	    tok = skip_whitespace (tok);
	    if (*tok == '=')
	    {
	      printcommand = TRUE;
	      if (entry)
	      {
		tok = skip_whitespace (++tok);
		safe_free ((void **) &entry->printcommand);
		entry->printcommand = safe_strdup(tok);
	      }
	    }
	    else 
	      mutt_error("Improperly formated mailcap entry for type %s", type);
	  }
	  else if (!strncasecmp (tok, "edit", 4))
	  {
	    tok+=4;
	    tok = skip_whitespace (tok);
	    if (*tok == '=')
	    {
	      editcommand = TRUE;
	      if (entry)
	      {
		tok = skip_whitespace (++tok);
		safe_free((void **) &entry->editcommand);
		entry->editcommand = safe_strdup(tok);
	      }
	    }
	    else 
	      mutt_error ("Improperly formated mailcap entry for type %s", type);
	  }
	  else if (!strncasecmp (tok, "nametemplate", 12))
	  {
	    tok += 12;
	    tok = skip_whitespace (tok);
	    if (*tok == '=')
	    {
	      if (entry)
	      {
		tok = skip_whitespace (++tok);
		safe_free ((void **) &entry->nametemplate);
		entry->nametemplate = safe_strdup (tok);
	      }
	    }
	    else
	      mutt_error ("Improperly formated mailcap entry for type %s", type);
	  }
	  else if (!strncasecmp (tok, "x-convert", 9))
	  {
	    tok += 9;
	    tok = skip_whitespace (tok);
	    if (*tok == '=')
	    {
	      if (entry)
	      {
		tok = skip_whitespace (++tok);
		safe_free ((void **) &entry->convert);
		entry->convert = safe_strdup (tok);
	      }
	    }
	    else
	      mutt_error ("Improperly formated mailcap entry for type %s", type);
	  }
	  else if (!strncasecmp (tok, "test", 4))
	  {
	    /* 
	     * This routine executes the given test command to determine
	     * if this is the right entry.
	     */
	    char path[_POSIX_PATH_MAX+STRING];
	    int retval;

            tok += 4;
            tok = skip_whitespace (tok);
            if (*tok == '=')
            {
              tok = skip_whitespace (++tok);
	      strfcpy (path, tok, sizeof (path));
	      rfc1524_expand_command (a, NULL, type, path, sizeof (path));
	      retval = mutt_system (path);
	      if (retval)
	      {
	        /* a non-zero exit code means test failed */
		match = FALSE;
	      }
              dprint (2, (debugfile,"testing entry: %s returns %d\n", path, retval));
            }
            else 
              mutt_error ("Improperly formated mailcap entry for type %s", type);
          }
          i = 0; 
          while(z < sizeof (buf) && i < sizeof (token) && buf[z] && 
                !(buf[z] == ';' && buf[z-1] != '\\'))
            token[i++] = buf[z++];
          token[i] = '\0';
          z++;
        }

        if (opt == M_AUTOVIEW)
	{
          if (!copiousoutput)
	    match = FALSE;
        }
	else if (opt == M_COMPOSE)
	{
          if (!composecommand)
	    match = FALSE;
        }
	else if (opt == M_EDIT)
	{
          if (!editcommand)
	    match = FALSE;
        }
	else if (opt == M_PRINT)
	{
          if (!printcommand)
	    match = FALSE;
        }

        if (!match)
	{
          found = FALSE;

          /* reset */
          match = TRUE;
          if (entry)
          {
	    safe_free ((void **) &entry->command);
	    safe_free ((void **) &entry->composecommand);
	    safe_free ((void **) &entry->editcommand);
	    safe_free ((void **) &entry->printcommand);
	    safe_free ((void **) &entry->nametemplate);
	    safe_free ((void **) &entry->convert);
	    entry->needsterminal = 0;
	    entry->copiousoutput = 0;
	  }
        }
      }
    }

    fclose(fp);
    return found;
  }
  else
    return FALSE;
}

rfc1524_entry *rfc1524_new_entry()
{
  rfc1524_entry *tmp;

  tmp = (rfc1524_entry *)safe_malloc(sizeof(rfc1524_entry));
  memset(tmp,0,sizeof(rfc1524_entry));

  return tmp;
}

void rfc1524_free_entry(rfc1524_entry **entry)
{
  rfc1524_entry *p = *entry;

  safe_free((void **)&p->command);
  safe_free((void **)&p->testcommand);
  safe_free((void **)&p->composecommand);
  safe_free((void **)&p->editcommand);
  safe_free((void **)&p->printcommand);
  safe_free((void **)&p->nametemplate);
  safe_free((void **)entry);
}

/*
 * rfc1524_mailcap_lookup attempts to find the given type in the
 * list of mailcap files.  On success, this returns the entry information
 * in *entry, and returns 1.  On failure (not found), returns 0.
 * If entry == NULL just return 1 if the given type is found.
 */
int rfc1524_mailcap_lookup (BODY *a, char *type, rfc1524_entry *entry, int opt)
{
  char path[_POSIX_PATH_MAX];
  int x;
  int found = FALSE;
  char *curr = MailcapPath;

  /* rfc1524 specifies that a path of mailcap files should be searched.
   * joy.  They say 
   * $HOME/.mailcap:/etc/mailcap:/usr/etc/mailcap:/usr/local/etc/mailcap, etc
   * and overriden by the MAILCAPS environment variable, and, just to be nice, 
   * we'll make it specifiable in .muttrc
   */
  if (!*curr)
  {
    mutt_error ("No mailcap path specified");
    return 0;
  }

  while (!found && *curr)
  {
    x = 0;
    while (*curr && *curr != ':' && x < sizeof (path) - 1)
    {
      path[x++] = *curr;
      curr++;
    }
    if (*curr)
      curr++;

    if (!x)
      continue;
    
    path[x] = '\0';
    mutt_expand_path (path, sizeof (path));

    dprint(2,(debugfile,"Checking mailcap file: %s\n",path));
    found = rfc1524_mailcap_parse (a, path, type, entry, opt);
  }

  if (entry && !found)
    mutt_error ("mailcap entry for type %s not found", type);

  return found;
}

/* Modified by blong to accept a "suggestion" for file name.  If
 * that file exists, then construct one with unique name but 
 * keep any extension.  This might fail, I guess.
 * Renamed to mutt_adv_mktemp so I only have to change where it's
 * called, and not all possible cases.
 */
void mutt_adv_mktemp (char *s)
{
  char buf[_POSIX_PATH_MAX];
  char tmp[_POSIX_PATH_MAX];
  char *period;

  strfcpy (buf, Tempdir, sizeof (buf));
  mutt_expand_path (buf, sizeof (buf));
  if (s[0] == '\0')
  {
    sprintf (s, "%s/muttXXXXXX", buf);
    mktemp (s);
  }
  else
  {
    strfcpy (tmp, s, sizeof (tmp));
    sprintf (s, "%s/%s", buf, tmp);
    if (access (s, F_OK) != 0)
      return;
    if ((period = strrchr (tmp, '.')) != NULL)
      *period = 0;
    sprintf (s, "%s/%s.XXXXXX", buf, tmp);
    mktemp (s);
    if (period != NULL)
    {
      *period = '.';
      strcat (s, period);
    }
  }
}

/* This routine expands the filename given to match the format of the
 * nametemplate given.  It returns various values based on what operations
 * it performs.
 *
 * Returns 0 if oldfile is fine as is.
 * Returns 1 if newfile specified
 */

int rfc1524_expand_filename (char *nametemplate,
			     char *oldfile, 
			     char *newfile,
			     size_t nflen)
{
  int z = 0;
  int i = 0, j = 0;
  int lmatch = TRUE;
  int match = TRUE;
  size_t len = 0;
  char *s;

  newfile[0] = 0;

  if (nametemplate && (s = strrchr (nametemplate, '/')))
    nametemplate = s + 1;

  if (oldfile && (s = strrchr (oldfile, '/')))
  {
    len = s - oldfile + 1;
    if (len > nflen)
      len = nflen;
    strfcpy (newfile, oldfile, len + 1);
    oldfile += len;
  }

  /* If nametemplate is NULL, create a newfile from oldfile and return 0 */
  if (!nametemplate)
  {
    if (oldfile)
      strfcpy (newfile, oldfile, nflen);
    mutt_adv_mktemp (newfile);
    return 0;
  }

  /* If oldfile is NULL, just return a newfile name */
  if (!oldfile)
  {
    snprintf (newfile, nflen, nametemplate, "mutt");
    mutt_adv_mktemp (newfile);
    return 0;
  }

  /* Next, attempt to determine if the oldfile already matches nametemplate */
  /* Nametemplate is of the form pre%spost, only replace pre or post if
   * they don't already match the oldfilename */
  /* Test pre */

  if ((s = strrchr (nametemplate, '%')) != NULL)
  {
    newfile[len] = '\0';

    z = s - nametemplate;

    for (i = 0; i < z && i < nflen; i++)
    {
      if (oldfile[i] != nametemplate[i])
      {
	lmatch=FALSE;
	break;
      }
    }

    if (!lmatch)
    {
      match = FALSE;
      i = nflen - len;
      if (i > z)
	i = z;
      strfcpy (newfile + len, nametemplate, i);
    }

    strfcpy (newfile + strlen (newfile), 
	    oldfile, nflen - strlen (newfile));

    dprint (1, (debugfile,"template: %s, oldfile: %s, newfile: %s\n",
	      nametemplate, oldfile, newfile));

    /* test post */
    lmatch = TRUE;

    for (z += 2, i = strlen (oldfile) - 1, j = strlen (nametemplate) - 1; 
	i && j > z; i--, j--)
      if (oldfile[i] != nametemplate[j])
      {
	lmatch = FALSE;
	break;
      }

    if (!lmatch)
    {
      match = FALSE;
      strfcpy (newfile + strlen (newfile),
	      nametemplate + z, nflen - strlen (newfile));
    }

    if (match) 
      return 0;

    return 1;
  }
  else
  {
    /* no %s in nametemplate, graft unto path of oldfile */
    strfcpy (newfile, nametemplate, nflen);
    return 1;
  }
}

/* For nametemplate support, we may need to rename a file.
 * If rfc1524_expand_command() is used on a recv'd message, then
 * the filename doesn't exist yet, but if its used while sending a message,
 * then we need to rename the existing file.
 *
 * This function returns 2 on successful move, 1 on old file doesn't exist,
 * and 0 on failure.  
 * This function returns 0 if the newfile already exists or an error occurs.
 */
int mutt_rename_file (char *oldfile, char *newfile)
{
  FILE *ofp, *nfp;

  if (access (oldfile, F_OK) != 0)
    return 1;
  if (access (newfile, F_OK) == 0)
    return 0;
  if ((ofp = fopen (oldfile,"r")) == NULL)
    return 0;
  if ((nfp = fopen (newfile,"w")) == NULL)
  {
    fclose(ofp);
    return 0;
  }
  mutt_copy_stream (ofp,nfp);
  fclose (nfp);
  fclose (ofp);
  mutt_unlink (oldfile);
  return 2;
}
