/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "mutt_menu.h"
#include "rfc1524.h"

#ifdef _PGPPATH
#include "pgp.h"
#endif

#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>
#include <string.h>
#include <errno.h>

static ATTACHPTR *gen_attach_list (BODY *m, int level, int compose)
{
  ATTACHPTR *top = NULL, *end = NULL, *new;

  while (m)
  {
    if (m->type == TYPEMULTIPART && m->parts)
    {
      if (top)
	end->next = gen_attach_list (m->parts, level, compose);
      else
	end = top = gen_attach_list (m->parts, level, compose);
      while (end->next)
	end = end->next;
    }
    else
    {
      new = (ATTACHPTR *) safe_calloc (1, sizeof (ATTACHPTR));
      new->content = m;
      new->level = level;

      if (top)
      {
        end->next = new;
        end = new;
      }
      else
        top = end = new;
        
      /* We don't support multipart messages in the compose menu yet */
      if (!compose && m->type == TYPEMESSAGE &&
	    (!strcasecmp (m->subtype, "rfc822") ||
	    !strcasecmp (m->subtype, "news")) &&
	    is_multipart (m->parts))
      {
	end->next = gen_attach_list (m->parts, level + 1, compose);
	while (end->next)
	  end = end->next;
      }
    }

    m = m->next;
  }

  return (top);
}

void mutt_update_tree (ATTACHPTR **idx, short idxlen)
{
  char buf[STRING];
  char *s;
  int x;

  for (x = 0; x < idxlen; x++)
  {
    if (idx[x]->level)
    {
      s = buf + 2 * (idx[x]->level - 1);
      *s++ = (idx[x]->content->next) ? '\003' : '\001';
      *s++ = '\004';
      *s++ = '\007';
    }
    else
      s = buf;
    *s = 0;

    if (idx[x]->tree)
    {
      if (strcmp (idx[x]->tree, buf) != 0)
      {
	safe_free ((void **) &idx[x]->tree);
	idx[x]->tree = safe_strdup (buf);
      }
    }
    else
      idx[x]->tree = safe_strdup (buf);

    if (idx[x]->level)
    {
      s = buf + 2 * (idx[x]->level - 1);
      *s++ = (idx[x]->content->next) ? '\005' : '\006';
      *s++ = '\006';
    }
  }
}

ATTACHPTR **mutt_gen_attach_list (BODY *cur, short *idxlen,
					      short *idxmax, int compose)
{
  ATTACHPTR *list, *plist;
  ATTACHPTR **idx;

  plist = list = gen_attach_list (cur, 0, compose);
  idx = (ATTACHPTR **) safe_malloc (sizeof (ATTACHPTR *) * (*idxmax = 5));

  for (*idxlen = 0; plist; (*idxlen)++, plist = plist->next)
  {
    if (*idxlen == *idxmax)
      safe_realloc ((void **) &idx, sizeof (ATTACHPTR *) * (*idxmax += 5));
    idx[*idxlen] = plist;
  }

  mutt_update_tree (idx, *idxlen);
  return (idx);
}

void attach_entry (char *b, size_t blen, MUTTMENU *menu, int num)
{
  char t[SHORT_STRING];
  char s[SHORT_STRING];
  char size[SHORT_STRING];
  ATTACHPTR **idx = (ATTACHPTR **) menu->data;
  BODY *m;

  m = idx[num]->content;
  s[0] = 0;
  if (m->type == TYPEMESSAGE && (!strcasecmp ("rfc822", m->subtype) ||
      !strcasecmp ("news", m->subtype)) && MsgFmt[0])
    _mutt_make_string (s, sizeof (s), MsgFmt, m->hdr, M_FORCESUBJ);

  mutt_pretty_size (size, sizeof (size), m->length);
  snprintf (t, sizeof (t), "[%.7s/%.10s, %.6s, %s]",
	  TYPE (m->type), m->subtype, ENCODING (m->encoding), size);
  snprintf (b, blen, " %c %2d %-34.34s %s%s",
	  idx[num]->tagged ? '*' : ' ',
	  num + 1,
	  t,
	  idx[num]->tree ? idx[num]->tree : "",
	  s[0] ? s : (m->description ? m->description :
	      (m->filename ? m->filename : "<no description>")));
}

void mutt_clear_markers (ATTACHPTR **idx, short indexlen)
{
  int x;

  for (x = 0; x < indexlen; x++)
    idx[x]->content->marked = 0;
}

void mutt_mark_parts (BODY *b, int value)
{
  BODY *m;

  if (b)
  {
    b->marked = value;
    if (is_multipart (b))
    {
      for (m = b->parts; m; m = m->next)
	mutt_mark_parts (m, value);
    }
  }
}

void mutt_tag_attachment (int tag, MUTTMENU *menu, ATTACHPTR **idx)
{
  int x;

  if (tag && !option (OPTAUTOTAG))
  {
    for (x = 0; x < menu->max; x++)
      idx[x]->tagged = 0;
    menu->redraw = REDRAW_INDEX | REDRAW_STATUS;
  }
  else
  {
    idx[menu->current]->tagged = !idx[menu->current]->tagged;
    if (option (OPTRESOLVE) && menu->current < menu->max - 1)
    {
      menu->oldcurrent = menu->current;
      menu->current++;
      if (menu->current >= menu->top + menu->pagelen)
      {
	menu->top = menu->current;
	menu->redraw = REDRAW_INDEX | REDRAW_STATUS;
      }
      else
	menu->redraw |= REDRAW_MOTION_RESYNCH;
    }
    else
      menu->redraw |= REDRAW_CURRENT;
  }
}

void mutt_save_attachment_list (FILE *fp, int tag, MUTTMENU *menu,
					      ATTACHPTR **idx, int decode)
{
  char buf[_POSIX_PATH_MAX], tfile[_POSIX_PATH_MAX];
  char prompt[SHORT_STRING];
  FILE *tfp;
  int rc;
  
  if (!tag)
  {
    if (fp || !idx[menu->current]->content->empty)
    {
      if (decode && !mutt_can_decode (idx[menu->current]->content))
      {
	mutt_error ("I don't know how to decode that!");
	return;
      }
      if (fp && !decode && idx[menu->current]->content->filename)
	strfcpy (buf, idx[menu->current]->content->filename, sizeof (buf));
      else
	buf[0] = 0;
      if (ci_get_field ("Save to file: ", buf, sizeof (buf),
					M_FILE | M_CLEAR) != 0 || !buf[0])
	return;
      mutt_expand_path (buf, sizeof (buf));
      if (mutt_check_overwrite (decode ?
	  NULL : idx[menu->current]->content->filename, buf,
	  tfile, sizeof (tfile), 0))
	return;

      if (decode)
	rc = mutt_decode_save_attachment (fp, idx[menu->current]->content,
								  tfile, 0, 0); 
      else
	rc = mutt_save_attachment (fp, idx[menu->current]->content, tfile, 0); 
      if (rc == 0)
	mutt_message ("Attachment saved.");
    }
  }
  else
  {
    int found = 0;
    int cnt = 0;
    int x;

    if (option (OPTATTACHSPLIT))
    {
      for (x = 0; x < menu->max ; x++)
      {
	if (idx[x]->tagged)
	{
	  found = 1;
	  if (fp == NULL && idx[x]->content->empty)
	    continue;
	  if (decode && !mutt_can_decode (idx[x]->content))
	    continue;
	  if (fp && idx[x]->content->filename)
	    strfcpy (buf, idx[x]->content->filename, sizeof (buf));
	  else
	    buf[0] = 0;
	  snprintf (prompt, sizeof (prompt), "Save attachment %d to file: ",
									  x+1);
	  if (ci_get_field (prompt, buf, sizeof (buf), M_FILE | M_CLEAR) == -1)
	    break;
	  if (!buf[0])
	    continue;

	  mutt_expand_path (buf, sizeof (buf));
	  if (mutt_check_overwrite (decode ?
	      NULL : idx[x]->content->filename, buf,
	      tfile, sizeof (tfile), 0))
	    continue;

	  if (decode)
	    rc = mutt_decode_save_attachment (fp, idx[x]->content, tfile, 0, 0); 
	  else
	    rc = mutt_save_attachment (fp, idx[x]->content, tfile, 0); 
	  if (rc == 0)
	    cnt++;
	}
      }
    }
    else
    {
      for (x = 0; x < menu->max ; x++)
      {
	if (idx[x]->tagged)
	{
	  found = 1;
	  if (fp == NULL && idx[x]->content->empty)
	    continue;
	}
      }

      if (fp && idx[x]->content->filename)
	strfcpy (buf, idx[x]->content->filename, sizeof (buf));
      else
	buf[0] = 0;
      if (ci_get_field ("Save to file: ", buf, sizeof (buf),
				      M_FILE | M_CLEAR) != 0 || !buf[0])
	return;
      mutt_expand_path (buf, sizeof (buf));
      if (mutt_check_overwrite (decode ?
	  NULL : idx[x]->content->filename, buf,
	  tfile, sizeof (tfile), M_SAVE_APPEND))
	return;

      mutt_clear_markers (idx, menu->max);
      for (; x < menu->max ; x++)
      {
	if (idx[x]->tagged && !idx[x]->content->marked && 
			(fp || !idx[x]->content->empty))
	{
	  if (decode && !mutt_can_decode (idx[x]->content))
	    continue;

	  if (decode)
	    rc = mutt_decode_save_attachment (fp, idx[x]->content, tfile, 0,
							  M_SAVE_APPEND);
	  else
	    rc = mutt_save_attachment (fp, idx[x]->content, tfile,
							  M_SAVE_APPEND);
	  if (rc == 0)
	  {
	    if (AttachSep[0] && (tfp = safe_fopen (tfile, "a")) != NULL)
	    {
	      fputs (AttachSep, tfp);
	      fclose (tfp);
	    }
	    cnt++;
	  }
	  /* avoid saving nested parts twice */
	  mutt_mark_parts (idx[x]->content, 1);
	}
      }
    }
    if (!found)
      mutt_error ("No tagged attachments!");
    else if (cnt)
      mutt_message ("%d attachment%s saved.", cnt, (cnt != 1) ? "s" : "");
  }
}

void mutt_pipe_attachment_list (FILE *fp, int tag, MUTTMENU *menu,
						ATTACHPTR **idx, int filter)
{
  char buf[_POSIX_PATH_MAX], tfile[_POSIX_PATH_MAX];
  char warning[STRING+_POSIX_PATH_MAX];
  FILE *tfp;

  if (fp)
    filter = 0; /* sanity check: we can't filter in the recv case yet */

  buf[0] = 0;
  if (ci_get_field ((filter ? "Filter through: " : "Pipe to: "),
				  buf, sizeof (buf), 0) == 0 && buf[0])
  {
    mutt_expand_path (buf, sizeof (buf));

    if (!tag)
    {
      if (fp || !idx[menu->current]->content->empty)
      {
	if (filter)
	{
	  snprintf (warning, sizeof (warning),
		    "WARNING! You are about to overwrite %s, continue?",
		    idx[menu->current]->content->filename);
	  if (mutt_yesorno (warning, M_NO) != M_YES)
	    return;
	  mutt_mktemp (tfile);
	}
	else
	  tfile[0] = 0;

	if (mutt_pipe_attachment (fp, idx[menu->current]->content, buf, tfile))
	{
	  if (filter)
	  {
	    mutt_unlink (idx[menu->current]->content->filename);
	    mutt_rename_file (tfile, idx[menu->current]->content->filename);
	    menu->redraw = REDRAW_CURRENT;
	    mutt_update_encoding (idx[menu->current]->content);
	    mutt_message ("Attachment filtered.");
	  }
	}
	else
	{
	  if (filter && tfile[0])
	    mutt_unlink (tfile);
	}
      }
    }
    else
    {
      int found = 0;
      int cnt = 0;
      int x;

      if (filter || option (OPTATTACHSPLIT))
      {
	for (x = 0; x < menu->max; x++)
	{
	  if (idx[x]->tagged)
	  {
	    found = 1;
	    if (fp == NULL && idx[x]->content->empty)
	      continue;

	    if (filter)
	    {
	      snprintf (warning, sizeof (warning),
			"WARNING! You are about to overwrite %s, continue?",
			idx[x]->content->filename);
	      if (mutt_yesorno (warning, M_NO) != M_YES)
		return;
		
	      mutt_mktemp (tfile);
	      if (mutt_pipe_attachment (fp, idx[x]->content, buf, tfile))
	      {
		mutt_unlink (idx[x]->content->filename);
		mutt_rename_file (tfile, idx[x]->content->filename);
		mutt_update_encoding (idx[x]->content);
		menu->redraw = REDRAW_CURRENT;
		cnt++;
	      }
	      else
		mutt_unlink (tfile);
	    }
	    else
	      mutt_pipe_attachment (fp, idx[x]->content, buf, NULL);
	  }
	}
      }
      else 
      {
	mutt_mktemp (tfile);

	mutt_clear_markers (idx, menu->max);
	for (x = 0; x < menu->max; x++)
	{
	  if (idx[x]->tagged && !idx[x]->content->marked)
	  {
	    found = 1;
	    if (fp == NULL && idx[x]->content->empty)
	      continue;

	    if (mutt_save_attachment (fp, idx[x]->content, tfile, 
						  M_SAVE_APPEND) == 0)
	    {
	      cnt++;
	      if (AttachSep[0] && (tfp = safe_fopen (tfile, "a")) != NULL)
	      {
		fputs (AttachSep, tfp);
		fclose (tfp);
	      }
	    }
	    /* avoid saving nested parts twice */
	    mutt_mark_parts (idx[x]->content, 1);
	  }
	}
	
	if (cnt && (tfp = safe_fopen (tfile, "r")) != NULL)
	{
	  pid_t thepid;
	  FILE *ofp;

	  endwin ();
	  thepid = mutt_create_filter (buf, &ofp, NULL, NULL);
	  mutt_copy_stream (tfp, ofp);
	  fclose (tfp);
	  fclose (ofp);
	  if (mutt_wait_filter (thepid) != 0 || option (OPTWAITKEY))
	    ci_any_key_to_continue (NULL);
	  mutt_unlink (tfile);
	}
	else
	{
	  mutt_unlink (tfile);
	  mutt_error ("Error %s attachment(s)!",
					filter ? "filtering" : "piping");
	}
      }
      if (!found)
	mutt_error ("No tagged attachments!");
      else if (cnt)
	mutt_message ("%d attachment%s %s.", cnt, (cnt != 1) ? "s" : "",
			filter ? "filtered" : "piped");
    }
    menu->redraw = REDRAW_FULL;
  }
}

void mutt_print_attachment_list (FILE *fp, int tag, MUTTMENU *menu,
							  ATTACHPTR **idx)
{
  char tfile[_POSIX_PATH_MAX];
  FILE *tfp;

  if (!tag)
  {
    if (query_quadoption (OPT_PRINT, "Print attachment?") == M_YES)
    {
      if (fp || !idx[menu->current]->content->empty)
      {
	if (mutt_print_attachment (fp, idx[menu->current]->content))
	  menu->redraw = REDRAW_FULL;
      }
    }
  }
  else 
  {
    int found = 0;
    int cnt = 0;
    int x;

    if (query_quadoption (OPT_PRINT, "Print tagged attachment(s)?") == M_YES)
    {
      if (option (OPTATTACHSPLIT))
      {
	for (x = 0; x < menu->max; x++)
	{
	  if (idx[x]->tagged)
	  {
	    found = 1;

	    if (fp == NULL && idx[x]->content->empty)
	      continue;
	    if (mutt_print_attachment (fp, idx[x]->content))
	    {
	      cnt++;
	      menu->redraw = REDRAW_FULL;
	    }
	  }
	}
      }
      else 
      {
	mutt_mktemp (tfile);

	mutt_clear_markers (idx, menu->max);
	for (x = 0; x < menu->max; x++)
	{
	  if (idx[x]->tagged && !idx[x]->content->marked)
	  {
	    found = 1;
	    if (mutt_save_attachment (fp, idx[x]->content, tfile, 
						  M_SAVE_APPEND) == 0)
	    {
	      if (AttachSep[0] && (tfp = safe_fopen (tfile, "a")) != NULL)
	      {
		fputs (AttachSep, tfp);
		fclose (tfp);
	      }
	    }
	    /* avoid saving nested parts twice */
	    mutt_mark_parts (idx[x]->content, 1);
	  }
	}

	if ((tfp = safe_fopen (tfile, "r")) != NULL)
	{
	  pid_t thepid;
	  FILE *ofp;

	  cnt++;
	  endwin ();
	  thepid = mutt_create_filter (PrintCmd, &ofp, NULL, NULL);
	  mutt_copy_stream (tfp, ofp);
	  fclose (tfp);
	  fclose (ofp);
	  if (mutt_wait_filter (thepid) != 0 || option (OPTWAITKEY))
	    ci_any_key_to_continue (NULL);
	  mutt_unlink (tfile);
	}
	else
	{
	  mutt_error ("Error printing attachment(s).");
	  mutt_unlink (tfile);
	}
	menu->redraw = REDRAW_FULL;
      }
    }
    if (!found)
      mutt_error ("No tagged attachments!");
    else if (cnt)
      mutt_message ("%d attachment%s printed.", cnt, (cnt != 1) ? "s" : "");
  }
}

static void copy_attachment_list (FILE *fp, int tag, MUTTMENU *menu,
				    ATTACHPTR **idx, HEADER *hdr, int decode)
{
  char tfile[_POSIX_PATH_MAX];
  char prompt[SHORT_STRING];
  CONTEXT ctx;
  BODY *b = NULL, *m = NULL, *tmp = NULL;
  int x;

  if (!decode) /* delete attachment not supported yet */
    return;

  if (tag)
  {
    for (x = 0; x < menu->max; x++)
    {
      if (idx[x]->tagged)
	break;
    }
    if (x >= menu->max)
    {
      mutt_error ("No tagged attachments!");
      return;
    }
  }

  snprintf (prompt, sizeof (prompt), "Decode-copy to mailbox");
  mutt_default_save (tfile, sizeof (tfile), hdr->env);

  mutt_pretty_mailbox (tfile);
  if (mutt_enter_fname (prompt, tfile, sizeof (tfile), &menu->redraw) ==
					    -1 || !tfile[0])
    return;

  mutt_expand_path (tfile, sizeof (tfile));
  if (!mutt_confirm_func (tfile))
  {
    CLEARLINE (LINES-1);
    return;
  }
  
  mutt_message ("Copying to %s...", tfile);

  if (mx_open_mailbox (tfile, M_APPEND, &ctx) != NULL)
  {
    mutt_clear_markers (idx, menu->max);
    for (x = 0; x < menu->max ; x++)
    {
      if (idx[x]->tagged && !idx[x]->content->marked)
      {
	if (b == NULL)
	{
	  b = mutt_dup_body (idx[x]->content);
	  mutt_mark_parts (b, 1);
	  m = b;
	}
	else
	{
	  b->next = mutt_dup_body (idx[x]->content);
	  mutt_mark_parts (b, 1);
	  b = b->next;
	  /* Should tagging a multipart include all of its parts? */
	  /* b->parts = NULL; */
	}
	b->next = NULL;
      }
    }
    if (m == NULL)
    {
      m = mutt_dup_body (idx[menu->current]->content);
      m->next = NULL;
    }
    
    tmp = b = mutt_new_body ();
    b->type = TYPEMULTIPART;
    b->subtype = safe_strdup ("mixed");
    b->parts = m;

    mutt_append_decoded (Context->fp, fp, hdr, b, &ctx);

    while (m)
    {
      b = m->next;
      safe_free ((void **) &m);
      m = b;
    }
    tmp->parts = NULL;
    mutt_free_body (&tmp);
    
    mx_close_mailbox (&ctx);
    mutt_clear_error ();
  }
}

static void bounce_attachment_list (int tag, MUTTMENU *menu,
                                    ATTACHPTR **idx, HEADER *hdr)
{
  char buf[HUGE_STRING];
  char prompt[SHORT_STRING];
  ADDRESS *adr = NULL;
  int count = 0;
  int x, rc;

  /* some sanity checks first */
  if (!tag)
  {
    if (!idx[menu->current]->content->hdr ||
	idx[menu->current]->content->type != TYPEMESSAGE ||
	(strcasecmp ("rfc822", idx[menu->current]->content->subtype) != 0 &&
	strcasecmp ("news", idx[menu->current]->content->subtype) != 0))
    {
      mutt_error ("You can only bounce message/rfc822 attachments.");
      return;
    }
    else if (idx[menu->current]->content->encoding == ENCBASE64 ||
	      idx[menu->current]->content->encoding == ENCQUOTEDPRINTABLE)
    {
      mutt_error ("Bouncing encoded messages is currently unsupported.");
      return;
    }
  }
  else
  {
    int tagged = 0;
    int found = 0;

    mutt_clear_markers (idx, menu->max);
    for (x = 0; x < menu->max ; x++)
    {
      if (idx[x]->tagged && !idx[x]->content->marked)
      {
	tagged = 1;
	if (idx[x]->content->hdr &&
	    idx[x]->content->type == TYPEMESSAGE &&
	    (strcasecmp ("rfc822", idx[x]->content->subtype) == 0 ||
	    strcasecmp ("news", idx[x]->content->subtype) == 0))
	{
	  if (idx[x]->content->encoding == ENCBASE64 ||
	      idx[x]->content->encoding == ENCQUOTEDPRINTABLE)
	  {
	    mutt_error ("Bouncing encoded messages is currently unsupported.");
	    return;
	  }
	  found = 1;
	  mutt_mark_parts (idx[x]->content, 1);
	}
      }
    }
    if (!tagged)
    {
      mutt_error ("No tagged attachments!");
      return;
    }
    else if (!found)
    {
      mutt_error ("You can only bounce message/rfc822 attachments.");
      return;
    }
  }

  buf[0] = 0;
  snprintf (prompt, sizeof (prompt), "Bounce %smessage%s to: ",
					tag ? "tagged " : "", tag ? "s" : "");
  rc = ci_get_field (prompt, buf, sizeof (buf), M_ALIAS);

  if (option (OPTNEEDREDRAW))
    menu->redraw = REDRAW_FULL;
  if (rc || !buf[0])
    return;

  rfc822_parse_adrlist (&adr, buf, "@");
  adr = mutt_expand_aliases (adr);
  buf[0] = 0;
  rfc822_write_address (buf, sizeof (buf), adr);
  snprintf (prompt, sizeof (prompt), "Bounce message%s to %s...?",
					(tag ? "s" : ""), buf);
  if (mutt_yesorno (prompt, 1) != 1)
  {
    mutt_free_address (&adr);
    CLEARLINE (LINES-1);
    return;
  }

  if (!tag)
  {
    idx[menu->current]->content->hdr->msgno = hdr->msgno;
    mutt_bounce_message (idx[menu->current]->content->hdr, adr);
    mutt_message ("Message bounced.");
  }
  else
  {
    for (x = 0; x < menu->max ; x++)
    {
      if (idx[x]->content->marked)
      {
	idx[x]->content->hdr->msgno = hdr->msgno;
	mutt_bounce_message (idx[x]->content->hdr, adr);
	mutt_mark_parts (idx[x]->content, 0);
	count++;
      }
    }
    mutt_message ("%d message%s bounced.", count, (count > 1 ? "s" : ""));
  }

  mutt_free_address (&adr);
}

void mutt_view_attachments (HEADER *hdr)
{
#ifdef _PGPPATH
  char tempfile[_POSIX_PATH_MAX];
  int pgp = 0;
#endif
  char buf[STRING];
  char helpstr[SHORT_STRING];
  MUTTMENU *menu;
  HEADER *hn;
  BODY *cur;
  BODY *b = NULL, *m = NULL;
  MESSAGE *msg;
  FILE *fp;
  ATTACHPTR **idx = NULL;
  short idxlen = 0;
  short idxmax = 0;
  int tag = 0;             /* has the tag-prefix command been pressed? */
  int op = 0;
  int flags = 0;
  int x;
  
  /* make sure we have parsed this message */
  mutt_parse_mime_message (hdr);

  if ((msg = mx_open_message (Context, hdr->msgno)) == NULL)
    return;

#ifdef _PGPPATH
  if (hdr->pgp == PGPENCRYPT && hdr->content->type == TYPEMULTIPART)
  {
    STATE s;

    memset (&s, 0, sizeof (s));
    s.fpin = msg->fp;
    mutt_mktemp (tempfile);
    if ((fp = safe_fopen (tempfile, "w+")) == NULL)
    {
      mutt_perror (tempfile);
      mx_close_message (&msg);
      return;
    }
    cur = pgp_decrypt_part (hdr->content->parts->next, &s, fp);
    rewind (fp);

    pgp = 1;
  }
  else
#endif /* _PGPPATH */
  {
    fp = msg->fp;
    cur = hdr->content;
  }

  idx = mutt_gen_attach_list (cur, &idxlen, &idxmax, 0);

  menu = mutt_new_menu ();
  menu->max = idxlen;
  menu->make_entry = attach_entry;
  menu->menu = MENU_ATTACH;
  menu->title = "Attachments";
  menu->data = idx;

  helpstr[0] = 0;
  mutt_make_help (buf, sizeof (buf), "Exit  ", MENU_ATTACH, OP_EXIT);
  strcat (helpstr, buf);
  mutt_make_help (buf, sizeof (buf), "Save  ", MENU_ATTACH, OP_SAVE);
  strcat (helpstr, buf);
  mutt_make_help (buf, sizeof (buf), "Pipe  ", MENU_ATTACH, OP_PIPE);
  strcat (helpstr, buf);
  mutt_make_help (buf, sizeof (buf), "Print  ", MENU_ATTACH, OP_PRINT);
  strcat (helpstr, buf);
  mutt_make_help (buf, sizeof (buf), "Help", MENU_ATTACH, OP_HELP);
  strcat (helpstr, buf);
  menu->help = helpstr;

  FOREVER
  {
    tag = 0; /* clear the tag-prefix */
    op = mutt_menuLoop (menu);
    if (op == OP_ATTACH_TAG_PREFIX) 
    {
      CLEARLINE (LINES-1);
      addstr ("tag-");
      op = mutt_menuLoop (menu);
      if (op == OP_ATTACH_TAG_PREFIX) 
      {
	CLEARLINE (LINES-1);
	continue;
      }
      tag = 1;
    }
    else if (option (OPTAUTOTAG))
    {
      for (x = 0; x < menu->max ; x++)
      {
	if (idx[x]->tagged)
	{
	  tag = 1;
	  break;
	}
      }
    }

    switch (op)
    {
      case OP_GENERIC_SELECT_ENTRY:
      case OP_DISPLAY_HEADERS:

	if (op == OP_DISPLAY_HEADERS)
	  unset_option (OPTWEED);
	while (mutt_view_attachment (fp,
			      idx[menu->current]->content, M_REGULAR) == 2)
	{
	  if (option (OPTWEED))
	    unset_option (OPTWEED);
	  else
	    set_option (OPTWEED);
	}
	set_option (OPTWEED);		/* turn header weeding back on. */
	menu->redraw = REDRAW_FULL;
	break;

      case OP_ATTACH_VIEW_MAILCAP:

	mutt_view_attachment (fp, idx[menu->current]->content, M_MAILCAP);
	menu->redraw = REDRAW_FULL;
	break;

      case OP_ATTACH_VIEW_TEXT:

	mutt_view_attachment (fp, idx[menu->current]->content, M_AS_TEXT);
	menu->redraw = REDRAW_FULL;
	break;

      case OP_PRINT:

	mutt_print_attachment_list (fp, tag, menu, idx);
	break;

      case OP_PIPE:

	mutt_pipe_attachment_list (fp, tag, menu, idx, 0);
	break;

      case OP_SAVE:
      case OP_ATTACH_DECODE_SAVE:

	mutt_save_attachment_list (fp, tag, menu, idx,
					      op == OP_ATTACH_DECODE_SAVE);
	break;

      case OP_COPY_MESSAGE:
      case OP_ATTACH_DECODE_COPY:

	copy_attachment_list (fp, tag, menu, idx, hdr,
					      op == OP_ATTACH_DECODE_COPY);
	break;
      
      case OP_ATTACH_TAG:

	mutt_tag_attachment (tag, menu, idx);
	break;

      case OP_BOUNCE_MESSAGE:

	bounce_attachment_list (tag, menu, idx, hdr);
	break;

      case OP_REPLY:
      case OP_GROUP_REPLY:
      case OP_LIST_REPLY:
      case OP_FORWARD_MESSAGE:

#ifdef _PGPPATH
	if (hdr->pgp == PGPENCRYPT && hdr->content->type == TYPEMULTIPART)
	{
	  mutt_error (
	    "This operation is not currently supported for PGP messages.");
	  break;
	}
#endif
	if (op == OP_FORWARD_MESSAGE)
	  flags = SENDFORWARD;
	else
	  flags = SENDREPLY | 
		  (op == OP_GROUP_REPLY ? SENDGROUPREPLY : 0) |
		  (op == OP_LIST_REPLY ? SENDLISTREPLY : 0);

	if (!tag)
	{
	  b = idx[menu->current]->content;

	  if (b->hdr && b->type == TYPEMESSAGE && 
	      (strcasecmp ("rfc822", b->subtype) == 0 ||
	      strcasecmp ("news", b->subtype) == 0))
	  {
	    b->hdr->msgno = hdr->msgno;
	    ci_send_message (flags, NULL, NULL, b->hdr);
	    hdr->replied = b->hdr->replied;
	    hdr->changed = b->hdr->changed;
	  }
	  else
	  {
	    hn = mutt_dup_header (hdr);
	    hn->content = idx[menu->current]->content;
	    ci_send_message (flags, NULL, NULL, hn);
	    hdr->replied = hn->replied;
	    hdr->changed = hn->changed;
	    safe_free ((void **) &hn);
	  }
	}
	else
	{
	  /* We need to assemble a completely fake message here 
	   * Currently selected message is used to find the HEADER
	   */

	  if (op == OP_FORWARD_MESSAGE)
	  {
	    mutt_error (
		"Fowarding of tagged attachments is currently unsupported.");
	    break;
	  }

	  b = m = NULL;
	  mutt_clear_markers (idx, menu->max);
	  for (x = 0; x < menu->max ; x++)
	  {
	    if (idx[x]->tagged && !idx[x]->content->marked)
	    {
	      if (b == NULL)
	      {
		b = mutt_dup_body (idx[x]->content);
		/* avoid including nested attachments twice */
		mutt_mark_parts (b, 1);
		m = b;
	      }
	      else
	      {
		b->next = mutt_dup_body (idx[x]->content);
		/* avoid including nested attachments twice */
		mutt_mark_parts (b, 1);
		b = b->next;
		/* Should tagging a multipart include all of its parts? */
		/* b->parts = NULL; */
	      }
	      b->next = NULL;
	    }
	  }

	  if (m == NULL)
	  {
	    mutt_error ("No tagged attachments!");
	    break;
	  }
	  for (x = 0; x < menu->max ; x++)
	  {
	    if (idx[x]->tagged)
	    {
	      b = idx[x]->content;
	      break;
	    }
	  }
	  if (b->hdr &&
	      b->type == TYPEMESSAGE &&
	      (strcasecmp ("rfc822", b->subtype) == 0 ||
	      strcasecmp ("news", b->subtype) == 0))
	  {
	    hn = mutt_dup_header (b->hdr);
	    hn->msgno = hdr->msgno;
	  }
	  else
	    hn = mutt_dup_header (hdr);

	  hn->content = mutt_new_body ();
	  hn->content->type = TYPEMULTIPART;
	  hn->content->subtype = safe_strdup ("mixed");
	  hn->content->parts = m;
	  ci_send_message (flags, NULL, NULL, hn);
	  hdr->replied = hn->replied;
	  hdr->changed = hn->changed;
	  while (m)
	  {
	    b = m->next;
	    safe_free ((void **) &m);
	    m = b;
	  }
	  hn->content->parts = NULL;
	  mutt_free_body (&hn->content);
	  safe_free ((void **) &hn);
	}
	menu->redraw = REDRAW_FULL;
	break;

      case OP_EXIT:

	mx_close_message (&msg);

	while (idxlen-- > 0)
	{
	  safe_free ((void **) &idx[idxlen]->tree);
	  safe_free ((void **) &idx[idxlen]);
	}
	safe_free ((void **) &idx);
	idxmax = 0;

#ifdef _PGPPATH
	if (pgp)
	{
	  fclose (fp);
	  mutt_free_body (&cur);
	  unlink (tempfile);
	}
#endif /* _PGPPATH */

	mutt_menuDestroy  (&menu);
	return;
    }
  }

  /* not reached */
}
