/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_regex.h"
#include "rfc2047.h"
#ifdef _PGPPATH
#include "pgp.h"
#endif /* _PGPPATH */

#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <stdlib.h>

/* Reads an arbitrarily long header field, and looks ahead for continuation
 * lines.  ``line'' is dynamically increased if more space is required to fit
 * the whole line.
 */
char *mutt_read_rfc822_line (FILE *f, char *line, size_t *linelen)
{
  size_t offset = 0, l;
  char buf[LONG_STRING];
  char ch, *pbuf, *q;
  int is_cont = 0; /* continuation line? */
  int is_full;

  if (line)
    *line = 0;

  FOREVER
  {
    if (fgets (buf, sizeof (buf), f) == NULL)
      return (line);

    /* check to see if we got a full line */
    l = strlen (buf);
    is_full = (!l || buf[l - 1] == '\n');

    pbuf = buf;
    if (is_cont)
    {
      /* unfold the continuation line, as described by MIME */
      for (; ISSPACE (*pbuf); pbuf++, l--) /* empty */
	;
      if (*pbuf)
      {
	*--pbuf = ' '; /* convert to a single space */
	l++;
      }
    }
    is_cont = 0;

    /* if we got a full line, remove trailing space */
    if (l && pbuf[l - 1] == '\n')
    {
      for (q = pbuf + l - 1 ; q >= pbuf && ISSPACE (*q) ; q--, l--)
        *q = 0;
    }

    if (offset + l > *linelen)
    {
      /* not enough space to hold the (rest of the) line, grow the buffer */
      if (line)
      {
	*linelen = offset + l;
	safe_realloc ((void **) &line, *linelen + 1);
      }
      else
      {
	*linelen = l;
	line = safe_malloc (*linelen + 1);
      }
    }
    else if (!l && !line)
    {
      /* Avoid a segfault if the first line we read is blank. */
      return (NULL);
    }

    if (l)
    {
      memcpy (line + offset, pbuf, l + 1);
      offset += l;
    }

    /* check to see if we got a full line */
    if (!is_full)
      continue; /* no, read the next part */

    /* a blank line indicates the end of the message header */
    if (! *line)
      return (line);

    /* check to see if the next line is a continuation line */
    ch = fgetc (f);
    ungetc (ch, f);
    if (ch != ' ' && ch != '\t')
      return (line); /* next line is a separate header field or EOH */
    is_cont = 1;
  }
  /* not reached */
}

static LIST *mutt_parse_references (char *s)
{
  LIST *t, *lst = NULL;

  while ((s = strtok (s, " \t")) != NULL)
  {
    /*
     * some mail clients add other garbage besides message-ids, so do a quick
     * check to make sure this looks like a valid message-id
     */
    if (*s == '<')
    {
      t = (LIST *)safe_malloc (sizeof (LIST));
      t->data = safe_strdup (s);
      t->next = lst;
      lst = t;
    }
    s = NULL;
  }

  return (lst);
}

int mutt_check_encoding (const char *c)
{
  if (strncasecmp ("7bit", c, sizeof ("7bit")-1) == 0)
    return (ENC7BIT);
  else if (strncasecmp ("8bit", c, sizeof ("8bit")-1) == 0)
    return (ENC8BIT);
  else if (strncasecmp ("binary", c, sizeof ("binary")-1) == 0)
    return (ENCBINARY);
  else if (strncasecmp ("quoted-printable", c, sizeof ("quoted-printable")-1) == 0)
    return(ENCQUOTEDPRINTABLE);
  else if (strncasecmp("base64", c, sizeof("base64")-1) == 0)
    return(ENCBASE64);
  else
    return(ENCOTHER);
}

static PARAMETER *parse_parameters (const char *s)
{
  PARAMETER *head = 0, *cur = 0, *new;
  char buffer[LONG_STRING];
  const char *p;
  size_t i;

  while (*s)
  {
    if ((p = strchr (s, '=')) == NULL)
    {
      dprint(1, (debugfile, "parse_parameters: malformed parameter: %s\n", s));
      return (head); /* just bail out now */
    }

    i = p - s;

    new = mutt_new_parameter ();

    new->attribute = safe_malloc (i + 1);
    memcpy (new->attribute, s, i);
    new->attribute[i] = 0;

    s = p + 1; /* skip over the = */
    SKIPWS (s);

    if (*s == '"')
    {
      s++;
      for (i=0; *s && *s != '"' && i < sizeof (buffer) - 1; i++, s++)
      {
        if (*s == '\\')
	{
          /* Quote the next character */
          buffer[i] = s[1];
	  if (!*++s)
	    break;
        }
        else
          buffer[i] = *s;
      }
      buffer[i] = 0;
      if (*s)
	s++; /* skip over the " */
    }
    else
    {
      for (i=0; *s && *s != ' ' && *s != ';' && i < sizeof (buffer) - 1; i++, s++)
        buffer[i] = *s;
      buffer[i] = 0;
    }

    new->value = safe_strdup (buffer);

    /* Add this parameter to the list */
    if (head)
    {
      cur->next = new;
      cur = cur->next;
    }
    else
      head = cur = new;

    /* Find the next parameter */
    if (*s == ';')
      s++;
    else
    {
      while (*s && *s != ';')
	s++;
      if (!*s)
	break; /* no more parameters! */
      s++;
    }

    /* Move past any leading whitespace */
    SKIPWS (s);

    /* skip empty parameters */
    while (*s == ';')
    {
      s++;
      SKIPWS (s);
    }
  }    

  return (head);
}

int mutt_check_mime_type (const char *s)
{
  if (strcasecmp ("text", s) == 0)
    return TYPETEXT;
  else if (strcasecmp ("multipart", s) == 0)
    return TYPEMULTIPART;
  else if (strcasecmp ("application", s) == 0)
    return TYPEAPPLICATION;
  else if (strcasecmp ("message", s) == 0)
    return TYPEMESSAGE;
  else if (strcasecmp ("image", s) == 0)
    return TYPEIMAGE;
  else if (strcasecmp ("audio", s) == 0)
    return TYPEAUDIO;
  else if (strcasecmp ("video", s) == 0)
    return TYPEVIDEO;
  else
    return TYPEOTHER;
}

static void parse_content_type (char *s, BODY *ct)
{
  char *pc;
  char buffer[SHORT_STRING];
  short i = 0;

  safe_free((void **)&ct->subtype);
  mutt_free_parameter(&ct->parameter);

  /* First extract any existing parameters */
  if ((pc = strchr(s, ';')) != NULL)
  {
    *pc++ = 0;
    while (*pc && whitespace(*pc)) pc++;
    ct->parameter = parse_parameters(pc);

    /* Some pre-RFC1521 gateways still use the "name=filename" convention */
    if ((pc = mutt_get_parameter("name", ct->parameter)) != 0)
      ct->filename = safe_strdup(pc);
  }
  
  /* Now get the subtype */
  if ((pc = strchr(s, '/')))
  {
    *pc++ = 0;
    while (*pc && !whitespace (*pc) && *pc != ';')
    {
      buffer[i++] = *pc;
      pc++;
    }
    buffer[i] = 0;
    ct->subtype = safe_strdup (buffer);
  }

  /* Finally, get the major type */
  ct->type = mutt_check_mime_type (s);

  if (ct->subtype == NULL)
  {
    /* Some older non-MIME mailers (i.e., mailtool, elm) have a content-type
     * field, so we can attempt to convert the type to BODY here.
     */
    if (ct->type == TYPETEXT)
      ct->subtype = safe_strdup ("plain");
    else if (ct->type == TYPEAUDIO)
      ct->subtype = safe_strdup ("basic");
    else if (ct->type == TYPEMESSAGE)
      ct->subtype = safe_strdup ("rfc822");
    else if (ct->type == TYPEOTHER)
    {
      ct->type = TYPEAPPLICATION;
      snprintf (buffer, sizeof (buffer), "x-%s", s);
      ct->subtype = safe_strdup (buffer);
    }
    else
      ct->subtype = safe_strdup ("x-unknown");
  }
}

static void parse_content_disposition (char *s, BODY *ct)
{
  PARAMETER *parms;

  if (!strncasecmp ("inline", s, 6))
    ct->disposition = DISPINLINE;
  else if (!strncasecmp ("form-data", s, 9))
    ct->disposition = DISPFORMDATA;
  else
    ct->disposition = DISPATTACH;

  /* Check to see if a default filename was given */
  if ((s = strchr (s, ';')) != NULL)
  {
    s++;
    SKIPWS (s);
    if ((s = mutt_get_parameter ("filename", (parms = parse_parameters (s)))) != 0)
    {
      /* free() here because the content-type parsing routine might
       * have allocated space if a "name=filename" parameter was
       * specified.
       */
      safe_free ((void **) &ct->filename);
      ct->filename = safe_strdup (s); 
    }
    if ((s = mutt_get_parameter ("name", parms)) != 0)
      ct->form_name = safe_strdup (s);
    mutt_free_parameter (&parms);
  }
}

/* args:
 *	fp	stream to read from
 *
 *	digest	1 if reading subparts of a multipart/digest, 0
 *		otherwise
 */

BODY *mutt_read_mime_header (FILE *fp, int digest)
{
  BODY *p = mutt_new_body();
  char *c, *line = NULL;
  size_t linelen = 0;
  
  p->hdr_offset = ftell(fp);

  p->encoding = ENC7BIT; /* default from RFC1521 */
  p->type = digest ? TYPEMESSAGE : TYPETEXT;

  while ((line = mutt_read_rfc822_line (fp, line, &linelen)) && *line != 0)
  {
    /* Find the value of the current header */
    if ((c = strchr (line, ':')))
    {
      c++;
      SKIPWS (c);
      if (!*c)
      {
	dprint (1, (debugfile, "mutt_read_mime_header(): skipping empty header field: %s\n", line));
	continue;
      }
    }
    else
    {
      dprint (1, (debugfile, "read_mime_header: bogus MIME header: %s\n", line));
      break;
    }

    switch (tolower (line[0]))
    {
      case 'c':

	if (!strncasecmp ("ontent-", line + 1, 7))
	{
	  if (!strncasecmp ("type", line + 8, 4))
	    parse_content_type (c, p);
	  else if (!strncasecmp ("transfer-encoding", line + 8, 17))
	    p->encoding = mutt_check_encoding (c);
	  else if (!strncasecmp ("disposition", line + 8, 11))
	    parse_content_disposition (c, p);
	  else if (!strncasecmp ("description", line + 8, 11))
	  {
	    safe_free ((void **) &p->description);
	    p->description = safe_strdup (c);
	    rfc2047_decode (p->description, p->description, strlen (p->description) + 1);
	  }
	}
	break;
    }
  }
  p->offset = ftell (fp); /* Mark the start of the real data */
  if (p->type == TYPETEXT && !p->subtype)
    p->subtype = safe_strdup ("plain");
  else if (p->type == TYPEMESSAGE && !p->subtype)
    p->subtype = safe_strdup ("rfc822");

  safe_free ((void **) &line);

  return (p);
}

/* parse a MESSAGE/RFC822 body
 *
 * args:
 *	fp		stream to read from
 *
 *	parent		structure which contains info about the message/rfc822
 *			body part
 *
 * NOTE: this assumes that `parent->length' has been set!
 */

BODY *mutt_parse_messageRFC822 (FILE *fp, BODY *parent)
{
  BODY *msg;

  parent->hdr = mutt_new_header();
  parent->hdr->offset = ftell(fp);
  parent->hdr->env = mutt_read_rfc822_header (fp, parent->hdr);
  msg = parent->hdr->content;

  if (msg->length == -1)
    msg->length = parent->length - (msg->offset - parent->offset);

  /* if body of this message is empty, we can end up with a negative length */
  if (msg->length < 0)
    msg->length = 0;

  if (msg->type == TYPEMULTIPART)
    msg->parts = mutt_parse_multipart (fp, mutt_get_parameter ("boundary", msg->parameter), msg->offset + msg->length, strcasecmp ("digest", msg->subtype) == 0);
  else if (msg->type == TYPEMESSAGE)
    msg->parts = mutt_parse_messageRFC822 (fp, msg);
  else
    return (msg);
  
  /* try to recover from parsing error */
  if (!msg->parts)
  {
    msg->type = TYPETEXT;
    safe_free ((void **) &msg->subtype);
    msg->subtype = safe_strdup ("plain");
  }

  return (msg);
}

/* parse a multipart structure
 *
 * args:
 *	fp		stream to read from
 *
 *	boundary	body separator
 *
 *	end_off		length of the multipart body (used when the final
 *			boundary is missing to avoid reading too far)
 *
 *	digest		1 if reading a multipart/digest, 0 otherwise
 */

BODY *mutt_parse_multipart (FILE *fp, const char *boundary, long end_off, int digest)
{
  int blen, len, crlf = 0;
  char buffer[LONG_STRING];
  BODY *head = 0, *last = 0, *new = 0;
  int i;
  int final = 0; /* did we see the ending boundary? */

  if (!boundary)
  {
    mutt_error ("multipart message has no boundary parameter!");
    return (NULL);
  }

  blen = strlen (boundary);
  while (ftell (fp) < end_off && fgets (buffer, LONG_STRING, fp) != NULL)
  {
    len = strlen (buffer);

    /* take note of the line ending.  I'm assuming that either all endings
     * will use <CR><LF> or none will.
     */
    if (len > 1 && buffer[len - 2] == '\r')
      crlf = 1;

    if (buffer[0] == '-' && buffer[1] == '-' &&
	strncmp (buffer + 2, boundary, blen) == 0)
    {
      if (last)
      {
	last->length = ftell (fp) - last->offset - len - 1 - crlf;
	if (last->parts && last->parts->length == 0)
	  last->parts->length = ftell (fp) - last->parts->offset - len - 1 - crlf;
	/* if the body is empty, we can end up with a -1 length */
	if (last->length < 0)
	  last->length = 0;
      }

      /* Remove any trailing whitespace, up to the length of the boundary */
      for (i = len - 1; whitespace (buffer[i]) && i >= blen + 2; i--)
        buffer[i] = 0;

      /* Check for the end boundary */
      if (strcmp (buffer + blen + 2, "--") == 0)
      {
	final = 1;
	break; /* done parsing */
      }
      else if (buffer[2 + blen] == 0)
      {
	new = mutt_read_mime_header (fp, digest);
	if (head)
	{
	  last->next = new;
	  last = new;
	}
	else
	  last = head = new;
      }
    }
  }

  /* in case of missing end boundary, set the length to something reasonable */
  if (last && last->length == 0 && !final)
    last->length = end_off - last->offset;

  /* parse recursive MIME parts */
  for (last = head; last; last = last->next)
  {
    switch (last->type)
    {
      case TYPEMULTIPART:
	fseek (fp, last->offset, 0);
	last->parts = mutt_parse_multipart (fp, mutt_get_parameter ("boundary", last->parameter), last->offset + last->length, strcasecmp ("digest", last->subtype) == 0);
	break;

      case TYPEMESSAGE:
	if (last->subtype &&
	    (strcasecmp (last->subtype, "rfc822") == 0 ||
	     strcasecmp (last->subtype, "news") == 0))
	{
	  fseek (fp, last->offset, 0);
	  last->parts = mutt_parse_messageRFC822 (fp, last);
	}
	break;
    }
  }

  return (head);
}

static const char *uncomment_timezone (char *buf, size_t buflen, const char *tz)
{
  char *p;
  size_t len;

  if (*tz != '(')
    return tz; /* no need to do anything */
  tz++;
  SKIPWS (tz);
  if ((p = strpbrk (tz, " )")) == NULL)
    return tz;
  len = p - tz;
  if (len > buflen - 1)
    len = buflen - 1;
  memcpy (buf, tz, len);
  buf[len] = 0;
  return buf;
}

static struct tz_t
{
  char *tzname;
  unsigned char zhours;
  unsigned char zminutes;
  unsigned char zoccident; /* west of UTC? */
  unsigned char xxx;       /* unused */
}
TimeZones[] =
{
  { "sst",  11,  0, 1, 0 }, /* Samoa */
  { "pst",   8,  0, 1, 0 },
  { "mst",   7,  0, 1, 0 },
  { "pdt",   7,  0, 1, 0 },
  { "cst",   6,  0, 1, 0 },
  { "mdt",   6,  0, 1, 0 },
  { "cdt",   5,  0, 1, 0 },
  { "est",   5,  0, 1, 0 },
  { "ast",   4,  0, 1, 0 }, /* Atlantic */
  { "edt",   4,  0, 1, 0 },
  { "wgt",   3,  0, 1, 0 }, /* Western Greenland */
  { "wgst",  2,  0, 1, 0 }, /* Western Greenland DST */
  { "aat",   1,  0, 1, 0 }, /* Atlantic Africa Time */
  { "egt",   1,  0, 1, 0 }, /* Eastern Greenland */
  { "egst",  0,  0, 0, 0 }, /* Eastern Greenland DST */
  { "gmt",   0,  0, 0, 0 },
  { "utc",   0,  0, 0, 0 },
  { "wat",   0,  0, 0, 0 }, /* West Africa */
  { "wet",   0,  0, 0, 0 }, /* Western Europe */
  { "bst",   1,  0, 0, 0 }, /* British DST */
  { "cat",   1,  0, 0, 0 }, /* Central Africa */
  { "cet",   1,  0, 0, 0 }, /* Central Europe */
  { "met",   1,  0, 0, 0 }, /* this is now officially CET */
  { "west",  1,  0, 0, 0 }, /* Western Europe DST */
  { "cest",  2,  0, 0, 0 }, /* Central Europe DST */
  { "eet",   2,  0, 0, 0 }, /* Eastern Europe */
  { "ist",   2,  0, 0, 0 }, /* Israel */
  { "sat",   2,  0, 0, 0 }, /* South Africa */
  { "ast",   3,  0, 0, 0 }, /* Arabia */
  { "eat",   3,  0, 0, 0 }, /* East Africa */
  { "eest",  3,  0, 0, 0 }, /* Eastern Europe DST */
  { "idt",   3,  0, 0, 0 }, /* Israel DST */
  { "msk",   3,  0, 0, 0 }, /* Moscow */
  { "adt",   4,  0, 0, 0 }, /* Arabia DST */
  { "msd",   4,  0, 0, 0 }, /* Moscow DST */
  { "gst",   4,  0, 0, 0 }, /* Presian Gulf */
  { "smt",   4,  0, 0, 0 }, /* Seychelles */
  { "ist",   5, 30, 0, 0 }, /* India */
  { "ict",   7,  0, 0, 0 }, /* Indochina */
/*{ "cst",   8,  0, 0, 0 },*/ /* China */
  { "hkt",   8,  0, 0, 0 }, /* Hong Kong */
/*{ "sst",   8,  0, 0, 0 },*/ /* Singapore */
  { "wst",   8,  0, 0, 0 }, /* Western Australia */
  { "jst",   9,  0, 0, 0 }, /* Japan */
/*{ "cst",   9, 30, 0, 0 },*/ /* Australian Central Standard Time */
  { "kst",  10,  0, 0, 0 }, /* Korea */
  { "nzst", 12,  0, 0, 0 }, /* New Zealand */
  { "nzdt", 13,  0, 0, 0 }, /* New Zealand DST */
  { NULL,    0,  0, 0, 0 }
};

/* parses a date string in RFC822 format:
 *
 * Date: [ weekday , ] day-of-month month year hour:minute:second timezone
 *
 * This routine assumes that `h' has been initialized to 0.  the `timezone'
 * field is optional, defaulting to +0000 if missing.
 */
static int parse_date (char *s, HEADER *h)
{
  int count = 0;
  char *t;
  int hour, min, sec;
  struct tm tm;
  int i;
  int tz_offset = 0;
  const char *ptz;
  char tzstr[SHORT_STRING];

  /* kill the day of the week, if it exists. */
  if ((t = strchr (s, ',')) != 0)
  {
    t++;
    SKIPWS (t);
  }
  else
    t = s;

  memset (&tm, 0, sizeof (tm));

  while ((t = strtok (t, " \t")) != NULL)
  {
    switch (count)
    {
      case 0: /* day of the month */
	if (!isdigit (*t))
	  return (-1);
	tm.tm_mday = atoi (t);
	if (tm.tm_mday > 31)
	  return (-1);
	break;

      case 1: /* month of the year */
	if ((i = mutt_check_month (t)) < 0)
	  return (-1);
	tm.tm_mon = i;
	break;

      case 2: /* year */
	tm.tm_year = atoi (t);
	if (tm.tm_year >= 1900)
	  tm.tm_year -= 1900;
	break;

      case 3: /* time of day */
	if (sscanf (t, "%d:%d:%d", &hour, &min, &sec) == 3)
	  ;
	else if (sscanf (t, "%d:%d", &hour, &min) == 2)
	  sec = 0;
	else
	{
	  dprint(1, (debugfile, "parse_date: could not process time format: %s\n", t));
	  return(-1);
	}
	tm.tm_hour = hour;
	tm.tm_min = min;
	tm.tm_sec = sec;
	break;

      case 4: /* timezone */

	/* sometimes we see things like (MST) or (-0700) so attempt to
	 * compensate by uncommenting the string if non-RFC822 compliant
	 */
	ptz = uncomment_timezone (tzstr, sizeof (tzstr), t);

	if (*ptz == '+' || *ptz == '-')
	{
	  if (ptz[1] && ptz[2] && ptz[3] && ptz[4])
	  {
	    h->zhours = (ptz[1] - '0') * 10 + (ptz[2] - '0');
	    h->zminutes = (ptz[3] - '0') * 10 + (ptz[4] - '0');

	    if (ptz[0] == '-')
	      h->zoccident = 1;
	  }
	}
	else
	{
	  for (i = 0; TimeZones[i].tzname; i++)
	    if (!strcasecmp (TimeZones[i].tzname, ptz))
	    {
	      h->zhours = TimeZones[i].zhours;
	      h->zminutes = TimeZones[i].zminutes;
	      h->zoccident = TimeZones[i].zoccident;
	      break;
	    }

	  /* ad hoc support for the European MET (now officially CET) TZ */
	  if (strcasecmp (t, "MET") == 0)
	  {
	    if ((t = strtok (NULL, " \t")) != NULL)
	    {
	      if (!strcasecmp (t, "DST"))
		h->zhours++;
	    }
	  }
	}
	tz_offset = h->zhours * 3600 + h->zminutes * 60;
	if (!h->zoccident)
	  tz_offset = -tz_offset;
	break;
    }
    count++;
    t = 0;
  }

  if (count < 4) /* don't check for missing timezone */
  {
    dprint(1,(debugfile, "parse_date(): error parsing date format, using received time\n"));
    return (-1);
  }

  h->date_sent = mutt_mktime (&tm) + tz_offset;

  return 0;
}

/* extract the first substring that looks like a message-id */
static char *extract_message_id (const char *s)
{
  const char *p;
  char *r;
  size_t l;

  if ((s = strchr (s, '<')) == NULL || (p = strchr (s, '>')) == NULL)
    return (NULL);
  l = (size_t)(p - s) + 1;
  r = safe_malloc (l + 1);
  memcpy (r, s, l);
  r[l] = 0;
  return (r);
}

void mutt_parse_mime_message (HEADER *cur)
{
  MESSAGE *msg;

  if (cur->content->type != TYPEMESSAGE && cur->content->type != TYPEMULTIPART)
    return; /* nothing to do */

  if ((msg = mx_open_message (Context, cur->msgno)))
  {
    fseek (msg->fp, cur->content->offset, 0);

    if (cur->content->type == TYPEMULTIPART)
    {
      if (!cur->content->parts)
	cur->content->parts = mutt_parse_multipart (msg->fp, mutt_get_parameter ("boundary", cur->content->parameter), cur->content->offset + cur->content->length, strcasecmp ("digest", cur->content->subtype) == 0);
    }
    else
    {
      if (!cur->content->parts)
	cur->content->parts = mutt_parse_messageRFC822 (msg->fp, cur->content);
    }
      

    /* try to recover from parsing error */
    if (!cur->content->parts)
    {
      cur->content->type = TYPETEXT;
      safe_free ((void **) &cur->content->subtype);
      cur->content->subtype = safe_strdup ("plain");
    }
    
#ifdef _PGPPATH
    cur->pgp = pgp_query (cur->content);
#endif /* _PGPPATH */
    mx_close_message (&msg);
  }
}

/* mutt_read_rfc822_header() -- parses a RFC822 header
 *
 * Args:
 *
 * f		stream to read from
 *
 * hdr		header structure of current message (optional).  If hdr is
 *		NULL, then we are reading a postponed message, or called
 *		from mutt_edit_headers() so we should keep a list of the
 *		user-defined headers.
 *
 * keep		1 to add the "mute" headers to the "userhdrs" list;
 *		used when building new headers for decode-save
 *
 */
ENVELOPE *_mutt_read_rfc822_header (FILE *f, HEADER *hdr, int keep)
{
  ENVELOPE *e = mutt_new_envelope();
  LIST *last = NULL;
  char *line = NULL, *p;
  char in_reply_to[SHORT_STRING] = "";
  long loc;
  int matched;
  int date_found = 0;
  size_t linelen = 0;

  if (hdr)
  {
    hdr->content = mutt_new_body ();

    /* set the defaults from RFC1521 */
    hdr->content->type = TYPETEXT;
    hdr->content->subtype = safe_strdup ("plain");
    hdr->content->encoding = ENC7BIT;
    hdr->content->length = -1;
  }

  loc = ftell (f);
  while ((line = mutt_read_rfc822_line (f, line, &linelen)) != NULL &&
	 *line != 0)
  {
    matched = 0;

    if ((p = strpbrk (line, ": \t")) == NULL || *p != ':')
    {
      char return_path[LONG_STRING] = { 0 };
      time_t t;

      /* some bogus MTAs will quote the original "From " line */
      if (strncmp (">From ", line, 6) == 0)
      {
	loc = ftell (f);
	continue; /* just ignore */
      }
      else if ((t = is_from (line, return_path, sizeof (return_path))))
      {
	/* MH somtimes has the From_ line in the middle of the header! */
	if (hdr && !hdr->received)
	  hdr->received = t;
	loc = ftell (f);
	continue;
      }

      fseek (f, loc, 0);
      break; /* end of header */
    }

    p++;
    SKIPWS (p);

    switch (tolower (line[0]))
    {
      case 'a':
	if (strncasecmp (line+1, "pparently-to:", 13) == 0)
	{
	  rfc822_parse_adrlist (&e->to, p, "@");
	  matched = 1;
	}
	else if (strncasecmp (line+1, "pparently-from:", 15) == 0)
	{
	  rfc822_parse_adrlist (&e->from, p, "@");
	  matched = 1;
	}
	break;

      case 'b':
	if (strncasecmp (line+1, "cc:", 3) == 0)
	{
	  rfc822_parse_adrlist (&e->bcc, p, "@");
	  matched = 1;
	}
	break;

      case 'c':
	if (strncasecmp (line+1, "c:", 2) == 0)
	{
	  rfc822_parse_adrlist (&e->cc, p, "@");
	  matched = 1;
	}
	else if (strncasecmp (line + 1, "ontent-", 7) == 0)
	{
	  if (strncasecmp (line+8, "type:", 5) == 0)
	  {
	    if (hdr && !keep)
	      parse_content_type (p, hdr->content);
	    matched = 1;
	  }
	  else if (strncasecmp (line+8, "transfer-encoding:", 18) == 0)
	  {
	    if (hdr && !keep)
	      hdr->content->encoding = mutt_check_encoding (p);
	    matched = 1;
	  }
	  else if (strncasecmp (line+8, "length:", 7) == 0)
	  {
	    if (hdr && !keep)
	      hdr->content->length = atoi (p);
	    matched = 1;
	  }
	  else if (strncasecmp (line+8, "description:", 12) == 0)
	  {
	    if (hdr && !keep)
	    {
	      safe_free ((void **) &hdr->content->description);
	      hdr->content->description = safe_strdup (p);
	      rfc2047_decode (hdr->content->description,
			      hdr->content->description,
			      strlen (hdr->content->description) + 1);
	    }
	    matched = 1;
	  }
	  else if (strncasecmp (line+8, "disposition:", 5) == 0)
	  {
	    if (hdr && !keep)
	      parse_content_disposition (p, hdr->content);
	    matched = 1;
	  }
	}
	break;

      case 'd':
	if (!strncasecmp ("ate:", line + 1, 4))
	{
	  if (hdr && parse_date (p, hdr) == 0)
	    date_found = 1;
	}
	break;

      case 'f':
	if (!strncasecmp ("rom:", line + 1, 4))
	{
	  rfc822_parse_adrlist (&e->from, p, "@");
	  matched = 1;
	}
	break;

      case 'i':
	if (strncasecmp (line+1, "n-reply-to:", 11) == 0)
	{
	  if (hdr)
	    strfcpy (in_reply_to, p, sizeof (in_reply_to));
	}
	break;

      case 'l':
	if (strncasecmp (line+1, "ines:", 5) == 0)
	{
	  if (hdr && !keep)
	    hdr->lines = atoi(p);
	  matched = 1;
	}
	break;

      case 'm':
	if (strncasecmp (line+1, "ime-version:", 12) == 0)
	{
	  if (hdr)
	    hdr->mime = 1;
	  matched = 1;
	}
	else if (strncasecmp (line+1, "essage-id:", 10) == 0)
	{
	  if (!keep)
	  {
	    /* We add a new "Message-Id:" when building a message */
	    safe_free ((void **)&e->message_id);
	    e->message_id = extract_message_id (p);
	  }
	  matched = 1;
	}
	break;

      case 'r':
	if (strncasecmp (line+1, "eferences:", 10) == 0)
	{
	  mutt_free_list (&e->references);
	  e->references = mutt_parse_references (p);
	  matched = 1;
	}
	else if (strncasecmp (line+1, "eply-to:", 8) == 0)
	{
	  rfc822_parse_adrlist (&e->reply_to, p, "@");
	  matched = 1;
	}
	else if (strncasecmp (line+1, "eturn-path:", 11) == 0)
	{
	  if (hdr) rfc822_parse_adrlist (&e->return_path, p, "@");
	  matched = 1;
	}
	break;

      case 's':
	if (strncasecmp (line+1, "ubject:", 7) == 0)
	{
	  if (!e->subject) e->subject = safe_strdup (p);
	  matched = 1;
	}
	else if (strncasecmp (line+1, "ender:", 6) == 0)
	{
	  rfc822_parse_adrlist (&e->sender, p, "@");
	  matched = 1;
	}
	else if (strncasecmp (line+1, "tatus:", 6) == 0)
	{
	  if (hdr)
	  {
	    while (*p)
	    {
	      switch(*p)
	      {
		case 'r':
		  hdr->replied = 1;
		  break;
		case 'O':
		  if (option (OPTMARKOLD))
		    hdr->old = 1;
		  break;
		case 'R':
		  hdr->read = 1;
		  break;
	      }
	      p++;
	    }
	  }
	  matched = 1;
	}
	break;

      case 't':
	if (strncasecmp (line+1, "o:", 2) == 0)
	{
	  rfc822_parse_adrlist (&e->to, p, "@");
	  matched = 1;
	}
	break;

      case 'x':
	if (strncasecmp (line+1, "-status:", 8) == 0)
	{
	  if (hdr)
	  {
	    while (*p)
	    {
	      switch (*p)
	      {
		case 'A':
		  hdr->replied = 1;
		  break;
		case 'D':
		  hdr->deleted = 1;
		  break;
		case 'F':
		  hdr->flagged = 1;
		  break;
		default:
		  break;
	      }
	      p++;
	    }
	  }
	  matched = 1;
	}
	    
	if (hdr && keep && strncasecmp (line+1, "-mailer:", 8) != 0)
	{
	  /* If we are building a new message we want to save all
	   * relevant headers in the "userhdrs" list
	   */
	  LIST *tmp = e->userhdrs;

	  while (tmp && tmp->next)
	    tmp = tmp->next;
	  if (tmp)
	  {
	    tmp->next = mutt_new_list();
	    tmp = tmp->next;
	  }
	  else
	    e->userhdrs = tmp = mutt_new_list();

	  tmp->data = safe_strdup (line);
	}
	break;

      default:
	break;
    }

    /* if hdr==NULL, then we are using this to parse either a postponed
     * message, or a outgoing message (edit_hdrs), so we want to keep
     * track of the user-defined headers
     */
    if (!matched && !hdr)
    {
      if (last)
      {
	last->next = mutt_new_list ();
	last = last->next;
      }
      else
	last = e->userhdrs = mutt_new_list ();
      last->data = safe_strdup (line);
    }

    loc = ftell (f);
  }

  safe_free ((void **) &line);

  if (hdr)
  {
    hdr->content->hdr_offset = hdr->offset;
    hdr->content->offset = ftell (f);

    /* if an in-reply-to was given, check to see if it is in the references
     * list already.  if not, add it so we can do a better job of threading.
     */
    if (!keep && in_reply_to[0] &&
		    (p = extract_message_id (in_reply_to)) != NULL)
    {
      if (!e->references ||
	  (e->references && strcmp (e->references->data, p) != 0))
      {
	LIST *tmp = mutt_new_list ();

	tmp->data = p;
	tmp->next = e->references;
	e->references = tmp;
      }
      else
	safe_free ((void **) &p);
    }

    /* do RFC2047 decoding */
    rfc2047_decode_adrlist (e->from);
    rfc2047_decode_adrlist (e->to);
    rfc2047_decode_adrlist (e->cc);
    rfc2047_decode_adrlist (e->reply_to);
    rfc2047_decode_adrlist (e->return_path);
    rfc2047_decode_adrlist (e->sender);

    if (e->subject)
    {
      rfc2047_decode (e->subject, e->subject, strlen (e->subject) + 1);

      if (!keep)
      {
	regmatch_t pmatch[1];

	if (regexec ((regex_t *) ReplyRegexp.rx, e->subject, 1, pmatch, 0) == 0)
	  e->real_subj = e->subject + pmatch[0].rm_eo;
	else
	  e->real_subj = e->subject;
      }
    }

    /* Check for missing or invalid date.  If this is the case, convert the
     * date the message was received to UTC and use that value.
     */
    if (!date_found)
    {
      dprint(1,(debugfile,"read_rfc822_header(): no date found, using received time from msg separator\n"));
      hdr->date_sent = hdr->received;
    }
  }

  return (e);
}

void mutt_parse_adrlist (ADDRESS **p, const char *s, const char *host)
{
  const char *q;

  /* check for a simple whitespace separated list of addresses */
  if ((q = strpbrk (s, "\"<>():;,\\")) == NULL)
  {
    char tmp[HUGE_STRING];
    char *r;

    strfcpy (tmp, s, sizeof (tmp));
    r = tmp;
    while ((r = strtok (r, " \t")) != NULL)
    {
      rfc822_parse_adrlist (p, r, host);
      r = NULL;
    }
  }
  else
    rfc822_parse_adrlist (p, s, host);
}
