/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "mutt_regex.h"
#include "init.h"

#include <pwd.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/utsname.h>
#include <errno.h>
#include <sys/wait.h>

void set_quadoption (int opt, int flag)
{
  QuadOptions &= ~(0x3 << (2 * opt)); /* first clear the bits */
  QuadOptions |= (flag & 0x3) << (2 * opt); /* now set them */
}

int quadoption (int opt)
{
  return ((QuadOptions >> (opt * 2)) & 0x3);
}

int query_quadoption (int opt, const char *prompt)
{
  int v = quadoption (opt);

  switch (v)
  {
    case M_YES:
    case M_NO:

      return (v);

    default:

      v = mutt_yesorno (prompt, (v == M_ASKYES));
      CLEARLINE (LINES - 1);
      return (v);
  }

  /* not reached */
}

/* given the variable ``s'', return the index into the rc_vars array which
 * matches, or -1 if the variable is not found.
 */
int mutt_option_index (char *s)
{
  int i;

  for (i=0; MuttVars[i].option; i++)
    if (strcmp (s, MuttVars[i].option) == 0)
      return (i);
  return (-1);
}

void remove_from_list (LIST **l, const char *s)
{
  size_t len;
  LIST *last = NULL;
  LIST *p;
  const char *q;

  do
  {
    if ((q = strpbrk (s, " \t#")))
      len = q - s;
    else
      len = strlen (s);

    if (strncmp ("*", s, len) == 0)
      mutt_free_list (l);    /* ``unCMD *'' means delete all current entries */
    else
    {
      p = *l;
      last = NULL;
      while (p)
      {
	if (strncasecmp (s, p->data, len) == 0 && strlen (p->data) == len)
	{
	  safe_free ((void **) &p->data);
	  if (last)
	    last->next = p->next;
	  else
	    (*l) = p->next;
	  safe_free ((void **) &p);
	}
	else
	{
	  last = p;
	  p = p->next;
	}
      }
    }

    if (q)
      SKIPWS (q);
    s = q;
  }
  while ((s = q) != NULL && *s && *s != '#');
}

static int parse_unignore (const char *s, void *data, char *err, size_t errlen)
{
  mutt_add_to_list (&UnIgnore, s);
  remove_from_list (&Ignore, s);
  return 0;
}

static int parse_ignore (const char *s, void *data, char *err, size_t errlen)
{
  mutt_add_to_list (&Ignore, s);
  remove_from_list (&UnIgnore, s);
  return 0;
}

static int parse_unalias (const char *s, void *data, char *err, size_t errlen)
{
  ALIAS *tmp;
  ALIAS *last = NULL;
  size_t l;
  const char *p;

  do
  {
    if ((p = strpbrk (s, " \t#")))
      l = p - s;
    else
      l = strlen (s);

    tmp = Aliases;
    while (tmp)
    {
      if (strncasecmp (s, tmp->name, l) == 0 && tmp->name[l] == 0)
      {
	if (last)
	  last->next = tmp->next;
	else
	  Aliases = tmp->next;
	tmp->next = NULL;
	mutt_free_alias (&tmp);
	break;
      }
      last = tmp;
      tmp = tmp->next;
    }

    if (p)
      SKIPWS (p);
  }
  while ((s = p) != NULL && *s && *s != '#');

  return 0;
}

static char *reap_comment (const char *p)
{
  char *ptr;
  char *rptr;
  char *wptr;

  /* make a copy and reap comments */

  ptr = safe_strdup (p);
  rptr = ptr;
  wptr = ptr;

  while (*rptr)
  {
    if (*rptr == '\\')
    {
      rptr++;
      *wptr++ = *rptr;
      if (!*rptr)
	break;
      rptr++;
    }
    else if (*rptr == '#')
      break;
    else
      *wptr++ = *rptr++;
  }
  *wptr = 0;

  return ptr;
}

static int parse_alias (const char *s, void *data, char *errmsg, size_t errlen)
{
  ALIAS *tmp = Aliases;
  ALIAS *last = NULL;
  const char *p;
  char *ptr;
  size_t len;

  if ((p = strpbrk (s, " \t")) == NULL)
  {
    strfcpy (errmsg, "alias has no address", errlen);
    return (-1);
  }

  len = p - s;

  /* check to see if an alias with this name already exists */
  while (tmp)
  {
    if (strncasecmp (tmp->name, s, len) == 0 && *(tmp->name + len) == 0)
      break;
    last = tmp;
    tmp = tmp->next;
  }

  if (!tmp)
  {
    /* create a new alias */
    tmp = (ALIAS *) safe_calloc (1, sizeof (ALIAS));
    tmp->name = safe_malloc (len + 1);
    memcpy (tmp->name, s, len);
    tmp->name[len] = 0;
  }
  else
  {
    /* override the previous value */
    mutt_free_address (&tmp->addr);
  }

  ptr = reap_comment (p);

  rfc822_parse_adrlist (&tmp->addr, ptr, "@");

  free (ptr);

  if (last)
    last->next = tmp;
  else
    Aliases = tmp;

  return 0;
}

static int parse_unmy_hdr (const char *s, void *data, char *err, size_t errlen)
{
  LIST *last = NULL;
  LIST *tmp = UserHeader;
  LIST *ptr;
  size_t l;
  const char *p;

  do
  {
    if ((p = strpbrk (s, " \t#")))
      l = p - s;
    else
      l = strlen (s);

    if (strncmp ("*", s, l) == 0 && s[l] == 0)
      mutt_free_list (&UserHeader);
    else
    {
      tmp = UserHeader;
      last = NULL;

      if (s[l-1] == ':')
	l--;

      while (tmp)
      {
	if (strncasecmp (s, tmp->data, l) == 0 && tmp->data[l] == ':')
	{
	  ptr = tmp;
	  if (last)
	    last->next = tmp->next;
	  else
	    UserHeader = tmp->next;
	  tmp = tmp->next;
	  ptr->next = NULL;
	  mutt_free_list (&ptr);
	}
	else
	{
	  last = tmp;
	  tmp = tmp->next;
	}
      }
    }

    if (p)
      SKIPWS (p);
  }
  while ((s = p) != NULL && *s && *s != '#');

  return 0;
}

static int parse_my_hdr (const char *s, void *data, char *errmsg, size_t errlen)
{
  LIST *tmp;
  size_t keylen;
  char *ptr;
  char *p;

  ptr = reap_comment (s);

  if ((p = strpbrk (ptr, ": \t")) == NULL || *p != ':')
  {
    free (ptr);
    strfcpy (errmsg, "invalid header field", errlen);
    return (-1);
  }
  keylen = p - ptr + 1;

  p++;
  SKIPWS (p);
  if (!*p)
  {
    free (ptr);
    snprintf (errmsg, errlen, "ignoring empty header field: %s", s);
    return (-1);
  }

  if (UserHeader)
  {
    tmp = UserHeader;

    FOREVER
    {
      /* see if there is already a field by this name */
      if (strncasecmp (ptr, tmp->data, keylen) == 0)
      {
	/* replace the old value */
	safe_free ((void **) &tmp->data);
	tmp->data = ptr;
	return 0;
      }

      if (!tmp->next)
	break;
      tmp = tmp->next;
    }

    tmp->next = mutt_new_list ();
    tmp = tmp->next;
  }
  else
  {
    tmp = mutt_new_list ();
    UserHeader = tmp;
  }

  tmp->data = ptr;

  return 0;
}

static int parse_sort (short *val, const char *s, char *err, size_t errlen)
{
  int i, flags = 0;

  if (strncmp ("reverse-", s, 8) == 0)
  {
    s += 8;
    flags = SORT_REVERSE;
  }
  
  if (strncmp ("last-", s, 5) == 0)
  {
    s += 5;
    flags |= SORT_LAST;
  }

  if ((i = mutt_getvaluebyname (s, SortMethods)) == -1)
  {
    snprintf (err, errlen, "%s: unknown sorting method", s);
    return (-1);
  }

  *val = i | flags;

  return 0;
}

/* flags
 *	M_EQUAL		'=' is a special char
 *	M_CONDENSE	condense ^(char) runs
 *	M_SPACE		don't treat whitespace as a terminator
 */

const char *
_mutt_extract_token (char *d, size_t dlen, const char *s, int flags)
{
  char qc = 0; /* quote char */

  *d = 0;

  if (!s || *s == '#')
    return NULL; /* nothing to do */

  SKIPWS (s);

  if (*s == '\'' || *s == '\"')
    qc = *s++;

  dlen--; /* save room for the terminal \0 */

  while (*s)
  {
    if ((!qc && ((ISSPACE (*s) && (flags & M_SPACE) == 0) || *s == '#')) ||
	*s == qc || ((flags & M_EQUAL) && *s == '='))
      break;

    if (*s == '\\')
    {
      if (! *++s)
	break;

      if (flags & M_SPACE)
      {
	if (dlen > 1)
	{
	  *d++ = '\\';
	  *d++ = *s;
	  dlen -= 2;
	}
      }
      else
	switch (*s)
	{
	  case 'c': /* control char: \cX */
	  case 'C':
	    if (*++s)
	    {
	      if (dlen)
	      {
		*d++ = (toupper (*s) - '@') & 0x7f;
		dlen--;
	      }
	    }
	    break;
	  case 'r':
	    if (dlen)
	    {
	      *d++ = '\r';
	      dlen--;
	    }
	    break;
	  case 'n':
	    if (dlen)
	    {
	      *d++ = '\n';
	      dlen--;
	    }
	    break;
	  case 't':
	    if (dlen)
	    {
	      *d++ = '\t';
	      dlen--;
	    }
	    break;
	  case 'f': /* formfeed */
	    if (dlen)
	    {
	      *d++ = '\f';
	      dlen--;
	    }
	    break;
	  case 'e': /* escape */
	    if (dlen)
	    {
	      *d++ = '\033';
	      dlen--;
	    }
	    break;
	  default:
	    if (dlen)
	    {
	      if (isdigit (s[0]) &&
		  s[1] && isdigit (s[1]) &&
		  s[2] && isdigit (s[2]))
	      {
		/* octal number */
		*d++ = s[2] + (s[1] << 3) + (s[0] << 6) - 3504;
		s += 2;
	      }
	      else
		*d++ = *s;
	      dlen--;
	    }
	    break;
	}
    }
    else if (*s == '^' && (flags & M_CONDENSE))
    {
      if (! *++s)
	break;

      if (*s == '^')
      {
	if (dlen)
	{
	  *d++ = *s;
	  dlen--;
	}
      }
      else if (*s == '[')
      {
	if (dlen)
	{
	  *d++ = '\033';
	  dlen--;
	}
      }
      else if (isalpha (*s))
      {
	if (dlen)
	{
	  *d++ = toupper (*s) - '@';
	  dlen--;
	}
      }
      else
      {
	dprint (1, (debugfile, "mutt_extract_token(): unknown escape sequence: ^%c\n", *s));
	if (dlen > 1)
	{
	  *d++ = '^';
	  *d++ = *s;
	  dlen -= 2;
	}
      }
    }
    else
    {
      if (dlen)
      {
	*d++ = *s;
	dlen--;
      }
    }

    s++;
  }

  *d = 0;

  if (qc && *s == qc)
    s++;

  SKIPWS (s);

  if (!*s || *s == '#')
    s = NULL;

  return (s);
}

static int parse_set (const char *s, void *data, char *errmsg, size_t errlen)
{
  char keyword[STRING];
  char tmp[LONG_STRING];
  int idx;
  int query;
  int unset;
  int inv;

  while (s && *s != '#')
  {
    /* reset state variables */
    query = 0;
    unset = ((int) data) & M_SET_UNSET;
    inv = ((int) data) & M_SET_INV;

    if (*s == '?')
    {
      query = 1;
      s++;
    }
    else if (strncmp ("no", s, 2) == 0)
    {
      unset = 1;
      s += 2;
    }
    else if (strncmp ("inv", s, 3) == 0)
    {
      inv = 1;
      s += 3;
    }

    /* get the variable name */
    s = _mutt_extract_token (keyword, sizeof (keyword), s, M_EQUAL);

    if ((idx = mutt_option_index (keyword)) == -1)
    {
      snprintf (errmsg, errlen, "%s: unknown variable", keyword);
      return (-1);
    }

    if (MuttVars[idx].type == DT_BOOL)
    {
      if (s && *s == '=')
      {
	snprintf (errmsg, errlen, "%s is a boolean var!", keyword);
	return (-1);
      }

      if (query)
      {
	snprintf (errmsg, errlen, "%s is %sset", keyword,
		  option (MuttVars[idx].bit) ? "" : "un");
	return 0;
      }

      if (unset)
	unset_option (MuttVars[idx].bit);
      else if (inv)
	toggle_option (MuttVars[idx].bit);
      else
	set_option (MuttVars[idx].bit);
    }
    else if (MuttVars[idx].type == DT_STR || MuttVars[idx].type == DT_PATH)
    {
      char *ptr = MuttVars[idx].data;

      if (query || !s)
      {
	/* user requested the value of this variable */
	snprintf (errmsg, errlen, "%s=\"%s\"", MuttVars[idx].option, ptr);
	return 0;
      }

      if (*s != '=')
      {
	strfcpy (errmsg, "missing =", errlen);
	return (-1);
      }

      s++;

      /* copy the value of the string */
      s = mutt_extract_token (ptr, MuttVars[idx].size, s);

      if (MuttVars[idx].type == DT_PATH)
	mutt_expand_path (ptr, MuttVars[idx].size);
    }
    else if (MuttVars[idx].type == DT_RX)
    {
      REGEXP *ptr = MuttVars[idx].data;
      regex_t *rx;
      int err, flags = REG_EXTENDED;

      if (query || !s)
      {
	/* user requested the value of this variable */
	snprintf (errmsg, errlen, "%s=\"%s\"", MuttVars[idx].option, ptr->pattern);
	return 0;
      }

      if (*s != '=')
      {
	strfcpy (errmsg, "missing =", errlen);
	return (-1);
      }

      s++;

      /* copy the value of the string */
      s = mutt_extract_token (tmp, sizeof (tmp), s);

      rx = (regex_t *) safe_malloc (sizeof (regex_t));
      if (!ptr->pattern || strcmp (ptr->pattern, tmp) != 0)
      {
	/* $url_regexp, $alternates and $local_site are case-insensitive,
	 * $mask is case-sensitive */
	if (strcmp (MuttVars[idx].option, "url_regexp") == 0 ||
	    strcmp (MuttVars[idx].option, "alternates") == 0 ||
	    strcmp (MuttVars[idx].option, "local_site") == 0)
	  flags |= REG_ICASE;
	else if (strcmp (MuttVars[idx].option, "mask") != 0)
	  flags |= mutt_which_case (tmp);
	
	if ((err = regcomp (rx, tmp, flags)) != 0)
	{
	  regerror (err, rx, errmsg, errlen);
	  regfree (rx);
	  safe_free ((void **) &rx);
	  return (-1);
	}

	/* get here only if everything went smootly */
	if (ptr->pattern)
	{
	  safe_free ((void **) &ptr->pattern);
	  regfree ((regex_t *) ptr->rx);
	  safe_free ((void **) &ptr->rx);
	}

	ptr->pattern = safe_strdup (tmp);
	ptr->rx = rx;

	/* $reply_regexp requires special treatment */
	if (Context && Context->msgcount &&
	    strcmp (MuttVars[idx].option, "reply_regexp") == 0)
	{
	  regmatch_t pmatch[1];
	  int i;
	  
#define CUR_ENV Context->hdrs[i]->env
	  for (i = 0; i < Context->msgcount; i++)
	  {
	    if (CUR_ENV && CUR_ENV->subject)
	    {
	      CUR_ENV->real_subj = (regexec ((regex_t *) ReplyRegexp.rx,
				    CUR_ENV->subject, 1, pmatch, 0)) ?
				    CUR_ENV->subject : 
				    CUR_ENV->subject + pmatch[0].rm_eo;
	    }
	  }
#undef CUR_ENV
	  if ((!option (OPTSTRICTTHREADS) && Sort == SORT_THREADS) || Sort == SORT_SUBJECT)
	    set_option (OPTNEEDRESORT);
	}
      }
    }
    else if (MuttVars[idx].type == DT_MAGIC)
    {
      if (query || !s)
      {
	char *tmp;

	switch (DefaultMagic)
	{
	  case M_MBOX:
	    tmp = "mbox";
	    break;

	  case M_MMDF:
	    tmp = "MMDF";
	    break;

	  case M_MH:
	    tmp = "MH";
	    break;

	  case M_MAILDIR:
	    tmp = "Maildir";
	    break;
	    
	  default:
	    tmp = "unknown";
	    break;
	}
	/* user requested the value of this variable */
	snprintf (errmsg, errlen, "%s=\"%s\"", MuttVars[idx].option, tmp);
	return 0;
      }

      if (*s != '=')
      {
	strfcpy (errmsg, "missing =", errlen);
	return (-1);
      }

      s++;

      /* copy the value of the string */
      s = mutt_extract_token (tmp, sizeof (tmp), s);
      if (mx_set_magic (tmp))
      {
	strfcpy (errmsg, "invalid mailbox type", errlen);
	return (-1);
      }
    }
    else if (MuttVars[idx].type == DT_NUM)
    {
      short *ptr = MuttVars[idx].data;

      if (query || !s)
      {
	snprintf (errmsg, errlen, "%s=%d", MuttVars[idx].option, *ptr);
	return 0;
      }

      if (*s != '=')
      {
	strfcpy (errmsg, "missing =", errlen);
	return (-1);
      }

      s++;

      s = mutt_extract_token (tmp, sizeof (tmp), s);
      *ptr = (short) atoi (tmp);

      /* these ones need a sanity check */
      if (strcmp (MuttVars[idx].option, "history") == 0)
      {
	if (*ptr < 0)
	  *ptr = 0;
	mutt_init_history ();
      }
      else if (strcmp (MuttVars[idx].option, "pager_index_lines") == 0)
      {
	if (*ptr < 0)
	  *ptr = 0;
      }
    }
    else if (MuttVars[idx].type == DT_QUAD)
    {
      if (query)
      {
	char *vals[] = { "no", "yes", "ask-no", "ask-yes" };

	snprintf (errmsg, errlen, "%s=%s", MuttVars[idx].option,
		  vals [ quadoption (MuttVars[idx].bit) ]);
	return 0;
      }

      if (s && *s == '=')
      {
	s++;

	s = mutt_extract_token (tmp, sizeof (tmp), s);

	if (strcasecmp ("yes", tmp) == 0)
	  set_quadoption (MuttVars[idx].bit, M_YES);
	else if (strcasecmp ("no", tmp) == 0)
	  set_quadoption (MuttVars[idx].bit, M_NO);
	else if (strcasecmp ("ask-yes", tmp) == 0)
	  set_quadoption (MuttVars[idx].bit, M_ASKYES);
	else if (strcasecmp ("ask-no", tmp) == 0)
	  set_quadoption (MuttVars[idx].bit, M_ASKNO);
	else
	{
	  snprintf (errmsg, errlen, "%s: invalid value", tmp);
	  return (-1);
	}
      }
      else
      {
	if (unset)
	  set_quadoption (MuttVars[idx].bit, M_NO);
	else
	  set_quadoption (MuttVars[idx].bit, M_YES);
      }
    }
    else if (MuttVars[idx].type == DT_SORT)
    {
      if (query || !s)
      {
	char *m = mutt_getnamebyvalue (*((short *) MuttVars[idx].data) & SORT_MASK, SortMethods);

	snprintf (errmsg, errlen, "%s=%s%s%s", MuttVars[idx].option,
		  (*((short *) MuttVars[idx].data) & SORT_REVERSE) ? "reverse-" : "",
		  (*((short *) MuttVars[idx].data) & SORT_LAST) ? "last-" : "", m);
	return 0;
      }

      if (*s != '=')
      {
	snprintf (errmsg, errlen, "missing =");
	return (-1);
      }

      s++;

      s = mutt_extract_token (tmp, sizeof (tmp), s);

      if (parse_sort ((short *) MuttVars[idx].data, tmp, errmsg, errlen) == -1)
	return (-1);

      if (strcmp (MuttVars[idx].option, "sort_browser"))
	set_option (OPTNEEDRESORT);
    }
    if (MuttVars[idx].redraw & R_INDEX)
      set_option (OPTFORCEREDRAWINDEX);
    if (MuttVars[idx].redraw & R_PAGER)
      set_option (OPTFORCEREDRAWPAGER);
  }

  return 0;
}

/* reads the specified initialization file.  returns -1 if errors were found
 * so that we can pause to let the user know...
 */
static int source_rc (const char *rcfile)
{
  FILE *f;
  int line = 0, rc = 0;
  int isContinuation = 0;
  char buf[LONG_STRING];
  char errbuf[SHORT_STRING];
  char *linebuf = NULL;
  size_t linelen = 0;
  size_t linemax = 0;
  size_t buflen;

  if ((f = fopen (rcfile, "r")) == NULL)
  {
    printf ("%s: unable to open file\n", rcfile);
    return (-1);
  }

  buf[sizeof (buf) - 1] = 0;
  while (fgets (buf, sizeof (buf) - 1, f) != NULL)
  {
    line++;

    buflen = strlen (buf);

    /* check to see if the whole line was read */

    isContinuation = 0;

    if (buf[buflen - 1] == '\n')
    {
      buflen--;
      buf[buflen] = 0; /* kill the newline */

      /* look for a continuation line */
      if (buflen && buf[buflen - 1] == '\\')
      {
	buflen--;
	isContinuation = 1;
      }
    }
    else
      isContinuation = 1;

    if (buflen)
    {
      if (buflen + linelen + 1 > linemax)
      {
	/* not enough room, so allocate more */
	safe_realloc ((void **) &linebuf, (linemax = buflen + linelen + 80));
      }

      memcpy (linebuf + linelen, buf, buflen + 1);
      linelen += buflen;

      if (isContinuation)
	continue;
    }

    if (linelen)
    {
      if (mutt_parse_rc_line (linebuf, errbuf, sizeof (errbuf)) == -1)
      {
	if (!option (OPTNOCURSES))
	  printf ("Error in %s, line %d: %s\n", rcfile, line, errbuf);
	rc = -1;
      }

      linelen = 0;
    }
  }

  safe_free ((void **) &linebuf);

  fclose (f);
  return (rc);
}

static int parse_source (const char *s, void *data, char *err, size_t errlen)
{
  char path[_POSIX_PATH_MAX];

  mutt_extract_token (path, sizeof (path), s);
  mutt_expand_path (path, sizeof (path));
  if (access (path, R_OK) == -1)
  {
    strfcpy (err, strerror (errno), errlen);
    return (-1);
  }

  return (source_rc (path));
}

static int parse_list (const char *s, void *data, char *err, size_t errlen)
{
  mutt_add_to_list ((LIST **) data, s);
  return 0;
}

static int
parse_unlist (const char *s, void *data, char *err, size_t errlen)
{
  remove_from_list ((LIST **) data, s);
  return 0;
}

void mutt_add_to_list (LIST **list, const char *s)
{
  LIST *last = 0;
  LIST *t;
  const char *p;
  size_t len;

  do
  {
    p = strpbrk (s, " \t,#");
    if (p)
      len = p - s;
    else
      len = strlen (s);

    /* check to make sure the item is not already on this list */
    last = *list;
    while (last)
    {
      if (strncasecmp (s, last->data, len) == 0 && last->data[len] == 0)
      {
	/* already on the list, so just ignore it */
	last = NULL;
	break;
      }
      if (!last->next)
	break;
      last = last->next;
    }

    if (!*list || last)
    {
      t = (LIST *) safe_calloc (1, sizeof(LIST));
      t->data = safe_malloc (len + 1);
      memcpy (t->data, s, len);
      t->data[len] = 0;

      if (last)
      {
	last->next = t;
	last = last->next;
      }
      else
	*list = last = t;
    }

    if (p)
      SKIPWS (p);
  }
  while ((s = p) != NULL && *s && *s != '#');
}

/*
 * Takes a string, and executes any `commands` and replaces the 
 * command string with the first line of the command output
 */
static void mutt_string_cmd_sub (char *s, int slen)
{
   char *tmp;
   int incmd = 0;
   char *cmd, *resp; 
   int cmdind = 0;
   int x,y;

   dprint(1,(debugfile,"mutt_string_cmd_sub(): Substring: %s, slen: %d\n",s,slen));
   tmp = (char *) safe_malloc (sizeof(char *) * slen);
   strfcpy (tmp, s, slen);
   cmd = (char *) safe_malloc (sizeof(char *) * slen);
   resp = (char *) safe_malloc (sizeof(char *) * slen);
   for (x = 0, y = 0; x < slen && y < slen && s[x]; x++)
   {
     if (incmd)
     {
       /* dprint(1,(debugfile,"mutt_string_cmd_sub(): cmd: %s\n",s)); */
       if (s[x] == '\\')
       {
	 x++;
	 if (s[x] == 'n')
	 {
	   cmd[cmdind++] = '\n';
	 } 
	 else if (s[x] == 't')
	 {
	   cmd[cmdind++] = '\t';
	 }
	 else if (s[x])
	   cmd[cmdind++] = s[x];
       }
       else if (s[x] == '`')
	 incmd = 0;
       else
	 cmd[cmdind++] = s[x];

       if (!incmd)
       {
	 cmd[cmdind] = 0;
	 if (cmd[0])
	 {
	   FILE *out = NULL;
	   int rind = 0;
	   pid_t thepid;

           dprint (1,(debugfile,"mutt_string_cmd_sub(): Command: %s\n",cmd));
	   thepid = mutt_create_filter (cmd, NULL, &out, NULL);
	   fgets (resp,slen,out);
	   fclose (out);
	   mutt_wait_filter (thepid);
           dprint (1,(debugfile,"mutt_string_cmd_sub(): Response: %s\n",resp));

	   /* copy response into temp string */
	   while (resp[rind] && y < slen)
	   {
	     if (resp[rind] != '\n')
	       tmp[y++] = resp[rind++];
	     else
	       rind++;
	   }
	   if (y < slen)
	     tmp[y] = 0;
	   else
	     tmp[slen-1] = 0;
           /* reset command */
           cmdind = 0;
         }
       }
     }
     else
     {
       if ((s[x] == '`') && (!x || (s[x-1] != '\\')))
	 incmd = 1;
       else
	 tmp[y++] = s[x];
     }
   }
   tmp[y] = 0;
   strfcpy (s, tmp, slen);
   safe_free ((void **) &tmp);
   safe_free ((void **) &cmd);
   safe_free ((void **) &resp);
}

int mutt_parse_rc_line (const char *cmd, char *errmsg, int errlen)
{
  char tmp[HUGE_STRING];
  const char *q;
  char *p;
  int i;
  size_t len;

  *errmsg = 0;

  SKIPWS (cmd);

  if (*cmd == '#')
    return 0; /* rest of line is a comment */

  q = cmd + strlen (cmd);

  /* skip any trailing whitespace */
  while ((q > cmd) && ISSPACE (*(q-1)))
    q--;

  len = q - cmd;

  if (len == 0)
    return 0; /* nothing on this line */

  if (len > sizeof (tmp) - 1)
    len = sizeof (tmp) - 1;
  memcpy (tmp, cmd, len);
  tmp[len] = 0;

  if ((p = strchr (tmp, '`')) && (*(p-1) != '\\')) 
    mutt_string_cmd_sub (tmp, sizeof (tmp));

  p = tmp;

  for (i = 0; Commands[i].name; i++)
  {
    len = strlen (Commands[i].name);
    if (strncmp (p, Commands[i].name, len) == 0)
    {
      if (ISSPACE (p[len]))
      {
	p += len;
	SKIPWS (p);
	return (Commands[i].func (p, Commands[i].data, errmsg, errlen));
      }
      else if (!p[len])
      {
	snprintf (errmsg, errlen, "%s: missing parameter", p);
	return (-1);
      }
    }
  }

  /* unknown command */

  snprintf (errmsg, errlen, "%s: unknown command", p);
  return (-1);
}

char *mutt_getnamebyvalue (int val, const struct mapping_t *map)
{
  int i;

  for (i=0; map[i].name; i++)
    if (map[i].value == val)
      return (map[i].name);
  return NULL;
}

int mutt_getvaluebyname (const char *name, const struct mapping_t *map)
{
  int i;

  for (i=0; map[i].name; i++)
    if (strcmp (map[i].name, name) == 0)
      return (map[i].value);
  return (-1);
}

#ifdef DEBUG
static void start_debug (void)
{
  time_t t;
  int i;
  char buf[_POSIX_PATH_MAX];
  char buf2[_POSIX_PATH_MAX];

  /* rotate the old debug logs */
  for (i=3; i>=0; i--)
  {
    snprintf (buf, sizeof(buf), "%s/.muttdebug%d", Homedir, i);
    snprintf (buf2, sizeof(buf2), "%s/.muttdebug%d", Homedir, i+1);
    rename (buf, buf2);
  }
  if ((debugfile = fopen(buf, "w")) != NULL)
  {
    t = time (0);
    fprintf (debugfile, "%s started at %s.\nDebugging at level %d.\n\n",
	     MuttVersion, asctime(localtime(&t)), debuglevel);
  }
}
#endif

void mutt_init (int skip_sys_rc)
{
  struct passwd *pw;
  struct utsname utsname;
  char buffer[STRING];
  char *p;

  /* Get some information about the user */
  if ((pw = getpwuid (getuid ())) == NULL)
  {
    mutt_error ("Out of memory!");
    sleep (1);
    mutt_exit (1);
  }
  strfcpy (Username, pw->pw_name, sizeof (Username));

  /* on one of the systems I use, getcwd() does not return the same prefix
   * as is listed in the passwd file
   */
  if ((p = getenv ("HOME")) != NULL)
    strfcpy (Homedir, p, sizeof (Homedir));
  else
    strfcpy (Homedir, pw->pw_dir, sizeof (Homedir));

  strfcpy (Realname, pw->pw_gecos, sizeof (Realname));
  if ((p = strchr (Realname, ','))) *p = 0;
  strfcpy (Shell, pw->pw_shell, sizeof (Shell));

#ifdef DEBUG
  /* Start up debugging mode if requested */
  if (debuglevel > 0)
    start_debug ();
#endif

  /* And about the host... */
  uname (&utsname);
  strfcpy (Hostname, utsname.nodename, sizeof (Hostname));

  /* some systems report the FQDN instead of just the hostname */
  if ((p = strchr (Hostname, '.')))
  {
    *p++ = 0;
    strfcpy (buffer, p, sizeof (buffer)); /* save the domain for below */
  }

#ifndef DOMAIN
#define DOMAIN buffer
  if (!p && getdnsdomainname (buffer, sizeof (buffer)) == -1)
  {
    Fqdn[0] = '@';
    Fqdn[1] = 0;
  }
  else
#endif /* DOMAIN */
  {
# ifdef HIDDEN_HOST
    strfcpy (Fqdn, DOMAIN, sizeof (Fqdn));
# else
    snprintf (Fqdn, sizeof (Fqdn), "%s.%s", Hostname, DOMAIN);
# endif /* HIDDEN_HOST */
  }

#ifndef LOCALES_HACK
  /* Do we have a locale definition? */
  if (((p = getenv ("LC_ALL")) != NULL && p[0]) ||
      ((p = getenv ("LANG")) != NULL && p[0]) ||
      ((p = getenv ("LC_CTYPE")) != NULL && p[0]))
    set_option (OPTLOCALES);
#endif

  /* Set some defaults */

  set_option (OPTALLOW8BIT);
  set_option (OPTATTACHSPLIT);
  set_option (OPTBEEP);
  set_option (OPTCHECKNEW);
  set_option (OPTCONFIRMCREATE);
  set_option (OPTCONFIRMAPPEND);
  set_option (OPTFCCATTACH);
  set_option (OPTHDRS);
  set_option (OPTHELP);
  set_option (OPTHOLD);
  set_option (OPTMARKERS);
  set_option (OPTMARKOLD);
  set_option (OPTPOINTNEW);
  set_option (OPTPROMPTAFTER);
  set_option (OPTRESOLVE);
  set_option (OPTSAVEEMPTY);
  set_option (OPTSIGDASHES);
  set_option (OPTSORTRE);
  set_option (OPTSUSPEND);
  set_option (OPTUSEDOMAIN);
  set_option (OPTUSEFROM);
  set_option (OPTWAITKEY);
  set_option (OPTWEED);
  set_option (OPTWRAP);
  
#ifdef _PGPPATH
  set_option (OPTPGPENCRYPTSELF);
  set_quadoption (OPT_VERIFYSIG, M_YES);
#endif

  set_quadoption (OPT_USEMAILCAP, M_ASKYES);
  set_quadoption (OPT_INCLUDE, M_ASKYES);
  set_quadoption (OPT_RECALL, M_ASKYES);
  set_quadoption (OPT_PRINT, M_ASKNO);
  set_quadoption (OPT_POSTPONE, M_ASKYES);
  set_quadoption (OPT_DELETE, M_ASKYES);
  set_quadoption (OPT_MOVE, M_ASKNO);
  set_quadoption (OPT_REPLYTO, M_ASKYES);
  set_quadoption (OPT_COPY, M_YES);
  set_quadoption (OPT_ABORT, M_YES);
  set_quadoption (OPT_SUBJECT, M_ASKYES);

  if ((p = getenv ("MAIL")))
    strfcpy (Spoolfile, p, sizeof (Spoolfile));
  else
  {
#ifdef HOMESPOOL
    snprintf (Spoolfile, sizeof (Spoolfile), "%s/%s", Homedir, MAILPATH);
#else
    snprintf (Spoolfile, sizeof (Spoolfile), "%s/%s", MAILPATH, Username);
#endif
  }

  strfcpy (Attribution, "On %d, %n wrote:", sizeof (Attribution));
  strfcpy (AttachSep, "\n", sizeof (AttachSep));
  strfcpy (Charset, "iso-8859-1", sizeof (Charset));
  strfcpy (DecodeFmt, "[-- Decoded from message %i --]\n", sizeof (DecodeFmt));
  strfcpy (DateFmt, "!%a, %b %d, %Y at %I:%M:%S%p %Z", sizeof (DateFmt));
  strfcpy (ForwFmt, "[%a: %s]", sizeof (ForwFmt));
  strfcpy (HdrFmt, "%4C %Z %{%b %d} %-15.15L (%4l) %s", sizeof (HdrFmt));
  strfcpy (InReplyTo, "%i; from %n on %{!%a, %b %d, %Y at %I:%M:%S%p %Z}", sizeof (InReplyTo));
  snprintf (Inbox, sizeof (Inbox), "%s/mbox", Homedir);

#ifdef ISPELL
  strfcpy (Ispell, ISPELL, sizeof (Ispell));
#else
  strfcpy (Ispell, "ispell", sizeof (Ispell));
#endif /* ISPELL */
  strfcpy (Locale, "C", sizeof (Locale));  

  if ((p = getenv ("MAILCAPS")))
    strfcpy (MailcapPath, p, sizeof (MailcapPath));
  else
  {
    /* Default search path from RFC1524 */
    strfcpy (MailcapPath, "~/.mailcap:" SHAREDIR "/mailcap:/etc/mailcap:/usr/etc/mailcap:/usr/local/etc/mailcap", sizeof (MailcapPath));
  }

  snprintf (Maildir, sizeof (Maildir), "%s/Mail", Homedir);
  strfcpy (MsgFmt, "%s", sizeof (MsgFmt));
  strfcpy (Pager, "builtin", sizeof (Pager));  
  strfcpy (PagerFmt, "-%S- %C/%m: %-20.20n   %s", sizeof (PagerFmt));
  strfcpy (PipeSep, "\n", sizeof (PipeSep));
  snprintf (Postponed, sizeof (Postponed), "%s/postponed", Homedir);
  strfcpy (Prefix, "> ", sizeof (Prefix));
  strfcpy (PrintCmd, "lpr", sizeof (PrintCmd));
  snprintf (Sendmail, sizeof (Sendmail), "%s -oi -oem -t", SENDMAIL);
  snprintf (SendmailBounce, sizeof (Sendmail), "%s -oi -oem", SENDMAIL);
  snprintf (Signature, sizeof (Signature), "%s/.signature", Homedir);
  strfcpy (SimpleSearch, "~f %s | ~s %s", sizeof (SimpleSearch));
  strfcpy (StChars, "-*%", sizeof (StChars));
  strfcpy (Status, "-%r-Mutt: %f [Msgs:%?M?%M/?%m%?n? New:%n?%?d? Del:%d?%?F? Flag:%F?%?t? Tag:%t?%?p? Post:%p?%?b? Inc:%b? %l]---(%s)%|-", sizeof (Status));

  if ((p = getenv ("TMPDIR")) != NULL)
    strfcpy (Tempdir, p, sizeof (Tempdir));
  else
    strfcpy (Tempdir, "/tmp", sizeof (Tempdir));

  strfcpy (Tochars, " +TCF", sizeof (Tochars));
#ifdef _PGPPATH
  strfcpy (Pgp, _PGPPATH, sizeof (Pgp));
#endif /* _PGPPATH */
  strfcpy (PopUser, Username, sizeof (PopUser));

  memset (&Alternates, 0, sizeof (REGEXP));
  memset (&Localsites, 0, sizeof (REGEXP));
  
  QuoteRegexp.pattern = safe_strdup ("^([ \t]*[|>:}#])+");
  QuoteRegexp.rx = safe_malloc (sizeof (regex_t));
  regcomp ((regex_t *) QuoteRegexp.rx, QuoteRegexp.pattern, 
	   REG_EXTENDED | mutt_which_case (QuoteRegexp.pattern));

  UrlRegexp.pattern = safe_strdup ("((ftp|http)://|mailto:)[^ \t\r\n\">]*[^., \"\r\t\n>]");
  UrlRegexp.rx = safe_malloc (sizeof (regex_t));
  regcomp ((regex_t *) UrlRegexp.rx, UrlRegexp.pattern, REG_EXTENDED | REG_ICASE);

  Mask.pattern = safe_strdup ("^(\\.\\.$|[^.])");
  Mask.rx = safe_malloc (sizeof (regex_t));
  regcomp ((regex_t *) Mask.rx, Mask.pattern, REG_EXTENDED);

  ReplyRegexp.pattern = safe_strdup ("^(re|aw):[ \t]*");
  ReplyRegexp.rx = safe_malloc (sizeof (regex_t));
  regcomp ((regex_t *) ReplyRegexp.rx, ReplyRegexp.pattern, 
	    REG_EXTENDED | mutt_which_case (ReplyRegexp.pattern));

  if ((p = getenv ("EDITOR")) != NULL)
    strfcpy (Editor, p, sizeof (Editor));
  else
    strfcpy (Editor, "vi", sizeof (Editor));

  if ((p = getenv ("VISUAL")) != NULL)
    strfcpy (Visual, p, sizeof (Visual));
  else
    strfcpy (Visual, Editor, sizeof (Visual));

  if ((p = getenv ("REPLYTO")) != NULL)
  {
    char error[SHORT_STRING];

    snprintf (buffer, sizeof (buffer), "Reply-To: %s", p);
    parse_my_hdr (buffer, NULL, error, sizeof (error));
  }

  mutt_init_history ();

  if (!Muttrc[0])
  {
    snprintf (Muttrc, sizeof (Muttrc), "%s/.muttrc-%s", Homedir, MuttVersion + 5);
    if (access (Muttrc, F_OK) == -1)
      snprintf (Muttrc, sizeof (Muttrc), "%s/.muttrc", Homedir);
  }
  else
    mutt_expand_path (Muttrc, sizeof (Muttrc));
  strfcpy (AliasFile, Muttrc, sizeof (AliasFile));

  /* Process the global rc file if it exists and the user hasn't explicity
   * requested not to via "-n".
   */
  if (!skip_sys_rc)
  {
    char sysrc[_POSIX_PATH_MAX];

    snprintf (sysrc, sizeof (sysrc), "%s/Muttrc-%s", SHAREDIR, MuttVersion + 5);
    if (access (sysrc, F_OK) == -1)
      snprintf (sysrc, sizeof (sysrc), "%s/Muttrc", SHAREDIR);
    if (access (sysrc, F_OK) != -1)
      source_rc (sysrc);
  }

  /* Read the user's initialization file.  */
  if (access (Muttrc, F_OK) != -1)
  {
    if (!option (OPTNOCURSES))
      endwin ();
    if (source_rc (Muttrc) != 0 && !option (OPTNOCURSES))
    {
      if (ci_any_key_to_continue (NULL) == -1)
	mutt_exit (1);
    }
  }
}
