/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "send.h"

#include <sys/stat.h>
#include <string.h>
#include <ctype.h>

void mutt_edit_headers (const char *editor, const char *body, ENVELOPE **env,
			char *fcc, size_t fcclen)
{
  char combined[_POSIX_PATH_MAX]; /* tempfile used to edit headers + body */
  FILE *ifp, *ofp;
  char buffer[LONG_STRING];
  int i;
  ENVELOPE *n;
  time_t mtime;
  struct stat st;

  mutt_mktemp (combined);
  if ((ofp = fopen (combined, "w")) == NULL)
  {
    mutt_perror (combined);
    return;
  }

  mutt_write_rfc822_header (ofp, *env, NULL, 1);
  fputc ('\n', ofp); /* tie off the header. */

  /* now copy the body of the message. */
  if ((ifp = fopen (body, "r")) == NULL)
  {
    mutt_perror (body);
    return;
  }

  mutt_copy_stream (ifp, ofp);

  fclose (ifp);
  fclose (ofp);

  if (stat (combined, &st) == -1)
  {
    mutt_perror (combined);
    return;
  }

  mtime = st.st_mtime;
  mutt_edit_file (editor, combined);
  stat (combined, &st);
  if (mtime == st.st_mtime)
  {
    dprint(1, (debugfile, "ci_edit_headers(): temp file was not modified.\n"));
    /* the file has not changed! */
    mutt_unlink (combined);
    return;
  }

  mutt_unlink (body);
  mutt_free_list (&(*env)->userhdrs);

  /* Read the temp file back in */
  ifp = fopen (combined, "r");
  ofp = fopen (body, "w");
  n = mutt_read_rfc822_header (ifp, NULL);
  while ((i = fread (buffer, 1, sizeof (buffer), ifp)) > 0)
    fwrite (buffer, 1, i, ofp);
  fclose (ofp);
  fclose (ifp);
  mutt_unlink (combined);

  /* restore old info. */
  n->references = (*env)->references;
  (*env)->references = NULL;
  mutt_free_envelope (env);
  *env = n;

  (*env)->to = mutt_expand_aliases ((*env)->to);
  (*env)->cc = mutt_expand_aliases ((*env)->cc);
  (*env)->bcc = mutt_expand_aliases ((*env)->bcc);
  (*env)->reply_to = mutt_expand_aliases ((*env)->reply_to);

  if (option (OPTUSEFROM))
  {
    if (!(*env)->from)
      (*env)->from = mutt_default_from ();
  }

  if (fcc)
  {
    /* see if the user specified an Fcc */

    LIST *cur = (*env)->userhdrs;
    LIST *last = NULL;
    LIST *tmp;
    char *p;

    while (cur)
    {
      if (strncasecmp ("fcc:", cur->data, 4) == 0)
      {
	if (last)
	  last->next = cur->next;
	else
	  (*env)->userhdrs = cur->next;
	tmp = cur;
	cur = cur->next;
	tmp->next = NULL;

	p = tmp->data + 4;
	SKIPWS (p);
	if (*p)
	{
	  strfcpy (fcc, p, fcclen);
	  mutt_pretty_mailbox (fcc);
	}
	mutt_free_list (&tmp);
      }
      else
      {
	last = cur;
	cur = cur->next;
      }
    }
  }
}
