/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "mutt.h"
#include "mutt_curses.h"
#include "keymap.h"

#include <string.h>

static struct mapping_t Menus[] = {
 { "index",	MENU_MAIN },
 { "pager",	MENU_PAGER },
 { "url",	MENU_URL },
 { "compose",	MENU_COMPOSE },
 { "attach",	MENU_ATTACH },
 { "browser",	MENU_FOLDER },
 { "alias",	MENU_ALIAS },
 { "editor",	MENU_EDITOR },
#ifdef _PGPPATH
 { "pgp",	MENU_PGP },
#endif /* _PGPPATH */
 { NULL,	0 }
};

#define mutt_check_menu(s) mutt_getvaluebyname(s, Menus)

/* expects to see: <menu-string> <key-string> */
static const char *parse_keymap (int *menu,
				 char *key,
				 size_t keylen,
				 const char *s,
				 char *err,
				 size_t errlen)
{
  char buf[SHORT_STRING];

  /* menu name */
  s = mutt_extract_token (buf, sizeof (buf), s);

  if (s)
  {
    if ((*menu = mutt_check_menu (buf)) == -1)
    {
      snprintf (err, errlen, "%s: no such menu", s);
      return (NULL);
    }

    /* key sequence */
    s = mutt_extract_token (key, keylen, s);

    if (s)
      return s;
  }

  strfcpy (err, "too few arguments", errlen);
  return (NULL);
}

/* bind menu-name '<key_sequence>' function-name */
int mutt_parse_bind (const char *s, void *data, char *err, size_t errlen)
{
  struct binding_t *bindings = NULL;
  char buf[SHORT_STRING];
  char key[SHORT_STRING];
  int menu, i;

  if ((s = parse_keymap (&menu, key, sizeof (key), s, err, errlen)) == NULL)
    return (-1);

  switch (menu)
  {
    case MENU_MAIN:
      bindings = OpMain;
      break;

    case MENU_COMPOSE:
      bindings = OpCompose;
      break;

    case MENU_PAGER:
      bindings = OpPager;
      break;

    case MENU_POST:
      bindings = OpPost;
      break;

    case MENU_FOLDER:
      bindings = OpBrowser;
      break;

    case MENU_ATTACH:
      bindings = OpAttach;
      break;

    case MENU_EDITOR:
      bindings = OpEditor;
      break;

    case MENU_ALIAS:
      bindings = OpAlias;
      break;

#ifdef _PGPPATH
    case MENU_PGP:
      bindings = OpPgp;
      break;
#endif /* _PGPPATH */
  }

  /* function to execute */
  s = mutt_extract_token (buf, sizeof (buf), s);
  if (s)
  {
    strfcpy (err, "too many arguments", errlen);
    return (-1);
  }

  if (strcasecmp ("noop", buf) == 0)
  {
    km_bindkey (key, menu, OP_NULL, NULL);
    return 0;
  }
  else if (menu == MENU_PAGER || menu == MENU_MAIN || menu == MENU_EDITOR)
  {
    /* These menus have keymaps of their own and do not refer to the
     * generic map.
     */
    for (i = 0; bindings[i].name; i++)
      if (strcmp (buf, bindings[i].name) == 0)
      {
	km_bindkey (key, menu, bindings[i].op, NULL);
	return (0);
      }
  }
  else
  {
    /*
     * First check the "generic" list of commands.
     */
    for (i=0; OpGeneric[i].name; i++)
    {
      if (strcmp (buf, OpGeneric[i].name) == 0)
      {
	km_bindkey (key, menu, OpGeneric[i].op, NULL);
	return (0);
      }
    }

    /*
     * Now check the menu-specific list of commands (if they exist).
     */
    if (bindings)
    {
      for (i=0; bindings[i].name; i++)
      {
	if (strcmp (buf, bindings[i].name) == 0)
	{
	  km_bindkey (key, menu, bindings[i].op, NULL);
	  return (0);
	}
      }
    }
  }

  snprintf (err, errlen, "%s: no such function in map", buf);
  return (-1);
}

/* macro <menu> <key> <macro> */
int mutt_parse_macro (const char *s, void *data, char *err, size_t errlen)
{
  int menu;
  char key[SHORT_STRING];
  char buf[SHORT_STRING];

  if ((s = parse_keymap (&menu, key, sizeof (key), s, err, errlen)) == NULL)
    return (-1);

  s = _mutt_extract_token (buf, sizeof (buf), s, M_CONDENSE);
  if (s)
  {
    strfcpy (err, "too many arguments", errlen);
    return (-1);
  }

  km_bindkey (key, menu, OP_MACRO, buf);

  return 0;
}
