/* $Id: w3mimgdisplay.c,v 1.1.4.9 2002/03/31 13:36:20 suto Exp $ */
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <limits.h>
#include <Imlib.h>
#include "w3mimgdisplay.h"
#include "config.h"
#include <time.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

static Display *display;
static Window window, parent;
static unsigned long background_pixel;
static char *background = NULL;
static int offset_x = 2, offset_y = 2, window_depth = sizeof(int) * CHAR_BIT;
static int defined_bg = 0, defined_x = 0, defined_y = 0, defined_test = 0;
static int defined_debug = 0;
static int defined_ignore_exposure = 0;
static int xsync_option = False;

typedef struct _Image {
  struct _Image *next, *prev;
  Pixmap pixmap;
  int index;
  int width;
  int height;
  int queued;
} Image;

static Image *imageHead;
static Image *imageTail;
static Image *imageFree;
static Image **imageTab;
static int maxImageSize = (1U << (11 * 2 + 2));
static int imageSize;
static int maxImage;
static int nImage;
static XPoint screenRectangle[5];
static Region screenRegion;
static Region clipRegion;
static Region exposureRegion;
static Region imageRegion;
static GC imageGC;

typedef struct _ImageQueue {
  Image *image;
  int sx, sy, sw, sh, x, y;
} ImageQueue;

static ImageQueue *q;
static int nq;
static int nq_max;

#ifndef PIPE_BUF
#define PIPE_BUF (512)
#endif
static char buf[PIPE_BUF];
static char *l;
static int nl, nl_max;

static void GetOption(int argc, char **argv);
static void DrawImage(char *buf);
static void ClipImage(char *p);
static void ClearImage(void);
static void FlushImage(Region region);
static void ResetImage(int clear_p);
static void ClearImageAndExit(void);
static MySignalHandler ExitBySignal(SIGNAL_ARG);

/*
  xterm/kterm/hanterm/cxterm
    top window (WINDOWID)
      +- text window
           +- scrollbar
  rxvt/aterm/Eterm/wterm
    top window (WINDOWID)
      +- text window
      +- scrollbar
      +- menubar (etc.)
  gnome-terminal
    top window
      +- text window (WINDOWID)
      +- scrollbar
      +- menubar
  mlterm (-s)
    top window
      +- text window (WINDOWID)
      +- scrollbar
  mlterm
    top window = text window (WINDOWID)

  powershell
    top window
      +- window
      |    +- text window
      |    +- scrollbar
      +- menubar (etc.)
  dtterm
    top window
      +- window
           +- window
           |    +- window
           |         +- text window
           |         +- scrollbar
           +- menubar
  hpterm
    top window
      +- window
           +- text window
           +- scrollbar
           +- (etc.)
*/

void
myExit(int status, FILE *out, const char *frmt, ...)
{
  va_list ap;

  va_start(ap, frmt);
  vfprintf(out, frmt, ap);
  fflush(out);
  va_end(ap);
  exit(status);
}

void
pushLine(char *s, int llen)
{
  if (nl + llen >= nl_max) {
    nl_max = ((nl + llen) / 2 + 1) * 3;

    if (l)
      l = realloc(l, nl_max);
    else
      l = malloc(nl_max);

    if (!l)
      myExit(1, stderr, "l = malloc(%d): %s\n", nl_max, strerror(errno));
  }

  memcpy(l + nl, s, llen);
  nl += llen;
  l[nl] = '\0';
}

int
main(int argc, char **argv)
{
  Window root, *children;
  XWindowAttributes attr;
  XColor screen_def, exact_def;
  int revert, nchildren, width, height, i;
  char *id;
  struct timeval itv;
  fd_set fds;
  int stdin_fd, n, llen, nsyncs, nevent, prev_xexpose_count;
  char *eol = NULL;

  GetOption(argc, argv);

  if (!(display = XOpenDisplay(NULL))) {
    if (errno)
      myExit(1, stderr, "XOpenDisplay(): %s\n", strerror(errno));
    else
      exit(0);
  }

  if ((id = getenv("WINDOWID")) && *id)
    window = (Window) atoi(id);
  else
    XGetInputFocus(display, &window, &revert);

  if (!window)
    myExit(1, stderr, "XGetInputFocus(): %s\n", strerror(errno));

  XGetWindowAttributes(display, window, &attr);
  width = attr.width;
  height = attr.height;
  window_depth = attr.depth;

  while (1) {
    Window p_window;

    XQueryTree(display, window, &root, &parent, &children, &nchildren);

    if (defined_debug)
      fprintf(stderr,
	      "window=%lx root=%lx parent=%lx nchildren=%d width=%d height=%d\n",
	      (unsigned long)window, (unsigned long)root, (unsigned long)parent,
	      nchildren, width, height);

    p_window = window;

    for (i = 0; i < nchildren; i++) {
      XGetWindowAttributes(display, children[i], &attr);

      if (defined_debug)
	fprintf(stderr,
		"children[%d]=%lx x=%d y=%d width=%d height=%d\n", i,
		(unsigned long)children[i], attr.x, attr.y, attr.width, attr.height);

      if (attr.width > width * 0.7 && attr.height > height * 0.7) {
	/* maybe text window */
	width = attr.width;
	height = attr.height;
	window_depth = attr.depth;
	window = children[i];
      }
    }

    if (p_window == window)
      break;
  }

  if (!defined_x) {
    for (i = 0; i < nchildren; i++) {
      XGetWindowAttributes(display, children[i], &attr);

      if (attr.x <= 0 && attr.width < 30 && attr.height > height * 0.7) {
	if (defined_debug)
	  fprintf(stderr,
		  "children[%d]=%lx x=%d y=%d width=%d height=%d\n",
		  i, (unsigned long)children[i], attr.x, attr.y, attr.width,
		  attr.height);

	/* scrollbar of xterm/kterm ? */
	offset_x += attr.x + attr.width + attr.border_width * 2;
	break;
      }
    }
  }

  if (defined_test)
    myExit(0, stdout, "%d;%d;\n", width - offset_x, height - offset_y);		/* test OK */

  screenRectangle[1].x = screenRectangle[2].x = width;
  screenRectangle[2].y = screenRectangle[3].y = height;

  if (defined_bg && XAllocNamedColor(display, DefaultColormap(display, 0),
				     background, &screen_def, &exact_def))
    background_pixel = screen_def.pixel;
  else {
    XImage *winimg;

    if ((winimg = XGetImage(display, window,
			    offset_x > 0 ? offset_x - 1 : 0,
			    offset_y > 0 ? offset_y - 1 : 0,
			    1, 1, AllPlanes, XYPixmap))) {
      background_pixel = XGetPixel(winimg, 0, 0);
      XDestroyImage(winimg);
    }
    else
      background_pixel = WhitePixel(display, 0);
  }

  if (defined_ignore_exposure)
    xsync_option = True;
  else
    XSelectInput(display, window, ExposureMask);

  FD_ZERO(&fds);
  stdin_fd = fileno(stdin);
  prev_xexpose_count = -1;
  signal(SIGHUP, ExitBySignal);
  signal(SIGTERM, ExitBySignal);
  signal(SIGINT, SIG_IGN);
  signal(SIGQUIT, SIG_IGN);

  for (nsyncs = nevent = 0 ;;) {
#if CLOCKS_PER_SEC > 50
    itv.tv_usec = 20000;
    itv.tv_sec = 0;
#elif CLOCKS_PER_SEC > 1
    itv.tv_usec = 1000000 / CLOCKS_PER_SEC;
    itv.tv_sec = 0;
#else
    itv.tv_usec = 0;
    itv.tv_sec = 1;
#endif
    FD_SET(stdin_fd, &fds);

    switch (n = select(stdin_fd + 1, &fds, NULL, NULL, &itv)) {
    case 0:
      if (nsyncs) {
	if (exposureRegion && !XEmptyRegion(exposureRegion))
	  XSubtractRegion(exposureRegion, exposureRegion, exposureRegion);

	if (!screenRegion &&
	    !(screenRegion = XPolygonRegion(screenRectangle,
					    sizeof(screenRectangle) / sizeof(screenRectangle[0]),
					    WindingRule)))
	  myExit(1, stderr, "screenRegion = XPolygonRegion(): %s\n", strerror(errno));

	FlushImage(screenRegion);
	--nsyncs;
	prev_xexpose_count = -1;
      }
      else if (!defined_ignore_exposure) {
	if (!prev_xexpose_count && exposureRegion && !XEmptyRegion(exposureRegion))
	  FlushImage(exposureRegion);

	if (XEventsQueued(display, QueuedAfterFlush)) {
	  XEvent event;

	  do {
	    XNextEvent(display, &event);

	    if (event.type == Expose && event.xexpose.window == window) {
	      XRectangle rect;

	      if (!prev_xexpose_count && exposureRegion && !XEmptyRegion(exposureRegion))
		XSubtractRegion(exposureRegion, exposureRegion, exposureRegion);

	      if (!exposureRegion &&
		  !(exposureRegion = XCreateRegion()))
		myExit(1, stderr, "exposureRegion = XCreateRegion(): %s\n", strerror(errno));

	      rect.x = event.xexpose.x;
	      rect.y = event.xexpose.y;
	      rect.width = event.xexpose.width;
	      rect.height = event.xexpose.height;
	      XUnionRectWithRegion(&rect, exposureRegion, exposureRegion);

	      if (!(prev_xexpose_count = event.xexpose.count))
		FlushImage(exposureRegion);
	    }
	  } while (XEventsQueued(display, QueuedAlready));
	}
      }

      break;
    case 1:
      if ((n = read(stdin_fd, buf, sizeof(buf))) > 0) {
	int llen;

	if ((eol = memchr(buf, '\n', n))) {
	  llen = eol - buf;

	  if (llen > 0 && buf[llen - 1] == '\r')
	    --llen;
	}
	else
	  llen = n;

	pushLine(buf, llen);

	if (!eol)
	  break;
      }
      else if (!n) {
	if (!nl)
	  goto end;

	eol = NULL;
      }
      else if (errno == EINTR)
	break;
      else
	myExit(1, stderr, "read(): %s\n", strerror(errno));
    line_processing:
      if (nl > 1 && l[1] == ';')
	switch (l[0] - '0') {
	case IMGDISPLAY_DRAW:
	  DrawImage(&l[2]);
	  break;
	case IMGDISPLAY_CLIP:
	  ClipImage(&l[2]);
	  break;
	case IMGDISPLAY_CLEAR:
	  ClearImage();
	  break;
	case IMGDISPLAY_XSYNC:
#ifndef IMGDISPLAY_XSYNC_FACTOR
#define IMGDISPLAY_XSYNC_FACTOR (3)
#endif
	  nsyncs += IMGDISPLAY_XSYNC_FACTOR;
	  break;
	case IMGDISPLAY_RESET:
	  ResetImage(0);
	  nsyncs = 0;
	  break;
	case IMGDISPLAY_CLRIMG:
	  ResetImage(1);
	  nsyncs = 0;
	  break;
	default:
	  putchar(l[0]);
	  putchar('\n');
	  fflush(stdout);
	  break;
	}

      nl = 0;

      if (eol) {
	if (eol + 1 < buf + n) {
	  char *bol;

	  bol = eol + 1;

	  if ((eol = memchr(bol, '\n', n - (bol - buf)))) {
	    llen = eol - bol;

	    if (llen > 0 && bol[llen - 1] == '\r')
	      --llen;

	    pushLine(bol, llen);
	    goto line_processing;
	  }

	  pushLine(bol, n - (bol - buf));
	}

	break;
      }
      else
	goto end;
    default:
      if (errno == EINTR)
	break;

      myExit(1, stderr, "select(): %s\n", strerror(errno));
    }
  }
end:
  ClearImageAndExit();
  return 0;
}

static void
GetOption(int argc, char **argv)
{
  int i;

  for (i = 1; i < argc; i++)
    if (!strcmp("-bg", argv[i])) {
      if (++i >= argc)
	myExit(1, stderr, "No argument to ``-bg''\n");

      background = argv[i];
      defined_bg = 1;
    }
    else if (!strcmp("-x", argv[i])) {
      if (++i >= argc)
	myExit(1, stderr, "No argument to ``-x''\n");

      offset_x = atoi(argv[i]);
      defined_x = 1;
    }
    else if (!strcmp("-y", argv[i])) {
      if (++i >= argc)
	myExit(1, stderr, "No argument to ``-y''\n");

      offset_y = atoi(argv[i]);
      defined_y = 1;
    }
    else if (!strcmp("-test", argv[i]))
      defined_test = 1;
    else if (!strcmp("-debug", argv[i]))
      defined_debug = 1;
    else if (!strcmp("-ignore_exposure", argv[i]))
      defined_ignore_exposure = 1;
    else if (!strcmp("-sizemax", argv[i])) {
      if (++i >= argc)
	myExit(1, stderr, "No argument to ``-sizemax''\n");

      maxImageSize = atoi(argv[i]);
    }
    else
      myExit(1, stderr, "Unknown option ``%s''\n", argv[i]);
}

Image *
BsearchImage(int key, int *pos)
{
  int b, e, i, j;
  Image *image = NULL;

  for (b = 0, e = j = nImage ; b < e ;) {
    i = (b + e) / 2;

    if (key < imageTab[i]->index)
      e = j = i;
    else if (key > imageTab[i]->index)
      b = i + 1;
    else {
      j = i;
      image = imageTab[i];
      break;
    }
  }

  if (pos)
    *pos = j;

  return image;
}

void
UnlinkImage(Image *image)
{
  if (image->prev)
    image->prev->next = image->next;
  else
    imageHead = image->next;

  if (image->next)
    image->next->prev = image->prev;
  else
    imageTail = image->prev;

  image->prev = image->next = NULL;
}

void
DrawImage(char *buf)
{
  static ImlibData *id = NULL;
  ImlibImage *im;
  char *p = buf;
  int n = 0, x = 0, y = 0, w = 0, h = 0, sx = 0, sy = 0, sw = 0, sh = 0, pos;
  Image *image;
  ImageQueue *iq;

  if (!p)
    return;

  for (; isdigit(*p); p++)
    n = 10 * n + (*p - '0');

  if (*(p++) != ';')
    return;

  for (; isdigit(*p); p++)
    x = 10 * x + (*p - '0');

  if (*(p++) != ';')
    return;

  for (; isdigit(*p); p++)
    y = 10 * y + (*p - '0');

  if (*(p++) != ';')
    return;

  for (; isdigit(*p); p++)
    w = 10 * w + (*p - '0');

  if (*(p++) != ';')
    return;

  for (; isdigit(*p); p++)
    h = 10 * h + (*p - '0');

  if (*(p++) != ';')
    return;

  for (; isdigit(*p); p++)
    sx = 10 * sx + (*p - '0');

  if (*(p++) != ';')
    return;

  for (; isdigit(*p); p++)
    sy = 10 * sy + (*p - '0');

  if (*(p++) != ';')
    return;

  for (; isdigit(*p); p++)
    sw = 10 * sw + (*p - '0');

  if (*(p++) != ';')
    return;

  for (; isdigit(*p); p++)
    sh = 10 * sh + (*p - '0');

  if (*(p++) != ';')
    return;

  if (--n < 0)
    return;

  if (!id &&
      !(id = Imlib_init(display)))
    return;

  if (!(image = BsearchImage(n, &pos))) {
    int newImageSize;

    newImageSize = imageSize + w * h * window_depth / CHAR_BIT;

    if (newImageSize > maxImageSize) {
      int tail, size;
      Image *prev;

      for (image = imageTail ; image ; image = prev) {
	prev = image->prev;

	if (image->queued)
	  continue;

	BsearchImage(image->index, &tail);

	if (tail + 1 < nImage)
	  memmove(&imageTab[tail], &imageTab[tail + 1], sizeof(Image *) * (nImage - tail - 1));

	--nImage;
	UnlinkImage(image);
	BsearchImage(n, &pos);

	if (image->pixmap) {
	  XFreePixmap(display, image->pixmap);
	  image->pixmap = 0;
	  size = image->width * image->height * window_depth / CHAR_BIT;
	  imageSize -= size;
	  newImageSize -= size;

	  if (newImageSize <= maxImageSize)
	    goto set_image;
	}

	image->next = imageFree;
	imageFree = image;
      }
    }

    if (nImage >= maxImage) {
      maxImage = (nImage / 2 + 1) * 3;

      if (imageTab)
	imageTab = realloc(imageTab, sizeof(Image *) * maxImage);
      else
	imageTab = malloc(sizeof(Image *) * maxImage);

      if (!imageTab)
	myExit(1, stderr, "imageTab =malloc(sizeof(Image *) * %d): %s\n", maxImage, strerror(errno));
    }

    if ((image = imageFree))
      imageFree = image->next;
    else if (!(image = malloc(sizeof(Image))))
      myExit(1, stderr, "malloc(sizeof(Image)): %s\n", strerror(errno));

    image->pixmap = 0;
    image->next = NULL;
  set_image:
    if ((image->prev = imageTail))
      imageTail->next = image;
    else
      imageHead = image;

    imageTail = image;

    if (pos < nImage)
      memmove(&imageTab[pos + 1], &imageTab[pos], sizeof(Image *) * (nImage - pos));

    ++nImage;
    image->index = n;
    image->queued = 0;
    imageTab[pos] = image;
  }

  if (!image->pixmap) {
    im = Imlib_load_image(id, p);

    if (!im) {
      UnlinkImage(image);

      if (pos + 1 < nImage)
	memmove(&imageTab[pos], &imageTab[pos + 1], sizeof(Image *) * (nImage - pos - 1));

      --nImage;
      image->next = imageFree;
      imageFree = image;
      return;
    }

    if (!w)
      w = im->rgb_width;

    if (!h)
      h = im->rgb_height;

    if (!(image->pixmap = XCreatePixmap(display, parent, w, h,
					DefaultDepth(display, 0)))) {
      Imlib_kill_image(id, im);
      myExit(1, stderr, "XCreatePixmap(): %s\n", strerror(errno));
    }

    if (!imageGC &&
	!(imageGC = XCreateGC(display, parent, 0, NULL)))
      myExit(1, stderr, "XCreateGC(): %s\n", strerror(errno));

    XSetClipMask(display, imageGC, None);
    XSetForeground(display, imageGC, background_pixel);
    XFillRectangle(display, image->pixmap, imageGC, 0, 0, w, h);
    Imlib_paste_image(id, im, image->pixmap, 0, 0, w, h);
    Imlib_kill_image(id, im);
    image->width = w;
    image->height = h;
    imageSize += w * h * window_depth / CHAR_BIT;
  }

  if (nq >= nq_max) {
    nq_max = (nq / 2 + 1) * 3;

    if (q)
      q = realloc(q, sizeof(ImageQueue) * nq_max);
    else
      q = malloc(sizeof(ImageQueue) * nq_max);

    if (!q)
      myExit(1, stderr, "q = malloc(%d): %s\n", sizeof(ImageQueue) * nq_max, strerror(errno));
  }

  iq = &q[nq++];
  iq->image = image;
  iq->sx = sx;
  iq->sy = sy;
  iq->sw = sw;
  iq->sh = sh;
  iq->x = x;
  iq->y = y;
  image->queued = 1;
  UnlinkImage(image);

  if ((image->next = imageHead))
    imageHead->prev = image;
  else
    imageTail = image;

  imageHead = image;
}

void
ClipImage(char *p)
{
  char *ep;
  int i;
  short val[4];
  XRectangle rect;

  for (i = 0 ; *p ;) {
    val[i++] = strtol(p, &ep, 10);

    if (i == sizeof(val) / sizeof(val[0])) {
      rect.x = val[0];
      rect.y = val[1];
      rect.width = val[2];
      rect.height = val[3];

      if (!clipRegion &&
	  !(clipRegion = XCreateRegion()))
	myExit(1, stderr, "clipRegion = XCreateRegion(): %s\n", strerror(errno));

      XUnionRectWithRegion(&rect, clipRegion, clipRegion);
      i = 0;
    }

    p = *ep ? ep + 1 : ep;
  }
}

void
ClearImage(void)
{
  Image *image;

  if (clipRegion) {
    XDestroyRegion(clipRegion);
    clipRegion = NULL;
  }

  if (exposureRegion) {
    XDestroyRegion(exposureRegion);
    exposureRegion = NULL;
  }

  if (screenRegion) {
    XDestroyRegion(screenRegion);
    screenRegion = NULL;
  }

  if (imageRegion) {
    XDestroyRegion(imageRegion);
    imageRegion = NULL;
  }

  if (imageGC) {
    XFreeGC(display, imageGC);
    imageGC = NULL;
  }

  if (imageTab) {
    int i;

    for (i = 0; i < nImage; i++) {
      if (imageTab[i]->pixmap)
	XFreePixmap(display, imageTab[i]->pixmap);

      free(imageTab[i]);
    }

    free(imageTab);
    imageTab = NULL;
  }

  imageHead = imageTail = NULL;

  if (imageFree)
    do {
      image = imageFree;
      imageFree = image->next;
      free(image);
    } while (imageFree);

  nImage = maxImage = imageSize = 0;
  nq = nl = 0;
}

void
FlushImage(Region region)
{
  int i;
  ImageQueue *p;

  if (XEmptyRegion(region)) {
    XSync(display, xsync_option);
    return;
  }

  if (!imageGC &&
      !(imageGC = XCreateGC(display, parent, 0, NULL)))
    myExit(1, stderr, "XCreateGC(): %s\n", strerror(errno));

  if (clipRegion && !XEmptyRegion(clipRegion)) {
    if (imageRegion) {
      if (!XEmptyRegion(imageRegion))
	XSubtractRegion(imageRegion, imageRegion, imageRegion);
    }
    else if (!(imageRegion = XCreateRegion()))
      myExit(1, stderr, "imageRegion = XCreateRegion(): %s\n", strerror(errno));

    XSubtractRegion(region, clipRegion, imageRegion);
    XSetRegion(display, imageGC, imageRegion);
  }
  else if (region == screenRegion)
    XSetClipMask(display, imageGC, None);
  else
    XSetRegion(display, imageGC, region);

  for (i = 0 ; i < nq ;) {
    p = &q[i++];
    XCopyArea(display, p->image->pixmap, window, imageGC,
	      p->sx, p->sy,
	      (p->sw ? p->sw : p->image->width),
	      (p->sh ? p->sh : p->image->height),
	      p->x + offset_x, p->y + offset_y);
  }

  XSync(display, xsync_option);
}

void
ResetImage(int clear_p)
{
  int i;
  ImageQueue *p;

  for (i = 0 ; i < nq ;) {
    p = &q[i++];

    if (clear_p)
      XClearArea(display, window,
		 p->x + offset_x, p->y + offset_y,
		 (p->sw ? p->sw : p->image->width),
		 (p->sh ? p->sh : p->image->height),
		 True);

    p->image->queued = 0;
  }

  if (exposureRegion && !XEmptyRegion(exposureRegion))
    XSubtractRegion(exposureRegion, exposureRegion, exposureRegion);

  if (clipRegion && !XEmptyRegion(clipRegion))
    XSubtractRegion(clipRegion, clipRegion, clipRegion);

  nq = 0;
}

void
ClearImageAndExit(void)
{
  signal(SIGHUP, SIG_IGN);
  ClearImage();
  exit(0);
}

MySignalHandler
ExitBySignal(SIGNAL_ARG)
{
  ClearImageAndExit();
  SIGNAL_RETURN;
}
