/*
** Copyright (C) 1998,1999,2000,2001 Martin Roesch <roesch@clark.net>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/****************************************************
* spp_unidecode.h
*
* decodes UTF-8 encoded requests.  
* 
* Notes:
*   Arguments are the same as the http_decode plugin
*   Checks for NULL CGI and Unicode Directory Trans
*****************************************************/

/* $Id: spp_unidecode.c,v 1.8 2001/07/12 16:23:28 fygrave Exp $ */

#include "spp_unidecode.h"
#include "codes.h"
#include "decode.h"
#include <ctype.h>

#define MODNAME "spp_unidecode"

#define NOCGINULL "-cginull"
#define NOUNICODE "-unicode"

int Ucheck_cgi_null = 1;
int Ucheck_iis_unicode = 1;
int Uparse_arg; 

extern char *file_name;
extern int file_line;
extern int do_detect;

/* Instantiate the list of ports we're going to watch */
UPortList UnidecodePorts;

/*local function declares*/
int TranslateUnicode(char* InBuff, int InLength, char* OutBuff, int OutLength, Packet* p);
int GetNextChar(char* Buff, char* NextChar, Packet* p);
void CheckNull(unsigned long c, Packet* p);
void CheckDirTrans(unsigned long c, Packet* p);
void LogUnknown(Packet* p);
void LogInvalid(Packet* p);

/*
 * Function: SetupHttpDecode()
 *
 * Purpose: Registers the preprocessor keyword and initialization 
 *          function into the preprocessor list.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupUnidecode()
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
    RegisterPreprocessor("unidecode", UnidecodeInit);

#ifdef DEBUG
    printf("Preprocessor: Unidecode in setup...\n");
#endif
}


/*
 * Function: HttpDecodeInit(u_char *)
 *
 * Purpose: Processes the args sent to the preprocessor, sets up the
 *          port list, links the processing function into the preproc
 *          function list
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void UnidecodeInit(u_char *args)
{
#ifdef DEBUG
    printf("Preprocessor: HttpDecode Initialized\n");
#endif

    /* parse the argument list into a list of ports to normalize */
    USetPorts(args);
	init_codes();

    /* Set the preprocessor function into the function list */
    AddFuncToPreprocList(UPreprocUrlDecode);
}



/*
 * Function: SetPorts(char *)
 *
 * Purpose: Reads the list of port numbers from the argument string and 
 *          parses them into the port list data struct
 *
 * Arguments: portlist => argument list
 *
 * Returns: void function
 *
 */
void USetPorts(char *portlist)
{
    char **toks;
    int num_toks;
    int num_ports = 0;
    int num;

    if(portlist == NULL)
    {
        FatalError("ERROR %s (%d)=> No arguments to http_decode preprocessor!\n", file_name, file_line);
    }

    /* tokenize the argument list */
    toks = mSplit(portlist, " ", 31, &num_toks, '\\');

    /* convert the tokens and place them into the port list */
    for(num = 0; num < num_toks; num++)
    {
        if(!strncmp(NOUNICODE, toks[num], sizeof NOUNICODE))
        {
            Ucheck_iis_unicode = 0;
        }
        else if(!strncmp(NOCGINULL, toks[num], sizeof NOCGINULL))
        {
            Ucheck_cgi_null = 0;
        }
        else
        {
            UnidecodePorts.ports[num_ports++] = atoi(toks[num]);
        }
    }   

    UnidecodePorts.num_entries = num_ports;

#ifdef DEBUG
    printf("Decoding HTTP on %d ports: ", UnidecodePorts.num_entries);

    for(num_ports = 0; num_ports < UnidecodePorts.num_entries; num_ports++)
    {
        printf("%d ", UnidecodePorts.ports[num_ports]);
    }

    printf("\n");
#endif

}

/*
 * Function: PreprocUrlDecode(Packet *)
 *
 * Purpose: Inspects the packet's payload for "Escaped" characters and 
 *          converts them back to their ASCII values.  This function
 *          is based on the NCSA code and was contributed by Michael Henry!
 *
 * Arguments: p => pointer to the current packet data struct 
 *
 * Returns: void function
 *
 */
void UPreprocUrlDecode(Packet *p)
{
    char *url;       /* this is where the converted data will be written */
    char *index;     /* this is the index pointer to walk thru the data */
    char *end;       /* points to the end of the payload, for loop control */
    u_int16_t psize; /* payload size */
    int i;           /* loop counter */

#ifdef DEBUG
    printf("http decoder init on %d bytes\n", p->dsize);
#endif

    /* check to make sure we're talking TCP and that the TWH has already
       completed before processing anything */
    if(!PacketIsTCP(p))
    {
#ifdef DEBUG
        printf("It isn't TCP session traffic\n");
#endif
        return;
    }

    if(!IsTcpSessionTraffic(p))
    {
#ifdef DEBUG
        printf("It isn't TCP session traffic\n");
#endif
        return;
    }

    /* check the port against the decode port list */
    for(i = 0; i < UnidecodePorts.num_entries; i++)
    {
        if(UnidecodePorts.ports[i] == p->dp)
        {
            /* on match, normalize the data */
#ifdef DEBUG
            printf("Got HTTP traffic (%d bytes)!\n", p->dsize);
            printf("%s\n", p->data);
#endif
            /* setup the pointers */
            url =   (char *) p->data;
            index = (char *) p->data;
            end =   (char *) p->data + p->dsize;
            psize = (u_int16_t) (p->dsize);

            Uparse_arg = 0; /* nullify cgi arguments flag */
            /* walk thru each char in the payload */
			p->dsize = TranslateUnicode(url, p->dsize, index, p->dsize, p);

            /* set the payload size to reflect the new size */ 
            //p->dsize = psize;

#ifdef DEBUG
            printf("New size: %d\n", p->dsize);
            printf("converted data:\n");
            PrintNetData(stdout, p->data, p->dsize);
#endif

            return;
        }
    }
}



int TranslateUnicode(char* InBuff, int InLength, char* OutBuff, 
                     int OutLength, Packet* p)
{	
	int	i;
	int	out_pos;
	int	num_this_char;
	char	new_char;
    int first_space_found = 0;
    int uri_set = 0;
    int uri_finished = 0;
    char *cur;


	DebugMessage(DEBUG_DECODE, "Decoding\n");
			
	out_pos=0;
	i=0;

	while((num_this_char=GetNextChar(&InBuff[i], &new_char, p)) && 
           !uri_finished)
    {
        cur = OutBuff+out_pos;

		if (out_pos!=OutLength){
			OutBuff[out_pos++]=new_char;
		}
        else
        {
			OutBuff[out_pos]=0;
			return out_pos;
		}

        if(*cur == ' ' && !first_space_found)
        {
            first_space_found = 1;
            DebugMessage(DEBUG_DECODE, "found URI first space\n");
        }

        if(first_space_found && *cur != ' ' && !uri_set)
        {
            p->URI.uri = cur;
            uri_set = 1;
            DebugMessage(DEBUG_DECODE, "set URI at %p\n", p->URI.uri);
        }

        if(!uri_finished && uri_set && 
          (*cur == ' ' || *cur == '\n'))
        {
            char *foo = p->URI.uri;

            p->URI.length = (cur - foo);
            uri_finished  = 1;
            DebugMessage(DEBUG_DECODE, "URI length: %d\n", p->URI.length);
            DebugMessage(DEBUG_DECODE, "URI: %s\n", p->URI.uri);
        }

		i=i+num_this_char;
		
		if (i>InLength){
			return out_pos;
		}
	}

    /* after the URI has been decoded, shift any remaining data left and return
     * the new size
     */
    while(i < InLength)
    {
        OutBuff[out_pos++] = InBuff[i++];
    }
	
	return out_pos;
}


/*******************************************
* returns the number of bytes used in the 
* IN buff to make the character
*******************************************/
int GetNextChar(char* Buff, char* NextChar, Packet* p)
{
    char			Code[3];
    unsigned long 	Hex;
    unsigned long 	Hex2;
    unsigned long 	Hex3;

    NextChar[0]=Buff[0];

    if (Buff[0]==0){
        return 0;
    }

    if (Buff[0]==0x25)
    {
        Code[0]=Buff[1];
        Code[1]=Buff[2];
        Code[2]=0;
        Hex = strtoul(Code, NULL, 16);

        if ((Hex & 128)==0)
        {
            NextChar[0] = codes[Hex];
            CheckNull(Hex, p);
            CheckDirTrans(Hex, p);
            return 3;
        }
        else if ((Hex & 224) == 192)
        {
            if (Buff[3]!=0x25)
            {
                DebugMessage(DEBUG_DECODE, "Invalid UTF-8 Code\n");
                LogInvalid(p);
                NextChar[0]=Buff[0];
                return 1;
            }

            Code[0]=Buff[4];
            Code[1]=Buff[5];
            Code[2]=0;
            Hex2 = strtoul(Code, NULL, 16);

            if ((Hex2 & 192)!=128)
            {
                DebugMessage(DEBUG_DECODE, "Second Byte Invalid\n");
                LogInvalid(p);
                NextChar[0]=Buff[0];
                return 1;
            }

            Hex = Hex & 31;
            Hex2 = Hex2 & 63;	
            CheckNull((Hex * 64) | Hex2, p);
            CheckDirTrans((Hex * 64) | Hex2, p);

            if (codes[(Hex * 64) | Hex2]==0)
            {
                DebugMessage(DEBUG_DECODE, "Unknown mapping\n");
                LogUnknown(p);
                NextChar[0]=Buff[0];
                return 1;
            }
            else
            {
                NextChar[0] = codes[(Hex * 64) | Hex2];
                return 6;
            }
        }
        else if ((Hex & 240) == 224)
        {
            if (Buff[3]!=0x25)
            {
                DebugMessage(DEBUG_DECODE, "Invalid UTF-8 Code\n");
                LogInvalid(p);
                NextChar[0]=Buff[0];
                return 1;
            }

            if (Buff[6]!=0x25)
            {
                DebugMessage(DEBUG_DECODE, "Invalid UTF-8 Code\n");
                LogInvalid(p);
                NextChar[0]=Buff[0];
                return 1;
            }

            Code[0]=Buff[4];
            Code[1]=Buff[5];
            Code[2]=0;
            Hex2 = strtoul(Code, NULL, 16);

            if ((Hex2 & 192)!=128)
            {
                DebugMessage(DEBUG_DECODE, "Second Byte Invalid\n");
                LogInvalid(p);
                NextChar[0]=Buff[0];
                return 1;
            }

            Code[0]=Buff[7];
            Code[1]=Buff[8];
            Code[2]=0;
            Hex3 = strtoul(Code, NULL, 16);

            if ((Hex3 & 192)!=128)
            {
                DebugMessage(DEBUG_DECODE, "Third Byte Invalid\n");
                LogInvalid(p);
                NextChar[0]=Buff[0];
                return 1;
            }

            Hex = Hex & 15;
            Hex2 = Hex2 & 63;	
            Hex3 = Hex3 & 63;	

            if ( codes[(Hex * 4096) | (Hex2 * 64) | Hex3] ==0)
            {
                DebugMessage(DEBUG_DECODE, "Unknown Mapping\n");
                LogUnknown(p);
                NextChar[0]=Buff[0];
                return 1;
            }
            else
            {
                NextChar[0] = codes[(Hex * 4096) | (Hex2 * 64) | Hex3];
                CheckNull((Hex * 4096) | (Hex2 * 64) | Hex3, p);
                CheckDirTrans((Hex * 4096) | (Hex2 * 64) | Hex3, p);
                return 9;
            }
        }		
    }
    else
    {
        if (Buff[0] == '?') Uparse_arg = 1; /* tail is presumably cgi params. */
        NextChar[0] = Buff[0];
    }

    return 1;
}

/*************************************************
* Check for Null cgi attack
*************************************************/
void CheckNull(unsigned long c, Packet* p){
	char logMessage[180];
    Event event;
	
	if (!Ucheck_cgi_null || Uparse_arg) return;
	
	if (c==0 && do_detect)
    {
		snprintf(logMessage, sizeof(logMessage), 
			MODNAME ": CGI Null Byte attack detected");
        
        SetEvent(&event, GENERATOR_SPP_UNIDECODE, 
                UNIDECODE_CGINULL_ATTACK, 1, 0, 0, 0);

		CallAlertFuncs(p, logMessage, NULL, &event);
		CallLogFuncs(p, logMessage, NULL, &event);
        do_detect = 0;
	}
}

/***************************************************
* Check for the unicode directory transveral attack
***************************************************/
void CheckDirTrans(unsigned long c, Packet* p){
	char logMessage[180];
    Event event;
	
	if (!Ucheck_iis_unicode || Uparse_arg) return;
	
	if (((c==0x5c) || (c==0x2f) || (c==0x2e)) && do_detect)
    {
		snprintf(logMessage, sizeof(logMessage), 
			MODNAME ": Unicode Directory Transversal attack detected");

        SetEvent(&event, GENERATOR_SPP_UNIDECODE, 
                UNIDECODE_DIRECTORY_TRAVERSAL, 1, 0, 0, 0);

		CallAlertFuncs(p, logMessage, NULL, &event);
		CallLogFuncs(p, logMessage, NULL, &event);
        do_detect = 0;
	}
}

/****************************************************
* We don't know what this is
****************************************************/
void LogUnknown(Packet* p)
{
    char logMessage[180];
    Event event;

    if(!do_detect)
        return;

    snprintf(logMessage, sizeof(logMessage), 
            MODNAME ": Unknown Unicode Mapping.  If you know what this maps "
            "to contact anonpoet@inconnu.isu.edu.");

    SetEvent(&event, GENERATOR_SPP_UNIDECODE, 
            UNIDECODE_UNKNOWN_MAPPING, 1, 0, 0, 0);

    CallAlertFuncs(p, logMessage, NULL, &event);
    CallLogFuncs(p, logMessage, NULL, &event);

    do_detect = 0;
}

/***************************************************
* This combination can't exist
***************************************************/
void LogInvalid(Packet* p)
{
	char logMessage[180];
    Event event;
	
    if(!do_detect)
        return;

	snprintf(logMessage, sizeof(logMessage), 
		MODNAME ": Invalid Unicode String detected");

    SetEvent(&event, GENERATOR_SPP_UNIDECODE, 
            UNIDECODE_INVALID_MAPPING, 1, 0, 0, 0);

	CallAlertFuncs(p, logMessage, NULL, &event);
	CallLogFuncs(p, logMessage, NULL, &event);

    do_detect = 0;
}
