package Net::SnortLog;
##############################################################################
# This is a perl module to parse the log files create by Snort
# Please read the man pages for a good description on how it works
#
# This module is copyrighted on the GNU GPL by Craig Smith (May 2000)
#############################################################################
use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;
require AutoLoader;

@ISA = qw(Exporter AutoLoader);

$VERSION='0.01';

my @PktStack;
my $PktCount;

###
# Constructor
###
sub new {
  my $proto = shift;
  my $class = ref($proto) || $proto;
  my $self = {};
  my @PktStack = undef;
  my $PktCount=0;
  bless($self, $class);
  return $self;
}

###
# Methods
###

## retrieves a list of files matching a certian type
# Usage: get_log_files(LOG_DIR, TYPE) # Eg: get_log_files("./log","TCP")
# Returns an array of file lists.
sub get_log_files {
 my $self = shift;
 my ($log_dir, $type) = @_;
 my @filelist;
 my @files;
 my $file;
 my @packets;
 my $packet;
 if($log_dir && $type) {
   if(-e $log_dir && -d $log_dir) {
     opendir LOGDIR, $log_dir || die "Could not open $log_dir: $!";
     @files=readdir(LOGDIR);
     foreach $file (@files) {
      if($file=~/(\d*).(\d*).(\d*).(\d*)/) {
	opendir PKTDIR, "$log_dir/$file" || die "Could not open $log_dir/file $!";
	@packets=readdir(PKTDIR);
	foreach $packet (@packets) {
	   if($packet=~/$type/i) {
		push @filelist, "$log_dir/$file/$packet";
	   }
	}
        close(PKTDIR);
      }
     }
     close(LOGDIR);
   } else {
     print "ERROR: $log_dir is not a valid directory\n";
   }
 } else {
   print "ERROR: use read_log_files(LOG_DIR, TYPE);\n";
 }
 return @filelist;
}

## Parses a snort file for packet info
# Usage: parse_file(FILE);
#
sub parse_file {
 my $self = shift;
 my $file = shift;
 my $line;
 my $packet = {};
 if($file && -e $file) {
   open LOG, $file || die "Coudn't open $file for parsing: $!";
   while(<LOG>) {
     $line=$_;
     chomp($line);
     if($line=~/(\d+)\/(\d+)-(\d+):(\d+):(\S*)/) {
	$packet->{TIME}="$1/$2-$3:$4:$5";
     }
     if($line=~/(.*)\[\*\*\]/) {
	$packet->{MSG}=$1;
        $packet->{MSG}=~s/\[\*\*\] //;
     }
     if($line=~/(\S*) -> (\S*) type:(\S*) len:(\S*)/) {
	$packet->{SRCMAC}=$1;
	$packet->{DSTMAC}=$2;
	$packet->{TYPE}=$3;
	$packet->{LEN}=$4;
     }
     if($line=~/(\S*):(\d*) -> (\S*):(\d*)/) {
	$packet->{SRCIP}=$1;
	$packet->{SRCPORT}=$2;
	$packet->{DSTIP}=$3;
	$packet->{DSTPORT}=$4;
     }
     if($line=~/(\S*) -> (\S*) ICMP/){
	$packet->{SRCIP}=$1;
	$packet->{DSTIP}=$2;
     }
     if($line=~/(\S*) TTL:(\d*) TOS:(\S*) ID:(\d*)/) {
	$packet->{PROTO}=$1;
	$packet->{TTL}=$2;
	$packet->{TOS}=$3;
	$packet->{ID}=$4;
     }
     if($line=~/ID:(\d*)  DF/) {
	$packet->{DF}=$1;
     }
     if($line=~/(\S*) Seq: (\S*)   Ack: (\S*)   Win: (\S*)/) {
	$packet->{FLAGS}=$1;
	$packet->{SEQ}=$2;
	$packet->{ACK}=$3;
	$packet->{WIN}=$4;
     }
     if($line=~/^Len: (\d*)/) {
	$packet->{UDP_LEN}=$1;
     }
     if($line=~/^(\w+) (\w+) (\w+) /) {	# Not the best but it works (usually)
	$packet->{DATA}.="$line\n";
     }
     if(!$line && $packet->{TIME}) {	# Time should be set
	push @PktStack, $packet;
	$packet = undef;	# I think this will clear it
     }
   }
   close LOG;
 } else {
   print "ERROR: Couldn't parse $file\n";
 } 
}

##
#
sub next_pkt {
  my $self=shift;
  my $packet={};
  if($PktStack[$PktCount]) {
     $packet=$PktStack[$PktCount];
     $PktCount++;
     return $packet;
  } else {
     @PktStack = undef;
     return;
  }
}

1; # return

__END__

=head1 NAME

Net::SnortLog - Perl module to analyze Snort log files

=head1 SYNOPSIS

  use Net::Snortlog;
  $data = new Net::SnortLog;
  @filelist=$data->get_log_files("/var/log/snort", "TCP");
  $data->parse_file("/var/log/snort/alert");
  $packet={};
  $packet=$data->next_pkt;

=head1 DESCRIPTION

This package is used to locate and parse log files that have been created
with the Snort Network application.  Snort is used as a lightweight and
customizable Intrusion Detection System.  This module allows you to quickly
and easily parse many different formats of the data that can be created
by snort.  It then returns the data into an easy to use data structure.

Function List:

B<get_log_files>  LOG_DIR, TYPE

	This function returns an array of target files.  You have to give
	the log directory just like you would Snort.  For example if you
	used the parameter Snort -l ./log then the LOG_DIR should be
	./log.  TYPE is the type of file you want to gather, eg: "TCP" or
	"UDP".  You can also use "TCP|UDP" to get both types.

B<parse_file>	FILE

	This function takes a snort file (either a log file or an alert 
	file) and parses the data into a nice internal array.  If the 
	array is not wiped out (See the next function) then each time
	you call this function it will append data to its internal i
	structure.  This can be benificial and is a feature not a bug :)

B<next_pkt>

	This function extracts data from the internal data structure
	created by parse_file.  Each time it is called in extracts 
	another chunk of data.  When the data is exhausted it clears 
	out the internal structure and returns NULL.  This is meant to
	run in a while loop.  (See EXAMPLES)

=head1 DATA STRUCTURE

Here is a list of the returned data structure.  If the option 
was not available due to wrong packet type or wrong parameters 
supplied to Snort then they will be undefined.

	$packet->{TIME}		# Time of packet
	$packet->{MSG}		# Alert Msg if applicable
	$packet->{SRCMAC}	# Source MAC address 
	$packet->{DSTMAC}	# Destination MAC address 
	$packet->{TYPE}		# Packet Type 
	$packet->{LEN}		# Packet Length in Hex 
	$packet->{SRCIP}	# Source IP Address
	$packet->{SRCPORT}	# Source Port Address
	$packet->{DSTIP}	# Destination IP Address
	$packet->{DSTPORT}	# Destination PORT Address
	$packet->{PROTO}	# Porotocal (TCP/UDP/ICMP)
	$packet->{TTL}		# TTL - Time To Live
	$packet->{TOS}		# TOS - Type Of Service
	$packet->{ID}		# ID - ID number
	$packet->{DF}		# Don't Fragment ('DF' if set else NULL)
	$packet->{FLAGS}	# TCP Flags (snort format)
	$packet->{SEQ}		# TCP Sequence Number
	$packet->{ACK}		# TCP Acknowledgment Number
	$packet->{WIN}		# TCP Window Size
	$packet->{UDP_LEN}	# UDP Packet Length
	$packet->{DATA}		# Packet Data (scalar with \n's)

=head1 EXAMPLES

	my @our_files;	# Will contain all files we want to parse
	my $packet={};	# Anonymous has reference is used for the data

	my $data=new Net::SnortLog;

	# Read in all TCP files
	@our_files=$data->get_log_files("/var/log/snort", "TCP");
	foreach $file (@our_files) {
		$data->parse_file($file);
	}

	# Display all TCP files (source ip and destination ip)
	while($packet=$data->next_pkt) {
		print "SOURCE: $packet->{SRCIP}\n";
		print "DESTINATION: $packet->{DSTIP}\n";
	}

=head1 AUTHOR

Craig Smith	smithc@cinstate.cc.oh.us

=cut

