/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLAuthenticationQuery.cpp - SAML authentication query implementation

   Scott Cantor
   5/7/04

   $History:$
*/

#include "internal.h"

using namespace saml;
using namespace std;

SAMLAuthenticationQuery::SAMLAuthenticationQuery(SAMLSubject* subject, const XMLCh* authMethod)
    : SAMLSubjectQuery(subject), m_authMethod(XML::assign(authMethod))
{
    RTTI(SAMLAuthenticationQuery);
}

SAMLAuthenticationQuery::SAMLAuthenticationQuery(DOMElement* e) : SAMLSubjectQuery(e), m_authMethod(NULL)
{
    RTTI(SAMLAuthenticationQuery);
    fromDOM(e);
}

SAMLAuthenticationQuery::SAMLAuthenticationQuery(std::istream& in) : SAMLSubjectQuery(in), m_authMethod(NULL)
{
    RTTI(SAMLAuthenticationQuery);
    fromDOM(m_document->getDocumentElement());
}

SAMLAuthenticationQuery::~SAMLAuthenticationQuery()
{
    if(m_bOwnStrings)
        XMLString::release(&m_authMethod);
}

void SAMLAuthenticationQuery::ownStrings()
{
    if (!m_bOwnStrings) {
        m_authMethod=XML::assign(m_authMethod);
        m_bOwnStrings=true;
    }
}

void SAMLAuthenticationQuery::fromDOM(DOMElement* e)
{
    if (SAMLConfig::getConfig().strict_dom_checking) {
        if (XMLString::compareString(XML::SAMLP_NS,e->getNamespaceURI()))
            throw MalformedException(SAMLException::REQUESTER,"SAMLAuthenticationQuery::fromDOM() root element isn't in samlp namespace");
        if (XMLString::compareString(L(AuthenticationQuery),e->getLocalName())) {
            auto_ptr<saml::QName> type(saml::QName::getQNameAttribute(e,XML::XSI_NS,L(type)));
            if ((XMLString::compareString(L(Query),e->getLocalName()) && XMLString::compareString(L(SubjectQuery),e->getLocalName())) ||
                !type.get() || XMLString::compareString(XML::SAMLP_NS,type->getNamespaceURI()) ||
                XMLString::compareString(L(AuthenticationQueryType),type->getLocalName()))
                throw MalformedException(SAMLException::REQUESTER,"SAMLAuthenticationQuery::fromDOM() missing samlp:AuthenticationQuery element at root");
        }
    }

    if (e->hasAttributeNS(NULL,L(AuthenticationMethod)))
        m_authMethod = const_cast<XMLCh*>(e->getAttributeNS(NULL,L(AuthenticationMethod)));
    checkValidity();
}

void SAMLAuthenticationQuery::setAuthMethod(const XMLCh* authMethod)
{
    if (XML::isEmpty(authMethod))
        throw SAMLException("authMethod cannot be null or empty");
        
    if (m_bOwnStrings)
        XMLString::release(&m_authMethod);
    else {
        m_authMethod=NULL;
        ownStrings();
    }
    m_authMethod=XML::assign(authMethod);
    setDirty();
}

DOMElement* SAMLAuthenticationQuery::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* q=doc->createElementNS(XML::SAMLP_NS,L(AuthenticationQuery));
    if (xmlns)
        q->setAttributeNS(XML::XMLNS_NS,L(xmlns),XML::SAMLP_NS);
    return q;
}

DOMNode* SAMLAuthenticationQuery::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLSubjectQuery::toDOM(doc,xmlns);
    DOMElement* q=static_cast<DOMElement*>(m_root);
    doc=q->getOwnerDocument();

    if (m_bDirty) {
        if (!XML::isEmpty(m_authMethod))
            q->setAttributeNS(NULL,L(AuthenticationMethod),m_authMethod);
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(q,XML::SAMLP_NS);
    }

    return q;
}

SAMLObject* SAMLAuthenticationQuery::clone() const
{
    return new SAMLAuthenticationQuery(static_cast<SAMLSubject*>(m_subject->clone()),m_authMethod);
}
