/***************************************************************************
                          theme.c  -  description
                             -------------------
    begin                : Fri Feb 15 2002
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#include "lbreakout.h"
#include "config.h"
#include "list.h"
#include "theme.h"
#include "levels.h"
#include "player.h"
#include "shrapnells.h"
#include "bricks.h"
#ifdef SOUND
#include "audio.h"
#endif

extern Sdl sdl;
extern Config config;
extern int ball_w, ball_h;
extern int shot_w, shot_h;
/* theme name list */
char **theme_names = 0;
int theme_count = 0;
/* current theme name */
char theme_name[32] = "Default";

/*
====================================================================
Themeable gfx and sounds.
====================================================================
*/
SDL_Surface *exp_pic = 0;
SDL_Surface *exp_dark_pic = 0;
Font        *font = 0;
#ifdef SOUND
Sound_Chunk *wav_click = 0;
Sound_Chunk *wav_damn = 0, *wav_dammit = 0, *wav_wontgiveup = 0, *wav_excellent = 0, *wav_verygood = 0;
#endif
SDL_Surface *brick_pic = 0;
SDL_Surface *brick_shadow = 0;
#ifdef SOUND
Sound_Chunk *wav_boom = 0;
#endif
SDL_Surface *paddle_pic = 0;
SDL_Surface *paddle_shadow = 0;
SDL_Surface *weapon_pic = 0;
SDL_Surface *weapon_shadow = 0;
#ifdef SOUND
Sound_Chunk *wav_expand = 0, *wav_shrink = 0, *wav_frozen = 0;
#endif
SDL_Surface *ball_pic = 0; /* ball pictures */
SDL_Surface *ball_shadow = 0;
#ifdef SOUND
Sound_Chunk *wav_reflect = 0;
Sound_Chunk *wav_attach = 0;
#endif
SDL_Surface *shot_pic = 0;
SDL_Surface *shot_shadow = 0;
#ifdef SOUND
Sound_Chunk *wav_shot = 0;
#endif
SDL_Surface *extra_pic = 0; /* graphics */
SDL_Surface *extra_shadow = 0;
SDL_Surface *wall_pic = 0;
#ifdef SOUND
Sound_Chunk *wav_score = 0, *wav_metal = 0, *wav_std = 0, *wav_wall = 0, *wav_joker = 0;
Sound_Chunk *wav_goldshower = 0;
Sound_Chunk *wav_speedup = 0, *wav_speeddown = 0;
Sound_Chunk *wav_chaos = 0, *wav_darkness = 0, *wav_ghost = 0;
Sound_Chunk *wav_timeadd = 0, *wav_expl_ball = 0, *wav_weak_ball = 0;
Sound_Chunk *wav_bonus_magnet = 0, *wav_malus_magnet = 0, *wav_disable = 0;
#endif
SDL_Surface *shine_pic = 0;
SDL_Surface *frame_left, *frame_top, *frame_right;
SDL_Surface *frame = 0; /* assembled frame */
SDL_Surface *frame_shadow = 0; /* shadow of frame */
SDL_Surface *lamps = 0; /* life lamps */
Font *frame_font;
#ifdef SOUND
Sound_Chunk *wav_life_up = 0, *wav_life_down = 0;
#endif
SDL_Surface *bkgnds[BACK_COUNT];
int bkgnd_count = 0;

/*
====================================================================
Locals
====================================================================
*/

/*
====================================================================
Load a picture or sound from theme directory or (if not found)
from original source directory.
====================================================================
*/
SDL_Surface* theme_load_surf( char *name )
{
    SDL_Surface *surf = 0;
    char path[512];
    sprintf( path, "%s/.lgames/lbreakout2-themes/%s/%s", getenv( "HOME" ), theme_name, name );
    if ( strequal( theme_name, "Default" ) || ( surf = load_surf( path, SDL_SWSURFACE | SDL_NONFATAL ) ) == 0 )
        surf = load_surf( name, SDL_SWSURFACE );
    return surf;
}
#ifdef SOUND
Sound_Chunk* theme_load_sound( char *name )
{
    Sound_Chunk *sound = 0;
    char path[512];
    sprintf( path, "%s/.lgames/lbreakout2-themes/%s/%s", getenv( "HOME" ), theme_name, name );
    if ( strequal( theme_name, "Default" ) ||  ( sound = sound_chunk_load( path ) ) == 0 )
        sound = sound_chunk_load( name );
    return sound;
}
#endif
Font* theme_load_font_fixed( char *name, int start, int len, int width ) 
{
    Font *font = 0;
    char path[512];
    sprintf( path, "%s/.lgames/lbreakout2-themes/%s/%s", getenv( "HOME" ), theme_name, name );
    if ( strequal( theme_name, "Default" ) ||  ( font = load_fixed_font( path, start, len, width, SDL_SWSURFACE | SDL_NONFATAL ) ) == 0 )
        font = load_fixed_font( name, start, len, width, SDL_SWSURFACE );
    return font;
}

/*
====================================================================
The frame is a bit complex to be done.
====================================================================
*/
void theme_load_frame()
{
    Uint32 ckey;
    SDL_Rect rect = { BRICK_WIDTH, BRICK_HEIGHT, sdl.screen->w - BRICK_WIDTH * 2, sdl.screen->h - BRICK_HEIGHT };

    /* load resources */
    frame_left = theme_load_surf( "fr_left.bmp" );
    frame_top = theme_load_surf( "fr_top.bmp" );
    frame_right = theme_load_surf( "fr_right.bmp" );

    /* use most right pixel of fr_right */
    ckey = get_pixel( frame_right, frame_right->w - 1, 0 );
    /* assemble frame */
    frame = create_surf( sdl.screen->w, sdl.screen->h, SDL_SWSURFACE );
    SDL_FillRect( frame, &rect, ckey );
    /* left part */
    DEST( frame, 0, 0, BRICK_WIDTH, sdl.screen->h );
    SOURCE( frame_left, 0, 0 );
    blit_surf();
    /* top part */
    DEST( frame, BRICK_WIDTH, 0, sdl.screen->w - BRICK_WIDTH * 2, BRICK_HEIGHT );
    SOURCE( frame_top, 0, 0 );
    blit_surf();
    /* right part */
    DEST( frame, sdl.screen->w - BRICK_WIDTH, 0, BRICK_WIDTH, sdl.screen->h );
    SOURCE( frame_right, 0, 0 );
    blit_surf();
    /* create shadow */
    SDL_SetColorKey( frame, SDL_SRCCOLORKEY, ckey );
    frame_shadow = create_shadow( frame, 0, 0, frame->w, frame->h );

    /* load other resources */
    lamps = theme_load_surf( "life.bmp" );
    frame_font = theme_load_font_fixed( "f_frame.bmp", 32, 96, 8 );
    frame_font->align = ALIGN_X_RIGHT | ALIGN_Y_CENTER;
	
	/* waves */
#ifdef SOUND
    wav_life_up = theme_load_sound( "gainlife.wav" );
    wav_life_down = theme_load_sound( "looselife.wav" );
#endif
}

/*
====================================================================
Load backgrounds. If no background was found use the original
backgrounds. Else count the number of new backgrounds and use these.
====================================================================
*/
void theme_load_bkgnds()
{
    int i;
    struct stat filestat;
    char path[512];
    char fname[512];
    sprintf( fname, "%s/.lgames/lbreakout2-themes/%s/back0.png", getenv( "HOME" ), theme_name );
    if ( strequal( theme_name, "Default" ) || stat( fname, &filestat ) == -1 ) {
        /* use original backs */
        bkgnd_count = BACK_COUNT;
        sprintf( path, "." );
    }
    else {
        /* use new backs */
        sprintf( path, "%s/.lgames/lbreakout2-themes/%s", getenv( "HOME" ), theme_name );
        bkgnd_count = -1;
        do {
            bkgnd_count++;
            if ( bkgnd_count == BACK_COUNT )
                break;
            sprintf( fname, "%s/back%i.png", path, bkgnd_count );
        } while ( stat( fname, &filestat ) == 0 );
    }
    /* load backs */
    for ( i = 0; i < bkgnd_count; i++ ) {
        sprintf( fname, "%s/back%i.png", path, i );
        bkgnds[i] = load_surf( fname, SDL_SWSURFACE );
    }
}

/*
====================================================================
Publics
====================================================================
*/

/*
====================================================================
Get/delete names of all themes found in themes directory.
====================================================================
*/
void theme_get_list()
{
    int i;
    char dir[256];
    char path[512];
    DIR *hdir = 0;
    struct dirent *entry = 0;
    struct stat estat;
    List *list = 0;
    /* auxilary list */
    list = list_create( LIST_NO_AUTO_DELETE, NO_CALLBACK );
    /* theme directory */
    sprintf( dir, "%s/.lgames/lbreakout2-themes", getenv( "HOME" ) );
    if ( ( hdir = opendir( dir ) ) != 0 ) {
        while ( ( entry = readdir( hdir ) ) ) {
            if ( entry->d_name[0] == '.' ) 
                continue;
            /* full path */
            sprintf( path, "%s/%s", dir, entry->d_name );
            stat( path, &estat );
            if ( S_ISDIR( estat.st_mode ) )
                list_add( list, strdup( entry->d_name ) );
        }
        closedir( hdir );
    }
    /* create static list. first entry is default */
    theme_count = list->count + 1;
    theme_names = calloc( theme_count, sizeof( char* ) );
    theme_names[0] = strdup( "Default" );
    for ( i = 0; i < list->count; i++ )
        theme_names[i + 1] = list_get( list, i );
    list_delete( list );
    /* if config's theme count is not equal reset the theme_id to 0 */
    if ( theme_count != config.theme_count ) {
        config.theme_count = theme_count;
        config.theme_id = 0;
    }
}
void theme_delete_list()
{
    int i;
    if ( theme_names ) {
        for ( i = 0; i < theme_count; i++ )
            if ( theme_names[i] ) free( theme_names[i] );
        free( theme_names );
        theme_names = 0;
    }
}

/*
====================================================================
Load theme by name or delete current theme (is done by load too
so it might be used to change a theme as well.)
====================================================================
*/
void theme_load( char *name )
{
    int i;
    /* clear resources */
    theme_delete();
    /* set theme name */
    strcpy( theme_name, name );
    /* load resources */
    /* explosions */
    exp_pic = theme_load_surf( "explosions.bmp" );
    exp_dark_pic = theme_load_surf( "explosions_dark.bmp" );
    /* main game font */
    font = theme_load_font_fixed( "f_game.bmp", 32, 96, 10 );
    font->align = ALIGN_X_CENTER | ALIGN_Y_CENTER;
    /* main game sounds */
#ifdef SOUND
    wav_click = theme_load_sound( "click.wav" );
    wav_damn = theme_load_sound( "damn.wav" );
    wav_dammit = theme_load_sound( "dammit.wav" );
    wav_wontgiveup = theme_load_sound( "wontgiveup.wav" );
    wav_excellent = theme_load_sound( "excellent.wav" );
    wav_verygood = theme_load_sound( "verygood.wav" );
#endif
    /* brick pics and shadow template */
    brick_pic = theme_load_surf( "bricks.bmp" );
    brick_shadow = create_shadow( brick_pic, 0, 0, BRICK_WIDTH, BRICK_HEIGHT );
    /* brick destruction sound */
#ifdef SOUND
    wav_boom = theme_load_sound( "exp.wav" );
#endif
    /* paddle and weapon */
    paddle_pic = theme_load_surf( "paddle.bmp" );
    SDL_SetColorKey( paddle_pic, SDL_SRCCOLORKEY, get_pixel( paddle_pic, 0, 0 ) );
    weapon_pic = theme_load_surf( "weapon.bmp" );
    paddle_shadow = create_shadow( paddle_pic, 0, 0, paddle_pic->w, paddle_pic->h );
    weapon_shadow = create_shadow( weapon_pic, 0, 0, weapon_pic->w, weapon_pic->h );
    /* paddle sounds */
#ifdef SOUND
    wav_expand = theme_load_sound( "expand.wav" );
    wav_shrink = theme_load_sound( "shrink.wav" );
    wav_frozen = theme_load_sound( "freeze.wav" );
#endif
    /* ball gfx */
    ball_pic = theme_load_surf( "ball.bmp" );
    SDL_SetColorKey( ball_pic, SDL_SRCCOLORKEY, get_pixel( ball_pic, 0, 0 ) );
    ball_shadow = create_shadow( ball_pic, 0, 0, ball_w, ball_h );
    /* ball sounds */
#ifdef SOUND
    wav_reflect = theme_load_sound( "reflect.wav" );
    wav_attach = theme_load_sound( "attach.wav" );
#endif
    /* shot gfx */
    shot_pic = theme_load_surf( "shot.bmp" );
    shot_shadow = create_shadow( shot_pic, 0, 0, shot_w, shot_h );
#ifdef SOUND
    wav_shot = theme_load_sound( "shot.wav" );
#endif
    /* extra pic */
    extra_pic = theme_load_surf( "extras.bmp" );
	SDL_SetColorKey( extra_pic, 0, 0 );
    extra_shadow = create_shadow( extra_pic, 0, 0, BRICK_WIDTH, BRICK_HEIGHT );
    /* wall */
    wall_pic = create_surf( sdl.screen->w - 2 * BRICK_WIDTH, BRICK_HEIGHT, SDL_SWSURFACE );
    /* bricks must have been loaded at this point! */
    for ( i = 0; i < MAP_WIDTH - 2; i++ ) {
        DEST( wall_pic, i * BRICK_WIDTH, 0, BRICK_WIDTH, BRICK_HEIGHT );
        SOURCE( brick_pic, BRICK_WIDTH, 0 );
        blit_surf();
    }
    /* extra sounds */
#ifdef SOUND
    wav_score = theme_load_sound( "score.wav" );
    wav_metal = theme_load_sound( "metal.wav" );
    wav_speedup = theme_load_sound( "speedup.wav" );
    wav_speeddown = theme_load_sound( "speeddown.wav" );
    wav_std = theme_load_sound( "standard.wav" );
    wav_wall = theme_load_sound( "wall.wav" );
    wav_joker = theme_load_sound( "joker.wav" );
    wav_goldshower = theme_load_sound( "goldshower.wav" );
    wav_chaos = theme_load_sound( "chaos.wav" );
    wav_darkness = theme_load_sound( "darkness.wav" );
    wav_ghost = theme_load_sound( "ghost.wav" );
    wav_timeadd = theme_load_sound( "timeadd.wav" );
    wav_expl_ball = theme_load_sound( "expl_ball.wav" );
    wav_weak_ball = theme_load_sound( "weak_ball.wav" );
    wav_bonus_magnet = theme_load_sound( "bonus_magnet.wav" );
    wav_malus_magnet = theme_load_sound( "malus_magnet.wav" );
    wav_disable = theme_load_sound( "disable.wav" );
#endif
    /* shine pic */
    shine_pic = theme_load_surf( "shine.bmp" );
    /* frame */
    theme_load_frame();
    /* backgrounds */
    theme_load_bkgnds();
}
void theme_delete()
{
    int i;
    free_surf( &exp_pic );
    free_surf( &exp_dark_pic );
    free_font( &font );
#ifdef SOUND
    sound_chunk_free( &wav_click );
    sound_chunk_free( &wav_damn );
    sound_chunk_free( &wav_dammit );
    sound_chunk_free( &wav_wontgiveup );
    sound_chunk_free( &wav_excellent );
    sound_chunk_free( &wav_verygood );
#endif
    free_surf( &brick_pic );
    free_surf( &brick_shadow );
#ifdef SOUND
    sound_chunk_free( &wav_boom );
#endif
    free_surf( &paddle_pic );
    free_surf( &weapon_pic );
    free_surf( &paddle_shadow );
    free_surf( &weapon_shadow );
#ifdef SOUND
    sound_chunk_free( &wav_expand );
    sound_chunk_free( &wav_shrink );
    sound_chunk_free( &wav_frozen );
#endif
    free_surf( &ball_pic );
    free_surf( &ball_shadow );
#ifdef SOUND
    sound_chunk_free( &wav_reflect );
    sound_chunk_free( &wav_attach );
#endif
    free_surf( &shot_pic );
    free_surf( &shot_shadow );
#ifdef SOUND
    sound_chunk_free( &wav_shot );
#endif
    free_surf( &extra_pic );
    free_surf( &extra_shadow );
    free_surf( &wall_pic );
#ifdef SOUND
    sound_chunk_free( &wav_score );
    sound_chunk_free( &wav_metal );
    sound_chunk_free( &wav_speedup );
    sound_chunk_free( &wav_speeddown );
    sound_chunk_free( &wav_std );
    sound_chunk_free( &wav_wall );
    sound_chunk_free( &wav_joker );
    sound_chunk_free( &wav_goldshower );
    sound_chunk_free( &wav_chaos );
    sound_chunk_free( &wav_darkness );
    sound_chunk_free( &wav_ghost );
    sound_chunk_free( &wav_timeadd );
    sound_chunk_free( &wav_expl_ball );
    sound_chunk_free( &wav_weak_ball );
    sound_chunk_free( &wav_bonus_magnet );
    sound_chunk_free( &wav_malus_magnet );
    sound_chunk_free( &wav_disable );
#endif
    free_surf( &shine_pic );
    free_surf( &frame_left );
    free_surf( &frame_right );
    free_surf( &frame_top );
    free_surf( &frame );
    free_surf( &frame_shadow );
    free_surf( &lamps );
    free_font( &frame_font );
#ifdef SOUND
    sound_chunk_free( &wav_life_up );
    sound_chunk_free( &wav_life_down );
#endif
    for ( i = 0; i < bkgnd_count; i++ )
        free_surf( &bkgnds[i] );
    bkgnd_count = 0;
}

