/* $Id: history.cc,v 1.12 2001/02/23 13:12:53 bergo Exp $ */

/*

    GPS - Graphical Process Statistics
    Copyright (C) 1999-2000 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "transient.h"

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <sys/time.h>
#include <gtk/gtk.h>
#include "diefast.h"
#include "history.h"
#include "netwatch.h"
#include "polling.h"
#include "netpolling.h"
#include "importglobals.h"

GtkWidget *G1,*G2,*H1,*H2; /* gauges and history */
GtkWidget *lg1,*ml[12];
GdkPixmap *g1canvas=NULL,*h1canvas=NULL,
          *g2canvas=NULL,*h2canvas=NULL;

int host_slot; /* current slot, -1 means all together */

History *history[20];

int history_watch_count; /* max 20 */
int today_is_a_good_day_to_die;

void pop_history(GtkWidget *widget, gpointer data) {
  GtkWidget *v1,*h1,*hs,*dis,*cput;
  GtkWidget *bh,*tb;
  char title[512],bu[256],bf[256];
  char *lcol[1];
  int i;
  GList *pt;
  NetworkSystemInfoProvider *wh;
  int no_list;

  if (hdlg!=NULL)
    return;

  if (netcritical)
    return;

  for(i=0;i<20;i++) history[i]=NULL;

  history_watch_count=watch_count+1;
  if (history_watch_count>20)
    history_watch_count=20;

  no_list=0;
  if ((poll_local_only)||(!watch_count))
    no_list=1;

  host_slot=0;
  history[0]=new History(info_provider->CpuCount);
  history[0]->sip=info_provider;

  if (no_list)
    history_watch_count=1;
  else
    for(i=1,pt=sysinfowatch;pt!=NULL;i++,pt=g_list_next(pt)) {
      if (i>history_watch_count)
	break;
      history[i]=new History(1);
      history[i]->sip=((SystemInfoProvider *)(pt->data));
    }

  hdlg=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_position(GTK_WINDOW(hdlg),GTK_WIN_POS_CENTER);
  gtk_widget_realize(hdlg);
  gtk_window_set_policy(GTK_WINDOW(hdlg),TRUE,TRUE,TRUE);
  gethostname(bu,256);
  sprintf(title,"gPS: CPU and Memory (%s)",bu);
  gtk_window_set_title(GTK_WINDOW(hdlg),title);
  gtk_container_set_border_width(GTK_CONTAINER(hdlg),4);

  bh=gtk_hbox_new(FALSE,2);
  gtk_container_add(GTK_CONTAINER(hdlg),bh);  

  /* the host list (if any) */

  if (!no_list) {
    tb=gtk_clist_new(1);
    gtk_clist_set_shadow_type(GTK_CLIST(tb),GTK_SHADOW_IN);
    gtk_clist_set_selection_mode(GTK_CLIST(tb),GTK_SELECTION_SINGLE);
    gtk_clist_set_column_title(GTK_CLIST(tb),0,"Views");
    gtk_clist_column_titles_passive(GTK_CLIST(tb));
    gtk_clist_column_titles_show(GTK_CLIST(tb));
    gtk_clist_set_column_width(GTK_CLIST(tb),0,100);

    gtk_box_pack_start(GTK_BOX(bh),tb,FALSE,TRUE,2);
    
    sprintf(bf,"%s (local)",this_host);
    lcol[0]=bf;
    gtk_clist_append(GTK_CLIST(tb),(gchar**)lcol);

    gtk_clist_select_row(GTK_CLIST(tb),0,0);
    
    for(i=1,pt=sysinfowatch;i<history_watch_count;i++,pt=g_list_next(pt)) {
      wh=(NetworkSystemInfoProvider *)(pt->data);
      lcol[0]=wh->getHostname();
      gtk_clist_append(GTK_CLIST(tb),(gchar**)lcol);
    }

    sprintf(bf,"All together");
    lcol[0]=bf;
    gtk_clist_append(GTK_CLIST(tb),(gchar**)lcol);

    gtk_signal_connect (GTK_OBJECT(tb),"select_row",
			GTK_SIGNAL_FUNC(view_selection_callback),NULL);
  }

  /* --- */

  v1=gtk_vbox_new(FALSE,2);
  gtk_box_pack_start(GTK_BOX(bh),v1,TRUE,TRUE,2);

  cput=gtk_table_new(6,3,FALSE);
  gtk_box_pack_start(GTK_BOX(v1),cput,TRUE,TRUE,4);

  mk_cput(cput);

  hs=gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(v1),hs,FALSE,FALSE,4);

  h1=gtk_hbox_new(FALSE,2);
  gtk_box_pack_start(GTK_BOX(v1),h1,FALSE,TRUE,2);

  dis=gtk_button_new_with_label("   Dismiss   ");
  gtk_box_pack_end(GTK_BOX(h1),dis,FALSE,FALSE,2);

  gtk_signal_connect(GTK_OBJECT(dis),"clicked",
		     GTK_SIGNAL_FUNC(close_history),NULL);
  gtk_signal_connect (GTK_OBJECT (hdlg), "destroy",
		      GTK_SIGNAL_FUNC (destroy_history), NULL);

  gtk_widget_show(v1);
  gtk_widget_show(h1);
  gtk_widget_show(hs);
  gtk_widget_show(dis);
  gtk_widget_show(cput);
  gtk_widget_show(bh);

  if (!no_list)
    gtk_widget_show(tb);

  gtk_widget_show(hdlg);

  /* start poller threads */
  today_is_a_good_day_to_die=0;
  History::reset();
  for(i=0;i<history_watch_count;i++) {
   pthread_create(&(history[i]->tid),
		   NULL,
		   &thread_history_poller,
		   (void *)history[i]);
  }

  to_history=gtk_timeout_add(history_interval,refresh_history,NULL);
}

void close_history(GtkWidget *w, gpointer data) {
  gtk_widget_destroy(hdlg);
  hdlg=NULL;
}

void destroy_history(GtkWidget *w,gpointer data) {
  int i;
  if (to_history>=0)
    gtk_timeout_remove(to_history);
  to_history=-1;

  today_is_a_good_day_to_die=1;
  for(i=0;i<history_watch_count;i++)
    pthread_join(history[i]->tid,NULL);

  for(i=0;i<history_watch_count;i++) {
    delete history[i];
    history[i]=NULL;
  }

  gdk_pixmap_unref(g1canvas); g1canvas=NULL;
  gdk_pixmap_unref(h1canvas); h1canvas=NULL;
  gdk_pixmap_unref(g2canvas); g2canvas=NULL;
  gdk_pixmap_unref(h2canvas); h2canvas=NULL;
  hdlg=NULL;
}

void mk_cput(GtkWidget *t) {
  GtkTable *cput;
  GtkWidget *l[4];
  GtkWidget *rt,*hm,*hs;
  GtkStyle *style;
  GdkPixmap *p;
  GdkBitmap *mask;
  GtkWidget *wp[2];
  int i;

  cput=GTK_TABLE(t);

  l[0]=gtk_label_new("CPU");
  l[1]=gtk_label_new("History");
  l[2]=gtk_label_new("Memory");
  l[3]=gtk_label_new("History");

  gtk_table_attach_defaults(cput,l[0],0,1,0,1);
  gtk_table_attach_defaults(cput,l[1],2,3,0,1);
  gtk_table_attach_defaults(cput,l[2],0,1,3,4);
  gtk_table_attach_defaults(cput,l[3],2,3,3,4);

  /* mem/swap  labels */
  rt=gtk_table_new(3,4,TRUE);
  gtk_table_attach_defaults(cput,rt,0,3,5,6);

  hm=gtk_hbox_new(FALSE,0);
  hs=gtk_hbox_new(FALSE,0);

  style=gtk_widget_get_style(hdlg);
  p=gdk_pixmap_create_from_xpm_d(hdlg->window,&mask,
				 &style->bg[GTK_STATE_NORMAL],
				 (gchar **)pix_mem);
  wp[0]=gtk_pixmap_new(p,mask);
  p=gdk_pixmap_create_from_xpm_d(hdlg->window,&mask,
				 &style->bg[GTK_STATE_NORMAL],
				 (gchar **)pix_swp);
  wp[1]=gtk_pixmap_new(p,mask);

  ml[1]=gtk_label_new("Used"); 
  ml[2]=gtk_label_new("Free");
  ml[3]=gtk_label_new("Total");

  ml[4]=gtk_label_new("Memory"); 
  ml[5]=gtk_label_new("0");
  ml[6]=gtk_label_new("0");
  ml[7]=gtk_label_new("0");

  ml[8]=gtk_label_new("Swap");
  ml[9]=gtk_label_new("0");
  ml[10]=gtk_label_new("0");
  ml[11]=gtk_label_new("0");

  gtk_box_pack_start(GTK_BOX(hm),wp[0],FALSE,FALSE,4);
  gtk_box_pack_start(GTK_BOX(hm),ml[4],FALSE,FALSE,0);

  gtk_box_pack_start(GTK_BOX(hs),wp[1],FALSE,FALSE,4);
  gtk_box_pack_start(GTK_BOX(hs),ml[8],FALSE,FALSE,0);

  gtk_table_attach_defaults(GTK_TABLE(rt),ml[1],1,2,0,1);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[2],2,3,0,1);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[3],3,4,0,1);

  gtk_table_attach_defaults(GTK_TABLE(rt),hm,0,1,1,2);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[5],1,2,1,2);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[6],2,3,1,2);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[7],3,4,1,2);

  gtk_table_attach_defaults(GTK_TABLE(rt),hs,0,1,2,3);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[9],1,2,2,3);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[10],2,3,2,3);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[11],3,4,2,3);

  /* cpu gauge */
  G1=gtk_drawing_area_new();
  gtk_widget_set_events (G1, GDK_EXPOSURE_MASK);

  gtk_drawing_area_size(GTK_DRAWING_AREA(G1),64,96);

  gtk_signal_connect (GTK_OBJECT (G1), "expose_event",
		      (GtkSignalFunc) g1_expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT (G1), "configure_event",
		      (GtkSignalFunc) g1_configure_event, NULL);

  gtk_table_attach_defaults(cput,G1,0,1,1,2);

  lg1=gtk_label_new(" ");
  gtk_table_attach_defaults(cput,lg1,0,1,2,3);

  /* cpu history */
  H1=gtk_drawing_area_new();
  gtk_widget_set_events (H1, GDK_EXPOSURE_MASK);

  gtk_drawing_area_size(GTK_DRAWING_AREA(H1),350,96);

  gtk_signal_connect (GTK_OBJECT (H1), "expose_event",
		      (GtkSignalFunc) h1_expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT (H1), "configure_event",
		      (GtkSignalFunc) h1_configure_event, NULL);

  gtk_table_attach_defaults(cput,H1,2,3,1,2);

  /* mem gauge */
  G2=gtk_drawing_area_new();
  gtk_widget_set_events (G2, GDK_EXPOSURE_MASK);

  gtk_drawing_area_size(GTK_DRAWING_AREA(G2),64,96);

  gtk_signal_connect (GTK_OBJECT (G2), "expose_event",
		      (GtkSignalFunc) g2_expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT (G2), "configure_event",
		      (GtkSignalFunc) g2_configure_event, NULL);

  gtk_table_attach_defaults(cput,G2,0,1,4,5);

  /* mem history */
  H2=gtk_drawing_area_new();
  gtk_widget_set_events (H2, GDK_EXPOSURE_MASK);
  gtk_drawing_area_size(GTK_DRAWING_AREA(H2),350,96);

  gtk_signal_connect (GTK_OBJECT (H2), "expose_event",
		      (GtkSignalFunc) h2_expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT (H2), "configure_event",
		      (GtkSignalFunc) h2_configure_event, NULL);

  gtk_table_attach_defaults(cput,H2,2,3,4,5);

  for(i=0;i<4;i++)
    gtk_widget_show(l[i]);

  for(i=1;i<12;i++)
    gtk_widget_show(ml[i]);

  gtk_widget_show(G1);
  gtk_widget_show(H1);
  gtk_widget_show(G2);
  gtk_widget_show(H2);
  gtk_widget_show(lg1);
  gtk_widget_show(wp[0]);
  gtk_widget_show(wp[1]);
  gtk_widget_show(hm);
  gtk_widget_show(hs);
  gtk_widget_show(rt);
}

/* gauge 1 (cpu) */

gboolean g1_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
			    gpointer data) {
  GdkGC *mygc;
  int ia,i,x,y;
  double fa;
  int smps[64],bw,ncpu;

  if (host_slot==-1) {
    return(g1_party_configure_event(widget,ce,data));
  }

  if (g1canvas!=NULL)
    gdk_pixmap_unref(g1canvas);

  mygc=gdk_gc_new(widget->window);
  
  g1canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(g1canvas,widget->style->black_gc,TRUE,0,0,x,y);

  gauge_grid(g1canvas,mygc,x,y);

  ncpu=history[host_slot]->NCPU;
  if (ncpu==1) {
    fa=(1.0-history[host_slot]->sip->cpu_usage)*((double)(y-5));
    ia=3+(int)fa;
    draw_bar(g1canvas,mygc,10,ia,x-20,y-ia,GREEN);
  } else {
    fa=(1.0-history[host_slot]->sip->cpu_usage)*((double)(y-5));
    smps[0]=3+(int)fa;
    for(i=0;i<ncpu;i++) {
      fa=(1.0-history[host_slot]->sip->cpus_load[i])*((double)(y-5));
      smps[i+1]=3+(int)fa;
    }
    bw=(x-10)/(ncpu+1);
    draw_bar(g1canvas,mygc,5,smps[0],bw-1,y-smps[0],GREEN);
    for(i=0;i<ncpu;i++)
      draw_bar(g1canvas,mygc,5+bw*(i+1),smps[i+1],bw-1,y-smps[i+1],
	       BLUE);
  }

  // outline
  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_a);
  gdk_draw_rectangle(g1canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);
  return FALSE;
}

gboolean g1_expose_event(GtkWidget *widget,GdkEventExpose *ee,
			    gpointer data) {
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  g1canvas,
		  ee->area.x, ee->area.y,
		  ee->area.x, ee->area.y,
		  ee->area.width, ee->area.height);
  return FALSE;
}

/* history 1 (cpu) */

gboolean h1_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
			    gpointer data) {
  GdkGC *mygc;
  int ia,ib,i,j,x,y,d,c;
  float fa,fb;
  long la;
  int ncpu;
  
  if (host_slot==-1) {
    return(h1_party_configure_event(widget,ce,data));
  }
  
  if (h1canvas!=NULL)
    gdk_pixmap_unref(h1canvas);
  
  mygc=gdk_gc_new(widget->window);
  
  h1canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(h1canvas,widget->style->black_gc,TRUE,0,0,x,y);
  
  /* should I fill or should I go ? */
  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_b);
  
  la=History::getCursor()+2*HISTORY_LENGTH-1;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-history[host_slot]->getValue(la%HISTORY_LENGTH,CPU_USAGE))*
      ((float)(y-5));
    ia=3+(int)fa;
    gdk_draw_line(h1canvas,mygc,i,ia,i,y-5+3);
  }
  
  gauge_grid(h1canvas,mygc,x,y);

  /* minute bars */
  gdk_rgb_gc_set_foreground(mygc,c_history_grid);
  for(i=x-2,c=0;i>1;i--,c++)
    if (((c%60)==0)&&(c!=0)) 
      gdk_draw_line(h1canvas,mygc,i,1,i,y-2);
  
  // Single CPU / SMP average history (pseudo-filled)
  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_a);
  la=History::getCursor()+2*HISTORY_LENGTH-1;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-history[host_slot]->getValue(la%HISTORY_LENGTH,CPU_USAGE))*
      ((float)(y-5)); 
    ia=3+(int)fa;
    fb=(1.0-history[host_slot]->getValue((la-1)%HISTORY_LENGTH,CPU_USAGE))*
      ((float)(y-5)); 
    ib=3+(int)fb;
    gdk_draw_line(h1canvas,mygc,i,ia,i-1,ib);
  }

  // SMP individual CPUs
  ncpu=history[host_slot]->NCPU;
  if (ncpu>1) {
    gdk_rgb_gc_set_foreground(mygc,c_blue_c);
    for(j=0;j<ncpu;j++) {
      la=History::getCursor()+2*HISTORY_LENGTH-1;
      for(i=x-2;i>1;i--,la--) {
	fa=(1.0-history[host_slot]->getCpuLoad(la%HISTORY_LENGTH,j))*
	((float)(y-5)); 
	ia=3+(int)fa;
	fb=(1.0-history[host_slot]->getCpuLoad((la-1)%HISTORY_LENGTH,j))*
	  ((float)(y-5)); 
	ib=3+(int)fb;
	gdk_draw_line(h1canvas,mygc,i,ia,i-1,ib);
      } // for i...
    } // for j...
  } // if ncpu>1
  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_a);
  gdk_draw_rectangle(h1canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);

  return FALSE;
}

gboolean h1_expose_event(GtkWidget *widget,GdkEventExpose *ee,
			    gpointer data) {
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  h1canvas,
		  ee->area.x, ee->area.y,
		  ee->area.x, ee->area.y,
		  ee->area.width, ee->area.height);
  return FALSE;
}

/* gauge 2 (memory & swap) */

gboolean g2_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
			    gpointer data) {
  GdkGC *mygc;
  int ia,x,y,hx,c;
  double fa;

  if (host_slot==-1) {
    return(g2_party_configure_event(widget,ce,data));
  }

  if (g2canvas!=NULL)
    gdk_pixmap_unref(g2canvas);

  mygc=gdk_gc_new(widget->window);
  
  g2canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(g2canvas,widget->style->black_gc,TRUE,0,0,x,y);

  hx=x/2;

  /* MEMORY */
  fa=(1.0-history[host_slot]->sip->mem_fraction())*((double)(y-5));
  ia=3+(int)fa;

  gauge_grid(g2canvas,mygc,x,y);

  draw_bar(g2canvas,mygc,10,ia,x-hx,y-ia,GREEN);

  /* SWAP */
  fa=(1.0-history[host_slot]->sip->swap_fraction())*((double)(y-5));
  ia=3+(int)fa;

  draw_bar(g2canvas,mygc,hx+1,ia,x-hx-10,y-ia,RED);

  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_a);
  gdk_draw_rectangle(g2canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);
  return FALSE;
}

gboolean g2_expose_event(GtkWidget *widget,GdkEventExpose *ee,
			    gpointer data) {
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  g2canvas,
		  ee->area.x, ee->area.y,
		  ee->area.x, ee->area.y,
		  ee->area.width, ee->area.height);
  return FALSE;
}

/* history 2 (mem/swap) */

gboolean h2_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
			    gpointer data) {
  GdkGC *mygc;
  int ia,ib,i,x,y,d,c;
  float fa,fb;
  long la;

  if (host_slot==-1) {
    return(h2_party_configure_event(widget,ce,data));
  }

  if (h2canvas!=NULL)
    gdk_pixmap_unref(h2canvas);

  mygc=gdk_gc_new(widget->window);
  
  h2canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(h2canvas,widget->style->black_gc,TRUE,0,0,x,y);

  /* should I fill or should I go ? */
  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_b);
  la=History::getCursor()+2*HISTORY_LENGTH-1;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-history[host_slot]->getValue(la%HISTORY_LENGTH,MEM_USAGE))*
      ((float)(y-5));
    ia=3+(int)fa;
    gdk_draw_line(h2canvas,mygc,i,ia,i,y-5+3);
  }

  /* swap fill (filled in front of memory, as 
     memory usage should be higher than swap) */
  gdk_rgb_gc_set_foreground(mygc,c_history_swap_b);
  la=History::getCursor()+2*HISTORY_LENGTH-1;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-history[host_slot]->getValue(la%HISTORY_LENGTH,SWAP_USAGE))*
      ((float)(y-5));
    ia=3+(int)fa;
    gdk_draw_line(h2canvas,mygc,i,ia,i,y-5+3);
  }

  /* 25%, 50%, 75% */
  gdk_rgb_gc_set_foreground(mygc,c_history_grid);
  d=3+(y-5)/2;
  gdk_draw_line(h2canvas,mygc,1,d,x-2,d);
  d=3+(y-5)/4;
  gdk_draw_line(h2canvas,mygc,1,d,x-2,d);
  d=3+3*(y-5)/4;
  gdk_draw_line(h2canvas,mygc,1,d,x-2,d);

  /* minute bars */
  gdk_rgb_gc_set_foreground(mygc,c_history_grid);
  for(i=x-2,c=0;i>1;i--,c++)
    if (((c%60)==0)&&(c!=0)) 
      gdk_draw_line(h2canvas,mygc,i,1,i,y-2);
  
  /* mem */
  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_a);
  la=History::getCursor()+2*HISTORY_LENGTH-1;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-history[host_slot]->getValue(la%HISTORY_LENGTH,MEM_USAGE))*
      ((float)(y-5));
    ia=3+(int)fa;
    fb=(1.0-history[host_slot]->getValue((la-1)%HISTORY_LENGTH,MEM_USAGE))*
      ((float)(y-5));
    ib=3+(int)fb;
    gdk_draw_line(h2canvas,mygc,i,ia,i-1,ib);
  }

  /* swap */
  gdk_rgb_gc_set_foreground(mygc,c_history_swap_a);
  la=History::getCursor()+2*HISTORY_LENGTH-1;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-history[host_slot]->getValue(la%HISTORY_LENGTH,SWAP_USAGE))*
      ((float)(y-5));
    ia=3+(int)fa;
    fb=(1.0-history[host_slot]->getValue((la-1)%HISTORY_LENGTH,SWAP_USAGE))*
      ((float)(y-5));
    ib=3+(int)fb;
    gdk_draw_line(h2canvas,mygc,i,ia,i-1,ib);
  }

  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_a);
  gdk_draw_rectangle(h2canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);
  return FALSE;
}

gboolean h2_expose_event(GtkWidget *widget,GdkEventExpose *ee,
			      gpointer data) {
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  h2canvas,
		  ee->area.x, ee->area.y,
		  ee->area.x, ee->area.y,
		  ee->area.width, ee->area.height);
  return FALSE;
}

void view_selection_callback(GtkWidget *widget,gint row,gint column,
			     GdkEventButton *event,gpointer data) {
  host_slot=row;
  if (host_slot>=history_watch_count)
    host_slot=-1;
}


gboolean g1_party_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
				  gpointer data) {
  GdkGC *mygc;
  int ia,i,j,x,y,c,dx,bw;
  double fa;
  int nhosts,ncpu;
  int smps[64];

  if (g1canvas!=NULL)
    gdk_pixmap_unref(g1canvas);

  mygc=gdk_gc_new(widget->window);  
  g1canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(g1canvas,widget->style->black_gc,TRUE,0,0,x,y);

  nhosts=history_watch_count;
  dx=(x-20)/nhosts;

  gauge_grid(g1canvas,mygc,x,y);

  for(j=0;j<nhosts;j++) {

    ncpu=history[j]->NCPU;
    if (ncpu==1) {
      fa=(1.0-history[j]->sip->cpu_usage)*((double)(y-5));
      ia=3+(int)fa;
      draw_bar(g1canvas,mygc,10+dx*j,ia,dx-2,y-ia,GREEN);
    } else {
      fa=(1.0-history[j]->sip->cpu_usage)*((double)(y-5));
      smps[0]=3+(int)fa;
      for(i=0;i<ncpu;i++) {
	fa=(1.0-history[j]->sip->cpus_load[i])*((double)(y-5));
	smps[i+1]=3+(int)fa;
      }
      bw=dx/(ncpu+1);
      draw_bar(g1canvas,mygc,10+dx*j,smps[0],bw-1,y-smps[0],GREEN);
      for(i=0;i<ncpu;i++)
	draw_bar(g1canvas,mygc,10+dx*j+bw*(i+1),smps[i+1],bw-1,y-smps[i+1],
		 BLUE);
    } // else    
  } // for

  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_a);
  gdk_draw_rectangle(g1canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);
  return FALSE;
}

gboolean h1_party_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
				  gpointer data) {
  GdkGC *mygc;
  int ia,ib,i,j527,x,y,d,c;
  float fa,fb;
  long la;
  float bubble[20],aux;
  long grass,bluegrass; // :-)
  int j,k,m,ncpu,nhosts;

  if (h1canvas!=NULL)
    gdk_pixmap_unref(h1canvas);

  mygc=gdk_gc_new(widget->window);
  
  h1canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(h1canvas,widget->style->black_gc,TRUE,0,0,x,y);

  nhosts=history_watch_count;

  /* should I fill or should I go ? */
  la=History::getCursor()+2*HISTORY_LENGTH-1;
  for(i=x-2;i>1;i--,la--) {
    for(j=0;j<nhosts;j++) {
      bubble[j]=history[j]->getValue(la%HISTORY_LENGTH,CPU_USAGE);
    }
    for(k=0;k<(nhosts-1);k++)
      for(m=k+1;m<nhosts;m++)
	if (bubble[k]>bubble[m]) {
	  aux=bubble[k];
	  bubble[k]=bubble[m];
	  bubble[m]=aux;
	}
    grass=c_history_cpu_c;
    k=y-5+3;
    for(j=0;j<nhosts;j++) {
      gdk_rgb_gc_set_foreground(mygc,grass);
      fa=(1.0-bubble[j])*((float)(y-5));
      ia=3+(int)fa;     
      gdk_draw_line(h1canvas,mygc,i,ia,i,k);
      k=ia;
      grass=lighten(grass);
    }
  }

  gauge_grid(h1canvas,mygc,x,y);

  /* minute bars */
  gdk_rgb_gc_set_foreground(mygc,c_history_grid);
  for(i=x-2,c=0;i>1;i--,c++)
    if (((c%60)==0)&&(c!=0)) 
      gdk_draw_line(h1canvas,mygc,i,1,i,y-2);
  
  grass=c_history_cpu_a;
  bluegrass=c_blue_c;

  for(j=0;j<nhosts;j++) {
    
    // avg / single cpu (green)
    gdk_rgb_gc_set_foreground(mygc,grass);
    la=History::getCursor()+2*HISTORY_LENGTH-1;
    for(i=x-2;i>1;i--,la--) {
      fa=(1.0-history[j]->getValue(la%HISTORY_LENGTH,CPU_USAGE))*
	((float)(y-5));
      ia=3+(int)fa;
      fb=(1.0-history[j]->getValue((la-1)%HISTORY_LENGTH,CPU_USAGE))*
	((float)(y-5));
      ib=3+(int)fb;
      gdk_draw_line(h1canvas,mygc,i,ia,i-1,ib);
    }
    grass=darken(grass);

    // SMP individual CPUs
    ncpu=history[j]->NCPU;
    if (ncpu>1) {
      gdk_rgb_gc_set_foreground(mygc,bluegrass);
      for(j527=0;j527<ncpu;j527++) {
	la=History::getCursor()+2*HISTORY_LENGTH-1;
	for(i=x-2;i>1;i--,la--) {
	  fa=(1.0-history[j]->getCpuLoad(la%HISTORY_LENGTH,j527))*
	    ((float)(y-5)); 
	  ia=3+(int)fa;
	  fb=(1.0-history[j]->getCpuLoad((la-1)%HISTORY_LENGTH,j527))*
	    ((float)(y-5)); 
	  ib=3+(int)fb;
	  gdk_draw_line(h1canvas,mygc,i,ia,i-1,ib);
	}
      }
      bluegrass=darken(bluegrass);
    } // if ncpu>1
  } // for j...
    
  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_a);
  gdk_draw_rectangle(h1canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);
  return FALSE;
}

gboolean g2_party_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
				  gpointer data) {
  GdkGC *mygc;
  int ia,i,x,y,hx,c,dx,ncpus,j;
  double fa;

  if (g2canvas!=NULL)
    gdk_pixmap_unref(g2canvas);

  mygc=gdk_gc_new(widget->window);
  
  g2canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(g2canvas,widget->style->black_gc,TRUE,0,0,x,y);

  hx=(3*x)/4;

  ncpus=history_watch_count;
  dx=hx/ncpus;

  /* MEMORY */
  gauge_grid(g2canvas,mygc,x,y);

  for(j=0;j<ncpus;j++) {
    fa=(1.0-history[j]->sip->mem_fraction())*((double)(y-5));
    ia=3+(int)fa;
    draw_bar(g2canvas,mygc,10+dx*j,ia,dx-2,y-ia,GREEN);
  }

  /* SWAP */

  for(j=0;j<ncpus;j++) {
    fa=(1.0-history[j]->sip->swap_fraction())*((double)(y-5));
    ia=3+(int)fa;
    draw_bar(g2canvas,mygc,x-10-hx+dx*j,ia,dx-2,y-ia,RED);
  }

  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_a);
  gdk_draw_rectangle(g2canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);
  return FALSE;
}

gboolean h2_party_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
				  gpointer data) {
  GdkGC *mygc;
  int ia,ib,i,x,y,d,c;
  float fa,fb;
  long la;
  float bubble[20],aux;
  int ncpus,j,k,m;
  long grass;

  if (h2canvas!=NULL)
    gdk_pixmap_unref(h2canvas);

  mygc=gdk_gc_new(widget->window);
  
  h2canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(h2canvas,widget->style->black_gc,TRUE,0,0,x,y);

  ncpus=history_watch_count;

  /* should I fill or should I go ? */
  la=History::getCursor()+2*HISTORY_LENGTH-1;
  for(i=x-2;i>1;i--,la--) {
    for(j=0;j<ncpus;j++) {
      bubble[j]=history[j]->getValue(la%HISTORY_LENGTH,MEM_USAGE);
    }
    for(k=0;k<(ncpus-1);k++)
      for(m=k+1;m<ncpus;m++)
	if (bubble[k]>bubble[m]) {
	  aux=bubble[k];
	  bubble[k]=bubble[m];
	  bubble[m]=aux;
	}
    
    grass=c_history_cpu_c;
    k=y-5+3;
    for(j=0;j<ncpus;j++) {
      gdk_rgb_gc_set_foreground(mygc,grass);
      fa=(1.0-bubble[j])*((float)(y-5));
      ia=3+(int)fa;
      gdk_draw_line(h2canvas,mygc,i,ia,i,k);
      k=ia;
      grass=lighten(grass);
    }

  }

  /* swap fill (filled in front of memory, as 
     memory usage should be higher than swap) */
  gdk_rgb_gc_set_foreground(mygc,c_history_swap_b);
  la=History::getCursor()+2*HISTORY_LENGTH-1;
  for(i=x-2;i>1;i--,la--) {
    for(j=0;j<ncpus;j++){
      bubble[j]=history[j]->getValue(la%HISTORY_LENGTH,SWAP_USAGE);
    }
    for(k=0;k<(ncpus-1);k++)
      for(m=k+1;m<ncpus;m++)
	if (bubble[k]>bubble[m]) {
	  aux=bubble[k];
	  bubble[k]=bubble[m];
	  bubble[m]=aux;
	}

    grass=c_history_swap_c;
    k=y-5+3;

    for(j=0;j<ncpus;j++) {
      gdk_rgb_gc_set_foreground(mygc,grass);
      fa=(1.0-bubble[j])*((float)(y-5));
      ia=3+(int)fa;
      gdk_draw_line(h2canvas,mygc,i,ia,i,k);

      k=ia;
      grass=lighten(grass);
    }
  }

  /* 25%, 50%, 75% */
  gdk_rgb_gc_set_foreground(mygc,c_history_grid);
  d=3+(y-5)/2;
  gdk_draw_line(h2canvas,mygc,1,d,x-2,d);
  d=3+(y-5)/4;
  gdk_draw_line(h2canvas,mygc,1,d,x-2,d);
  d=3+3*(y-5)/4;
  gdk_draw_line(h2canvas,mygc,1,d,x-2,d);

  /* minute bars */
  gdk_rgb_gc_set_foreground(mygc,c_history_grid);
  for(i=x-2,c=0;i>1;i--,c++)
    if (((c%60)==0)&&(c!=0)) 
      gdk_draw_line(h2canvas,mygc,i,1,i,y-2);

  /* mem */
  grass=c_history_cpu_a;
  for(j=0;j<ncpus;j++) {
    gdk_rgb_gc_set_foreground(mygc,grass);
    la=History::getCursor()+2*HISTORY_LENGTH-1;
    for(i=x-2;i>1;i--,la--) {
      fa=(1.0-history[j]->getValue(la%HISTORY_LENGTH,MEM_USAGE))*
	((float)(y-5));
      ia=3+(int)fa;
      fb=(1.0-history[j]->getValue((la-1)%HISTORY_LENGTH,MEM_USAGE))*
	((float)(y-5));
      ib=3+(int)fb;
      gdk_draw_line(h2canvas,mygc,i,ia,i-1,ib);
    }
    grass=darken(grass);
  }

  /* swap */
  grass=c_history_swap_a;
  for(j=0;j<ncpus;j++) {
    gdk_rgb_gc_set_foreground(mygc,grass);
    la=History::getCursor()+2*HISTORY_LENGTH-1;
    for(i=x-2;i>1;i--,la--) {
      fa=(1.0-history[j]->getValue(la%HISTORY_LENGTH,SWAP_USAGE))*
	((float)(y-5));
      ia=3+(int)fa;
      fb=(1.0-history[j]->getValue((la-1)%HISTORY_LENGTH,SWAP_USAGE))*
	((float)(y-5));
      ib=3+(int)fb;
      gdk_draw_line(h2canvas,mygc,i,ia,i-1,ib);
    }
    grass=darken(grass);
  }

  gdk_rgb_gc_set_foreground(mygc,c_history_cpu_a);
  gdk_draw_rectangle(h2canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);
  return FALSE;
}

/* **************************************************


    History class


   ************************************************* */ 

int History::cursor=0;
int History::adv=0;
pthread_mutex_t History::ohshit=PTHREAD_MUTEX_INITIALIZER;

History::History(int cpus) {
  int i,j;
  NCPU=cpus;
  dcpu=(float *)g_malloc0(sizeof(float)*HISTORY_LENGTH);
  dmem=(float *)g_malloc0(sizeof(float)*HISTORY_LENGTH);
  dswp=(float *)g_malloc0(sizeof(float)*HISTORY_LENGTH);
  for(i=0;i<NCPU;i++)
    smpcpu[i]=(float *)g_malloc0(sizeof(float)*HISTORY_LENGTH);
  for(i=0;i<HISTORY_LENGTH;i++) {
    dcpu[i]=0.0;
    dmem[i]=0.0;
    dswp[i]=0.0;
    for(j=0;j<NCPU;j++)
      smpcpu[j][i]=0.0;
  }
}

History::~History() {
  int i;
  g_free(dcpu);
  g_free(dmem);
  g_free(dswp);
  for(i=0;i<NCPU;i++)
    g_free(smpcpu[i]);
  cursor=0;
}

int History::getCursor() {
  return(cursor);
}

float History::getValue(int pos,HistoryData q) {
  float rv;

  pthread_mutex_lock(&ohshit);

  switch(q) {
  case CPU_USAGE: rv=dcpu[pos]; break;
  case MEM_USAGE: rv=dmem[pos]; break;
  case SWAP_USAGE: rv=dswp[pos]; break;
  default: rv=0.0;
  }
  pthread_mutex_unlock(&ohshit);

  return(rv);
}

void History::setValue(float vc,float vm,float vs) {
  pthread_mutex_lock(&ohshit);
  dcpu[cursor]=vc;
  dmem[cursor]=vm;
  dswp[cursor]=vs;
  pthread_mutex_unlock(&ohshit);
}

void History::setCpuLoad(int cpu,float load) {
  pthread_mutex_lock(&ohshit);
  if (cpu>=NCPU) cpu=NCPU-1;
  smpcpu[cpu][cursor]=load;
  pthread_mutex_unlock(&ohshit);
}

float History::getCpuLoad(int pos,int cpu) {
  float val;
  pthread_mutex_lock(&ohshit);
  if (cpu>=NCPU) cpu=NCPU-1;
  val=smpcpu[cpu][pos];
  pthread_mutex_unlock(&ohshit);
  return val;
}

void History::advance() {
  pthread_mutex_lock(&ohshit);
  adv++;
  if (adv>=history_watch_count) {
    adv=0;
    cursor=(++cursor)%HISTORY_LENGTH;
  }
  pthread_mutex_unlock(&ohshit);
}

void History::reset() {
  cursor=0;
  adv=0;
}

void *thread_history_poller(void *p) {
  History *h;
  float c,m,s,smp[64];
  int i,j;
  struct timeval tv1,tv2;
  struct timezone tz;
  long microdiff;
  int ocur,ncur;
  int ncpu;

  h=(History *)p;
  ncpu=h->NCPU;
  while(!today_is_a_good_day_to_die) {
    c=m=s=0.0;
    for(i=0;i<ncpu;i++) smp[i]=0.0;
    for(i=0;i<4;i++) {
      gettimeofday(&tv1,&tz);
      h->sip->update();
      c+=h->sip->cpu_usage;
      m+=h->sip->mem_fraction();
      s+=h->sip->swap_fraction();
      for(j=0;j<ncpu;j++)
	smp[j]+=h->sip->cpus_load[j];
      gettimeofday(&tv2,&tz);
      microdiff=1000*(tv2.tv_sec-tv1.tv_sec)+(tv2.tv_usec-tv1.tv_usec)/1000;
      if (microdiff<history_interval)
	usleep((history_interval-microdiff)*1000);
    }

    h->setValue(c/4.0,m/4.0,s/4.0);
    for(j=0;j<ncpu;j++)
      h->setCpuLoad(j,smp[j]/4.0);

    ocur=History::getCursor();
    h->advance();
    while(History::getCursor()==ocur) {
      if (today_is_a_good_day_to_die) break;
      usleep(10003);
    }
  }
  pthread_exit(NULL);
}

gint refresh_history(gpointer data) {
  static int round=0;
  long pv[6];
  int i;
  char c[256],b[256];

  if (hdlg==NULL)
    return FALSE;

  round=(++round)%4;

  g1_configure_event(G1,NULL,NULL);
  h1_configure_event(H1,NULL,NULL);

  g2_configure_event(G2,NULL,NULL);
  h2_configure_event(H2,NULL,NULL);

  gtk_widget_queue_draw(G2);
  gtk_widget_queue_draw(G1);

  if (!round) {
    gtk_widget_queue_draw(H1);
    gtk_widget_queue_draw(H2);
  }
  
  if (host_slot<0) {
    for(i=0;i<6;i++)
      pv[i]=0L;
    for(i=0;i<history_watch_count;i++) {
      pv[0]+=(history[i]->sip->memory_used>>10);
      pv[1]+=(history[i]->sip->memory_free>>10);
      pv[2]+=(history[i]->sip->memory_total>>10);
      pv[3]+=(history[i]->sip->swap_used>>10);
      pv[4]+=(history[i]->sip->swap_free>>10);
      pv[5]+=(history[i]->sip->swap_total>>10);
    }
    strcpy(c,"MB");
  }
  else {
    pv[0]=(history[host_slot]->sip->memory_used);
    pv[1]=(history[host_slot]->sip->memory_free);
    pv[2]=(history[host_slot]->sip->memory_total);
    pv[3]=(history[host_slot]->sip->swap_used);
    pv[4]=(history[host_slot]->sip->swap_free);
    pv[5]=(history[host_slot]->sip->swap_total);
    strcpy(c,"KB");
  }
  for(i=0;i<6;i++) pv[i]>>=10;
  
  sprintf(b,"%lu %s",pv[0],c);
  gtk_label_set_text(GTK_LABEL(ml[5]),b);
  sprintf(b,"%lu %s",pv[1],c);
  gtk_label_set_text(GTK_LABEL(ml[6]),b);
  sprintf(b,"%lu %s",pv[2],c);
  gtk_label_set_text(GTK_LABEL(ml[7]),b);
  
  sprintf(b,"%lu %s",pv[3],c);
  gtk_label_set_text(GTK_LABEL(ml[9]),b);
  sprintf(b,"%lu %s",pv[4],c);
  gtk_label_set_text(GTK_LABEL(ml[10]),b);
  sprintf(b,"%lu %s",pv[5],c);
  gtk_label_set_text(GTK_LABEL(ml[11]),b);

  return TRUE;
}

/* JELLY */

void draw_jelly_bar(GdkPixmap *drawable,GdkGC *gc,int x,int y,int w,
		    int h,BarColor bc) {

  int xxx,j1,j2,j3,i,k1,k2,ec;

  k1=x;
  k2=x+1;
  j1=x+w-1;
  j2=x+w-2;

  for(i=x;i<(x+w);i++) {
    xxx=y;
    if (i==k1) xxx=y+2;
    if (i==k2) xxx=y+1;
    if (i==j1) xxx=y+2;
    if (i==j2) xxx=y+1;
    j3=abs(i-(x+3))*3;
    if (j3>255) j3=255;

    switch(bc) {
    case GREEN:  j3=(255-j3)<<8;  break;
    case RED:    j3=(255-j3)<<16; break;
    case BLUE:   j3=(255-j3);     break;
    case YELLOW: j3=((255-j3)<<8)|((255-j3)<<16); break;
    }

    gdk_rgb_gc_set_foreground(gc,j3);
    gdk_draw_line(drawable,gc,i,xxx,i,y+h-1);
  }

  switch(bc) {
  case GREEN:  ec=0xc0e0c0; break;
  case RED:    ec=0xe0c0c0; break;
  case BLUE:   ec=0xc0c0e0; break;
  case YELLOW: ec=0xe0e0c0; break;
  default: ec=0;
  }
  gdk_rgb_gc_set_foreground(gc,ec);
  gdk_draw_line(drawable,gc,x+2,y+2,x+2,y+h-1);
  gdk_draw_line(drawable,gc,x+4,y+1,x+4,y+h-1);

  switch(bc) {
  case GREEN:  ec=0xe0ffe0; break;
  case RED:    ec=0xffe0e0; break;
  case BLUE:   ec=0xe0e0ff; break;
  case YELLOW: ec=0xffffe0; break;
  default: ec=0;
  }
  gdk_rgb_gc_set_foreground(gc,ec);
  gdk_draw_line(drawable,gc,x+3,y+2,x+3,y+h-1);
}

void draw_usual_bar(GdkPixmap *drawable,GdkGC *gc,int x,int y,int w,
		    int h,BarColor bc) {
  int i,c,d;
  for(i=(y+h-1),c=0;i>=y;i--,c=1-c) {
    switch(bc) {
    case GREEN:  d=c?c_history_cpu_a:c_history_cpu_b;   break;
    case RED:    d=c?c_history_swap_a:c_history_swap_b; break;
    case BLUE:   d=c?c_blue_a:c_blue_b;                 break;
    case YELLOW: d=c?c_yellow_a:c_yellow_b;             break;
    default:     d=0;
    }
    gdk_rgb_gc_set_foreground(gc,d);
    gdk_draw_line(drawable,gc,x,i,x+w-1,i);
  }
}

int darken(int c) {
  int r,g,b,v;

  r=c>>16;
  g=(c>>8)&0xff;
  b=c&0xff;

  r=(r*9)/10;
  g=(g*9)/10;
  b=(b*9)/10;

  v=((r&0xff)<<16)|((g&0xff)<<8)|(b&0xff);
  return v;
}

int lighten(int c) {
  int r,g,b,v;

  r=c>>16;
  g=(c>>8)&0xff;
  b=c&0xff;

  r=(r*10)/9;
  g=(g*10)/9;
  b=(b*10)/9;

  if (r>255) r=255;
  if (g>255) g=255;
  if (b>255) b=255;

  v=((r&0xff)<<16)|((g&0xff)<<8)|(b&0xff);
  return v;
}

void gauge_grid(GdkPixmap *drw,GdkGC *gc,int x,int y) {
  int d;
  gdk_rgb_gc_set_foreground(gc,c_history_grid);
  d=3+(y-5)/2;
  gdk_draw_line(drw,gc,1,d,x-2,d);
  d=3+(y-5)/4;
  gdk_draw_line(drw,gc,1,d,x-2,d);
  d=3+3*(y-5)/4;
  gdk_draw_line(drw,gc,1,d,x-2,d);
}

void draw_bar(GdkPixmap *drawable,GdkGC *gc,int x,int y,int w,
	      int h,BarColor bc) {
  if (jelly_bar)
    draw_jelly_bar(drawable,gc,x,y,w,h,bc);
  else
    draw_usual_bar(drawable,gc,x,y,w,h,bc);
}
