/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * This file contains all the global types used within Galeon.
 */

#ifndef __galeon_types_h
#define __galeon_types_h

#include <gtk/gtkwidget.h>

/* FIXME Remove these */
#include <gtk/gtktooltips.h>
#include <gtk/gtkctree.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkdnd.h>

/* forward declarations of types */
typedef struct _GaleonWindow             GaleonWindow;
typedef struct _GaleonEmbed              GaleonEmbed;
typedef struct _HistoryHost              HistoryHost;
typedef struct _HistoryItem              HistoryItem;

#ifdef ENABLE_NAUTILUS_VIEW
#include "galeon-nautilus-view.h"
#endif

/**
 * FocusType: what caused the switch to a embed object
 */
typedef enum
{
	FOCUS_ON_CREATE,	/* auto_jump or opened via menu */
	FOCUS_ON_REQUEST	/* user explicitly switched (clicked on tab) */
} FocusType;

/**
 * DockType: type of dock
 */
typedef enum
{
	DOCK_NONE,
	DOCK_BOOKMARKS,
	DOCK_HISTORY
} DockType;

/**
 * EmbedCreateFlags: flags to specify special instructions to use when
 * creating an embed and adding it to a window
 */
typedef enum
{
	EMBED_CREATE_FORCE_APPEND	= 1 << 0, /* put the embed at the end
						     of the notebook */
	EMBED_CREATE_NEVER_JUMP		= 1 << 1, /* do not jump to the new
						     embed */
	EMBED_CREATE_FORCE_JUMP		= 1 << 2, /* force a jump to the new
						     embed */
	EMBED_CREATE_RAISE_WINDOW	= 1 << 3, /* raise the window that
						     the embed is added to */
	EMBED_CREATE_LOAD_DEFAULT_URL	= 1 << 4, /* load the default url
						     in the embed */
	EMBED_CREATE_DONT_SHOW		= 1 << 5, /* don't load about:blank,
						     instead keeping the
						     embed not visible */
	EMBED_CREATE_GRAB_FOCUS		= 1 << 6, /* grab the focus on the
						     first page */
	EMBED_CREATE_DONT_AUTOSAVE_SESSION = 1 << 7  /* set the embed's
						     dont_autosave_session
						     flag */
} EmbedCreateFlags;

/**
 * DisplayType: Passed through to mozilla when creating a new view from the
 * contents of an old one.
 **/

typedef enum
{
	DISPLAY_AS_SOURCE = 1U,
	DISPLAY_NORMAL = 2U
} DisplayType;

/**
 * AutoreloadInfo: this structure contains information about when
 * and how the embed should be reloaded automatically
 */
typedef struct
{
	gboolean enabled;  /* is autoreload enabled ?             */
	gint delay;        /* autoreload delay                    */
	gint delay_unit;   /* what unit does the previous int use */
	guint timeout_id;  /* ID of the autoreload timeout func   */
} AutoreloadInfo;

/**
 * GaleonEmbed: this is the type of a single web view (e.g. a tab):
 * this is effectively Galeon's wrapping of a GtkMozEmbed with extra
 * state information.
 */ 
struct _GaleonEmbed
{
	guint32 magic;

	/* parent window in which this embed is visible */
	/* note that this can be NULL if the Embed is in a NautilusView */
	GaleonWindow *parent_window;

	/* embedding widget */
	GtkWidget *mozembed;

	/* tab widgets */
	GtkWidget *notebook_hbox;                /* packing box           */
	GtkWidget *notebook_icon;                /* the icon              */
	GtkWidget *notebook_label;               /* the label             */
	GtkWidget *notebook_close_button;        /* active close button   */
	GtkWidget *notebook_close_pixmap;	 /* inactive close pixmap */

	/* C++ helper, a "GaleonWrapper" */
        gpointer wrapper;

	/* location and title */
	gchar *location;
	gchar *modified_location;
	gchar *title;
	gchar *title_utf8;

	gint load_started; /* count of currently active connections */
	gint load_percent;
	gint bytes_loaded;
	gint max_bytes_loaded;
	const gchar *load_status_message;

	/* time when the load was actually started */
	GTime when_started;

	gchar *temp_statusbar_message;

	/* currently set zoom level */
	gint zoom;

	/* true if the zoom was set automatically */
	gboolean zoom_auto_set : 1;

	/* true if this is the "active_embed" of the parent window */
	gboolean is_active : 1;

	/* is this browser actually visible */
	gboolean is_visible : 1;

	/* ignore mime processing of next url */
	gboolean mime_ignore_once : 1;

	/* has this browser content been viewed yet? */
	gboolean has_been_viewed : 1;

	/* is this a xul dialog ? */
	gboolean is_chrome : 1;

	/* secure page ? */
	gboolean secure : 1;

	/* if true, don't autosave the session file
	 * at the location_changed signal */
	gboolean dont_autosave_session : 1;

	/* has user stylesheet been applied ? */
	gboolean usercss_applied : 1;

	/* creation flags */
	EmbedCreateFlags create_flags;

	/* user stylesheets */
	GHashTable *usersheets;
	
	/* how this embed was focused if applicable */
	FocusType focus_type;

	/* security tooltip */
	gchar *security_tooltip;
	
	/* dialogs */
	GtkWidget *print_dialog;
	GtkWidget *find_dialog;

	/* autoreload information */
	AutoreloadInfo *autoreload_info;

#ifdef ENABLE_NAUTILUS_VIEW 
	/* Points to the parent GaleonNautilusView, or NULL if none */
	GaleonNautilusView *nautilus_view;
#endif
	
	/* used for gestures */
	gint gesture_motion_signal;
	gint gesture_press_signal;
	gint gesture_release_signal;
	guint gesture_button;

 	/* used for scrolling */
 	gint scroll_motion_signal;
 	gint scroll_release_signal;
 	gint scroll_timeout_id;

	/* Hack to work around javascript close called
	   when we are already closing the window */
	gboolean being_closed;
};

#define	MAIN_TOOLBAR 1
#define	BOOKMARKS_TOOLBAR 2
#define	TABS_APPEAREANCE 4
#define	AUTOBOOKMARKS 8

/**
 * GaleonWindow: this is the type of Galeon windows, i.e. a wrapping of
 * the toplevel GnomeApp window. A single window can contain many 
 * GaleonEmbeds.
 */
struct _GaleonWindow
{
	guint32 magic;

	/* window parts */
	GtkWidget *wmain;
	GaleonEmbed *active_embed;
	GList *embed_list; /* list of GaleonEmbeds */
	gint visible_embeds; /* number of visible GaleonEmbeds, used to
			      * track whether or not the window is visible */

	/* true if the spinner is running; false otherwise */
	gboolean spinner_running;

	/* assigned size */
	gboolean set_size : 1;

	/* still running the activity bar? */
	gboolean progress_timeout : 1;

	/* have we set the encodings menu yet? */
	gboolean encodings_menu_set : 1;

	/* GUI components */
	GtkWidget *notebook;
	GtkWidget *hpaned;
	GtkWidget *hbox;
	GtkWidget *docked;                /* currently docked control      */
	const gchar *docked_width_pref;   /* where width of dock is stored */
	
	/* These specify the non-fullscreen *bar state, so we can switch
	 * back from fullscreen properly */
	gboolean show_menubar      : 1;
	gboolean show_toolbar      : 1;
	gboolean show_bookmarks    : 1;
	gboolean show_statusbar    : 1;

	/* if we are in full screen or not */
	gboolean fullscreen_active;

	/* Toolbar components */
	GtkWidget *dock;
	GtkWidget *appbar;
	GtkWidget *menubar;
	GtkWidget *main_toolbar;
	GtkWidget *main_dockitem;
	GtkWidget *right_toolbar;
	GtkWidget *location_toolbar;
	GtkWidget *location_dockitem;

	/* Toolbar buttons etc */
	GtkWidget *new_button;
	GtkWidget *back_button;
	GtkWidget *up_button;
	GtkWidget *forward_button;
	GtkWidget *refresh_button;
	GtkWidget *home_button;
	GtkWidget *stop_button;
	GtkWidget *go_button;
	GtkWidget *zoom_spin;
	GtkWidget *drag_event_box;
	GtkWidget *drag_pixmap;
	GtkWidget *location_gnomeentry;
	GtkWidget *location_entry;
	GtkWidget *spinner;               /* the GnomeAnimator spinner */
	GtkWidget *history_button;
	GtkWidget *bookmarks_button;
	GtkWidget *back_history_button;
	GtkWidget *forward_history_button;
	GtkWidget *clear_location_button;
	GtkWidget *link_menu_button;
	GtkWidget *fullscreen_button;
	GtkWidget *find_entry;
	
	/* menus */
	GtkWidget *file_menu;
	GtkWidget *bookmarks_menu;
	GtkWidget *go_menu;

	/* menu items */
	GtkWidget *recent_sessions_menuitem;
	GtkWidget *view_toolbar;
	GtkWidget *view_bookmarks;
	GtkWidget *view_statusbar;
	GtkWidget *view_menubar;
	GtkWidget *view_fullscreen;
	GtkWidget *detach_tab;
	GtkWidget *move_tab_to_window_menu;
	GtkWidget *next_tab;
	GtkWidget *prev_tab;
	GtkWidget *move_tab_left;
	GtkWidget *move_tab_right;
	GtkWidget *load_images_always;
	GtkWidget *load_images_from_current_server_only;
	GtkWidget *load_images_never;
	GtkWidget *animate_always;
	GtkWidget *animate_once_through;
	GtkWidget *animate_never;
	GtkWidget *no_proxy;
	GtkWidget *manual_proxy;
	GtkWidget *autoproxy;
	GtkWidget *use_own_fonts;
	GtkWidget *use_own_colors;
	GtkWidget *enable_java;
	GtkWidget *enable_javascript;
	GtkWidget *allow_popups;
	GtkWidget *add_bookmark;
	GtkWidget *bookmarks_separator;
	GtkWidget *encoding;
	GtkWidget *csslist;
	GtkWidget *java_console;
	GtkWidget *back_menuitem;
	GtkWidget *forward_menuitem;
	GtkWidget *up_menuitem;
	GtkWidget *refresh_menuitem;
	GtkWidget *homepage_menuitem;
	GtkWidget *related_links_menuitem;

	/* edit menu menuitems */
	GtkWidget *edit_cut;
	GtkWidget *edit_copy;
	GtkWidget *edit_paste;

	/* dialogs and views */
	GtkWidget *open_url_entry;

	/* security status */
	GtkWidget *security_frame;
	GtkWidget *security_icon;
	GtkWidget *security_eventbox;

	/* list of toolbars */
	GList *bookmarks_toolbars; 
	GtkTooltips *bookmarks_tooltips;

	/* saved window geometry for fullscreen windows */
	gint x, y, width, height;

	/* visibility */
	gboolean menubar_on : 1;
	gboolean toolbar_on : 1;
	gboolean location_bar_on : 1;
	gboolean statusbar_on : 1;

	/* is this a popup window ? */
	gboolean is_popup : 1;

	/* has the layer been already set ? */
	gboolean layered : 1;

	/* dock type */
	DockType dock_type;

	/* saved layout */
	gchar *layout;

	/* notifiers IDs to remove them when the window is destroyed */
	GList *notifiers_id;

	/* used for draggable tabs */
	gint drag_motion_signal;
	gint drag_leave_signal;
	gboolean in_predrag : 1;
	gboolean in_drag : 1;
	gint drag_min_x, drag_max_x, drag_min_y, drag_max_y;

	/* list of dummy widgets used to bind accelerators to */
	GList *dummy_widgets;

	/* window appeareance settings that need UI update */
	int to_update;

	/* requested window width & height */
	gint requested_width;
	gint requested_height;

	/* Recently visited sites menu entries */
	GList *recent_menu_entries;  /* list of RecentMenuEntry* */
};

/**
 * Magic numbers used to detect if the structure has been accidentally 
 * free'd or corrupted
 */
#define GALEON_WINDOW_MAGIC   (0x87de130f)
#define GALEON_EMBED_MAGIC    (0x45a0ebed)

/**
 * Checking macros used to validate window and embed pointers and
 * their magic. At some point we'll probably do all of this with
 * GObjects, but for now this is it. This will introduce some (very
 * small) overhead, and we should probably provide a #ifdef DEBUG
 * which disables these (around v1.0).
 */

#define return_if_not_window(window) \
  g_return_if_fail (window != NULL); \
  g_return_if_fail (window->magic == GALEON_WINDOW_MAGIC);

#define return_val_if_not_window(window, val) \
  g_return_val_if_fail (window != NULL, val); \
  g_return_val_if_fail (window->magic == GALEON_WINDOW_MAGIC, val);

#define return_if_not_embed(embed) \
  g_return_if_fail (embed != NULL); \
  g_return_if_fail (embed->magic == GALEON_EMBED_MAGIC);

#define return_val_if_not_embed(embed, val) \
  g_return_val_if_fail (embed != NULL, val); \
  g_return_val_if_fail (embed->magic == GALEON_EMBED_MAGIC, val);

#define return_if_not_sane_click(xBUTTON, xEVENT) \
  if (GTK_IS_BUTTON (xBUTTON) && xEVENT->button == 1 && \
      !GTK_BUTTON (xBUTTON)->button_down) \
        return FALSE; \
  if (xEVENT->button > 3) \
	return FALSE; \
  if (GTK_WIDGET (xBUTTON)->window != xEVENT->window) \
	return FALSE;

#ifdef ENABLE_NAUTILUS_VIEW
/**
 * These macros check if the embed is an embed and it hasa parent GaleonWindow
 * or a parent GaleonNautilusView.
 */
#define return_if_not_sane_embed(embed) \
  g_return_if_fail ((embed != NULL) \
		    && (embed->magic = GALEON_EMBED_MAGIC) \
		    && (((embed->parent_window != NULL) \
			 && (embed->parent_window->magic == GALEON_WINDOW_MAGIC)) \
			|| ((embed->nautilus_view != NULL) \
			    && GALEON_IS_NAUTILUS_VIEW \
			    (embed->nautilus_view))))
#define return_val_if_not_sane_embed(embed, val) \
  g_return_val_if_fail ((embed != NULL) \
		        && (embed->magic = GALEON_EMBED_MAGIC) \
			&& (((embed->parent_window != NULL) \
			     && (embed->parent_window->magic == GALEON_WINDOW_MAGIC))\
			    || ((embed->nautilus_view != NULL) \
				&& GALEON_IS_NAUTILUS_VIEW \
				(embed->nautilus_view))), val)
/**
 * Checks if the embed is in a NautilusView instead of a GaleonWindow
 */
#define EMBED_IN_NAUTILUS(embed) (embed && ((embed)->nautilus_view != NULL))

#else

#define return_if_not_sane_embed(embed) \
  g_return_if_fail ((embed != NULL) \
		    && (embed->magic = GALEON_EMBED_MAGIC) \
		    && (((embed->parent_window != NULL) \
			 && (embed->parent_window->magic == GALEON_WINDOW_MAGIC))))
#define return_val_if_not_sane_embed(embed, val) \
  g_return_val_if_fail ((embed != NULL) \
		        && (embed->magic = GALEON_EMBED_MAGIC) \
			&& (((embed->parent_window != NULL) \
			     && (embed->parent_window->magic == GALEON_WINDOW_MAGIC))), val)
#endif

/**
 * WidgetLookup: a mapping from strings to pointers, used to lookup
 * a group of widgets from a Glade description.
 */
typedef struct 
{ 
	const gchar *name; 
	GtkWidget **ptr;
} WidgetLookup;

/**
 * MozillaKeyModifier: key modifiers
 */
typedef enum
{
	ALT_KEY   = (1 << 0),
	CTRL_KEY  = (1 << 1),
	SHIFT_KEY = (1 << 2),
	META_KEY  = (1 << 3),
	KEY_CODE  = (1 << 4)
} MozillaKeyModifier;

/**
 * BookmarkTitleBehaviour: type of behaviour when making a new bookmark
 */
typedef enum 
{ 
	BOOKMARK_PAGE_TITLE,
	BOOKMARK_ASK_TITLE
} BookmarkTitleBehaviour;

/**
 * StartPageType: start site type
 */
typedef enum
{
	STARTPAGE_HOME,
	STARTPAGE_LAST,
	STARTPAGE_BLANK,
} StartPageType;

/**
 * MimeAction: types of actions to take for opening files
 */
typedef enum
{
	MIME_UNKNOWN,
	MIME_SAVE_TO_DISK,
	MIME_RUN_PROGRAM,      /* should run the GNOME association */
	MIME_ASK_ACTION,
	MIME_IGNORE
} MimeAction;

/**
 * FilePickerMode: What mode FilePicker should run in
 */

typedef enum
{
	modeOpen = 0,
	modeSave = 1,
	modeGetFolder =2
} FilePickerMode;

/**
 * ImageLoadingType: various image loading behaviours
 */
typedef enum
{
	IMAGE_LOADING_ALWAYS                   = 0,
	IMAGE_LOADING_FROM_CURRENT_SERVER_ONLY = 1,
	IMAGE_LOADING_NEVER                    = 2
} ImageLoadingType;

/**
 * AnimateType: various image animation behaviours
 */
typedef enum
{
	ANIMATE_ALWAYS       = 0,
	ANIMATE_ONCE_THROUGH = 1,
	ANIMATE_NEVER        = 2
} AnimateType;

/**
 * ViewState: destination for loading URLs
 */
typedef enum
{
	VIEWSTATE_EXISTING,		/* open in existing browser */
	VIEWSTATE_WINDOW,		/* open in new window */
	VIEWSTATE_TAB_FORCE,		/* open in tab (requested) */
	VIEWSTATE_TAB_FORCE_NORAISE, 	/* same, but don't raise window */
	VIEWSTATE_TAB_DEFAULT,		/* open in tab (by default) */
	VIEWSTATE_TAB_DEFAULT_NORAISE	/* same, but don't raise window */
} ViewState;

/**
 * LinkState: generalized URL activation types
 */
typedef enum
{
	LINKSTATE_BUTTON_MASK = 7,
	LINKSTATE_SHIFTED     = (1 << 3),
	LINKSTATE_CTRLED      = (1 << 4),
	LINKSTATE_ALTED       = (1 << 5),
	/* force a certain mode rather than basing off the pref */
	LINKSTATE_NEWTAB      = 1 + (1 << 9),
	LINKSTATE_NEWWIN      = 1 + (1 << 10),
} LinkState;

/**
 * ContextMenuType: various types of context menu
 */
typedef enum
{
	CONTEXT_NONE     = 0,
	CONTEXT_DEFAULT  = (1 << 1),
	CONTEXT_LINK     = (1 << 2),
	CONTEXT_IMAGE    = (1 << 3),
	CONTEXT_DOCUMENT = (1 << 4),
	CONTEXT_INPUT    = (1 << 5),
	CONTEXT_XUL      = (1 << 7),
} ContextMenuType;

/**
 * MimeItem: a particular MIME type binding
 */
typedef struct
{
	gchar *name; 
	MimeAction action;
	gchar *helperApp;
	gboolean terminalHelper;
	gboolean urlHelper;
	gboolean alwaysUse;
} MimeItem;

typedef struct
{
	gchar *serif;
	gchar *sansserif;
	gchar *cursive;
	gchar *fantasy;
	gchar *monospace;

	int min_size;
} LanguageFontInfo;

/**
 * HistoryView: a history view (either dialog or dockeD)
 */
typedef struct
{
	GaleonWindow *window;            /* parent window */
	GtkWidget *view;                 /* either the dock or the dialog */
	GtkWidget *ctree;
	GtkWidget *host_checkbutton;
	GtkWidget *time_optionmenu;
	GtkWidget *search_entry;
	GtkWidget *dock_close_button;
	GtkWidget *dock_detach_button;
	gboolean group;
	gchar *search;
	gint time;
	GtkWidget *sibling;              /* used in tree construction */
	GHashTable *host_to_node;        /* maps HistoryHosts to tree nodes */
	GHashTable *item_to_node;        /* maps HistoryItems to tree nodes */
	GList *selected_items;
} HistoryView;

/**
 * HistoryItem: an item in the history
 */
struct _HistoryItem
{
	gchar *url;
	gchar *title_locale;
	gchar *title_utf8;
	GTime first;
	GTime last;
	gint visits;
	GList *children;
	gint selected_authorcss;

	/* parent host */
	HistoryHost *host;
};

/**
 * HistoryHost: a group of history elements assembled by host
 */
struct _HistoryHost
{
	/* the first fields are like HistoryItem */
	struct _HistoryItem item;

	/* tree specific data */
	GtkCTreeNode *node;
	GtkCTreeNode *sibling;

	/* stored zoom for this host */
	gint zoom;

	/* dominant child -- the location visited the most of this server */
	HistoryItem *dominant_item;
};

/**
 * An entry in the recent menu of any window
 */
typedef struct 
{
	gchar *url;
	gchar *title;
	GtkMenuItem *menuitem;
} RecentMenuEntry;

/**
 * ToolbarItemType: various toolbar item types
 */
typedef enum
{
	TOOLBAR_ITEM_NEW = 0,
	TOOLBAR_ITEM_BACK,
	TOOLBAR_ITEM_UP,
	TOOLBAR_ITEM_FORWARD,
	TOOLBAR_ITEM_REFRESH,
	TOOLBAR_ITEM_START,
	TOOLBAR_ITEM_STOP,
	TOOLBAR_ITEM_GO,
	TOOLBAR_ITEM_ZOOM_SPIN,
	TOOLBAR_ITEM_DRAG_HANDLE,
	TOOLBAR_ITEM_LOCATION_ENTRY,
	TOOLBAR_ITEM_SPINNER,
	TOOLBAR_ITEM_SEPARATOR,
	TOOLBAR_ITEM_FIND_ENTRY,
	TOOLBAR_ITEM_BOOKMARKS,
	TOOLBAR_ITEM_HISTORY,
	TOOLBAR_ITEM_PRINT,
	TOOLBAR_ITEM_BACK_HISTORY,
	TOOLBAR_ITEM_FORWARD_HISTORY,
	TOOLBAR_ITEM_CLEAR_LOCATION,
	TOOLBAR_ITEM_LINK_MENU,
	TOOLBAR_ITEM_FIND,
	TOOLBAR_ITEM_FULLSCREEN,

	/* must be last */
	TOOLBAR_ITEM_COUNT
} ToolbarItemType;	

/**
 * ToolbarItem: a toolbar item
 */
typedef struct
{
	const gchar *label;
	const gchar *tooltip;
	const gchar *stock_icon;
	const gchar *theme_icon;
	gboolean priority; /* i.e. show the text in horizontal mode */
	gboolean togglebutton;
	gint order;        /* order value for default insertion place */
	void (*clicked_cb) (GtkButton *button, 
			    GaleonWindow *window);
	gint (*button_press_event_cb) (GtkButton *button, 
				       GdkEventButton *event,
				       GaleonWindow *window);
	gint (*button_release_event_cb) (GtkButton *button, 
				         GdkEventButton *event,
				         GaleonWindow *window);
	void (*drag_data_received_cb) (GtkWidget *widget, 
				       GdkDragContext *drag_context,
				       gint x, gint y,
				       GtkSelectionData *selection_data,
				       guint info, guint time,
				       GaleonWindow *window);
} ToolbarItem;

/**
 * PixmapData: a single pixmap (stored on the X server)
 */
typedef struct
{
	GdkPixmap *pixmap;
	GdkBitmap *mask;
} PixmapData;

/**
 * ToolbarStyle: various toolbar styles
 */
typedef enum
{
	TOOLBAR_STYLE_HORIZONTAL,
	TOOLBAR_STYLE_VERTICAL,
	TOOLBAR_STYLE_TEXT_ONLY,
	TOOLBAR_STYLE_ICONS_ONLY,       /* not used for bookmark toolbars */
} ToolbarStyle;

/**
 * The "folder extra items" actions
 */
typedef enum
{
	EXTRA_ITEM_LOCATION_EXCLUSIVE = -1,
	EXTRA_ITEM_LOCATION_TOP_SUBMENU = 0,
	EXTRA_ITEM_LOCATION_BOTTOM_SUBMENU,
	EXTRA_ITEM_LOCATION_CONTEXT,
} BookmarksExtraItemsLocation;

/**
 * DnDType: various types that can be dragged and dropped
 */
typedef enum
{
	DND_TARGET_GALEON_BOOKMARK, 
	DND_TARGET_GALEON_URL,
	DND_TARGET_NETSCAPE_URL, 
	DND_TARGET_TEXT_URI_LIST,
	DND_TARGET_STRING,
} DnDType;

/**
 * BlockedHost: a blocked host
 */
typedef struct
{
	gchar *type;
	gchar *domain;
} BlockedHost;

/**
 * Cookie: the type of cookies
 */
typedef struct
{
	BlockedHost base;
	gchar *name;
	gchar *value;
	gchar *path;
	gchar *secure;
	gchar *expire;
} Cookie;

/**
 * Password: a password manager entry
 */
typedef struct
{
	gchar *host;
	gchar *username;
} Password;

/**
 * PasswordType: To distinguish actual passwords from blocked password sites
 */
typedef enum
{
	PASSWORD_PASSWORD,
	PASSWORD_REJECT
} PasswordType;

/**
 * PDMPage: For selecting which page in the persistent data manager is active
 */
typedef enum
{
	PDM_COOKIES = 0,
	PDM_COOKIESITES = 1,
	PDM_IMAGESITES = 2,
	PDM_PASSWORDS = 3,
	PDM_PASSWORDSITES = 4
} PDMPage;

/**
 * PDMDialog: persistent data manger dialog
 */
typedef struct
{
	GtkWidget *dialog;
	GtkWidget *notebook;

	GtkWidget *cookie_clist;
	GtkWidget *cookie_remove_button;
	GtkWidget *cookie_remove_and_block_button;
	GtkWidget *cookie_remove_all_button;
	GtkWidget *cookie_name;
	GtkWidget *cookie_info;
	GtkWidget *cookie_host;
	GtkWidget *cookie_path;
	GtkWidget *cookie_secure;
	GtkWidget *cookie_expires;
	
	GtkWidget *cookie_site_clist;
	GtkWidget *cookie_site_remove_button;
	GtkWidget *cookie_site_remove_all_button;
	
	GtkWidget *image_site_clist;
	GtkWidget *image_site_remove_button;
	GtkWidget *image_site_remove_all_button;
	
	GtkWidget *password_clist;
	GtkWidget *password_remove_button;
	GtkWidget *password_remove_all_button;

	GtkWidget *password_site_clist;
	GtkWidget *password_site_remove_button;
	GtkWidget *password_site_remove_all_button;
} PDMDialog;

/**
 * Downloader: the choice between the builtin (mozilla) downloader, the
 * external GTM downloader (via CORBA), or using some external command.
 */
typedef enum
{
	DOWNLOADER_BUILTIN  = 0,
	DOWNLOADER_GTM      = 1,
	DOWNLOADER_EXTERNAL = 2
} Downloader;

/**
 * PreferencesPageId: different pages of the prefs dialog
 */
typedef enum
{
	PREFS_PAGE_ID_BROWSING_GENERAL = 1,
	PREFS_PAGE_ID_BROWSING_BOOKMARKS,
	PREFS_PAGE_ID_BROWSING_HISTORY,
	PREFS_PAGE_ID_USER_INTERFACE_TABS,
	PREFS_PAGE_ID_USER_INTERFACE_WINDOWS,
	PREFS_PAGE_ID_USER_INTERFACE_TOOLBARS,
	PREFS_PAGE_ID_USER_INTERFACE_MOUSE,
	PREFS_PAGE_ID_HANDLERS_DOWNLOADING,
	PREFS_PAGE_ID_HANDLERS_PROGRAMS,
	PREFS_PAGE_ID_HANDLERS_MIME_TYPES,
	PREFS_PAGE_ID_RENDERING_FONTS_COLORS,
	PREFS_PAGE_ID_RENDERING_LANGUAGES,
	PREFS_PAGE_ID_ADVANCED_NETWORK,
	PREFS_PAGE_ID_ADVANCED_FILTERING,
	PREFS_PAGE_ID_ADVANCED_PERSISTENT_DATA,
	PREFS_PAGE_ID_ADVANCED_SECURITY,
	PREFS_PAGE_COUNT /* must be last */
} PreferencesPageId;

/**
 * AlternateStyleSheetType: Kind of stylesheet
 */
typedef enum
{
	STYLESHEET_NONE,
	STYLESHEET_DEFAULT,
	STYLESHEET_ALTERNATE
} AlternateStyleSheetType;

/**
 * AlternateStyleSheet: An alternate stylesheet
 */
typedef struct
{
	gchar *name;    /* stylesheet name */
	gpointer sheet; /* pointer to the mozilla sheet itself */
	AlternateStyleSheetType type;
} AlternateStyleSheet;

/**
 * LinkInterfaceItemType: link toolbar item type
 */
typedef enum
{
	LINK_ITEM_START,
	LINK_ITEM_NEXT,
	LINK_ITEM_PREV,
	LINK_ITEM_PREVIOUS,
	LINK_ITEM_CONTENTS,
	LINK_ITEM_TOC,
	LINK_ITEM_INDEX,
	LINK_ITEM_GLOSSARY,
	LINK_ITEM_COPYRIGHT,
	LINK_ITEM_CHAPTER,
	LINK_ITEM_SECTION,
	LINK_ITEM_SUBSECTION,
	LINK_ITEM_APPENDIX,
	LINK_ITEM_HELP,
	LINK_ITEM_TOP,
	LINK_ITEM_SEARCH,
	LINK_ITEM_MADE,
	LINK_ITEM_BOOKMARK,
	LINK_ITEM_HOME
} LinkInterfaceItemType;

/**
 * LinkInterfaceItem: an item for in the link toolbar
 */
typedef struct
{
	LinkInterfaceItemType type;
	gchar *href;
	gchar *title;
} LinkInterfaceItem;


/**
 * LinkContextMenuItem: an item for in the generic link context menu
 */
typedef struct
{
	gchar *url;
	gchar *title;
} LinkContextMenuItem;

/**
 * GaleonGestureAction: different actions that can be performed by gestures
 */
typedef enum
{
	UNRECOGNIZED,
	NEW_WINDOW,
	NEW_TAB,
	RELOAD,
	RELOAD_BYPASS,
	HOMEPAGE,
	CLONE_WINDOW,
	CLONE_TAB,
	UP,
	CLOSE,
	BACK,
	FORWARD,
	FULLSCREEN,
	NEXT_TAB,
	PREV_TAB,
	VIEW_SOURCE,
	STOP,
	PREV_LINK,
	NEXT_LINK,
	CONTENTS_LINK,
} GaleonGestureAction;

/**
 * FileCallbackFunc: the type of functions used in file callbacks
 */
typedef void (*FileCallbackFunc) (const gchar *file, gpointer data);

#endif /* __galeon_types_h */
