/*
 *  apcreports.c --
 *
 *  apcupsd.c	 -- Simple Daemon to catch power failure signals from a
 *		    BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *		 -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *			  <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 */

/*
 *		       GNU GENERAL PUBLIC LICENSE
 *			  Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *			     675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

#include "apc.h"

static char largebuf[4096];
static int  stat_recs;
static int  logstats = 0;
static time_t last_time_status;
static time_t last_time_logging;
static time_t last_time_events;
static FILE *statusfile = NULL;  /* initial value set to NULL for terminate */


/*********************************************************************/
void clear_files(void)
{
    if (statusfile != NULL) {
	fflush(statusfile);
	fclose(statusfile);
    }
}

static void log_status_open(UPSINFO *ups)
{
    largebuf[0] = 0;
    stat_recs = 0;
    rewind(statusfile);
    logstats = ups->logstats;
}

#define STAT_REV 1
static int log_status_close(UPSINFO *ups, int fd)
{
    int i;
    char buf[MAXSTRING];
    char *sptr, *eptr;

    i = strlen(largebuf);
    if (i > sizeof(largebuf)-1) {
        log_event(ups, LOG_ERR, _("Status buffer overflow %d bytes\n"), i-sizeof(largebuf));
	return -1;
    }
    sprintf(buf, "APC      : %03d,%03d,%04d\n", STAT_REV, stat_recs, i);
    fputs(buf, statusfile);
    fputs(largebuf, statusfile);
    fflush(statusfile);
    
    /*
     * Write out the status log to syslog() one record at a 
     * time.
     */
    if (logstats) {
	buf[strlen(buf)-1] = 0;
	log_event(ups, LOG_NOTICE, buf);
	sptr = eptr = largebuf;
	for ( ; i > 0; i--) {
            if (*eptr == '\n') {
	       *eptr++ = 0;
	       log_event(ups, LOG_NOTICE, sptr);
	       sptr = eptr;
	    } else {
	       eptr++;
	    }
	}
    }		     
    return 0;
}



/********************************************************************* 
 * log one line of the status file
 * also send it to system log
 *
 */
static void log_status_write(UPSINFO *ups, char *fmt, ...)
{
    va_list ap;
    char buf[MAXSTRING];

    va_start(ap, fmt);

    vsprintf(buf, fmt, ap);
    va_end(ap);

    strcat(largebuf, buf);
    stat_recs++;
}


/********************************************************************
 *
 * Log current UPS data
 *
 *  Format of output for SmartUPS
 *    vmin, vmax, vout, vbat, freq, load%,temp,amb-temp,humid, vline   
 *    223.6,230.1,227.5,27.74,50.00,020.8,034.6,,,229.8
 *  currently ambient temp and humidity are not included
 */ 

static void log_data(UPSINFO *ups)
{
    char *ptr;
    char msg[MAXSTRING];

    if (ups->datatime == 0)
	return;

    switch (ups->mode.type) {
    case BK:
    case SHAREBASIC:
    case NETUPS:
	if (ups->OnBatt == 0) 
            log_event(ups, LOG_INFO,"LINEFAIL:OK BATTSTAT:OK");
	else {
	    if ((ups->BattLow == 0) ||
		((ups->BattLow != 0) &&
		((ups->sp_flags & dtr_bit) == 0)))
		    
                ptr = "RUNNING";
	    else
                ptr = "FAILING";
            log_event(ups, LOG_INFO, "LINEFAIL:DOWN BATTSTAT:%s", ptr);
	}
	break;
    case BKPRO:
    case VS:
	if (ups->OnBatt == 0) {
	    if (ups->LineLevel == -1) 
                ptr = "LOW";
	    else if (ups->LineLevel == 1) 
                ptr = "HIGH";
	    else 
                ptr = "OK";
            log_event(ups, LOG_INFO, "LINEFAIL:OK BATTSTAT:OK LINEVOLT:%s", ptr);
	} else {
	    if (ups->BattLow == 0) 
                ptr = "RUNNING";
	    else 
                ptr = "FAILING";
            log_event(ups, LOG_INFO, "LINEFAIL:DOWN BATTSTAT:%s", ptr);
	}

	switch ((*ups).G[0]) {
        case 'O':
            ptr = "POWER UP";
	    break;
        case 'S':
            ptr = "SELF TEST";
	    break;
        case 'L':
            ptr = "LINE VOLTAGE DECREASE";
	    break;
        case 'H':
            ptr = "LINE VOLTAGE INCREASE";
	    break;
        case 'T':
            ptr = "POWER FAILURE";
	    break;
        case 'R':
            ptr = "R-EVENT";
	    break;
	default :
            sprintf(msg, "UNKNOWN EVENT %c %c", (*ups).G[0], (*ups).G[1]);
	    ptr = msg;
	    break;
	}
        log_event(ups, LOG_INFO, "LASTEVENT: %s", ptr);
	break;
    case NBKPRO:
    case SMART:
    case SHARESMART:
    case MATRIX:
	log_event(ups, LOG_INFO, 
                "%05.1f,%05.1f,%05.1f,%05.2f,%05.2f,%04.1f,%04.1f,,,%05.1f",
		 ups->LineMin,
		 ups->LineMax,
		 ups->OutputVoltage,
		 ups->BattVoltage,
		 ups->LineFreq,
		 ups->BattChg,
		 ups->UPSTemp,
		 ups->LineVoltage);
	break;
    default:
	break;
    }
    return;
}


void do_reports(UPSINFO *ups)
{
    static int first_time = TRUE;
    time_t now = time(NULL);

    if (first_time) {
	first_time = FALSE;
	/*
	 * Set up logging and status timers.
	 */
	last_time_logging = 0;
	last_time_status  = 0;
	last_time_events  = 0;

	if (ups->stattime != 0) {
            if ((statusfile = fopen(ups->statfile,"w")) == NULL)
                log_event(ups, LOG_ERR, "Cannot open STATUS file %s: %s\n",
			  ups->statfile, strerror(errno));
	}  else {
	    unlink(ups->statfile);
	}
    }

    /* Check if it is time to log DATA record */
    if ((ups->datatime > 0) && (now - last_time_logging) > ups->datatime) {
	time(&last_time_logging);
	log_data(ups);
    }

    /* Check if it is time to write STATUS file */
    if ((statusfile != NULL) && (now - last_time_status) > ups->stattime) {
	time(&last_time_status);
	output_status(ups, 0, log_status_open, log_status_write, 
	   log_status_close);
    }

    /* Check if it is time to truncate EVENTS file once every 5 hours */
    if ((ups->eventfile[0] != 0) && (now - last_time_events) > 5*3600) {
	time(&last_time_events);
	if (truncate_events_file(ups)) {
	    close(ups->event_fd);
	    ups->event_fd = open(ups->eventfile, O_WRONLY|O_CREAT|O_APPEND, 0644);
	}
    }
}
