/*
 * libpam-sfs - Pluggable Authentication Module for SFS
 * 
 * Copyright (C) 2000, 2001 Luca Filipozzi
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free 
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 59 
 * Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <errno.h>
#include <limits.h>
#include <nana.h>
#include <pwd.h>
#include <regex.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#define PAM_SM_AUTH
#define PAM_SM_SESSION
#include <security/pam_modules.h>

#include "handler.h"

#ifdef PIC
# include <syslog.h>
# undef  L_DEFAULT_HANDLER
# define L_DEFAULT_HANDLER syslog
# undef  L_DEFAULT_PARAMS
# define L_DEFAULT_PARAMS  LOG_DEBUG
#endif

handler::handler() : service_(NULL), flags_(0),
                     username_(NULL), password_(NULL),
                     hostname_(NULL), keyname_(NULL),
                     pid_(0), pw_(NULL), debug_(0)
{
    // intentionally empty
}

handler::~handler()
{
    // intentionally empty
}

int handler::pam_sm_authenticate(const char *service, int flags,
                                 int argc, const char *argv[],
                                 const char *username, const char *password)
{
    int pid;
    int status;

    service_  = strdup(username);
    flags_    = flags;              // note: flags_ unused

    ParseOptions(argc, argv);

    username_ = strdup(username);
    password_ = strdup(password);

    if (!IsUsernameValid())         // has side effect of filling in pw_
    {
        LG(debug_, "username is invalid\n");
        return PAM_USER_UNKNOWN;
    }

    if (!IsPasswordValid())
    {
        LG(debug_, "password is invalid\n");
        return PAM_AUTH_ERR;
    }

    if (!IsHomeInSFS())             // has side effect of filling in hostname_
    {
        LG(debug_, "home is not in /sfs\n");
        return PAM_SUCCESS;
    }

    pid = fork();
    switch(pid)
    {
        case -1:    // internal error
            return PAM_AUTH_ERR;
            break;

        case 0:     // child
#ifdef PIC
            close(1);   // don't want to spew stuff to stdout
            close(2);   // don't want to spew stuff to stderr
#endif
            exit(AuthenticateUser());
            break;

        default:    // parent
            if (waitpid(pid, &status, 0) == -1)
            {
                LG(debug_, "waitpid() failed: %s\n", strerror(errno));
                return PAM_AUTHINFO_UNAVAIL;
            }
            if (WIFEXITED(status))
            {
                return WEXITSTATUS(status);
            }
            else
            {
                return PAM_AUTHINFO_UNAVAIL;
            }
            break;
    }
    return PAM_SUCCESS;
}

int handler::pam_sm_open_session(const char *service, int flags,
                                 int argc, const char *argv[],
                                 const char *username, int *lockfilefdp)
{
    int     pid;
    int     status;

    service_  = strdup(username);
    flags_    = flags;              // note: flags_ unused

    ParseOptions(argc, argv);

    username_ = strdup(username);

    if (!IsUsernameValid())         // has side effect of filling in pw_
    {
        LG(debug_, "username is invalid\n");
        return PAM_SESSION_ERR;
    }

    if (!IsHomeInSFS())             // has side effect of filling in hostname_
    {
        LG(debug_, "home is not in /sfs\n");
        return PAM_SUCCESS;
    }

    // open the lockfile
    {
        int     fd;
        int     len = strlen(username_)+strlen(hostname_)+7;
        char    buf[len];

        snprintf(buf, len, "/tmp/%s@%s", username_, hostname_);
        fd = open(buf, O_RDWR | O_CREAT);
        if (fd == -1)
        {
            LG(debug_, "open(lockfile) failed: %s\n", strerror(errno));
            return PAM_SESSION_ERR;
        }
        fchown(fd, pw_->pw_uid, (gid_t)-1);
        *lockfilefdp = fd;
    }

    pid = fork();
    switch(pid)
    {
        case -1:    // internal error
            return PAM_AUTH_ERR;
            break;

        case 0:     // child
#ifdef PIC
            close(1);   // don't want to spew stuff to stdout
            close(2);   // don't want to spew stuff to stderr
#endif
            if (!BecomeUser())
            {
                LG(debug_, "could not become the user\n");
                return PAM_SESSION_ERR;
            }

            L("incrementing refcount for %s@%s\n", username_, hostname_);
            if (flock(*lockfilefdp, LOCK_SH) == -1)
            {
                LG(debug_, "could not increment refcount\n");
                exit(PAM_SESSION_ERR);
            }
            exit(PAM_SUCCESS);
            break;

        default:    // parent
            if (waitpid(pid, &status, 0) == -1)
            {
                LG(debug_, "waitpid() failed: %s\n", strerror(errno));
                return PAM_SESSION_ERR;
            }
            if (WIFEXITED(status))
            {
                return WEXITSTATUS(status);
            }
            else
            {
                return PAM_SESSION_ERR;
            }
            break;
    }

    return PAM_SUCCESS;
}

int handler::pam_sm_close_session(const char *service, int flags,
                                  int argc, const char *argv[],
                                  const char *username, int lockfilefd)
{
    service_  = strdup(username);
    flags_    = flags;              // note: flags_ unused

    ParseOptions(argc, argv);

    username_ = strdup(username);

    if (!IsUsernameValid())         // has side effect of filling in pw_
    {
        LG(debug_, "username is invalid\n");
        return PAM_SESSION_ERR;
    }

    if (!IsHomeInSFS())             // has side effect of filling in hostname_
    {
        LG(debug_, "home is not in /sfs\n");
        return PAM_SUCCESS;
    }

    int     pid;
    int     status;
    pid = fork();
    switch(pid)
    {
        case -1:    // internal error
            return PAM_AUTH_ERR;
            break;

        case 0:     // child
#ifdef PIC
            close(1);   // don't want to spew stuff to stdout
            close(2);   // don't want to spew stuff to stderr
#endif
            if (!BecomeUser())
            {
                LG(debug_, "could not become the user\n");
                return PAM_SESSION_ERR;
            }

            L("decrementing refcount for %s@%s\n", username_, hostname_);
            if (flock(lockfilefd, LOCK_UN) == -1)
            {
                LG(debug_, "could not decrement refcount\n");
                exit(PAM_SESSION_ERR);
            }

            LG(debug_, "attempting to get an exclusive lock\n");
            if (flock(lockfilefd, LOCK_EX|LOCK_NB) != -1)
            {
                LG(debug_, "got an exclusive lock!\n");
                LG(debug_, "connecting to sfsagent\n");
                if (!OpenAgentConnection())
                {
                    L("could not connect to sfsagent\n");
                    exit(PAM_SESSION_ERR);
                }

                L("killing sfsagent for %s@%s\n", username_, hostname_);
                if (!KillAgent())
                {
                    L("could not kill sfsagent\n");
                    CloseAgentConnection();
                    exit(PAM_SESSION_ERR);
                }

                CloseAgentConnection();

                close(lockfilefd);

                {
                    int     len = strlen(username_)+strlen(hostname_)+7;
                    char    buf[len];

                    snprintf(buf, len, "/tmp/%s@%s", username_, hostname_);
                    if (unlink(buf) == -1)
                    {
                        LG(debug_, "unlink() failed: %s\n", strerror(errno));
                    }
                }
            }

            exit(PAM_SUCCESS);
            break;

        default:    // parent
            if (waitpid(pid, &status, 0) == -1)
            {
                LG(debug_, "waitpid() failed: %s\n", strerror(errno));
                return PAM_SESSION_ERR;
            }
            if (WIFEXITED(status))
            {
                return WEXITSTATUS(status);
            }
            else
            {
                return PAM_SESSION_ERR;
            }
            break;
    }

    return PAM_SUCCESS;
}

bool handler::ParseOptions(int argc, const char *argv[])
{
    int i;

    for (i = 0; i < argc; i++)
    {
        if (!strcmp(argv[i], "debug"))
        {
            debug_ = true;
        }
        else
        {
            L("unkown or unused option: %s\n", argv[i]);
        }
    }
    return true;
}

bool handler::IsUsernameValid()
{
    if (username_ == NULL || !isalnum(*username_))
    {
        return false;
    }

    pw_ = getpwnam(username_);
    if (pw_ == NULL)
    {
        return false;
    }

    // TODO: is root ok?
    /*
    if (pw_->pw_uid == 0)
    {
        return false;
    }
    */

    return true;
}

bool handler::IsPasswordValid()
{
    // TODO: is null ok?
    if (password_ == NULL /* && !null_ok */)
    {
        return false;
    }

    return true;
}

bool handler::IsHomeInSFS()
{
    // determine the hostname from the user'shome dir
    if (!FindServer())      // has side effect of filling in hostname_
    {
        return false;
    }

    // construct the keyname
    {
        int len  = strlen(username_)+strlen(hostname_)+2;
        keyname_ = new char[len];

        snprintf(keyname_, len, "%s@%s", username_, hostname_);
    }

    return true;
}

// this function is based on the GNU C Library function named canonicalize()
// the original function is
// Copyright (C) 1996, 1997, 1998, 1999, 2000 Free Software Foundation, Inc.
// the changes are
// Copyright (C) 2001 Luca Filipozzi
bool handler::FindServer()
{
    long int    path_max;   // equal to PATH_MAX on most architectures
    char*       path;
    char*       rpath;      // resolved path
    char*       dest;       // append pointer for rpath
    const char* limit;      // pointer to end of rpath buffer (protection)
    const char* beg;        // pointer to beg of element in path
    const char* end;        // pointer to end of element in path
    int         debug = 0;

    path = pw_->pw_dir;
    if (path == NULL)
    {
        LG(debug, "path arg is NULL\n");
        errno = EINVAL;
        return false;
    }

    if (path[0] == '\0')
    {
        LG(debug, "path arg is empty\n");
        errno = ENOENT;
        return false;
    }

#ifdef PATH_MAX
    path_max = PATH_MAX;
#else
    path_max = pathconf (name, _PC_PATH_MAX);
    if (path_max <= 0)
    {
        path_max = 1024;
    }
#endif

    rpath = (char*)alloca(path_max);
    if (rpath == NULL)
    {
        LG(debug, "E: alloca() failed: %s\n", strerror(errno));
        errno = ENOMEM;
        return false;
    }
    limit = rpath + path_max;

    LG(debug, "I: path = %s\n", path);
    LG(debug, "I: is path absolute... ");
    if (path[0] != '/')
    {
        LG(debug, "no\n");
        if (!getcwd(rpath, path_max))
        {
            LG(debug, "E: getcwd() failed: %s\n", strerror(errno));
            rpath[0] = '\0';
            goto error;
        }
        dest = strchr(rpath, '\0');
    }
    else
    {
        LG(debug, "yes\n");
        rpath[0] = '/';
        dest = rpath + 1;
    }

    // iterated over path
    for (beg = end = path; *beg; beg = end)
    {
        // skip sequences of path-separators
        while (*beg == '/')
        {
            ++beg;
        }

        // find the end of the path element
        for (end = beg; *end && *end != '/'; ++end)
        {
            // intentionally empty
        }

        // evaluate whether we found the end of a path element
        if (end - beg == 0)
        {
            // path element is of zero length ... we must be finished
            break;
        }
        else if (end - beg == 1 && beg[0] == '.')
        {
            // remain in cwd ... loop again
        }
        else if (end - beg == 2 && beg[0] == '.' && beg[1] == '.')
        {
            // back up one directory, ignoring if already at root
            if (dest > rpath + 1)   // check if already at root
            {
                while ((--dest)[-1] != '/')
                {
                    // intentionally empty
                }
            }
        }
        else
        {
            struct stat st;
            int         n;

            LG(debug, "I: found a real path element\n");
            // found a real path element

            // add a path separator, if necessary
            if (dest[-1] != '/')
            {
                *dest++ = '/';
            }

            // ensure that we don't overflow rpath
            if (dest + (end - beg) >= limit)
            {
                LG(debug, "E: too long!\n");
                errno = ENAMETOOLONG;
                if (dest > rpath + 1)
                {
                    dest--;
                    *dest = '\0';
                    goto error;
                }
            }

            // copy the path element from path into rpath
            memcpy(dest, beg, end - beg);
            dest += end - beg;
            *dest = '\0';

            LG(debug, "I: rpath = %s\n", rpath);

            // test for /sfs ... return true if we found the hostname
            if (strncmp(rpath, "/sfs", 4) == 0)
            {
                if (strlen(rpath) > 4)
                {
                    hostname_ = strdup(rpath+5);
                    LG(debug, "I: hostname = %s\n", hostname_);
                    return true;
                }
            }

            if (lstat(rpath, &st) < 0)
            {
                LG(debug, "E: lstat() failed: %s\n", strerror(errno));
                goto error;
            }

            if (S_ISLNK(st.st_mode))
            {
                char   *link = (char*)alloca(path_max);
                char   *buf  = (char*)alloca(path_max);
                size_t  len;

                if (link == NULL || buf == NULL)
                {
                    LG(debug, "E: alloca() failed: %s\n", strerror(errno));
                    errno = ENOMEM;
                    goto error;
                }

                n = readlink(rpath, link, path_max);
                if (n < 0)
                {
                    LG(debug, "E: readlink() failed: %s\n", strerror(errno));
                    goto error;
                }
                link[n] = '\0';
                LG(debug, "I: link = %s\n", link);

                len = strlen (end);
                if ((long int)(n+len) >= path_max)
                {
                    errno = ENAMETOOLONG;
                    goto error;
                }

                memmove(&buf[n], end, len + 1);
                path = (char*)memcpy(buf, link, n);
                end = path;

                LG(debug, "I: is the symlink absolute...");
                if (link[0] == '/')
                {
                    LG(debug, " yes\n");
                    dest = rpath + 1;
                }
                else
                {
                    LG(debug, " no\n");
                    // back up one directory, ignoring if already at root
                    if (dest > rpath + 1)   // check if already at root
                    {
                        while ((--dest)[-1] != '/')
                        {
                            // intentionally empty
                        }
                    }
                }
            }
        }
    }

    if (dest > rpath + 1 && dest[-1] == '/')
    {
        --dest;
    }
    *dest = '\0';

error:
    return false;
}

int handler::AuthenticateUser()
{
    int i;

    if (!BecomeUser())
    {
        LG(debug_, "could not become the user\n");
        return PAM_CRED_INSUFFICIENT;
    }

    if (!OpenAgentConnection())     // is sfsagent running?
    {
        LG(debug_, "sfsagent is not running\n");

        L("starting sfsagent for %s@%s\n", username_, hostname_);
        if (!ExecAgent())
        {
            L("could not start sfsagent for %s@%s\n", username_, hostname_);
            return PAM_AUTHINFO_UNAVAIL;
        }

        LG(debug_, "connecting to sfsagent\n");
        for (i = 0; i < 10; i++)
        {
            if (OpenAgentConnection())
            {
                LG(debug_, "connected to sfsagent\n");
                break;
            }
            sleep(2);
        }
        if (i == 10)
        {
            L("could not connect to sfsagent for %s@%s\n",username_,hostname_);
            return PAM_AUTHINFO_UNAVAIL;
        }

        LG(debug_, "adding key to sfsagent\n");
        if (!AddKey())
        {
            KillAgent();
            CloseAgentConnection();
            L("could not add key to sfsagent for %s@%s\n",username_,hostname_);
            return PAM_AUTHINFO_UNAVAIL;
        }

    }
    else
    {
        LG(debug_, "sfsagent is already running\n");

        LG(debug_, "checking if sfsagent has key\n");
        if (!HasKey())
        {
            // we should never get here... the agent is running but doesn't
            // have the key... so who started the agent??
            LG(debug_, "sfsagent does not have key\n");

            if(!AddKey())
            {
                //KillAgent();  // we didn't start the agent
                CloseAgentConnection();
                L("could not add key to sfsagent for %s@%s\n",
                  username_, hostname_);
                return PAM_AUTHINFO_UNAVAIL;
            }
        }
    }

    LG(debug_, "sfs mounting home directory\n");
    for (i = 0; i < 10; i++)
    {
        LG(debug_, "mount attempt %d of 10\n", i+1);
        if (chdir(pw_->pw_dir) != -1)
        {
            L("sfs mount of %s@%s's home succeeded\n",username_,hostname_);
            break;
        }
        sleep(2);
    }
    if (i == 10)
    {
        L("sfs mount of %s@%s's home failed\n",username_,hostname_);
    }

    CloseAgentConnection();
    return PAM_SUCCESS;

}

bool handler::BecomeUser()
{
    if (getuid() != pw_->pw_uid)
    {
        if (setuid(pw_->pw_uid) == -1)
        {
            LG(debug_, "setuid() failed: %s\n", strerror(errno));
            return false;
        }
    }
    return true;
}

bool handler::ExecAgent()
{
    pid_t   pid;

    switch((pid = fork()))
    {
        case -1:    // internal error
            return false;
            break;

        case 0:     // child
            close(2);
            execl(SFSAGENT_PATH, "sfsagent", "-c", NULL);
            break;

        default:    // parent
            pid_ = pid;
            // no need to wait() on a daemon()
            break;
    }
    return true;
}

bool handler::OpenAgentConnection()
{
    clnt_stat   err;
    int32_t     res;

    sfscdfd_   = suidgetfd("agent");
    if (sfscdfd_ == -1)
    {
        LG(debug_, "suidgetfd() failed\n");
        return false;
    }

    sfscdxprt_ = axprt_unix::alloc(sfscdfd_);
    if (sfscdxprt_ == NULL)
    {
        LG(debug_, "axprt_unix::alloc() failed\n");
        return false;
    }

    sfscdclnt_ = aclnt::alloc(sfscdxprt_, agent_prog_1);
    if (sfscdclnt_ == NULL)
    {
        LG(debug_, "aclnt::alloc() failed\n");
        return false;
    }

    err = sfscdclnt_->scall(AGENT_GETAGENT, NULL, &res);
    if (err != RPC_SUCCESS)
    {
        LG(debug_, "AGENT_GETAGENT failed: %s\n", clnt_sperrno(err));
        return false;
    }
    if (res != SFS_OK)
    {
        LG(debug_, "AGENT_GETAGENT failed: %s\n", strerror(res));
        return false;
    }

    agentfd_ = sfscdxprt_->recvfd();
    if (agentfd_ == -1)
    {
        LG(debug_, "recvfd() failed\n");
        return false;
    }

    agentxprt_ = axprt_stream::alloc(agentfd_);
    if (agentxprt_ == NULL)
    {
        LG(debug_, "axprt_unix::alloc() failed\n");
        return false;
    }

    agentclnt_ = aclnt::alloc(agentxprt_, agentctl_prog_1);
    if (agentclnt_ == NULL)
    {
        LG(debug_, "aclnt::alloc() failed\n");
        return false;
    }

    return true;
}

bool handler::CloseAgentConnection()
{
    //delete agentclnt_; agentclnt_ = NULL;
    //delete agentxprt_; agentxprt_ = NULL;
    //delete sfscdclnt_; sfscdclnt_ = NULL;
    //delete sfscdxprt_; sfscdxprt_ = NULL;

    close(sfscdfd_);   sfscdfd_ = 0;
    close(agentfd_);   agentfd_ = 0;

    return true;
}

bool handler::HasKey()
{
    clnt_stat   err;
    sfs_keylist kl;

    err = agentclnt_->scall(AGENTCTL_DUMPKEYS, NULL, &kl);
    if (err != RPC_SUCCESS)
    {
        LG(debug_, "AGENTCTL_DUMPKEYS failed: %s\n", clnt_sperrno(err));
        return false;
    }

    for (sfs_keylistelm *kle = kl; kle; kle = kle->next)
    {
        if (kle->comment == keyname_)
        {
            return true;
        }
    }

    return false;
}

bool handler::AddKey()
{
    int     rval;
    int     len = strlen(SFSKEY_PATH)+strlen(keyname_)+39;
    char    cmd[len];
    FILE   *sfskey;
    
    rval = snprintf(cmd, len, "export HOME=/ ; %s -p 0 add %s 2>/dev/null",
                    SFSKEY_PATH, keyname_);
    if (rval == -1)
    {
        LG(debug_, "snprintf() failed: %s\n", strerror(errno));
        return false;
    }

    sfskey = popen(cmd, "w");
    if (sfskey == NULL)
    {
        LG(debug_, "popen() failed: %s\n", strerror(errno));
        return false;
    }

    fprintf(sfskey, "%s", password_);

    if (pclose(sfskey) != 0)
    {
        return false;
    }

    return true;
}

bool handler::KillAgent()
{
    clnt_stat   err;
    int32_t     res;

    err = sfscdclnt_->scall(AGENT_KILL, NULL, &res);
    if (err != RPC_SUCCESS)
    {
        LG(debug_, "AGENT_KILL failed: %s\n", clnt_sperrno(err));
        return false;
    }
    if (res != SFS_OK)
    {
        LG(debug_, "AGENT_KILL failed: %s\n", strerror(res));
        return false;
    }

    return true;
}

