/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>

#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include "config.h"

#include "glue-gui-gtk.h"
#include "glue.h"

#include "seg7_sa5211cgkwa_gui.h"

#ifdef HAVE_GTK
#include "seg7_sa5211cgkwa_gui_gtk.h"
#endif

#define HZ	10

#define OFFX    5
#define OFFY    5
#define LX      20
#define LY      20
#define LSPACE  16
#define DX      (LX + LSPACE)

#define COMP_(x) seg7_sa5211cgkwa_gui_gtk_ ## x

struct cpssp {
	int state_seg[8];
	unsigned long long time_seg[8];
	unsigned long long sum_seg[8];

	GtkWidget *area;
};

static void
COMP_(hor)(
	GtkWidget *widget,
	int state,
	gint x,
	gint y
)
{
	GdkGC *gc = state
			? widget->style->fg_gc[widget->state]
			: widget->style->bg_gc[widget->state];

	gdk_draw_line(widget->window, gc,
			x + 4, y - 2, x + LX - 4, y - 2);
	gdk_draw_line(widget->window, gc,
			x + 3, y - 1, x + LX - 3, y - 1);
	gdk_draw_line(widget->window, gc,
			x + 2, y, x + LX - 2, y);
	gdk_draw_line(widget->window, gc,
			x + 3, y + 1, x + LX - 3, y + 1);
	gdk_draw_line(widget->window, gc,
			x + 4, y + 2, x + LX - 4, y + 2);
}

static void
COMP_(vert)(
	GtkWidget *widget,
	int state,
	gint x,
	gint y
)
{
	GdkGC *gc = state
			? widget->style->fg_gc[widget->state]
			: widget->style->bg_gc[widget->state];

	gdk_draw_line(widget->window, gc,
			x - 2, y + 4, x - 2, y + LY - 4);
	gdk_draw_line(widget->window, gc,
			x - 1, y + 3, x - 1, y + LY - 3);
	gdk_draw_line(widget->window, gc,
			x, y + 2, x, y + LY - 2);
	gdk_draw_line(widget->window, gc,
			x + 1, y + 3, x + 1, y + LY - 3);
	gdk_draw_line(widget->window, gc,
			x + 2, y + 4, x + 2, y + LY - 4);
}

static void
COMP_(dot)(
	GtkWidget *widget,
	int state,
	gint x,
	gint y
)
{
	GdkGC *gc = state
			? widget->style->fg_gc[widget->state]
			: widget->style->bg_gc[widget->state];

	gdk_draw_line(widget->window, gc,
			x - 2, y - 2, x + 2, y - 2);
	gdk_draw_line(widget->window, gc,
			x - 2, y - 1, x + 2, y - 1);
	gdk_draw_line(widget->window, gc,
			x - 2, y + 0, x + 2, y + 0);
	gdk_draw_line(widget->window, gc,
			x - 2, y + 1, x + 2, y + 1);
	gdk_draw_line(widget->window, gc,
			x - 2, y + 2, x + 2, y + 2);
}

static const struct {
	void (*func)(GtkWidget *widget, int state, gint x, gint y);
	gint x;
	gint y;
} COMP_(table)[8] = {
	/* A */  { COMP_(hor), 0 * LX, 0 * LY },
	/* B */  { COMP_(vert), 1 * LX, 0 * LY },
	/* C */  { COMP_(vert), 1 * LX, 1 * LY },
	/* D */  { COMP_(hor), 0 * LX, 2 * LY },
	/* E */  { COMP_(vert), 0 * LX, 1 * LY },
	/* F */  { COMP_(vert), 0 * LX, 0 * LY },
	/* G */  { COMP_(hor), 0 * LX, 1 * LY },
	/* DP */ { COMP_(dot), 1 * LX + LSPACE / 2, LY * 2 },
};

static int
COMP_(expose)(
	GtkWidget *widget,
	GdkEventExpose *event,
	gpointer _cpssp
)
{
	struct cpssp *cpssp = _cpssp;
	int i;

	if (0 < event->count)
		return FALSE;

	gdk_window_clear_area(cpssp->area->window,
			0, 0,
			cpssp->area->allocation.width,
			cpssp->area->allocation.height);

	for (i = 0; i < 8; i++) {
		(*COMP_(table)[i].func)(cpssp->area,
				cpssp->state_seg[i],
				OFFX + COMP_(table)[i].x,
				OFFY + COMP_(table)[i].y);
	}
	gui_gtk_flush();

	return FALSE;
}

static void
COMP_(timer)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	int i;
	unsigned long long tv;
	unsigned long long t;

	tv = time_virt();
	for (i = 0; i < 8; i++) {
		t = cpssp->sum_seg[i];
		cpssp->sum_seg[i] = 0;
		if (cpssp->state_seg[i]) {
			t += tv - cpssp->time_seg[i];
			cpssp->time_seg[i] = tv;
		}

		(*COMP_(table)[i].func)(cpssp->area,
				TIME_HZ / HZ / 4 < t,
				OFFX + COMP_(table)[i].x,
				OFFY + COMP_(table)[i].y);
	}
	gui_gtk_flush();

	time_call_after(TIME_HZ / HZ, COMP_(timer), cpssp);
}

static void
COMP_(seg_n_set)(struct cpssp *cpssp, int n, int on)
{
	if (cpssp->state_seg[n] == on) {
		return;
	}

	if (on) {
		cpssp->time_seg[n] = time_virt();
	} else {
		cpssp->sum_seg[n] += cpssp->time_seg[n] - time_virt();
	}
	cpssp->state_seg[n] = on;
}

#define FUNC(n) \
static void \
COMP_(seg ## n ## _set)( \
	void *_cpssp, \
	unsigned int x, unsigned int y, \
	uint8_t r, uint8_t g, uint8_t b \
) \
{ \
	struct cpssp *cpssp = _cpssp; \
	\
	COMP_(seg_n_set)(cpssp, n, g ? 0 : 1); \
}
FUNC(0) FUNC(1) FUNC(2) FUNC(3)
FUNC(4) FUNC(5) FUNC(6) FUNC(7)
#undef FUNC

void *
COMP_(create)(
	unsigned int page,
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_e,
	struct sig_std_logic *port_d,
	struct sig_std_logic *port_anode0,
	struct sig_std_logic *port_c,
	struct sig_std_logic *port_dp,
	struct sig_std_logic *port_b,
	struct sig_std_logic *port_a,
	struct sig_std_logic *port_anode1,
	struct sig_std_logic *port_f,
	struct sig_std_logic *port_g,
	struct sig_opt_rgb *port_opt_a,
	struct sig_opt_rgb *port_opt_b,
	struct sig_opt_rgb *port_opt_c,
	struct sig_opt_rgb *port_opt_d,
	struct sig_opt_rgb *port_opt_e,
	struct sig_opt_rgb *port_opt_f,
	struct sig_opt_rgb *port_opt_g,
	struct sig_opt_rgb *port_opt_dp
)
{
#define FUNC(n) \
	static const struct sig_opt_rgb_funcs seg ## n ## _funcs = { \
		.pixel_set = COMP_(seg ## n ## _set), \
	}
	FUNC(0); FUNC(1); FUNC(2); FUNC(3);
	FUNC(4); FUNC(5); FUNC(6); FUNC(7);
#undef FUNC
	struct cpssp *cpssp;
	int i;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	for (i = 0; i < 8; i++) {
		cpssp->state_seg[i] = 0;
		cpssp->time_seg[i] = 0;
		cpssp->sum_seg[i] = 0;
	}

	sig_opt_rgb_connect(port_opt_a, cpssp, &seg0_funcs);
	sig_opt_rgb_connect(port_opt_b, cpssp, &seg1_funcs);
	sig_opt_rgb_connect(port_opt_c, cpssp, &seg2_funcs);
	sig_opt_rgb_connect(port_opt_d, cpssp, &seg3_funcs);
	sig_opt_rgb_connect(port_opt_e, cpssp, &seg4_funcs);
	sig_opt_rgb_connect(port_opt_f, cpssp, &seg5_funcs);
	sig_opt_rgb_connect(port_opt_g, cpssp, &seg6_funcs);
	sig_opt_rgb_connect(port_opt_dp, cpssp, &seg7_funcs);

	cpssp->area = gtk_drawing_area_new();
	gtk_widget_set_size_request(cpssp->area, OFFX + LX + OFFX, OFFY + 2 * LY + OFFY);
	g_signal_connect(G_OBJECT(cpssp->area), "expose_event",
			G_CALLBACK(COMP_(expose)), cpssp);
	gtk_widget_show(cpssp->area);

	gui_gtk_comp_add(page, "Digit", name, cpssp->area, TRUE, TRUE, NULL);

	time_call_after(0, COMP_(timer), cpssp);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
