// vs_menu.cc

#include "vs_menu.h"
#include "vscreen.h"
#include "config/colors.h"

#include <sigc++/bind.h>
#include <sigc++/object_slot.h>
#include <sigc++/retype_return.h>

#include <algorithm>
#include <ctype.h>

using namespace std;

keybindings *vs_menu::bindings=NULL;

vs_menu_info::vs_menu_info(item_types type, const char *name, const char *binding,
			   const char *description, SigC::Slot0<void> slot)
  :item_type(type), item_name(name), item_binding(binding),
   item_description(description), item_slot(slot), item_enabled(NULL)
{
}

vs_menu_info::vs_menu_info(item_types type, const char *name, const char *binding,
			   const char *description, SigC::Slot0<void> *slot)
  :item_type(type), item_name(name), item_binding(binding),
   item_description(description), item_slot(slot), item_enabled(NULL)
{
}

vs_menu_info::vs_menu_info(item_types type, const char *name, const char *binding,
			   const char *description, SigC::Slot0<void> slot,
			   SigC::Slot0<bool> enabled)
  :item_type(type), item_name(name), item_binding(binding),
   item_description(description), item_slot(slot), item_enabled(enabled)
{
}

vs_menu_info::vs_menu_info(item_types type, const char *name, const char *binding,
			   const char *description, SigC::Slot0<void> *slot,
			   SigC::Slot0<bool> enabled)
  :item_type(type), item_name(name), item_binding(binding),
   item_description(description), item_slot(slot), item_enabled(enabled)
{
}

vs_menu_info::vs_menu_info(item_types type, const char *name, const char *binding,
			   const char *description, SigC::Slot0<bool> slot,
			   SigC::Slot0<bool> enabled)
  :item_type(type), item_name(name), item_binding(binding),
   item_description(description), item_slot(SigC::retype_return<void>(slot)),
   item_enabled(enabled)
{
}

vs_menu_info::vs_menu_info(item_types type)
  :item_type(type), item_name(NULL), item_binding(NULL),
   item_description(NULL), item_slot(NULL), item_enabled(NULL)
{
}

vs_menu_item::vs_menu_item(const string &_title, const string &_binding,
			   const string &_description)
  :title(_title), description(_description), binding(_binding),
   hotkey((chtype) ERR)
{
  for(string::size_type i=0; i<title.size(); i++)
    if(title[i]=='^' && i+1<title.size())
      {
	hotkey=title[i+1];
	break;
      }
}

bool vs_menu_item::is_enabled()
{
  if(enabled.empty())
    return !selected.empty();
  else
    return enabled();
}

vs_menu::vs_menu()
  :vscreen_widget(), cursorloc(0), req_width(2)
{
  shown_sig.connect(slot(*this, &vs_menu::appear));
}

vs_menu::~vs_menu()
{
  for(itemlist::iterator i=items.begin(); i!=items.end(); i++)
    delete *i;
}

vs_menu::vs_menu(int x, int y, int w, vs_menu_info *inf)
  :vscreen_widget(), cursorloc(0), req_width(w)
{
  while(inf->item_type!=vs_menu_info::VS_MENU_END)
    {
      switch(inf->item_type)
	{
	case vs_menu_info::VS_MENU_ITEM:
	  assert(inf->item_name!=NULL);

	  {
	    vs_menu_item *newitem=new vs_menu_item(inf->item_name,
						   inf->item_binding?inf->item_binding:"",
						   inf->item_description?inf->item_description:"");

	    if(inf->item_slot)
	      newitem->selected.connect(inf->item_slot);

	    if(inf->item_enabled)
	      newitem->enabled.connect(inf->item_enabled);

	    append_item(newitem);
	  }
	  break;
	case vs_menu_info::VS_MENU_SEPARATOR:
	  assert(inf->item_name==NULL);

	  append_item(NULL);
	  break;
	default:
	  fprintf(stderr, "ERROR: unknown item type code %i\n", inf->item_type);
	  abort();
	}

      inf++;
    }

  shown_sig.connect(slot(*this, &vs_menu::appear));
  hidden_sig.connect(slot(*this, &vs_menu::disappear));
}

void vs_menu::append_item(vs_menu_item *newitem)
{
  items.push_back(newitem);
  if(newitem)
    {
      string::size_type shortcutsize=0;
      string::size_type menusize=0;

      for(string::size_type i=0; i<newitem->get_title().size(); ++i)
	if(newitem->get_title()[i]!='^')
	  ++menusize;

      if(newitem->get_binding().empty())
	shortcutsize=0;
      else
	shortcutsize=global_bindings.keyname(newitem->get_binding()).size()+1;

      req_width=max<int>(menusize+2+shortcutsize,req_width);
    }

  if(get_visible())
    {
      vscreen_queuelayout();
      vscreen_update();
    }
}

void vs_menu::remove_item(vs_menu_item *item)
{
  itemlist::size_type idx=0;

  while(idx<items.size() && items[idx]!=item)
    ++idx;

  assert(idx<items.size());

  for(itemlist::size_type newidx=idx; newidx<items.size()-1; ++newidx)
    items[newidx]=items[newidx+1];

  items.pop_back();

  if(items.size()==0)
    set_cursor(0);
  else if(idx==cursorloc)
    set_cursor(prev_selectable(next_selectable(items.size()-1)));

  if(get_visible())
    vscreen_queuelayout();
}

size vs_menu::size_request()
{
  return size(req_width, items.size()+2);
}

void vs_menu::highlight_current()
{
  if(cursorloc>=0 && cursorloc<items.size())
    item_highlighted(items[cursorloc]);
  else
    item_highlighted(NULL);
}

void vs_menu::set_cursor(itemlist::size_type pos)
{
  if(cursorloc!=pos)
    {
      cursorloc=pos;
      highlight_current();

      if(get_visible())
	vscreen_update();
    }
}

void vs_menu::appear()
{
  cursorloc=next_selectable(0);

  highlight_current();
}

void vs_menu::disappear()
{
  set_cursor(items.size());
}

bool vs_menu::focus_me()
{
  return true;
}

bool vs_menu::selectable(itemlist::size_type pos)
{
  return pos>=0 && pos<items.size() && items[pos] && items[pos]->is_enabled();
}

vs_menu::itemlist::size_type vs_menu::next_selectable(itemlist::size_type pos)
{
  if(pos<0 || pos>=items.size())
    pos=0;

  while(pos<items.size() && (!items[pos] ||
			     !items[pos]->is_enabled()))
    ++pos;

  return pos;
}

vs_menu::itemlist::size_type vs_menu::prev_selectable(itemlist::size_type pos)
{
  if(pos<0 || pos>=items.size())
    pos=items.size()-1;

  while(pos>=0 && pos<items.size() && (!items[pos] ||
				       !items[pos]->is_enabled()))
    --pos;

  if(pos<0 || pos>=items.size())
    pos=items.size();

  return pos;
}

void vs_menu::sanitize_cursor(bool forward)
{
  if(forward)
    // Double-invoking next_selectable will search from the
    // start of the menu if searching forwards fails.
    cursorloc=next_selectable(next_selectable(cursorloc));
  else
    cursorloc=prev_selectable(prev_selectable(cursorloc));

  highlight_current();
}

bool vs_menu::handle_char(chtype ch)
{
  // This will ensure that the cursor is in bounds if possible, and that
  // if it is in bounds, a "real" item is selected.
  sanitize_cursor(true);

  if(bindings->key_matches(ch, "Up"))
    {
      if(cursorloc>0)
	{
	  itemlist::size_type newloc=prev_selectable(cursorloc-1);

	  if(newloc>=0 && newloc<items.size())
	    set_cursor(newloc);

	  vscreen_update();
	}
    }
  else if(bindings->key_matches(ch, "Down"))
    {
      if(cursorloc<items.size()-1)
	{
	  itemlist::size_type newloc=next_selectable(cursorloc+1);

	  if(newloc>=0 && newloc<items.size())
	    set_cursor(newloc);

	  vscreen_update();
	}
    }
  else if(bindings->key_matches(ch, "Begin"))
    set_cursor(next_selectable(0));
  else if(bindings->key_matches(ch, "End"))
    set_cursor(prev_selectable(items.size()-1));
  else if(bindings->key_matches(ch, "Confirm"))
    {
      itemlist::size_type selected=cursorloc;

      menus_goaway();
      item_highlighted(NULL);

      if(selectable(selected))
	items[selected]->selected();
    }
  else
    {
      for(itemlist::iterator i=items.begin(); i!=items.end(); i++)
	if(*i && (*i)->is_enabled() &&
	   toupper(ch)==toupper((*i)->get_hotkey()))
	  {
	    menus_goaway();
	    item_highlighted(NULL);

	    (*i)->selected();

	    return true;
	  }
      return vscreen_widget::handle_char(ch);
    }

  return true;
}

void vs_menu::dispatch_mouse(short id, int x, int y, int z, mmask_t bmask)
{
  if(bmask & (BUTTON1_RELEASED | BUTTON2_RELEASED |
	      BUTTON3_RELEASED | BUTTON4_RELEASED |
	      BUTTON1_CLICKED | BUTTON2_CLICKED |
	      BUTTON3_CLICKED | BUTTON4_CLICKED))
    {
      itemlist::size_type num=y-1;

      if(selectable(num))
	{
	  menus_goaway();
	  item_highlighted(NULL);
	  items[num]->selected();
	}
    }
  else if(bmask & (BUTTON1_PRESSED | BUTTON2_PRESSED |
		   BUTTON3_PRESSED | BUTTON4_PRESSED))
    {
      itemlist::size_type num=y-1;

      if(selectable(num))
	set_cursor(num);
    }
}

// FIXME: handle truncated menus (eg, from menus appearing that are
// larger than the screen).  REALLY FIX THIS!  It's incredibly annoying in, eg,
// GTK+, and text screens have even less space to play with..
void vs_menu::paint()
{
  int width, height;
  getmaxyx(height, width);

  attrset(get_color("MenuBorder"));
  mvaddch(0, 0, ACS_ULCORNER);
  for(int i=1; i<width-1; i++)
    addch(ACS_HLINE);
  addch(ACS_URCORNER);

  string::size_type maxlen=width-2;

  // Make sure that whatever is selected is really selectable.
  sanitize_cursor(true);

  for(itemlist::size_type i=0; i<items.size(); i++)
    if(items[i])
      {
	bool boldthis=false;

	attrset(get_color("MenuBorder"));
	mvaddch(1+i, 0, ACS_VLINE);
	mvaddch(1+i, width-1, ACS_VLINE);

	string title=items[i]->get_title();
	string righttext=items[i]->get_binding().empty()?"":global_bindings.keyname(items[i]->get_binding());

	bool enabled=items[i]->is_enabled();

	if(i==cursorloc)
	  attrset(get_color("HighlightedMenuEntry"));
	else if(enabled)
	  attrset(get_color("MenuEntry"));
	else
	  attrset(get_color("DisabledMenuEntry"));

	move(1+i, 1);

	string::size_type titleloc=0, rightloc=0;

	for(string::size_type j=0; j<maxlen; j++)
	  {
	    while(titleloc<title.size() && title[titleloc]=='^')
	      {
		boldthis=enabled;
		++titleloc;
	      }

	    if(titleloc==title.size())
	      {
		addch(' ');
		titleloc++;
	      }
	    else if(titleloc>title.size())
	      {
		if(j<maxlen-righttext.size())
		  addch(' ');
		else
		  addch((unsigned char) righttext[rightloc++]);
	      }
	    else if(boldthis)
	      {
		addch(((unsigned char) title[titleloc++])|A_BOLD);
		boldthis=false;
	      }
	    else
	      addch((unsigned char) title[titleloc++]);
	  }
      }
    else
      {
	attrset(get_color("MenuBorder"));
	mvaddch(1+i, 0, ACS_LTEE);
	for(int j=1; j<width-1; j++)
	  addch(ACS_HLINE);
	addch(ACS_RTEE);
      }

  attrset(get_color("MenuBorder"));
  for(int i=items.size()+1; i<height-1; i++)
    {
      move(i, 0);
      addch(ACS_VLINE);
      for(int j=0; j<width-2; j++)
	addch(' ');
      addch(ACS_VLINE);
    }

  mvaddch(height-1, 0, ACS_LLCORNER);
  for(int i=1; i<width-1; i++)
    addch(ACS_HLINE);
  addch(ACS_LRCORNER);
}

bool vs_menu::get_cursorvisible()
{
  sanitize_cursor(true);

  return cursorloc>=0 && cursorloc<items.size();
}

point vs_menu::get_cursorloc()
{
  sanitize_cursor(true);

  return point(0, 1+cursorloc);
}

void vs_menu::init_bindings()
{
  bindings=new keybindings(&global_bindings);
}
