#include <vector>
#include <string>
#include <sstream>
#include <iostream>

#include <glibmm/refptr.h>
#include <gdkmm/pixbuf.h>
#include <gtkmm/widget.h>
#include <gtkmm/stock.h>

using namespace std;

#include "cataloginfo.h"
#include "config.h"
#include "musiccatalog.h"
#include "facontext.h"
#include "utf8.h"

#include "../res/album_pix.xpm"
#include "../res/all_pix.xpm"
#include "../res/artist_pix.xpm"
#include "../res/catalog_pix.xpm"
#include "../res/playlist_pix.xpm"
#include "../res/track_pix.xpm"
#include "../res/uncatagorized_pix.xpm"
#include "../res/cd_pix.xpm"
#include "../res/streams_pix.xpm"

//Static Variables
//MusicCatalog *CatalogInfo::m_catalog = NULL;
MusicCollection *CatalogInfo::m_collection = NULL;

Glib::RefPtr<Gdk::Pixbuf> CatalogInfo::m_pic_Catalog;
Glib::RefPtr<Gdk::Pixbuf> CatalogInfo::m_pic_AllItems;
Glib::RefPtr<Gdk::Pixbuf> CatalogInfo::m_pic_UnCatItems;
Glib::RefPtr<Gdk::Pixbuf> CatalogInfo::m_pic_Artist;
Glib::RefPtr<Gdk::Pixbuf> CatalogInfo::m_pic_Album;
Glib::RefPtr<Gdk::Pixbuf> CatalogInfo::m_pic_Track;
Glib::RefPtr<Gdk::Pixbuf> CatalogInfo::m_pic_Playlist;
Glib::RefPtr<Gdk::Pixbuf> CatalogInfo::m_pic_Stream;
Glib::RefPtr<Gdk::Pixbuf> CatalogInfo::m_pic_CD;
Glib::RefPtr<Gdk::Pixbuf> CatalogInfo::m_pic_nothing;

void CatalogInfo::init(FAContext *context, Gtk::Widget *widget)
{
	//Save Catalog
	//m_catalog = context->catalog;
    m_collection = context->collection;
	
	//Create Icons
	m_pic_Catalog = Gdk::Pixbuf::create_from_xpm_data(catalog_pix);
	m_pic_AllItems = Gdk::Pixbuf::create_from_xpm_data(all_pix);
	m_pic_UnCatItems = Gdk::Pixbuf::create_from_xpm_data(uncatagorized_pix);
	m_pic_Artist = Gdk::Pixbuf::create_from_xpm_data(artist_pix);
	m_pic_Album = Gdk::Pixbuf::create_from_xpm_data(album_pix);
	m_pic_Track = Gdk::Pixbuf::create_from_xpm_data(track_pix);
	m_pic_Playlist = Gdk::Pixbuf::create_from_xpm_data(playlist_pix);
	m_pic_Stream = Gdk::Pixbuf::create_from_xpm_data(streams_pix);
	m_pic_CD = Gdk::Pixbuf::create_from_xpm_data(cd_pix);
	//Need Gtk::Widget To Create Stock Icons
	m_pic_nothing = widget->render_icon(Gtk::Stock::MISSING_IMAGE, Gtk::ICON_SIZE_MENU);
}



/*******************
 All Tracks Info Class
*******************/
bool AllTracksInfo::refresh_item_data()
{
	//Clear Existing Data
	m_urls.clear();
    m_collection->getTrackList(m_urls);
	
	return true;
}

uint32_t AllTracksInfo::child_count()
{
     //Make Sure Vector Has Been Filled In
	if(!m_urls.size()) refresh_item_data();

    return m_urls.size();
}

CatalogInfo* AllTracksInfo::child_item(uint32_t item_num)
{
	if(item_num < child_count()) { //Make Sure We Have Children
		//Create New Track Info
		TrackInfo* new_item = new TrackInfo(m_urls[item_num], true);
		return new_item;
	} else
		return CatalogInfo::child_item(item_num);
}


void AllTracksInfo::get_tracks(std::vector<std::string>&tracks)
{
	//Make Sure Vector Has Been Filled In
	if(!m_urls.size())
		refresh_item_data();
	tracks  =  m_urls;
}


/*******************
 Artist Info Class
*******************/
ArtistInfo::ArtistInfo(const std::string& name)
    :m_name(name), m_albums() 
{ 
    char *tu;
    utf8_encode(name.c_str(), &tu);
    m_text = tu; 
    delete[] tu;
}


bool ArtistInfo::refresh_item_data()
{
    m_collection->getAlbumList(m_name, m_albums);
    return true;
}

uint32_t ArtistInfo::child_count()
{
	if(m_albums.size() || refresh_item_data())
		return m_albums.size();
	else
		return 0;
}

CatalogInfo* ArtistInfo::child_item(uint32_t item_num)
{
	if(item_num < child_count()) { //Make Sure We Have Children
		//Create New Album Info
		AlbumInfo* new_item = new AlbumInfo(m_name, m_albums[item_num]);
		return new_item;
	} else
		return CatalogInfo::child_item(item_num);
}

void ArtistInfo::get_tracks(std::vector<std::string>&tracks)
{
    m_collection->getTrackList(m_name, tracks);
}


/**********************
 Album Info Class
**********************/
AlbumInfo::AlbumInfo(const std::string& artist, const std::string& name)
    : m_artist(artist), m_name(name), m_tracks() 
{
    char *tu;
    utf8_encode(name.c_str(), &tu);
    m_text = tu; 
    delete[] tu;
}



bool AlbumInfo::refresh_item_data()
{
    m_collection->getTrackList(m_artist, m_name, m_tracks);
    return true;
}

uint32_t AlbumInfo::child_count()
{
	if(m_tracks.size() || refresh_item_data())
		return m_tracks.size();
	else
		return 0;
}

CatalogInfo* AlbumInfo::child_item(uint32_t item_num)
{
	if(item_num < child_count()) { //Make Sure We Have Children
		//Create New Track Info
		TrackInfo* new_item = new TrackInfo(m_tracks[item_num]);
		return new_item;
	} else
		return CatalogInfo::child_item(item_num);
}

void AlbumInfo::get_tracks(std::vector<std::string>&tracks)
{
	if(!m_tracks.size()) //Check We Have An Array
		refresh_item_data();
    tracks = m_tracks;
}


/********************
 Track Info Class
********************/
bool TrackInfo::refresh_item_data()
{
    m_data[Metadata::kTitle];
    m_data[Metadata::kArtist];
    m_data[Metadata::kTrack];

    return m_collection->readMetadata(m_url, m_data);
}

const Glib::ustring& TrackInfo::item_text()
{
	//Read Data If Needed
    if (m_text.size() == 0) {
        if(!m_data.size())
            refresh_item_data();

        if (m_full_text){
            m_text = m_data[Metadata::kArtist];
            m_text += " - ";
        } else if (m_data.hasKey(Metadata::kTrack)){
            m_text = m_data[Metadata::kTrack];
            m_text += " - ";
        }
        m_text += m_data[Metadata::kTitle];
    }
    return m_text;
}

Metadata& TrackInfo::item_info()
{
	if(m_data.size() || refresh_item_data()) //Make Sure We Have Metadata
		return m_data;
	else
		return CatalogInfo::item_info();
}

void TrackInfo::get_tracks(vector<string>&tracks)
{
	//Only One URL, our own
	tracks.push_back(m_url);
}


/***********************
 Playlist Info Class
***********************/
PlaylistInfo::PlaylistInfo(const std::string& url)
{ 
    m_url = url; 
    
	//Parse Name From URL
	std::string::size_type posStart, posEnd;
	
	posStart = m_url.rfind(DIR_MARKER, m_url.length()) + 1;
	posEnd = m_url.rfind('.', m_url.length());
    string name = m_url.substr(posStart, posEnd - posStart);
    char *tu;
    utf8_encode(name.c_str(), &tu);
    m_text = tu;
    delete[] tu;
}


bool PlaylistInfo::refresh_item_data()
{
    return m_collection->containsURL(m_url);
}

void PlaylistInfo::get_tracks(vector<string>&tracks)
{
    m_collection->readEntries(m_url, tracks);
}


/*********************
 Catalog Root Class
*********************/
uint32_t CatalogRoot::child_count()
{
	return 4;
}

CatalogInfo* CatalogRoot::child_item(uint32_t item_num)
{
	CatalogInfo* new_node;
	if(item_num == 0) { //My Music
		new_node = new MyMusicInfo;
	} else if(item_num == 1) { //My Playlists
		new_node = new MyPlaylistsInfo;
	} else if(item_num == 2) { //My Streams
		new_node = new MyStreamsInfo;
	} else if(item_num == 3) {
		new_node = new CDInfo;
	} else
		return CatalogInfo::child_item(item_num);

	return new_node;
}


/*********************
 My Music Info Class
*********************/
bool MyMusicInfo::refresh_item_data()
{
    m_collection->getArtistList(m_artists);
    return true;
}


uint32_t MyMusicInfo::child_count()
{
    if (!m_artists.size()) refresh_item_data();
    return (m_artists.size() + 2);
}

CatalogInfo* MyMusicInfo::child_item(uint32_t item_num)
{
	CatalogInfo* new_item = NULL;
	if(item_num == 0) { //All Tracks
		new_item = new AllTracksInfo;
	} else if(item_num == 1) { //Unsorted Tracks
		new_item = new UnsortedTracksInfo;
	} else {
		if(item_num < child_count()) { //Make Sure We Have Children
			item_num -= 2; //Ignore All/Unsorted Tracks
			//Create New Artist Info
			new_item = new ArtistInfo(m_artists[item_num]);
		}
	}
	if(!new_item) 
		new_item =  CatalogInfo::child_item(item_num);
    return new_item;
}


/*************************
 My Playlists Info Class
*************************/



bool MyPlaylistsInfo::refresh_item_data()
{
    m_collection->getPlaylists(m_urls);
    return true;
}


uint32_t MyPlaylistsInfo::child_count()
{
    if (!m_urls.size()) refresh_item_data();
    return m_urls.size();
}

CatalogInfo* MyPlaylistsInfo::child_item(uint32_t item_num)
{
	if(item_num < child_count()) { //Make Sure We Have Children
		//Create New PlaylistInfo
		PlaylistInfo* new_item = new PlaylistInfo(m_urls[item_num]);
		return new_item;
	} else
		return CatalogInfo::child_item(item_num);
}



/**********************
 My Streams Info Class
**********************/
bool MyStreamsInfo::refresh_item_data()
{
    m_collection->getStreamList(m_urls);
    return true;
}


uint32_t MyStreamsInfo::child_count()
{
    if (!m_urls.size()) refresh_item_data();
    return m_urls.size();
}

CatalogInfo* MyStreamsInfo::child_item(uint32_t item_num)
{
	if(item_num < child_count()) { //Make Sure We Have Children
		//Create New StreamInfo
		StreamInfo* new_item = new StreamInfo(m_urls[item_num]);
		return new_item;
	} else
		return CatalogInfo::child_item(item_num);
}

void MyStreamsInfo::get_tracks(vector<string>&tracks)
{
    tracks =  m_urls;
}


/**********************
 * Streams Info Class
 **********************/
StreamInfo::StreamInfo(const std::string& url) : TrackInfo(url) 
{
}



/***************************
 Unsorted Tracks Info Class
***************************/
bool UnsortedTracksInfo::refresh_item_data()
{
    m_collection->getTrackList(m_urls);
    return true;
}


uint32_t UnsortedTracksInfo::child_count()
{
    if (!m_urls.size()) refresh_item_data();
    return m_urls.size();
}

CatalogInfo* UnsortedTracksInfo::child_item(uint32_t item_num)
{
	if(item_num < child_count()) { //Make Sure We Have Children
		//Create New Track Info
		TrackInfo* new_item = new TrackInfo(m_urls[item_num]);
		return new_item;
	} else
		return CatalogInfo::child_item(item_num);
}

void UnsortedTracksInfo::get_tracks(vector<string>&tracks)
{
    tracks =  m_urls;
}

/**********************
 Track List Info Class
**********************/

bool TrackListInfo::refresh_item_data()
{
	if(m_parent->refresh_item_data()) {
		//Get All Tracks From Parent
        m_parent->get_tracks(m_all_tracks);
		return true;
	} else
		return false;
}

uint32_t TrackListInfo::child_count()
{
	//Check We've Read The Track List
	if(m_all_tracks.size() == 0)
		refresh_item_data();
	
	return m_all_tracks.size();
}

CatalogInfo* TrackListInfo::child_item(uint32_t item_num)
{
	//Check We've Read The Track List
	if(m_all_tracks.size() == 0)
		refresh_item_data();
	
	if(item_num < m_all_tracks.size()) {
		//Create New Track Info
		return new TrackInfo(m_all_tracks.at(item_num));
 	}
	else
		return CatalogInfo::child_item(item_num);
}



FilesystemInfo::FilesystemInfo(const std::string& url) 
{ 
    m_url = url;
    //string path;
    //URLToFilePath(url, path);
}






/* arch-tag: 959b5730-341b-4b37-b32b-02ade298ee7b */
