
#ifndef __HASHTABLE_H__
#define __HASHTABLE_H__

#include "Hashable.h"
#include "DictValue.h"
#include "XPtrList.h"
#include "UtilStr.h"


/*
	A Hashtable is a special list.  It works like a word dictionary: words are "keys" that can be looked-up
to find definitions.  It has the exceptional and unique property of being able to "lookup" defininations (ie, data)
for a given key in O(1) time.  "Order one time" means the time it takes to lookup a piece of data doesn't depend
on the list size.  Even a sorted list, for example, takes O(log N) time to fetch data because you have to
still have to skip around in the list to search for your data.

A hashtable is desirable when you need lookup data from keys in a minimum amount of time.  There's lots more below...
*/


#define HT_KEY_FOUND		0x4000

#define HT_NO_FLAGS			0

// mFlags in KEntry only gives us 8 bits
#define HT_USER_FLAGS		0x1F
#define HT_VALUE_OWNED		0x20
#define HT_KEY_OWNED		0x40
#define HT_VALUE_UNUSED2	0x80



struct KEntry {
	public:
		unsigned long		mKey;
		const Hashable*		mHashable;
		void*				mValue;
		KEntry*				mNext;
		unsigned char		mFlags;
};


class UtilStrValue : public UtilStr, public DictValue {

};



typedef int (*IteratorFuncT)(const KEntry* inEntry, const void* inArg );



class Hashtable : public DictValue {


	public:
		//		Makes a new Hashtable.  The load factor how efficient memory usage is. The higher
		// the load factor, the more compact the Hashtable it, but at a penalty of performance.
		// In this implementaion, the performance gains out weight the more mem used by 4 or 5 to 1.
							Hashtable( int inLoadFactor = 50 );
		virtual				~Hashtable();

		// Returns the number of entries (ie, key-value pairs) in this hashtable
		inline long 		NumEntries() const											{ return mNumEntries;											}


		/*
			Webster's dictionary is an example of a Hashtable.  You use the first letter to get us in the proximity and then
		 do a fine search to find the word, making the entire process very fast, independent of the dictionary.
		 In Hashtable lingo in our example, the key is the word we're looking up and the value is its definition.
		 Like our dictionary, a Hashtable can only be used to fetch values from keys and can't get keys from values.
		 This class allows the client to own the value, the key, neither, or both.  You use Put( long, ) and PutOwned( long, )
		 when the key are already unique by nature (ex, memory address).  You'd use Put( Hashable*, ) and PutOwned( Hashable*, )
		 for keys that match in abstract ways (ex, two UtilStr containing the same text have different addresses still "match").
		 	To tell a Hashtable you a key owned, turn on the HT_KEY_OWNED bit flag in Put( Hashable*, ) or PutOwned( Hashable*, )
		 (a Hashable* is something able to be deleted while a long is just a number).  When a hashtable is destroyed or when
		 a match for an incoming Hashable* already exists, keys with HT_KEY_OWNED set will be deleted.  If you Put() a key that's
		 not owned in a hashtable containing a matching key that's owned, the owned key is kept (vs. getting destroyed and replaced
		 by the new key).  If you Put() a key that's owned in a hashtable containing a matching key that's not owned, the initial key
		 is replaced by the owned key.   If you Put() a key that's owned in a hashtable containing a matching key that's owned, the
		 initial key is kept and the incoming key is destroyed.
		 	To tell a Hashtable to delete values when they are replaced or when the Hashtable is destroyed, you must
		 derive your value class from DictValue and use one of the the PutOwned() functions.  When you call Put and that key already
		 exists, the old value is replaced the new value. The old value is returned *only* if it's NOT owned by this Hashtable,
		 otherwise NULL is returned.
			All bits within HT_USER_FLAGS passed are stored with the value.  They can be accessed via Get() and can be
		set via SetFlags().  User flags are often used to classify what 4 bytes are in the value (ex, a ptr? a code?). If
		the key or value isn't to be owned by a hashtable and you have no user flags to set, pass HT_NO_FLAGS.
			Put() and PutOwned() run in O(1) time.  Even looking up a word in Webster's dictionary is an O(log N)
		running time operation because the time needed to lookup a given word involves 'closing in' on the word using
		the alphabet headings on each page.  An example of an O(1) running time operation is fetching the 83rd element of an
		array--the time needed to fetch the 83rd element doesn't depend on how big the array is!  As you may have figured out,
		O(1) running time is exceptional and is the primary feature of hashtables.  */
		inline void*		Put( long					inKey, void* inValue,		long inFlags = HT_NO_FLAGS )	{ return put( inKey, NULL, inValue, inFlags );												}
		inline void*		Put( const Hashable*		inKey, void* inValue,		long inFlags )					{ return put( inKey->Hash( mHashParam ), inKey, inValue, inFlags );							}
		inline void*		PutOwned( long				inKey, DictValue* inValue,	long inFlags = HT_NO_FLAGS )	{ return put( inKey, NULL, inValue, HT_VALUE_OWNED | inFlags );								}
		inline void*		PutOwned( const Hashable*	inKey, DictValue* inValue,	long inFlags )					{ return put( inKey->Hash( mHashParam ), inKey, inValue, HT_VALUE_OWNED | inFlags );		}

		// 	Sets the user flags associated with the value for the given key.  Each key-value pair has an associated number of bit
		//    flags associated with it, some of which are user definable.  The bits you can set are HT_USER_FLAGS and are set using this fcn.
		//    You access key-value flags using Get().
		//	Returns: true if key-value was found and flags were set.
		//	Note: All user flag bits are zeroed when Put() or PutOwned() is called.
		inline bool			SetFlags( long				inKey, long inFlags )			{ return setFlags( inKey, NULL, inFlags );							}
		inline bool			SetFlags( const Hashable*	inKey, long inFlags )			{ return setFlags( inKey->Hash( mHashParam ), inKey, inFlags );		}

		//	Fetches the value associated with inKey (from a Put()) in O(1) time
		//	Note: If a matching key is found, a non-zero value is returned.
		//  Note: If outValue is NULL, the value is not returned.
		//	Returns:  HT_KEY_FOUND bit is on if matching key was found plus any user flags are returned (within the mask of HT_USER_FLAGS)
		long				Get( long				inKey, void** outValue = NULL ) const	{	return get( inKey, NULL, outValue );						}
		long				Get( const Hashable*	inKey, void** outValue = NULL ) const	{	return get( inKey->Hash( mHashParam ), inKey, outValue );	}

		//	Returns a list of all the values in this hashtable in no particular order.
		//	NOTE: The values are added one by one, so the list can have any sorting/order flag set on it.
		//	NOTE: O(N) running time.
		void				GetValues( XPtrList& outValues );

		// 	Returns a list of all the keys in this hashtable in no particular order
		//	NOTE: The keys are added one by one, so the list can have any sorting/order flag set on it.
		//	NOTE: O(N) running time.
		void				GetKeys( XPtrList& outKeys );

		/*	   Returns the key stored in this hashtable matching inKey.  GetKey() is useful when keys are owned and you're
			using a Hashtable to store keys and need to fetch a key stored in the table.  Suppose you were using a
			hashtable to make a webster's dictionary and the keys were UtilStrs and were owned.  If you needed to pass
			someone an original UtilStr holding "dog", you'd need to use GetKey() to fetch the original key:
			original = new UtilStr( "dog" );
			dict.Put( original, "an animal" );
			UtilStr str( "dog" );
			// dict.GetKey( str ) == original		*/
		const Hashable*		GetKey( const Hashable* inKey );

		//	Returns the nth key in this hashtable, with the values in no particular order. NULL is returned
		//  if inN invalid.
		//	NOTE: O(N) running time.
		// 	NOTE: Zero-based index is used.
		void*				GetValue( long inN );

		// 	These are 2 other handy ways of invoking Get(long, void**).  The only difference is that if the key isn't
		//	found, it is entered in this dictionary with a value set to zero/NULL.
		//  WARNING: *Only* use these to read values if any owned values exist in your
		//  Hashtable.  Otherwise, you risk overwriting a ptr to a value
		//  that's owned.  The following would be a memory leak:
		//  myHashTable.PutOwned( 55, new DataClass( "This is leaked" ) );
		//  myHashTable[ 55 ] = new DataClass( "Overwriter" ); // Instead, use: myHashTable.PutOwned( 55, new DataClass( "Overwriter" ) );
		void*&				operator[] ( const void* inKey );
		long&				operator[] ( const long inKey );

		//	Removes inKey from this Hashtable.  If inKey is not found, NULL is returned.
		//	If inKey was found, the key (and its associated value) are removed and the key's value is returned.
		//	Note:  if keys are owned, this fcn deletes the hashable that was used in Put()
		inline void*		Remove( long			inKey )								{ return remove( inKey, NULL );								}
		inline void*		Remove( const Hashable*	inKey )								{ return remove( inKey -> Hash( mHashParam ), inKey );		}

		//	Removes/clears all keys in this Dictionary.
		void				RemoveAll();

		//	Pre:	The compare function takes in two values and returns if the first is less, equal, or greater than (See XPtrList.h)
		//	Note:	The compare fcn is arbitrary, but if it's it left out (ie. NULL) then it sorts by value size, low to high.
		//	Note:	The compare fcn has the same prototype as the one in XPtrList.h, but this compare fcn is one qsort uses, which
		//			means the void ptrs comin in point to the array elements (not the 4 bytes in the cur pt in the array)
		//	Post: 	Get( outKeys[ i ] ) is the ith largest value in this list (in reference to the given compariston fcn)
		//	Post: 	outKeys.Count() == inNumToRank  (ie, only inNumToRank values of the ranking are returned)
		//	Note:	If inNumToRank is invalid, the full ranking is returned
		void				Rank( XPtrList& outKeys, ANSI_QSCompFcnT inCompFcn = NULL, long inNumToRank = -1 );

		//	This changes the param passed to Hashables when in Hash() or Equals().  If changing this affects the behavior of Hash() or Equals()
		//	for your keys DO NOT call this unless there are no entries in this Hashtable!
		void 				SetHashParam( long inParam )								{ mHashParam = inParam;										}

		//	Iterates through every key-value pair in this hashtable (duh, O(N) time)
		//  Warning: Don't modify values in KEntries unless you know what you're doing.
		void				Iterate( IteratorFuncT inFunc, void* inArg );

	protected:
		void				Rehash();

		KEntry*				fetchEntry( long inKey, const Hashable* inHashable ) const;
		void*				remove( long inKey, const Hashable* inHashable );
		void*				put( long inKey, const Hashable* inHashable, void* inValue, long inFlags );
		long				get( long inKey, const Hashable* inHashable, void** outValue ) const;
		bool				setFlags( long inKey, const Hashable* inHashable, long inFlags );

		void*				deleteEntry( KEntry* inEntry );

		static int			sLongComparitor( const void* inA, const void* inB );
		enum {
			NUM_SIZES		= 13
		};


		char				mLoadFactor;		// Percent: from 0 means 0% and 128 means 100%;
		static long			sTableSizes[ NUM_SIZES ];

		KEntry**			mTable;
		unsigned long		mTableSize;
		long				mNumEntries;
		long				mHashParam;
};

#endif // __HASHTABLE_H__
