
#ifndef __EG_OS_UTILS__
#define __EG_OS_UTILS__

#include "Eg Common.h"

#include "CEgFileSpec.h"
#include "UtilStr.h"
#include "CEgFileSpec.h"
#include "CEgErr.h"
#include "V3.h"

#if (defined EG_MAC || defined WIN_QUICKTIME_PRESENT)
#include <QuickDraw.h>
#endif

#define INITAL_DIR_STRLEN	502

class EgOSUtils {

public:
	// 			Call this once at program startup/end!!
	//  If you're compiling under windows and making a DLL, pass the DLL instance otherwise NULL is fine.
	static void			Initialize( void* inModuleInstance = NULL );
	static void			Shutdown();

	//	Post:	Assigns spec info for the next file in the dir specified by <folderSpec>.  If the directory doesn't exist or there
	//			aren't any more files, <false> is returned.  If a next file is found, <true> is returned and <outSpec> contains a spec to that file.
	//			If inFolders is true, only folders are retuned
	#define OSU_DO_FILES			0x1
	#define	OSU_DO_FOLDERS			0x2
	#define OSU_RESOLVE_ALIASES		0x4
	#define OSU_FIRST_FILE			1
	static bool			GetNextFile( const CEgFileSpec& inFolder, CEgFileSpec& outSpec, long inFlags, long* inIdx );

	//  Looks for an extension to remove from <inFileName> and stores the result in <outName> and the
	//  extension in <outExtension>
	//	Note:	Pass NULL in <outName> or <outExtension> if you don't want that information
	//	Note:	The same string is allowed to be used for both <inFileName> and <outName>.
	static void			FindExtension( const UtilStr& inFileName, UtilStr* outName, UtilStr* outExtension );

	//	Returns the 1 to 4 chars right of the lastmost '.' in the given string in uppercase (ex, '.JPG', 'MPEG', '.C')
	//	Note:  If the name contains no extension 0 is returned
	//	Note:  If the name contains greater than 4 chars, the first four are returned.
	static long			FindExtension( const UtilStr* inFileName );

	//			After Initialize, these contain the screen pixels per inch
	static int			sXdpi;
	static int			sYdpi;
	static int			sXSize;
	static int			sYSize;

	// 	Post:	This static method will display the given C string in a window with an OK button.
	static void			ShowMsg( const char* inMsg );
	static void			ShowMsg( const UtilStr& inMsg );

	//	Post:	Prints <inMsg> in a dialog box with an "OK" button and a "cancel" button.  <true>
	//			is returned if the use selects the "OK" button
	static bool			AreYouSure( const UtilStr& inMsg );
	static bool			AreYouSure( const char* inMsg );

	//	Post:	Asks "Save changes to '<inName>' before closing?"
	//	Note:	Returns 	0 if Cancel button, 1 for OK, and -1 for Don't Save
	static int			AskSaveChanges( const UtilStr& inName );
	static int			AskSaveChanges( const char* inName );

	//	Post:	With the file dialog, this prompts the user for a file name, etc.  If the func returns <true>,
	//			<outSpec> contains the file spec the user just entered
	static bool			AskSaveAs( const char* inPrompt, const UtilStr& inDefaultName, CEgFileSpec& outSpec, long inFileType = 0 );
	static bool			AskSaveAs( const char* inPrompt, const char* inDefaultName, CEgFileSpec& outSpec, long inFileType = 0 );

	//	Post:	With the file dialog, this prompts the user to select a file..  If the func returns <true>,
	//			<outSpec> contains the file spec the user just entered.
	//	Note:	Only file of type <inFileType> are visible. Zero means to accept all files types.
	static bool			AskOpen( const char* inPrompt, CEgFileSpec& outSpec, long inFileType = 0 );

	//	Post:	Puts the error message in a message box on the screen, saying
	//			"Error reading/writing "inName" -- inErrMsg
	//			<wasReading> determines chooses the word "reading" vs. "writing"
	static void			ShowFileErr( const UtilStr* inName, CEgErr& inErr, 			bool wasReading );
	static void			ShowFileErr( const UtilStr* inName, char* inErrMsg, 		bool wasReading );
	static void			ShowFileErr( const CEgFileSpec& inFileSpec, CEgErr& inErr,	bool wasReading );

	//	Post:	Makes the computer beep
	static void			Beep();

	//	Pre:	Designed to be called continously during a long process.
	//	Post:	Displays a spinning curor after a certain period of time.
	static void			SpinCursor();

	//	Post:	Call this to restore the cursor if it's been altered (ie, after a SpinCursor).
	//	Note:	If This proc is installed in the main event loop, you can call SpinCursor() whenever things are intensive
	static void			ResetCursor();

	// 	Post:	Hides/Shows the mouse cursor.
	static void			ShowCursor();
	static void			HideCursor();

	// 	Returns the current time in milliseconds since the system start.
	static long			CurTimeMS();

	//	Returns the global cords of the mouse
	static void			GetMouse( Point& outPt );

	//	Post:	Returns a random number from <min> to <max>, inclusive.
	static long 			Rnd( long min, long max );

	// Callback on some OSes to prevent processor sleep and/or cycling
	static void			PreventSleep();

	//	Post:	Reverses the byte order of the given long
	static unsigned long		RevBytes( unsigned long inNum );

	// Returns true if the given app was launched
	static bool			LaunchProcess( CEgFileSpec& inSpec );

	static bool			LaunchURL( const UtilStr& inURL, long inCreator_ID )														{ return LaunchURL( inURL.getCStr(), inCreator_ID );		}
	static bool			LaunchURL( const char* inURL, long inCreator_ID );
	static bool			LaunchHTMLFile( const CEgFileSpec& inHTMLFile, long inCreator_ID );
	static bool			LaunchDoc( const CEgFileSpec& inSpec, long inCreator_ID );

	// Is the directory/folder of this app/lib
	static CEgFileSpec		sAppSpec;

	// Converts a float HSV (each component is 0 to 1) into a 3x16 bit RGB value
	// H = .2 == 1.2 == 7.2 == -.8 == -3.8  (ie, H values are 'wrapped')
	// S,V = 1.0 == 1.1 == 6.9
	// S,V = 0 == -.1 == -4
	static void			HSV2RGB( float H, float S, float V, RGBColor& outRGB );

	// Converts a float RGB (each component is 0 to 1) into a HSV touple (each component is 0 to 1)
	static void			RGB2HSV( float R, float G, float B, float& outH, float& outS, float& outV );
	static void			RGB2HSV( RGBColor& inRGB, float& outH, float& outS, float& outV )						{ RGB2HSV( ((float)inRGB.red/65535.0), ((float)inRGB.green/65535.0), ((float)inRGB.blue/65535.0), outH, outS, outV );	}

	static void			CalcContrast( int inR, int inG, int inB, V3& outContrast );
	static void			CalcContrast( const RGBColor& inC, V3& outContrast )									{ CalcContrast( inC.red >> 8, inC.green >> 8, inC.blue >> 8, outContrast );												}
	static void			CalcContrast( const PixPalEntry& inC, V3& outContrast );

	static int			CalcIntensity( const PixPalEntry& inC );
	static int			CalcIntensity( const RGBColor& inC ) 													{ return ( 54 * inC.red + 183 * inC.green + 19 * inC.blue ) >> 16;	}
	static int			CalcIntensity( int inR, int inG, int inB ) 												{ return ( 54 * inR + 183 * inG + 19 * inB ) >> 8;					}

	// Returns a value from 0 to 100, 0 is zero contrast, 100 is max contrast
	static int			CalcContrast( const RGBColor& inC1, const RGBColor& inC2 )								{ return CalcContrast( inC1.red >> 8, inC1.green >> 8, inC1.blue >> 8, inC2.red >> 8, inC2.green >> 8, inC2.blue >> 8 );}
	static int			CalcContrast( const PixPalEntry& inC1, const PixPalEntry& inC2 );
	static int			CalcContrast( int inR, int inG, int inB, int inR2, int inG2, int inB2 );

	static void			CalcInverted( const RGBColor& inC, RGBColor& outInverted );

	static bool			sQTAvailable;

protected:
	static bool			sCanPreventSleep;
	static long			sLastCursor;
	static long			sLastCursorChange;
	static char			sInitialDir[ INITAL_DIR_STRLEN + 2 ];
};

#endif // __EG_OS_UTILS__
