/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include <config.h>
#include <stdlib.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qfont.h>
#include <qstyle.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qregexp.h>

#include <kwin.h>
#include <kfilemetainfo.h>
#include <klocale.h>

#ifdef HAVE_SMPEG
#include <smpeg.h>
#include "smpegplayer.h"
#include "smpegplayer.moc"

SMPEGPlayer::SMPEGPlayer(QStringList playList, QPoint playListPos, 
			 QSize playListSize, bool readID3Tags,
			 bool convertUnderscores, bool repeat, bool shuffle,
			 QPixmap *playListIcon) {

  running = false;
  playing = false;

  mpg = NULL;
  char buf[32];

  //this->playList = playList;
  this->readID3Tags = readID3Tags;
  this->convertUnderscores = convertUnderscores;

  qDebug("xmms-kde: created smpegplayer interface");

  if ((SDL_Init(SDL_INIT_AUDIO) < 0) || !SDL_AudioDriverName(buf, 1)) {
    fprintf(stderr, "Warning: Couldn't init SDL audio: %s\n",
	    SDL_GetError());
    fprintf(stderr, "Will ignore audio stream\n");
  }

  playListFrame = new SMPEGPlayList(playListPos, playListSize, playListIcon);
   
  connect((QObject *) playListFrame, SIGNAL(sigSelectionChanged()),
	  (QObject *) this, SLOT(selectionChanged()));

  title = QString("");
  volume = 100;

  QString files = playList.join("\r\n");

  playlistAdd(files);

  stopButtonPressed = false;

  if (repeat) toggleRepeat();
  if (shuffle) toggleShuffle();
}

SMPEGPlayer::~SMPEGPlayer() {

  if (mpg) {
    SMPEG_stop(mpg);
    SMPEG_delete(mpg);
  }
  
  if (playListFrame) 
    delete playListFrame;

  SDL_Quit();

}

void SMPEGPlayer::startPlayer() {

  running = true;

  playListFrame->show();

  if (!playList.isEmpty()) {
    playListFrame->setSelected(0, true);
  }
}

void SMPEGPlayer::exitPlayer() {

  playing = false;
  running = false;
  
  if (mpg) {
    SMPEG_stop(mpg);
    SMPEG_delete(mpg);
    mpg = NULL;
  }
}

void SMPEGPlayer::previous() {

  if (listIndex > 0) listIndex--;
  playListFrame->setSelected(listIndex, true);
}

void SMPEGPlayer::play() {

  if (mpg) {
    stopButtonPressed = false;
    SMPEG_rewind(mpg);
    SMPEG_play(mpg);
    playing = true;
  } else {
    selectionChanged();
  }
}

void SMPEGPlayer::pause() {

  if (mpg) {
    stopButtonPressed = !stopButtonPressed;
    SMPEG_pause(mpg);
  }
}

void SMPEGPlayer::stop() {

  if (mpg) {
    stopButtonPressed = true;
    SMPEG_rewind(mpg);
  }
}

void SMPEGPlayer::next() {
  
  playNext();
}

void SMPEGPlayer::toggleRepeat() {

  playListFrame->setRepeat(!playListFrame->getRepeat());
}

void SMPEGPlayer::toggleShuffle() {

  playListFrame->setShuffle(!playListFrame->getShuffle());
}

void SMPEGPlayer::setVolume(int volume) {

  if (volume > 100) volume = 100;
  if (volume < 0) volume = 0;

  this->volume = volume;
  if (mpg) SMPEG_setvolume(mpg, volume);
}

void SMPEGPlayer::setCurrentTime(int time) {

  if (mpg) {
    stopButtonPressed = false;
    getInfo();    

    SMPEG_rewind(mpg);
    SMPEG_skip(mpg, (int) (time / 1000));
    SMPEG_play(mpg);
  }
}

int SMPEGPlayer::getVolume() {

  return volume;
}

QString SMPEGPlayer::getTitle() {

  return title;
}

int SMPEGPlayer::getCurrentTime() {
  
  if (mpg) {
    getInfo();

    SMPEGstatus status = SMPEG_status(mpg);

    if (!stopButtonPressed && status == 0) {

      playNext();
    }

    return (int) (((double) info.current_offset / (double) info.total_size) * 
		  info.total_time * 1000);
  } else { 
    return 0;
  }  
}

int SMPEGPlayer::getTrackTime() {

  if (mpg) {
    getInfo();        
    return (int) (info.total_time * 1000);
  } else {
    return 0;
  }
}

bool SMPEGPlayer::getRepeat() {

  return playListFrame->getRepeat();
}

bool SMPEGPlayer::getShuffle() {

  return playListFrame->getShuffle();
}

bool SMPEGPlayer::isPlaying() {  

  return playing;
}

bool SMPEGPlayer::isPlayerRunning() {

  return running;
}

void SMPEGPlayer::openFileSelector() {

  int d = KWin::info(playListFrame->winId()).desktop;

  if (d != KWin::currentDesktop()) {
    
    KWin::setOnDesktop(playListFrame->winId(), KWin::currentDesktop());
  }

  if (playListFrame->isVisible()) 
    playListFrame->hide();
  else
    if (running) playListFrame->show();
}

void SMPEGPlayer::playlistClear() {

    if (running) {
      if (mpg) SMPEG_stop(mpg);
    }

    playList.clear();
    playListFrame->clear();
}


QString SMPEGPlayer::decodeString(QString s) {

  s = s.replace(QRegExp("%20"), " ");
  s = s.replace(QRegExp("%26"), "&");
  s = s.replace(QRegExp("%27"), "'");
  s = s.replace(QRegExp("%2C"), ",");

  s = s.replace(QRegExp("%C2%B4"), "");

  s = s.replace(QRegExp("%C3%A4"), "");
  s = s.replace(QRegExp("%C3%A6"), "");
  s = s.replace(QRegExp("%C3%84"), "");
  s = s.replace(QRegExp("%C3%B6"), "");
  s = s.replace(QRegExp("%C3%96"), "");
  s = s.replace(QRegExp("%C3%BC"), "");
  s = s.replace(QRegExp("%C3%9C"), "");

  s = s.replace(QRegExp("%C3%AD"), "");
  s = s.replace(QRegExp("%C3%B0"), "");
  s = s.replace(QRegExp("%C3%B3"), "");
  s = s.replace(QRegExp("%C3%BA"), "");

  return s;
}


void SMPEGPlayer::playlistAdd(QString file) {

  //qDebug("playlistAdd");
  // change the encoding of the file to unix filenames
  file = decodeString(file);

  inserting = true;
  
  int index = playListFrame->getSelected();

  addFilesToPlaylist(file);

  QStringList names;
  for (QStringList::Iterator it = playList.begin(); 
       it != playList.end(); ++it ) {

    if (readID3Tags) {
      QString s(readID3Tag(*it));
      if (convertUnderscores)
	s.replace(QRegExp("_"), " ");
      names << s;
    } else {
      QString s(QFileInfo(*it).fileName());
      if (convertUnderscores)
	s.replace(QRegExp("_"), " ");      
      names << s;
    }
  }
  
  playListFrame->insertStringList(names);
  playListFrame->setSelected(index, true);
  
  inserting = false;

  if (index == -1) 
      playListFrame->setSelected(0, true);

  playListFrame->centerCurrentItem();
}


void SMPEGPlayer::playlistAdd(QStringList list) {

  playlistAdd(list.join("\r\n"));
}

void SMPEGPlayer::addFilesToPlaylist(QString files) {

  //qDebug("addFilesToPlaylist");
  QStringList f = QStringList::split(QString("\r\n"), files);

  for (QStringList::Iterator it = f.begin(); it != f.end(); ++it) {

    //qDebug((QString) *it);

    QString fileName = (QString) *it;

    fileName = decodeString(fileName);

    if (fileName.startsWith("file:")) {
      fileName.remove(0, 5);
    }
   
    // don't add "." and ".." ("hidden" files also get sortet out...)
    if (QFileInfo(fileName).fileName().startsWith(".")) return;

    QFileInfo file(fileName);
  
    if (file.isDir()) {
      QDir dir(fileName);
      QStringList filelist(dir.entryList());
    
      // add the files in the directory, recursive
      for (QStringList::Iterator it = filelist.begin(); 
	   it != filelist.end(); ++it ) {
	addFilesToPlaylist(dir.absPath() + "/" + (*it));
      }
    } else {
      // add the file to the playlist
      KFileMetaInfo m(fileName);      
      if (m.isValid() && !m.mimeType().compare("audio/x-mp3")) {
	playList << fileName;
	//qDebug("add %s", fileName.latin1());
      }
    }    
  }

  //playListFrame->setSelected(0, true);
  //qDebug("");
}

void SMPEGPlayer::getInfo() {

  SMPEG_getinfo(mpg, &info);
}

void SMPEGPlayer::loadMPG(QString file, bool play) { 

  if (!running) return;

  if (mpg) {
    SMPEG_delete(mpg);
  }

  mpg = SMPEG_new(file, &info, true);
  if (SMPEG_error(mpg)) {
    fprintf(stderr, file, SMPEG_error(mpg));
    //SMPEG_delete(mpg);
    playing = false;
    mpg = NULL;
  } else {
  
    SMPEG_enableaudio(mpg, true);
    SMPEG_setvolume(mpg, volume);

    if (convertUnderscores) {
      QString s(readID3Tag(file).replace(QRegExp("_"), " "));
      title = s;
    } else title = readID3Tag(file);    

    if (play) SMPEG_play(mpg);
  }
}

/*
 * read the id3 tag of the mp3 file
 */
QString SMPEGPlayer::readID3Tag(QString filename) {

  QString t("");

  if (filename.right(4).lower() != ".mp3")
    return t;
  QFile file(filename);
  if (!file.open(IO_ReadOnly))
    return t;
  file.at(file.size() - 128);
  id3tag data;
  
  file.readBlock((char*) &data, sizeof(data));
  
  if (parse(data.field, 3) == "TAG") {    

    t = "[" + parse(data.artist) + "] " + parse(data.name);
  } else {

    QString name = QFileInfo(filename).fileName();
    t = name.left(name.length() - 4);
  }

  return t;
}

QString SMPEGPlayer::parse(char *d, int len) {

  char *temp = new char[len + 1];
  memcpy(temp, d, len);
  temp[len] = '\0';
  QString str(QString::fromLocal8Bit(temp));
  delete [] temp;
  str = str.simplifyWhiteSpace();
  return str;
}

void SMPEGPlayer::selectionChanged() {

  stopButtonPressed = false;
  if (inserting) return;
  
  listIndex = playListFrame->getSelected();

  if (!readID3Tags) {
    inserting = true;
    playListFrame->changeItem(readID3Tag(playList[listIndex]), listIndex);
    inserting = false;
  }

  if (!playList.isEmpty())
    loadMPG(playList[listIndex], true);
}

QStringList SMPEGPlayer::getPlayList() {
  
  return playList;
}

QPoint SMPEGPlayer::getPlayListPos() {

  return playListFrame->pos();
}

QSize SMPEGPlayer::getPlayListSize() {

  return playListFrame->size();
}

void SMPEGPlayer::playNext() {

  unsigned int last = playListFrame->getSelected();

  if (playListFrame->getShuffle()) {

    unsigned int j = (int) (playList.count() * (rand() / (RAND_MAX + 1.0)));
    playListFrame->setSelected(j, true);

    if (last == j)
      selectionChanged();

  } else {
    listIndex ++;
    if (listIndex >= playList.count()) {
      listIndex = 0;
      if (playListFrame->getRepeat()) {	
	playListFrame->setSelected(listIndex, true);
	if (listIndex == last)
	  selectionChanged();
      } else {
	stop();
      }
    } else {
      playListFrame->setSelected(listIndex, true);
    }
  }

  playListFrame->centerCurrentItem();
}


SMPEGPlayList::SMPEGPlayList(QPoint pos, QSize size, QPixmap *icon) 
  : QVBox(0, "Playlist", WStyle_Title) {

  playListBox = new QListBox(this);

  QHBox *hbox = new QHBox(this);
  repeatBox = new QCheckBox(i18n("repeat"), hbox);
  shuffleBox = new QCheckBox(i18n("shuffle"), hbox);

  connect((QObject *) playListBox, SIGNAL(highlighted(int)),
	  (QObject *) this, SLOT(selectionChanged()));

  QFont f = QFont(font());
  f.setPointSize(f.pointSize() - 1);
  setFont(f);

  resize(size);
  move(pos);
  setCaption(i18n("Playlist"));

  KWin::setIcons(winId(), *icon, *icon);
}

SMPEGPlayList::~SMPEGPlayList() {
}

void SMPEGPlayList::insertStringList(QStringList list) {

  playListBox->clear();

  for (QStringList::Iterator it = list.begin(); it != list.end(); ++it) {

     playListBox->insertItem(new QListBoxText(*it));
  } 
}

void SMPEGPlayList::setSelected(int num, bool s) {
  playListBox->setSelected(num, s);
}

int SMPEGPlayList::getSelected() {
  return playListBox->currentItem();
}

void SMPEGPlayList::clear() {
  playListBox->clear();
}


void SMPEGPlayList::changeItem(QString text, int index) {  
}

void SMPEGPlayList::centerCurrentItem() {
  playListBox->centerCurrentItem();
}

void SMPEGPlayList::setRepeat(bool r) {
  repeatBox->setChecked(r);
}

bool SMPEGPlayList::getRepeat() {
  return repeatBox->isChecked();
}

void SMPEGPlayList::setShuffle(bool s) {
  shuffleBox->setChecked(s);
}

bool SMPEGPlayList::getShuffle() {
  return shuffleBox->isChecked();
}

void SMPEGPlayList::selectionChanged() {
  
  emit sigSelectionChanged();
}


/*
SMPEGPlayListItem::SMPEGPlayListItem(QString title) 
  : QListBoxItem() {
     
     setText(title);
}


SMPEGPlayListItem::~SMPEGPlayListItem() {

}

void SMPEGPlayListItem::paint(QPainter *p) {

   //p->drawLine(0, 0, 10, 10);
}

int SMPEGPlayListItem::height(const QListBox *lb) {

   return 16;
}

int SMPEGPlayListItem::width(const QListBox *lb) {

   return 100;
}
*/

#endif









