/***************************************************************************
 *   copyright           : (C) 2003 by Hendrik Sattler                     *
 *   mail                : pingos@hendrik-sattler.de                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "ConfigFile.H"
#include <wx/settings.h>
#include <wx/fontutil.h>

TtConfigFile* TtConfigFile::ptr = 0;

TtConfigFile* TtConfigFile::getPtr() {
  if (ptr == 0) {
    ptr = new TtConfigFile();
  }
  return ptr;
}

TtConfigFile::TtConfigFile () {
  this->pConfig = new wxConfig(wxEmptyString, wxEmptyString, wxEmptyString,
#ifdef __WXMSW__
			       wxEmptyString,
#else
			       _T("tipptrainer.conf"),
#endif
			       wxCONFIG_USE_LOCAL_FILE|wxCONFIG_USE_GLOBAL_FILE);
  delete wxConfigBase::Set(this->pConfig);
}

TtConfigFile::~TtConfigFile () {
  delete wxConfigBase::Set((wxConfigBase*)0);
}

bool TtConfigFile::getBellStatus() {
  pConfig->SetPath(_T("/Settings"));
  bool retval;
  pConfig->Read(_T("Bell"),&retval,true);
  return retval;
}

void TtConfigFile::toggleBellStatus () {
  pConfig->SetPath(_T("/Settings"));
  if (getBellStatus()) {
    pConfig->Write(_T("Bell"), false);
  } else {
    pConfig->Write(_T("Bell"), true);
  }
}

wxFont TtConfigFile::getMainWindowFont () {
  pConfig->SetPath(_T("/Hauptfenster"));
  if (pConfig->Exists(_T("Font"))) {
    // due to a bug SetNativeFontInfo() has to be called
    // as wxFontBase::SetNativeFontInfo()
    wxFont retval;
    retval.wxFontBase::SetNativeFontInfo(pConfig->Read(_T("Font"),_T("")));
    return retval;
  } else {
    // don't use the wxFont copy constructor
    // or the default font will not work
    // for the font dialog in GTK
#ifdef __WXMSW__
    return *wxNORMAL_FONT;
#else
    return *wxSMALL_FONT;
#endif
  }
}

void TtConfigFile::setMainWindowFont (wxFont& storeFont) {
  pConfig->SetPath(_T("/Hauptfenster"));
  pConfig->Write(_T("Font"),storeFont.GetNativeFontInfoDesc());
  pConfig->Flush();
}

wxSize TtConfigFile::getMainWindowSize () {
  wxSize retval;
  pConfig->SetPath(_T("/Hauptfenster"));
  //needs to be lower that 800x600 because this might still
  //be a used screen resolution
  retval.SetWidth(pConfig->Read(_T("w"),(long) 750));
  retval.SetHeight(pConfig->Read(_T("h"),(long) 500));
  return retval;
}

void TtConfigFile::setMainWindowSize (wxSize size) {
  pConfig->SetPath(_T("/Hauptfenster"));
  pConfig->Write(_T("w"), size.GetWidth());
  pConfig->Write(_T("h"), size.GetHeight());
}

wxPoint TtConfigFile::getMainWindowPosition () {
  wxPoint retval;
  pConfig->SetPath(_T("/Hauptfenster"));
  //default should be to not move the windows at all
  retval.x = pConfig->Read(_T("x"),(long) 0);
  retval.y = pConfig->Read(_T("y"),(long) 0);
  return retval;
}

void TtConfigFile::setMainWindowPosition (wxPoint pos) {
  pConfig->SetPath(_T("/Hauptfenster"));
  pConfig->Write(_T("x"), pos.x);
  pConfig->Write(_T("y"), pos.y);
}

int TtConfigFile::getSashPosition (int defaultPos){
  return pConfig->Read(_T("s"),(long) defaultPos);
}

void TtConfigFile::setSashPosition (int pos) {
  pConfig->SetPath(_T("/Hauptfenster"));
  pConfig->Write(_T("s"),(long) pos);
}


bool TtConfigFile::issetLanguage () {
  return pConfig->Exists(_T("/Settings/language"));
}

void TtConfigFile::setLanguage (const wxString& lang) {
  wxString entry = _T("/Settings/language");
  if (lang.Length() == 0) {
    if (pConfig->Exists(entry)) {
      pConfig->DeleteEntry(entry);
    }
  } else {
    pConfig->Write(entry, lang);
  }
  pConfig->Flush();
}

wxString TtConfigFile::getLanguage () {
  wxString lang =  pConfig->Read(_T("/Settings/language"), _T(""));
  //some compatibility with old versions
  if (lang == _T("deutsch")) {
    return wxString(_T("german"));
  } else if (lang == _T("englisch")) {
    return wxString(_T("english"));
  } else {
    return lang;
  }
}

bool TtConfigFile::issetKeyboardLayout () {
  return pConfig->Exists(_T("/Settings/keybLayout"));
}

void TtConfigFile::setKeyboardLayout (const wxString& layout) {
  wxString entry = _T("/Settings/keybLayout");
  if (layout.Length() == 0) {
    if (pConfig->Exists(entry)) {
      pConfig->DeleteEntry(entry);
    }
  } else {
    pConfig->Write(entry, layout);
  }
  pConfig->Flush();
}

wxString TtConfigFile::getKeyboardLayout () {
  wxString ldefault;
  int language=wxLocale::GetSystemLanguage();
  if (language==wxLANGUAGE_GERMAN ||
      language==wxLANGUAGE_GERMAN_AUSTRIAN ||
      language==wxLANGUAGE_GERMAN_BELGIUM ||
      language==wxLANGUAGE_GERMAN_LIECHTENSTEIN ||
      language==wxLANGUAGE_GERMAN_LUXEMBOURG ||
      language==wxLANGUAGE_GERMAN_SWISS) {
    ldefault = _T("german");
  } else {
    ldefault = _T("english");
  }
  wxString layout =  pConfig->Read(_T("/Settings/keybLayout"), ldefault);
  //test the supported types
  if (layout != _T("german") &&
      layout != _T("english")) {
    return wxString(ldefault);
  } else {
    return wxString(layout);
  }
}

void TtConfigFile::storeKursNummer (const wxString& name,
				    const wxString& key,
				    int kursNummer) {
  wxString temp;
  temp << _T("/Kurs/") << name << _T("_") << key;
  pConfig->SetPath(temp);
  pConfig->Write(_T("kursnummer"), (long) kursNummer);
  pConfig->SetPath(_T("/"));
  pConfig->Flush();
}

int TtConfigFile::getKursNummer (const wxString& name,
				 const wxString& key) {
  wxString path;
  path << _T("/Kurs/") << name << _T("_") << key;
  pConfig->SetPath(path);
  //-1 is default because this is an invalid value
  //If kursnummer is missing, someone messed with the config file
  return pConfig->Read(_T("kursnummer"), -1);
}

std::vector<LEntry>* TtConfigFile::loadKursDaten (const wxString& name,
						  const wxString& key) {
  std::vector<LEntry>* retval = new std::vector<LEntry>();
  wxString s;
  s << _T("/Kurs/") << name << _T("_") << key;
  pConfig->SetPath(s);
  wxString t;
  //actually, counter should start at 1
  //but we leave it at 0 for compatibility reasons
  unsigned int counter = 0;
  t << counter;
  while (pConfig->HasGroup(t)) {
    pConfig->SetPath(t);
    retval->push_back(LEntry(pConfig->Read(_T("Fehler"),(long)0),
			     pConfig->Read(_T("Richtig"),(long)0),
			     pConfig->Read(_T("Zeit"),(long)0)));
    pConfig->SetPath(_T(".."));
    t.Empty();
    t << ++counter;
  }
  return retval;
}

void TtConfigFile::storeKursDaten (const wxString& name,
				   const wxString& key,
                                   std::vector<LEntry>& entries) {
  std::vector<LEntry>::iterator i = entries.begin();
  const std::vector<LEntry>::iterator k = entries.end();
  //actually, counter should start at 1
  //but we leave it at 0 for compatibility reasons
  unsigned int counter = 0;
  wxString s;
  while (i != k) {
    s.Empty();
    s << _T("/Kurs/") << name << _T("_") << key << _T("/") << counter++;
    pConfig->SetPath(s);
    pConfig->Write(_T("Fehler"), (long) i->getFehler());
    pConfig->Write(_T("Richtig"), (long) i->getRichtig());
    pConfig->Write(_T("Zeit"), (long) i->getZeit());
    i++;
  }
  pConfig->Flush();
}

std::vector<KursName>* TtConfigFile::getKursList () {
  std::vector<KursName>* retval = new std::vector<KursName>();
  pConfig->SetPath(_T("/Kurs"));
  long l=0;
  wxString s;
  while (pConfig->GetNextGroup(s, l)) {
    retval->push_back(KursName(s.BeforeLast('_'), s.AfterLast('_')));
  }
  return retval;
}

void TtConfigFile::storeLessonData (int index, LEntry& entry) {
  wxString s;
  s << _T("/Lektionen/Lektion") << index;
  pConfig->SetPath(s);
  pConfig->Write(_T("Fehler"), (long)entry.getFehler());
  pConfig->Write(_T("Richtig"), (long)entry.getRichtig());
  pConfig->Write(_T("Zeit"), (long)entry.getZeit());
  pConfig->Write(_T("Line"), (long)entry.getLine());
  pConfig->Write(_T("Column"), (long)entry.getColumn());
  pConfig->Flush();
}

LEntry* TtConfigFile::loadLessonData (int index) {
  LEntry* retval;
  wxString s;
  s << _T("/Lektionen/Lektion") << index;
  if (pConfig->Exists(s)) {
    pConfig->SetPath(s);
    retval = new LEntry();
    retval->setFehler(pConfig->Read(_T("Fehler"),(long)0));
    retval->setRichtig(pConfig->Read(_T("Richtig"),(long)0));
    retval->setLine(pConfig->Read(_T("Line"),(long)0));
    retval->setColumn(pConfig->Read(_T("Column"),(long)0));
    //support for the old xx:xx:xx time format
    s.Empty();
    s=pConfig->Read(_T("Zeit"), _T(""));
    if (s.Length()) {
      if (s.Freq(':')==2) {
	//moved format conversion to setZeit function of LEntry class
	retval->setZeit(s);
      } else if (s.Freq(':')==0) {
	retval->setZeit(pConfig->Read(_T("Zeit"),(long)0));
      }
      //default =0 already set in constructor, so omit generic else case
    } 
  } else {
    retval = NULL;
  }
  return retval;
}

bool TtConfigFile::contestsExist() {
  pConfig->SetPath(_T("/Contest"));
  return pConfig->GetNumberOfGroups()!=0;
}

std::vector<ContestDesc>* TtConfigFile::getContests() {
  pConfig->SetPath(_T("/Contest"));
  std::vector<ContestDesc>* retval = new std::vector<ContestDesc>;
  long l=0;
  wxString s;
  long id;
  if (pConfig->GetFirstGroup(s, l)) {
    do {
      pConfig->SetPath(s);
      //there are no default files, yet
      if (pConfig->Exists(_T("title")) &&
	  pConfig->Exists(_T("pictureFile")) &&
	  pConfig->Exists(_T("descrFile")) &&
	  pConfig->Exists(_T("lessonFile")) &&
	  pConfig->Exists(_T("dataFile"))) {
	s.ToLong(&id);
	retval->push_back(getContest(id));
      }
      pConfig->SetPath(_T(".."));
    } while (pConfig->GetNextGroup(s, l));
  }
  return retval;
}

ContestDesc TtConfigFile::getContest (long id) {
  ContestDesc temp;
  if (id >= 0) {
    pConfig->SetPath(_T("/Contest"));
    wxString subPath;
    subPath << id;
    pConfig->SetPath(subPath);
    temp.id = id;
    temp.title = pConfig->Read(_T("title"),temp.title);;

    temp.pictureFile = makeURL(pConfig->Read(_T("pictureFile"),temp.pictureFile.getURLstr()));
    temp.descrFile = makeURL(pConfig->Read(_T("descrFile"),temp.descrFile.getURLstr()));
    temp.descrFile.setCharset(pConfig->Read(_T("descrCharset"),_T("")));
    temp.lessonFile = makeURL(pConfig->Read(_T("lessonFile"),temp.lessonFile.getURLstr()));
    temp.lessonFile.setCharset(pConfig->Read(_T("lessonCharset"),_T("")));

    temp.dataFile = pConfig->Read(_T("dataFile"),temp.dataFile.GetFullPath());
    temp.weight = pConfig->Read(_T("weight"),(long)temp.weight);
  }
  
  return temp;
}

wxString TtConfigFile::makeURL (const wxString& input) {
  const wxString types[3] = {
    _T("file:"),
    _T("ftp:"),
    _T("http:")
  };

  for(unsigned int i=0; i<3; ++i) {
    if (input.compare(0,types[i].Len(),types[i]) == 0) {
      return input;
    }
  }
  return types[0]+input;
}

void TtConfigFile::setContest (ContestDesc& data) {
  pConfig->SetPath(_T("/Contest"));
  wxString subPath;
  //get the next possible id
  if (data.id < 0) {
    long dummy;
    if (pConfig->GetFirstGroup(subPath,dummy)) {
      do {
	subPath.ToLong(&data.id);
      } while (pConfig->GetNextGroup(subPath,dummy));
      do {
	++data.id;
	subPath.Empty();
	subPath << data.id;
      } while (pConfig->Exists(subPath)); //if someone messed up the config file
      subPath.Empty();
    } else {
      data.id=0;
    }
  }
  subPath << data.id;
  pConfig->SetPath(subPath);
  pConfig->Write(_T("title"),data.title);
  if (data.pictureFile.getURLstr().Len()) {
    pConfig->Write(_T("pictureFile"),data.pictureFile.getURLstr());
  } else {
    pConfig->DeleteEntry(_T("pictureFile"));
  }
  if (data.descrFile.getURLstr().Len()) {
    pConfig->Write(_T("descrFile"),data.descrFile.getURLstr());
    if (data.descrFile.getCharset().Len()) {
      pConfig->Write(_T("descrCharset"),data.descrFile.getCharset());
    } else {
      pConfig->DeleteEntry(_T("descrCharset"));
    }
  } else {
    pConfig->DeleteEntry(_T("descrFile"));
  }
  if (data.lessonFile.getURLstr().Len()) {
    pConfig->Write(_T("lessonFile"),data.lessonFile.getURLstr());
    if (data.lessonFile.getCharset().Len()) {
      pConfig->Write(_T("lessonCharset"),data.lessonFile.getCharset());
    } else {
      pConfig->DeleteEntry(_T("lessonCharset"));
    }
  } else {
    pConfig->DeleteEntry(_T("lessonFile"));
  }
  pConfig->Write(_T("dataFile"),data.dataFile.GetFullPath());
  pConfig->Write(_T("weight"),(long)data.weight);
  pConfig->Flush();
}

void TtConfigFile::deleteContest (long id) {
  if (id >= 0) {
    pConfig->SetPath(_T("/Contest"));
    wxString subPath;
    subPath << id;
    if (pConfig->Exists(subPath)) {
      pConfig->DeleteGroup(subPath);
    }
    if (getSelectedContestId()==id &&
	pConfig->Exists(_T("selected"))) {
      pConfig->DeleteEntry(_T("selected"));
    }
    pConfig->Flush();
  }
}

void TtConfigFile::setSelectedContestId (long id) {
  if (id >= 0) {
    wxString s;
    s << id;
    pConfig->SetPath(_T("/Contest"));
    if (pConfig->Exists(s)) {
      pConfig->Write(_T("selected"),id);
    }
  }
  pConfig->Flush();
}

long TtConfigFile::getSelectedContestId () {
  pConfig->SetPath(_T("/Contest"));
  long id = pConfig->Read(_T("selected"),(long)-1);
  wxString s;
  s << id;
  if (id < 0 || !pConfig->Exists(s)) {
    long dummy;
    pConfig->GetFirstGroup(s,dummy);
    s.ToLong(&id);
    setSelectedContestId(id);
  }
  pConfig->Flush();
  return id;
}


bool TtConfigFile::issetDataBaseUrl () {
  return pConfig->Exists(_T("/Data/baseUrl"));
}

void TtConfigFile::setDataBaseUrl (const wxString& url) {
  const wxString entry = _T("/Data/baseUrl");
  if (url.size()) {
    pConfig->Write(entry,url);
  } else {
    if (pConfig->Exists(entry)) {
      pConfig->DeleteEntry(entry);
    }
  }
  pConfig->Flush();
}

#include <wx/filename.h>

wxString TtConfigFile::getDataBaseUrl () {
  wxFileName l(wxString::FromAscii(LECTDIR),_T(""));
  l.Normalize();
  return pConfig->Read(_T("/Data/baseUrl"),UrlFile::FileNameToUrl(l));
}

unsigned int TtConfigFile::getToolTipDelay () {
  return (unsigned int) pConfig->Read(_T("/Settings/toolTipDelay"),(long)2);
}

void TtConfigFile::setToolTipDealy (unsigned int seconds) {
  pConfig->Write(_T("/Settings/toolTipDelay"),(long)seconds);
}
