///###////////////////////////////////////////////////////////////////////////
//
// Burton Computer Corporation
// http://www.burton-computer.com
// $Id: MimeHeader.cc,v 1.9 2003/08/30 21:42:24 bburton Exp $
//
// Copyright (C) 2000 Burton Computer Corporation
// ALL RIGHTS RESERVED
//
// This program is open source software; you can redistribute it
// and/or modify it under the terms of the Q Public License (QPL)
// version 1.0. Use of this software in whole or in part, including
// linking it (modified or unmodified) into other programs is
// subject to the terms of the QPL.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// Q Public License for more details.
//
// You should have received a copy of the Q Public License
// along with this program; see the file LICENSE.txt.  If not, visit
// the Burton Computer Corporation or CoolDevTools web site
// QPL pages at:
//
//    http://www.burton-computer.com/qpl.html
//

#include "RegularExpression.h"
#include "MimeHeader.h"

typedef vector<string>::size_type vsindex_t;

static const char QUOTE_CHAR('"');
static const string QUOTE("\"");
static const string COLON(": ");
static const string CONTENT_TYPE("content-type");
static const string ENCODING("content-transfer-encoding");
static const string BOUNDARY("boundary=");
static const string CHARSET("charset=");
static const string BOUNDARY_PREFIX("--");

MimeHeader::MimeHeader()
{
}

MimeHeader::~MimeHeader()
{
}

static bool is_boundary_char(char ch)
{
  if (is_cntrl(ch) || is_space(ch)) {
    return false;
  }
  switch (ch) {
  case '(':
  case ')':
  case '<':
  case '>':
  case '@':
  case ',':
  case ';':
  case ':':
  case '\\':
  case '"':
  case '/':
  case '[':
  case ']':
  case '?':
  case '=':
    return false;

  default:
    return true;
  }
}

bool MimeHeader::read(MimeLineReader &in,
                      bool skip_current,
                      bool allow_from)
{
  m_lines.clear();
  m_names.clear();
  m_isMime = false;
  m_isText = false;
  m_isQuoted = false;
  m_isBase64 = false;
  m_isMultiPart = false;
  m_isMessage = false;
  m_contentType.erase();
  m_boundary.erase();
  m_encoding.erase();
  m_charset.erase();

  if (skip_current && !in.readLine()) {
    return false;
  }

  while (in.getLine().length() == 0) {
    if (!in.readLine()) {
      return false;
    }
  }

  if (!allow_from && isFromLine(true, in.getLine())) {
    return false;
  }

  string last_line(in.getLine());
  while (in.readLine()) {
    string line = in.getLine();

    if (line.length() == 0) {
      break;
    }

    if (is_space(line[0])) {
      last_line += line;
    } else {
      addField(last_line);
      last_line = line;
    }
  }

  if (last_line.length() > 0) {
    addField(last_line);
  }

  assert(m_lines.size() == m_names.size());

  for (vsindex_t i = 0; i < m_lines.size(); ++i) {
    string lower = to_lower(m_lines[i]);
    if (m_names[i] == ENCODING) {
      m_isMime = true;
      m_encoding = trim_copy(lower, ENCODING.length() + 1);
      m_isQuoted = starts_with(m_encoding, "quoted-printable");
      m_isBase64 = starts_with(m_encoding, "base64");
    } else if (m_names[i] == CONTENT_TYPE) {
      m_isMime = true;
      m_contentType = trim_copy(lower, CONTENT_TYPE.length() + 1);
      string boundary = findParam(m_lines[i], lower, BOUNDARY, EMPTY_STRING);
      if (boundary.length() > 0) {
        m_boundary = BOUNDARY_PREFIX + boundary;
        m_terminator = BOUNDARY_PREFIX + boundary + BOUNDARY_PREFIX;
        if (is_debug) {
          cerr << "** BOUNDARY: " << m_boundary << endl;
        }
      }
      m_charset = findParam(m_lines[i], lower, CHARSET, EMPTY_STRING);
      m_isText = m_contentType.length() == 0 ||
                 starts_with(m_contentType, "text/") ||
                 starts_with(m_contentType, "message/rfc822");
      m_isMultiPart = starts_with(m_contentType, "multipart/");
      m_isMessage = starts_with(m_contentType, "message/rfc822");
    }
  }

  return m_lines.size() > 0;
}

string MimeHeader::findParam(const string &line,
                             const string &lower,
                             const string &param_name,
                             const string &default_value)
{
  assert(line.length() == lower.length());

  string::size_type start_offset = lower.find(param_name);
  if (start_offset == string::npos) {
    return default_value;
  }

  start_offset += param_name.length();
  while (is_space(lower[start_offset])) {
    ++start_offset;
    if (start_offset >= lower.length()) {
      return default_value;
    }
  }

  if (lower[start_offset] == QUOTE_CHAR) {
    ++start_offset;
    string::size_type quote_offset = lower.find(QUOTE, start_offset);
    if (quote_offset == string::npos) {
      return string(line, start_offset);
    } else {
      return string(line, start_offset, quote_offset - start_offset);
    }
  }

  string answer;
  char ch = line[start_offset];
  while (is_boundary_char(ch)) {
    answer += ch;
    ++start_offset;
    if (start_offset >= line.length()) {
      break;
    }
    ch = line[start_offset];
  }
  return answer;
}

void MimeHeader::addField(const string &line)
{
  if (is_debug) {
    cerr << "** HEAD " << line << endl;
  }
  m_lines.push_back(line);

  string::size_type colon_pos = line.find(": ");
  if (colon_pos != string::npos) {
    m_names.push_back(to_lower(string(line, 0, colon_pos)));
  } else {
    m_names.push_back("");
  }
}

bool MimeHeader::isMime()
{
  return m_isMime;
}

bool MimeHeader::getField(int index,
                          string &value) const
{
  assert(index >= 0 && index < (int)m_lines.size());

  value = m_lines[index];
  return true;
}

bool MimeHeader::getFieldName(int index,
                              string &name) const
{
  assert(index >= 0 && index < (int)m_lines.size());

  name = m_names[index];
  return true;
}

bool MimeHeader::getField(const string &name,
                          string &value) const
{
  assert(m_names.size() == m_lines.size());

  string lower = to_lower(name);
  for (vsindex_t i = 0; i < m_names.size(); ++i) {
    if (m_names[i] == lower) {
      value = trim_copy(m_lines[i], name.length() + 1);
      return true;
    }
  }
  return false;
}

bool MimeHeader::getField(const string &name,
                          vector<string> &values) const
{
  assert(m_names.size() == m_lines.size());

  values.clear();
  string lower = to_lower(name);
  for (vsindex_t i = 0; i < m_names.size(); ++i) {
    if (m_names[i] == lower) {
      values.push_back(trim_copy(m_lines[i], name.length() + 1));
    }
  }
  return values.size() > 0;
}

bool MimeHeader::isFromLine(bool prev_blank,
                            const string &line)
{
  static RegularExpression from_expr("^From[ \t]+[^ \t]+[ \t]+[a-zA-Z][a-zA-Z][a-zA-Z][ \t]+[a-zA-Z][a-zA-Z][a-zA-Z][ \t]+[0-9][0-9]?[ \t]+[0-9][0-9][ \t]*:[ \t]*[0-9][0-9][ \t]*:[ \t]*[0-9][0-9][ \t]+[0-9][0-9][0-9][0-9]");

  bool is_from = (prev_blank && from_expr.match(line));

  if (is_debug) {
    if (prev_blank && !is_from && starts_with(line, "From ")) {
      cerr << "NOT A FROM LINE: " << line << endl;
    } else if (is_from) {
      cerr << "FROM: " << line << endl;
      from_expr.dumpMatches(cerr);
    }
  }

  return is_from;
}
