/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#ifdef ENABLE_OGG

#include "../../player.h"
#include "../../other.h"
#include "../../audio.h"
#include "../../util.h"
#include "../../output.h"
#include "fconfig.h"
#include "vorbis.h"
#include "common.h"

audio_data_vorbis *
vorbis_audio_device (audio * output)
{

  audio_data_vorbis *vorbis;
  int ch = 0;
  int len;
  struct stat st;

  if (!(vorbis = (audio_data_vorbis *) malloc (sizeof (audio_data_vorbis))))
    fatal (_("Error: memory."));

  if (!lstat (output->audio_dev, &st))
    {
      if (vorbis_read_config (output->audio_dev, vorbis))
	fatal (_("Vorbis init error."));

    }
  else
    {

      vorbis->rate = 44100;
      vorbis->quality = 8;
      vorbis->bitrate = 96;
      vorbis->channels = OUT_STEREO;
      vorbis->lowpass = 0;
      vorbis->highpass = 0;
      vorbis->file = NULL;

      vorbis->fd = NULL;
      vorbis->vorbis = NULL;

      len = strlen (output->audio_dev);

      while (ch < len)
	{

	  switch (*(output->audio_dev + ch))
	    {

	    case 'b':
	      if (!strncmp (output->audio_dev + ch, "bitrate=", 6))
		{
		  ch += 8;
		  vorbis->bitrate =
		    audio_parse_int (output->audio_dev, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (vorbis);
		  return NULL;
		}

	      break;

	    case 'c':
	      if (!strncmp (output->audio_dev + ch, "channels=", 9))
		{
		  int a;

		  ch += 9;
		  a = audio_parse_int (output->audio_dev, &ch, len);

		  switch (a)
		    {
		    case 1:
		      vorbis->channels = OUT_MONO;
		      break;
		    case 2:
		      vorbis->channels = OUT_STEREO;
		      break;
		    default:
		      msg_error (_("Error in audiodevice parameter mode: %s"),
				 output->audio_dev);
		      free (vorbis);
		      return NULL;
		    }
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (vorbis);
		  return NULL;
		}

	      break;

	    case 'f':

	      if (!strncmp (output->audio_dev + ch, "file=", 5))
		{
		  ch += 5;
		  vorbis->file =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (vorbis);
		  return NULL;
		}

	      break;

	    case 'h':

	      if (!strncmp (output->audio_dev + ch, "highpass=", 9))
		{
		  ch += 9;
		  vorbis->highpass =
		    audio_parse_int (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (vorbis);
		  return NULL;
		}

	      break;

	    case 'l':

	      if (!strncmp (output->audio_dev + ch, "lowpass=", 8))
		{
		  ch += 8;
		  vorbis->lowpass =
		    audio_parse_int (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (vorbis);
		  return NULL;
		}

	      break;

	    case 'q':
	      if (!strncmp (output->audio_dev + ch, "quality=", 5))
		{
		  ch += 8;
		  vorbis->quality =
		    audio_parse_int (output->audio_dev, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (vorbis);
		  return NULL;
		}

	      break;

	    case 'r':

	      if (!strncmp (output->audio_dev + ch, "rate=", 5))
		{
		  ch += 5;
		  vorbis->rate =
		    audio_parse_int (output->audio_dev, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (vorbis);
		  return NULL;
		}

	      break;

	    default:
	      msg_error (_("Error in audiodevice parameter: %s"),
			 output->audio_dev);
	      free (vorbis);
	      return NULL;
	    }

	  ch++;
	}
    }

  if (!vorbis->file)
    {
      msg_error (_("No file in vorbis config."));
      free (vorbis);
      return NULL;
    }

  return vorbis;
}

int
vorbis_start (audio * output, long rate, int channels, int bitrate)
{
  audio_data_vorbis *data = (audio_data_vorbis *) output->data;
  char *fl;

  if (!data)
    fatal (_("Internal error."));

  fl = output_check_file (data->file);

  if (!(data->fd = fopen (fl, "wb")))
    {
      msg_error (_("Open error: %s."), fl);
      free (fl);

      return 1;
    }

  free (fl);

  if (!
      (data->vorbis =
       vorbis_config (data->rate, channels, data->channels, data->quality,
		      data->bitrate, data->lowpass, "", output,
		      vorbis_write_file)))
    {
      return 1;
    }

  return 0;
}

int
vorbis_pre_write (audio * output, int channels, int bitrate, void *what, size_t size)
{

  audio_data_vorbis *data = (audio_data_vorbis *) output->data;

  if (!data)
    fatal (_("Internal error."));

  return vorbis_write (data->vorbis, channels, bitrate, what, size, output,
		       vorbis_write_file);
}

int
vorbis_write_file (audio * output, void *a, size_t s)
{
  audio_data_vorbis *data = (audio_data_vorbis *) output->data;

  if (!data)
    fatal (_("Internal error."));

  return fwrite (a, 1, s, data->fd);
}

void
vorbis_pre_close (audio * output)
{
  audio_data_vorbis *data = (audio_data_vorbis *) output->data;

  if (!data)
    return;

  if (data->vorbis)
    {
      vorbis_quit (data->vorbis);
      if (data->fd)
	fclose (data->fd);
    }
}

void
vorbis_show (audio * output)
{
  audio_data_vorbis *data = (audio_data_vorbis *) output->data;

  if (!data)
    fatal (_("Internal error."));

  msg (_("Vorbis %d\n" "\tRate %d\n" "\tBitrate %d\n" "\tchannels %d\n" "\tQuality %d\n" "\tFile %s\n"), data->vorbis->vi.version, data->rate, data->bitrate, data->channels, data->quality, data->file);

}
#endif
