/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_backend.cpp,v 1.25.2.2 2004/12/30 04:51:55 suzhe Exp $
 *
 */

#define Uses_SCIM_BACKEND
#define Uses_SCIM_IMENGINE
#define Uses_SCIM_IMENGINE_MODULE
#define Uses_SCIM_CONFIG_PATH
#define Uses_STL_ALGORITHM
#define Uses_STL_MAP
#include "scim_private.h"
#include "scim.h"

namespace scim {

typedef Signal1<void, int>
        BackEndSignalVoid;

typedef Signal2<void, int, int>
        BackEndSignalInt;

typedef Signal2<void, int, bool>
        BackEndSignalBool;

typedef Signal2<void, int, const WideString&>
        BackEndSignalWideString;

typedef Signal2<void, int, const KeyEvent&>
        BackEndSignalKeyEvent;

typedef Signal2<void, int, const Property&>
        BackEndSignalProperty;

typedef Signal2<void, int, const PropertyList&>
        BackEndSignalPropertyList;

typedef Signal2<void, int, const LookupTable&>
        BackEndSignalLookupTable;

typedef Signal3<void, int, const WideString&,const AttributeList&>
        BackEndSignalWideStringAttributeList;

typedef std::map <int, IMEngineInstancePointer> IMEngineInstanceRepository;
typedef std::map <String, IMEngineFactoryPointer> IMEngineFactoryRepository;

class LocaleEqual
{
    String m_lhs;
public:
    LocaleEqual (const String &lhs) : m_lhs (lhs) { }

    bool operator () (const String &rhs) const {
        if (m_lhs == rhs) return true;
        if (scim_get_locale_language (m_lhs) == scim_get_locale_language (rhs) &&
            scim_get_locale_encoding (m_lhs) == scim_get_locale_encoding (rhs) &&
            m_lhs.find ('.') != String::npos && rhs.find ('.') != String::npos)
            return true;
        return false;
    }
};

class BackEndBase::BackEndBaseImpl
{
public:
    IMEngineInstanceRepository    m_instance_repository;
    IMEngineFactoryRepository     m_factory_repository;

    BackEndSignalVoid             m_signal_show_preedit_string;
    BackEndSignalVoid             m_signal_show_aux_string;
    BackEndSignalVoid             m_signal_show_lookup_table;

    BackEndSignalVoid             m_signal_hide_preedit_string;
    BackEndSignalVoid             m_signal_hide_aux_string;
    BackEndSignalVoid             m_signal_hide_lookup_table;

    BackEndSignalInt              m_signal_update_preedit_caret;
    BackEndSignalWideStringAttributeList m_signal_update_preedit_string;
    BackEndSignalWideStringAttributeList m_signal_update_aux_string;
    BackEndSignalWideString       m_signal_commit_string;
    BackEndSignalLookupTable      m_signal_update_lookup_table;

    BackEndSignalKeyEvent         m_signal_forward_key_event;

    BackEndSignalPropertyList     m_signal_register_properties;
    BackEndSignalProperty         m_signal_update_property;

    String                        m_supported_unicode_locales;

    int                           m_instance_id_count;
public:

    BackEndBaseImpl () : m_instance_id_count (0) { }

    void slot_show_preedit_string   (IMEngineInstanceBase * si) {
        if (si) m_signal_show_preedit_string (si->get_id ());
    }

    void slot_show_aux_string       (IMEngineInstanceBase * si) {
        if (si) m_signal_show_aux_string (si->get_id ());
    }

    void slot_show_lookup_table     (IMEngineInstanceBase * si) {
        if (si) m_signal_show_lookup_table (si->get_id ());
    }

    void slot_hide_preedit_string   (IMEngineInstanceBase * si) {
        if (si) m_signal_hide_preedit_string (si->get_id ());
    }

    void slot_hide_aux_string       (IMEngineInstanceBase * si) {
        if (si) m_signal_hide_aux_string (si->get_id ());
    }

    void slot_hide_lookup_table     (IMEngineInstanceBase * si) {
        if (si) m_signal_hide_lookup_table (si->get_id ());
    }

    void slot_update_preedit_caret  (IMEngineInstanceBase * si, int caret) {
        if (si) m_signal_update_preedit_caret (si->get_id (), caret);
    }

    void slot_update_preedit_string (IMEngineInstanceBase * si, const WideString & str, const AttributeList & attrs) {
        if (si) m_signal_update_preedit_string (si->get_id (), str, attrs);
    }

    void slot_update_aux_string     (IMEngineInstanceBase * si, const WideString & str, const AttributeList & attrs) {
        if (si) m_signal_update_aux_string (si->get_id (), str, attrs);
    }

    void slot_update_lookup_table   (IMEngineInstanceBase * si, const LookupTable & table) {
        if (si) m_signal_update_lookup_table (si->get_id (), table);
    }

    void slot_commit_string         (IMEngineInstanceBase * si, const WideString & str) {
        if (si) m_signal_commit_string (si->get_id (), str);
    }

    void slot_forward_key_event     (IMEngineInstanceBase * si, const KeyEvent & key) {
        if (si) m_signal_forward_key_event (si->get_id (), key);
    }

    void slot_register_properties   (IMEngineInstanceBase * si, const PropertyList & properties) {
        if (si) m_signal_register_properties (si->get_id (), properties);
    }

    void slot_update_property       (IMEngineInstanceBase * si, const Property & property) {
        if (si) m_signal_update_property (si->get_id (), property);
    }

    IMEngineFactoryPointer  find_factory (const String &uuid) const {
        IMEngineFactoryRepository::const_iterator it = m_factory_repository.find (uuid);

        if (it != m_factory_repository.end ())
            return it->second;

        return IMEngineFactoryPointer (0);
    }

    IMEngineInstancePointer find_instance (int id) const {
        IMEngineInstanceRepository::const_iterator it = m_instance_repository.find (id);

        if (it != m_instance_repository.end ())
            return it->second;

        return IMEngineInstancePointer (0);
    }

    void add_instance (const IMEngineInstancePointer &si) {
        if (si.null () || si->get_id () < 0) return;
 
        m_instance_repository [si->get_id ()] = si;
    }

    void attach_instance (const IMEngineInstancePointer &si) {
        if (si.null () || si->get_id () < 0) return;
 
        si->signal_connect_show_preedit_string (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_show_preedit_string));
        si->signal_connect_show_aux_string (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_show_aux_string));
        si->signal_connect_show_lookup_table (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_show_lookup_table));
 
        si->signal_connect_hide_preedit_string (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_hide_preedit_string));
        si->signal_connect_hide_aux_string (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_hide_aux_string));
        si->signal_connect_hide_lookup_table (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_hide_lookup_table));
 
        si->signal_connect_update_preedit_caret (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_update_preedit_caret));
        si->signal_connect_update_preedit_string (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_update_preedit_string));
        si->signal_connect_update_aux_string (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_update_aux_string));
        si->signal_connect_update_lookup_table (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_update_lookup_table));
 
        si->signal_connect_commit_string (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_commit_string));
 
        si->signal_connect_forward_key_event (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_forward_key_event));
 
        si->signal_connect_register_properties (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_register_properties));
 
        si->signal_connect_update_property (
            slot (this, &BackEndBase::BackEndBaseImpl::slot_update_property));
    }
};

BackEndBase::BackEndBase ()
    : m_impl (new BackEndBase::BackEndBaseImpl)
{
}

BackEndBase::~BackEndBase ()
{
    delete m_impl;
}

String
BackEndBase::get_all_locales () const
{
    String locale;

    std::vector <String> locale_list;
    std::vector <String> real_list;

    IMEngineFactoryRepository::const_iterator it; 

    for (it = m_impl->m_factory_repository.begin (); it != m_impl->m_factory_repository.end (); ++it) {
        if (locale.length () == 0)
            locale += it->second->get_locales ();
        else
            locale += (String (",") + it->second->get_locales ());
    }

    if (m_impl->m_supported_unicode_locales.length ())
        locale += (String (",") + m_impl->m_supported_unicode_locales);

    scim_split_string_list (locale_list, locale);

    for (std::vector <String>::iterator i = locale_list.begin (); i!= locale_list.end (); i++) {
        locale = scim_validate_locale (*i);
        if (locale.length () &&
            std::find_if (real_list.begin (), real_list.end (), LocaleEqual (locale)) == real_list.end ())
            real_list.push_back (locale);
    }

    return scim_combine_string_list (real_list);
}

uint32
BackEndBase::number_of_factories () const
{
    return m_impl->m_factory_repository.size ();
}

IMEngineFactoryPointer
BackEndBase::get_factory_pointer (uint32 idx) const
{
    if (idx < number_of_factories ()) {
        IMEngineFactoryRepository::const_iterator it = m_impl->m_factory_repository.begin ();

        for (size_t i = 0; i < idx; ++i) ++it;

        if (it != m_impl->m_factory_repository.end ())
            return it->second;
    }

    return IMEngineFactoryPointer (0);
}

Connection 
BackEndBase::signal_connect_show_preedit_string   (BackEndSlotVoid *slot)
{
    return m_impl->m_signal_show_preedit_string.connect (slot);
}

Connection 
BackEndBase::signal_connect_show_aux_string       (BackEndSlotVoid *slot)
{
    return m_impl->m_signal_show_aux_string.connect (slot);
}

Connection 
BackEndBase::signal_connect_show_lookup_table     (BackEndSlotVoid *slot)
{
    return m_impl->m_signal_show_lookup_table.connect (slot);
}

Connection 
BackEndBase::signal_connect_hide_preedit_string   (BackEndSlotVoid *slot)
{
    return m_impl->m_signal_hide_preedit_string.connect (slot);
}

Connection 
BackEndBase::signal_connect_hide_aux_string       (BackEndSlotVoid *slot)
{
    return m_impl->m_signal_hide_aux_string.connect (slot);
}

Connection 
BackEndBase::signal_connect_hide_lookup_table     (BackEndSlotVoid *slot)
{
    return m_impl->m_signal_hide_lookup_table.connect (slot);
}

Connection 
BackEndBase::signal_connect_update_preedit_caret  (BackEndSlotInt *slot)
{
    return m_impl->m_signal_update_preedit_caret.connect (slot);
}

Connection 
BackEndBase::signal_connect_update_preedit_string (BackEndSlotWideStringAttributeList *slot)
{
    return m_impl->m_signal_update_preedit_string.connect (slot);
}

Connection 
BackEndBase::signal_connect_update_aux_string     (BackEndSlotWideStringAttributeList *slot)
{
    return m_impl->m_signal_update_aux_string.connect (slot);
}

Connection 
BackEndBase::signal_connect_update_lookup_table   (BackEndSlotLookupTable *slot)
{
    return m_impl->m_signal_update_lookup_table.connect (slot);
}

Connection 
BackEndBase::signal_connect_commit_string         (BackEndSlotWideString *slot)
{
    return m_impl->m_signal_commit_string.connect (slot);
}

Connection 
BackEndBase::signal_connect_forward_key_event     (BackEndSlotKeyEvent *slot)
{
    return m_impl->m_signal_forward_key_event.connect (slot);
}

Connection 
BackEndBase::signal_connect_register_properties   (BackEndSlotPropertyList *slot)
{
    return m_impl->m_signal_register_properties.connect (slot);
}

Connection 
BackEndBase::signal_connect_update_property       (BackEndSlotProperty *slot)
{
    return m_impl->m_signal_update_property.connect (slot);
}

uint32
BackEndBase::get_factory_list (std::vector<String>& uuids, const String &encoding) const
{
    IMEngineFactoryRepository::const_iterator it;

    uuids.clear ();

    for (it = m_impl->m_factory_repository.begin (); it != m_impl->m_factory_repository.end (); ++it) {
        if ((encoding.length () == 0 || it->second->validate_encoding (encoding)))
            uuids.push_back (it->second->get_uuid ());
    }

    return uuids.size ();
}

WideString
BackEndBase::get_factory_name (const String &uuid) const
{
    IMEngineFactoryPointer sf = m_impl->find_factory (uuid);
    if (!sf.null ())
        return sf->get_name ();

    return WideString ();
}

WideString
BackEndBase::get_factory_authors (const String &uuid) const
{
    IMEngineFactoryPointer sf = m_impl->find_factory (uuid);
    if (!sf.null ())
        return sf->get_authors ();

    return WideString ();
}

WideString
BackEndBase::get_factory_credits (const String &uuid) const
{
    IMEngineFactoryPointer sf = m_impl->find_factory (uuid);
    if (!sf.null ())
        return sf->get_credits ();

    return WideString ();
}

WideString
BackEndBase::get_factory_help (const String &uuid) const
{
    IMEngineFactoryPointer sf = m_impl->find_factory (uuid);
    if (!sf.null ())
        return sf->get_help ();

    return WideString ();
}

String
BackEndBase::get_factory_locales (const String &uuid) const
{
    IMEngineFactoryPointer sf = m_impl->find_factory (uuid);
    if (!sf.null ())
        return sf->get_locales ();

    return String ();
}

String
BackEndBase::get_factory_language (const String &uuid) const
{
    IMEngineFactoryPointer sf = m_impl->find_factory (uuid);
    if (!sf.null ())
        return sf->get_language ();

    return String ("~other");
}

String
BackEndBase::get_factory_icon_file (const String &uuid) const
{
    IMEngineFactoryPointer sf = m_impl->find_factory (uuid);
    if (!sf.null ())
        return sf->get_icon_file ();

    return String ();
}

int
BackEndBase::new_instance (const String &sf_uuid, const String& encoding)
{
    IMEngineFactoryPointer sf = m_impl->find_factory (sf_uuid);

    if (sf.null () || !sf->validate_encoding (encoding)) {
        SCIM_DEBUG_BACKEND (1) << "IMEngineFactory " << sf_uuid
            << " does not support encoding " << encoding << "\n";
        return -1;
    }

    IMEngineInstancePointer si =
        sf->create_instance (encoding, m_impl->m_instance_id_count);

    if (si.null ()) {
        SCIM_DEBUG_BACKEND(1) << "IMEngineFactory " << sf_uuid
            << " failed to create new instance!\n";
        return -1;
    }

    ++ m_impl->m_instance_id_count;

    if (m_impl->m_instance_id_count < 0)
        m_impl->m_instance_id_count = 0;

    m_impl->add_instance (si);
    m_impl->attach_instance (si);

    return si->get_id ();
}

bool
BackEndBase::replace_instance (int si_id, const String &sf_uuid)
{
    IMEngineFactoryPointer sf = m_impl->find_factory (sf_uuid);

    if (sf.null ()) return false;

    IMEngineInstanceRepository::iterator it = m_impl->m_instance_repository.find (si_id);

    if (it != m_impl->m_instance_repository.end ()) {
        if (it->second->get_factory_uuid () == sf_uuid)
            return true;

        String encoding = it->second->get_encoding ();
        if (sf->validate_encoding (encoding)) {
            IMEngineInstancePointer si = sf->create_instance (encoding, si_id);
            if (!si.null ()) {
                it->second = si;
                m_impl->attach_instance (it->second);
                return true;
            }
        }
    }

    SCIM_DEBUG_BACKEND(1) << "Cannot find IMEngine Instance " << si_id << " to replace.\n";

    return false;
}

bool
BackEndBase::delete_instance (int id)
{
    IMEngineInstanceRepository::iterator it = m_impl->m_instance_repository.find (id);

    if (it != m_impl->m_instance_repository.end ()) {
        m_impl->m_instance_repository.erase (it);
        return true;
    }

    return false;
}

void
BackEndBase::delete_all_instances ()
{
    m_impl->m_instance_repository.clear ();
}

String
BackEndBase::get_instance_uuid (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) return si->get_factory_uuid ();

    return String ();
}

String
BackEndBase::get_instance_encoding (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) return si->get_encoding ();

    return String ();
}

WideString
BackEndBase::get_instance_name (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) return get_factory_name (si->get_factory_uuid ());

    return WideString ();
}

WideString
BackEndBase::get_instance_authors (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) return get_factory_authors (si->get_factory_uuid ());

    return WideString ();
}

WideString
BackEndBase::get_instance_credits (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) return get_factory_credits (si->get_factory_uuid ());

    return WideString ();
}

WideString
BackEndBase::get_instance_help (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) return get_factory_help (si->get_factory_uuid ());

    return WideString ();
}

String
BackEndBase::get_instance_icon_file (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) return get_factory_icon_file (si->get_factory_uuid ());

    return String ();
}

void
BackEndBase::focus_in (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) si->focus_in ();
}

void
BackEndBase::focus_out (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) si->focus_out ();
}

void
BackEndBase::reset (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) si->reset ();
}

bool
BackEndBase::process_key_event (int id, const KeyEvent& key) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) return si->process_key_event (key);

    return false;
}

void
BackEndBase::move_preedit_caret (int id, unsigned int pos) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) si->move_preedit_caret (pos);
}

void
BackEndBase::select_candidate (int id, unsigned int index) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) si->select_candidate (index);
}

void
BackEndBase::update_lookup_table_page_size (int id, unsigned int page_size) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) si->update_lookup_table_page_size (page_size);
}

void
BackEndBase::lookup_table_page_up (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) si->lookup_table_page_up ();
}

void
BackEndBase::lookup_table_page_down (int id) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) si->lookup_table_page_down ();
}

void
BackEndBase::trigger_property (int id, const String & property) const
{
    IMEngineInstancePointer si = m_impl->find_instance (id);

    if (!si.null ()) si->trigger_property (property);
}

bool
BackEndBase::add_factory (const IMEngineFactoryPointer &factory)
{
    if (!factory.null ()) {
        String uuid = factory->get_uuid ();

        if (uuid.length () && m_impl->m_factory_repository.find (uuid) == m_impl->m_factory_repository.end ()) {
            m_impl->m_factory_repository [uuid] = factory;
            return true;
        }
    }

    return false;
}

void
BackEndBase::set_supported_unicode_locales (const String &locales)
{
    std::vector <String> locale_list;
    std::vector <String> real_list;

    scim_split_string_list (locale_list, locales);

    for (std::vector <String>::iterator i = locale_list.begin (); i!= locale_list.end (); ++i) {
        *i = scim_validate_locale (*i);
        if (i->length () && scim_get_locale_encoding (*i) == "UTF-8" &&
            std::find_if (real_list.begin (), real_list.end (), LocaleEqual (*i)) == real_list.end ())
            real_list.push_back (*i);
    }

    m_impl->m_supported_unicode_locales = scim_combine_string_list (real_list);
}

void
BackEndBase::destroy_all_factories ()
{
    m_impl->m_instance_repository.clear ();
    m_impl->m_factory_repository.clear ();
}


// Implementation of CommonBackEnd.
struct CommonBackEnd::CommonBackEndImpl {
    IMEngineModule      *m_engine_modules;
};

CommonBackEnd::CommonBackEnd (const ConfigPointer       &config,
                              const std::vector<String> &modules)
    : m_impl (new CommonBackEndImpl)
{
    IMEngineFactoryPointer factory;
    std::vector<String>    disabled_factories;
    String                 utf8_locales;

    int all_factories_count = 0;
    int module_factories_count = 0;

    if (config.null ()) return;

    // Get disabled factories list.
    disabled_factories = scim_global_config_read (SCIM_GLOBAL_CONFIG_DISABLED_IMENGINE_FACTORIES, disabled_factories);

    // Set the default supported locales.
    utf8_locales = scim_global_config_read (SCIM_GLOBAL_CONFIG_SUPPORTED_UNICODE_LOCALES, String ("en_US.UTF-8"));

    set_supported_unicode_locales (utf8_locales);

    // Try to load all IMEngine modules
    try {
        m_impl->m_engine_modules = new IMEngineModule [modules.size ()];
    } catch (const std::exception & err) {
        std::cerr << err.what () << "\n";
        return;
    }

    //load IMEngine modules
    for (size_t i = 0; i < modules.size (); ++i) {
        SCIM_DEBUG_BACKEND (1) << "Loading IMEngine module: " << modules [i] << " ...\n";

        module_factories_count = 0;

        if (m_impl->m_engine_modules [i].load (modules [i], config) &&
            m_impl->m_engine_modules [i].valid ()) {
            for (size_t j=0; j < m_impl->m_engine_modules [i].number_of_factories (); ++j) {

                // Try to load a IMEngine Factory.
                try {
                    factory = m_impl->m_engine_modules [i].create_factory (j);
                } catch (const std::exception & err) {
                    std::cerr << err.what () << "\n";
                    factory.reset ();
                }

                if (!factory.null ()) {
                    // Check if it's disabled.
                    if (std::find (disabled_factories.begin (),
                                   disabled_factories.end (),
                                   factory->get_uuid ()) == disabled_factories.end () &&
                        add_factory (factory)) {

                        all_factories_count ++;
                        module_factories_count ++;
                        SCIM_DEBUG_BACKEND (1) << "    Loading IMEngine Factory " << j << " : " << "OK\n";
                    } else {
                        SCIM_DEBUG_BACKEND (1) << "    Loading IMEngine Factory " << j << " : " << "Disabled\n";
                        factory.reset ();
                    }
                } else {
                    SCIM_DEBUG_BACKEND (1) << "    Loading IMEngine Factory " << j << " : " << "Failed\n";
                }
            }
            if (module_factories_count) {
                SCIM_DEBUG_BACKEND (1) << modules [i] << " IMEngine module is successfully loaded.\n";
            } else {
                SCIM_DEBUG_BACKEND (1) << "No Factory loaded from " << modules [i] << " IMEngine module!\n";
                m_impl->m_engine_modules [i].unload ();
            }
        } else {
            SCIM_DEBUG_BACKEND (1) << "Failed to load " << modules [i] << " IMEngine module.\n";
        }
    }

    factory = new ComposeKeyFactory ();

    if (all_factories_count == 0 ||
        std::find (disabled_factories.begin (),
                   disabled_factories.end (),
                   factory->get_uuid ()) == disabled_factories.end ()) {
        add_factory (factory);
    }

    factory.reset ();
}

CommonBackEnd::~CommonBackEnd ()
{
    destroy_all_factories ();

    delete [] m_impl->m_engine_modules;
    delete m_impl;
}

} // namespace scim

/*
vi:ts=4:nowrap:ai:expandtab
*/
