#if !defined( FITSCHAN_INCLUDED ) /* Include this file only once */
#define FITSCHAN_INCLUDED
/*
*+
*  Name:
*     fitschan.h

*  Type:
*     C include file.

*  Purpose:
*     Define the interface to the FitsChan class.

*  Invocation:
*     #include "fitschan.h"

*  Description:
*     This include file defines the interface to the FitsChan class and
*     provides the type definitions, function prototypes and macros,
*     etc.  needed to use this class.
*
*     The FitsChan class provides facilities for reading and writing AST 
*     Objects in the form of FITS header cards.

*  Inheritance:
*     The FitsChan class inherits from the Channel class.

*  Macros:
*     Public:
*        None.
*
*     Protected:
*        AST__NOTYPE
*           Integer dentifier for an illegal FITS data type.
*        AST__COMMENT
*           Integer dentifier for a FITS comment keyword.
*        AST__INT    
*           Integer dentifier for the integer FITS data type.
*        AST__FLOAT  
*           Integer dentifier for the floating point FITS data type.
*        AST__STRING 
*           Integer dentifier for the string FITS data type.
*        AST__CONTINUE 
*           Integer dentifier for the continuation string FITS data type.
*        AST__COMPLEXF
*           Integer dentifier for the complex floating point FITS data type.
*        AST__COMPLEXI
*           Integer dentifier for the complex integer FITS data type.
*        AST__LOGICAL 
*           Integer dentifier for the logical FITS data type.

*  Feature Test Macros:
*     astCLASS
*        If the astCLASS macro is undefined, only public symbols are
*        made available, otherwise protected symbols (for use in other
*        class implementations) are defined. This macro also affects
*        the reporting of error context information, which is only
*        provided for external calls to the AST library.

*  Copyright:
*     Copyright (C) 2004 Central Laboratory of the Research Councils

*  Authors:
*     DSB: D.S. Berry (Starlink)

*  History:
*     11-DEC-1996 (DSB):
*        Original version.
*     30-Jun-1997 (DSB):
*        Changed character pointer argument to character array in PutFits.
*     26-SEP-1997 (DSB):
*        Added CDMatrix attribute.
*     21-OCT-1997 (DSB):
*        o  Renamed astFields as astKeyFields.
*     1-APR-2000 (DSB):
*        Changes for CDi_j based FITS-WCS standard.
*     18-MAY-2000 (DSB):
*        Added Warnings attribute.
*     4-APR-2001 (DSB):
*        Added AllWarnings attribute.
*     20-FEB-2002 (DSB):
*        Added CarLin attribute.
*     8-JAN-2003 (DSB):
*        Added protected astInitFitsChanVtab method.
*     13-FEB-2003 (DSB):
*        Added Clean attribute.
*     19-MAR-2004 (DSB):
*        Added astPutCards function.
*-
*/

/* Include files. */
/* ============== */
/* Interface definitions. */
/* ---------------------- */
#include "channel.h"             /* I/O channels (parent class) */

/* C header files. */
/* --------------- */
#if defined(astCLASS)            /* Protected */
#include <stddef.h>
#endif

/* Macros. */
/* ------- */
#if defined(astCLASS)            /* Protected */
#define AST__NOTYPE       -1
#define AST__COMMENT       0
#define AST__INT           1
#define AST__FLOAT         2  
#define AST__STRING        3
#define AST__COMPLEXF      4
#define AST__COMPLEXI      5
#define AST__LOGICAL       6
#define AST__CONTINUE      7
#endif

/* Type Definitions. */
/* ================= */

/* FitsChan structure. */
/* ------------------ */
/* This structure contains all information that is unique to each object in
   the class (e.g. its instance variables). */
typedef struct AstFitsChan {

/* Attributes inherited from the parent class. */
   AstChannel channel;             /* Parent class structure */

/* Attributes specific to objects in this class. */
   int encoding;    /* System for encoding AST objects ito FITS headers */
   int defb1950;    /* Use FK4 B1950 as defaults? */
   int cdmatrix;    /* Use a CD matrix in FITS-WCS Encoding? */
   int carlin;      /* Use linear CAR mappings? */
   int iwc;         /* Include an IWC Frame? */
   int clean;       /* Remove used cards even if an error occurs? */
   int fitsdigits;  /* No. of decmial places in formatted floating point keyword values */
   char *warnings;  /* Pointer to a string containing warning conditions */
   void *card;      /* Pointer to next FitsCard to be read */
   void *head;      /* Pointer to first FitsCard in the circular linked list */
   void *keyseq;    /* List of keyword sequence numbers used */
   const char *(* source)( void ); /* Pointer to source function */
   char *(* source_wrap)( const char *(*)( void ) );
                                   /* Source wrapper function pointer */
   void (* sink)( const char * );  /* Pointer to sink function */
   void (* sink_wrap)( void (*)( const char * ), const char * );
                                   /* Sink wrapper function pointer */

} AstFitsChan;

/* Virtual function table. */
/* ----------------------- */
/* This table contains all information that is the same for all
   objects in the class (e.g. pointers to its virtual functions). */
#if defined(astCLASS)            /* Protected */
typedef struct AstFitsChanVtab {

/* Properties (e.g. methods) inherited from the parent class. */
   AstChannelVtab channel_vtab;  /* Parent class virtual function table */

/* Unique flag value to determine class membership. */
   int *check;                   /* Check value */

/* Properties (e.g. methods) specific to this class. */
   void (* Empty)( AstFitsChan * );
   void (* DelFits)( AstFitsChan * );
   int (* FindFits)( AstFitsChan *, const char *, char [81], int );
   void (* PutFits)( AstFitsChan *, const char [81], int );
   void (* PutCards)( AstFitsChan *, const char * );
   int (* KeyFields)( AstFitsChan *, const char *, int, int *, int *);
   int (* FitsEof)( AstFitsChan * );
   int (* FitsGetCF)( AstFitsChan *, const char *, double * );
   int (* FitsGetCI)( AstFitsChan *, const char *, int * );
   int (* FitsGetF)( AstFitsChan *, const char *, double * );
   int (* FitsGetI)( AstFitsChan *, const char *, int * );
   int (* FitsGetL)( AstFitsChan *, const char *, int * );
   int (* FitsGetS)( AstFitsChan *, const char *, char ** );
   int (* FitsGetCN)( AstFitsChan *, const char *, char ** );
   int (* FitsGetCom)( AstFitsChan *, const char *, char ** );
   void (* FitsSetCom)( AstFitsChan *, const char *, const char *, int  );
   void (* FitsSetCF)( AstFitsChan *, const char *, double *, const char *, int  );
   void (* FitsSetCI)( AstFitsChan *, const char *, int *, const char *, int  );
   void (* FitsSetF)( AstFitsChan *, const char *, double, const char *, int  );
   void (* FitsSetI)( AstFitsChan *, const char *, int, const char *, int  );
   void (* FitsSetL)( AstFitsChan *, const char *, int, const char *, int  );
   void (* FitsSetS)( AstFitsChan *, const char *, const char *, const char *, int  );
   void (* FitsSetCN)( AstFitsChan *, const char *, const char *, const char *, int  );
   int (* GetCard)( AstFitsChan * );
   int (* TestCard)( AstFitsChan * );
   void (* SetCard)( AstFitsChan *, int );
   void (* ClearCard)( AstFitsChan * );
   int (* GetFitsDigits)( AstFitsChan * );
   int (* TestFitsDigits)( AstFitsChan * );
   void (* SetFitsDigits)( AstFitsChan *, int );
   void (* ClearFitsDigits)( AstFitsChan * );
   int (* GetDefB1950)( AstFitsChan * );
   int (* TestDefB1950)( AstFitsChan * );
   void (* SetDefB1950)( AstFitsChan *, int );
   void (* ClearDefB1950)( AstFitsChan * );
   int (* GetCarLin)( AstFitsChan * );
   int (* TestCarLin)( AstFitsChan * );
   void (* SetCarLin)( AstFitsChan *, int );
   void (* ClearCarLin)( AstFitsChan * );
   int (* GetNcard)( AstFitsChan * );
   int (* GetEncoding)( AstFitsChan * );
   int (* TestEncoding)( AstFitsChan * );
   void (* SetEncoding)( AstFitsChan *, int );
   void (* ClearEncoding)( AstFitsChan * );
   const char *(* GetWarnings)( AstFitsChan * );
   const char *(* GetAllWarnings)( AstFitsChan * );
   int (* TestWarnings)( AstFitsChan * );
   void (* ClearWarnings)( AstFitsChan * );
   void (* SetWarnings)( AstFitsChan *, const char * );
   int (* GetClean)( AstFitsChan * );
   int (* TestClean)( AstFitsChan * );
   void (* SetClean)( AstFitsChan *, int );
   void (* ClearClean)( AstFitsChan * );
   int (* GetCDMatrix)( AstFitsChan * );
   int (* TestCDMatrix)( AstFitsChan * );
   void (* SetCDMatrix)( AstFitsChan *, int );
   void (* ClearCDMatrix)( AstFitsChan * );

   int (* GetIwc)( AstFitsChan * );
   int (* TestIwc)( AstFitsChan * );
   void (* SetIwc)( AstFitsChan *, int );
   void (* ClearIwc)( AstFitsChan * );

} AstFitsChanVtab;
#endif

/* Function prototypes. */
/* ==================== */
/* Prototypes for standard class functions. */
/* ---------------------------------------- */
astPROTO_CHECK(FitsChan)          /* Check class membership */
astPROTO_ISA(FitsChan)            /* Test class membership */

/* Constructor. */
#if defined(astCLASS)            /* Protected. */
AstFitsChan *astFitsChan_( const char *(*)( void ), void (*)( const char * ),
                          const char *, ... );
#else
AstFitsChan *astFitsChanId_( const char *(*)( void ), void (*)( const char * ),
                            const char *, ... );
AstFitsChan *astFitsChanForId_( const char *(*)( void ),
                              char *(*)( const char *(*)( void ) ),
                              void (*)( const char * ),
                              void (*)( void (*)( const char * ),
                                        const char * ),
                              const char *, ... );
#endif

#if defined(astCLASS)            /* Protected */

/* Initialiser. */
AstFitsChan *astInitFitsChan_( void *, size_t, int, AstFitsChanVtab *, 
                               const char *, 
                               const char *(*)( void ),
                               char *(*)( const char *(*)( void ) ),
                               void (*)( const char * ),
                               void (*)( void (*)( const char * ), const char * ) );

/* Vtab initialiser. */
void astInitFitsChanVtab_( AstFitsChanVtab *, const char * );



/* Loader. */
AstFitsChan *astLoadFitsChan_( void *, size_t, AstFitsChanVtab *,
                               const char *, AstChannel * );
#endif

/* Prototypes for member functions. */
/* -------------------------------- */
   void astPutFits_( AstFitsChan *, const char [81], int );
   void astPutCards_( AstFitsChan *, const char * );
   int  astFindFits_( AstFitsChan *, const char *, char [81], int );
   void astDelFits_( AstFitsChan * );

# if defined(astCLASS)           /* Protected */

   int  astFitsEof_( AstFitsChan * );
   int  astFitsGetCF_( AstFitsChan *, const char *, double * );
   int  astFitsGetCI_( AstFitsChan *, const char *, int * );
   int  astFitsGetF_( AstFitsChan *, const char *, double * );
   int  astFitsGetI_( AstFitsChan *, const char *, int * );
   int  astFitsGetL_( AstFitsChan *, const char *, int * );
   int  astFitsGetS_( AstFitsChan *, const char *, char ** );
   int  astFitsGetCN_( AstFitsChan *, const char *, char ** );
   int  astFitsGetCom_( AstFitsChan *, const char *, char ** );
   void astFitsSetCom_( AstFitsChan *, const char *, const char *, int  );
   void astFitsSetCF_( AstFitsChan *, const char *, double *, const char *, int  );
   void astFitsSetCI_( AstFitsChan *, const char *, int *, const char *, int  );
   void astFitsSetF_( AstFitsChan *, const char *, double, const char *, int  );
   void astFitsSetI_( AstFitsChan *, const char *, int, const char *, int  );
   void astFitsSetL_( AstFitsChan *, const char *, int, const char *, int  );
   void astFitsSetS_( AstFitsChan *, const char *, const char *, const char *, int  );
   void astFitsSetCN_( AstFitsChan *, const char *, const char *, const char *, int  );

   void astEmpty_( AstFitsChan * );
   int  astKeyFields_( AstFitsChan *, const char *, int, int *, int *);

   int astGetCard_( AstFitsChan * );
   int astTestCard_( AstFitsChan * );
   void astSetCard_( AstFitsChan *, int );
   void astClearCard_( AstFitsChan * );

   int astGetDefB1950_( AstFitsChan * );
   int astTestDefB1950_( AstFitsChan * );
   void astSetDefB1950_( AstFitsChan *, int );
   void astClearDefB1950_( AstFitsChan * );

   int astGetCDMatrix_( AstFitsChan * );
   int astTestCDMatrix_( AstFitsChan * );
   void astSetCDMatrix_( AstFitsChan *, int );
   void astClearCDMatrix_( AstFitsChan * );

   int astGetCarLin_( AstFitsChan * );
   int astTestCarLin_( AstFitsChan * );
   void astSetCarLin_( AstFitsChan *, int );
   void astClearCarLin_( AstFitsChan * );

   int astGetIwc_( AstFitsChan * );
   int astTestIwc_( AstFitsChan * );
   void astSetIwc_( AstFitsChan *, int );
   void astClearIwc_( AstFitsChan * );

   int astGetClean_( AstFitsChan * );
   int astTestClean_( AstFitsChan * );
   void astSetClean_( AstFitsChan *, int );
   void astClearClean_( AstFitsChan * );

   int astGetFitsDigits_( AstFitsChan * );
   int astTestFitsDigits_( AstFitsChan * );
   void astSetFitsDigits_( AstFitsChan *, int );
   void astClearFitsDigits_( AstFitsChan * );

   const char *astGetAllWarnings_( AstFitsChan * );

   const char *astGetWarnings_( AstFitsChan * );
   int astTestWarnings_( AstFitsChan * );
   void astClearWarnings_( AstFitsChan * );
   void astSetWarnings_( AstFitsChan *, const char * );

   int astGetNcard_( AstFitsChan * );

   int astGetEncoding_( AstFitsChan * );
   int astTestEncoding_( AstFitsChan * );
   void astSetEncoding_( AstFitsChan *, int );
   void astClearEncoding_( AstFitsChan * );

#endif

/* Function interfaces. */
/* ==================== */
/* These macros are wrap-ups for the functions defined by this class
   to make them easier to invoke (e.g. to avoid type mis-matches when
   passing pointers to objects from derived classes). */

/* Interfaces to standard class functions. */
/* --------------------------------------- */
/* Some of these functions provide validation, so we cannot use them
   to validate their own arguments. We must use a cast when passing
   object pointers (so that they can accept objects from derived
   classes). */

/* Check class membership. */
#define astCheckFitsChan(this) astINVOKE_CHECK(FitsChan,this)

/* Test class membership. */
#define astIsAFitsChan(this) astINVOKE_ISA(FitsChan,this)

/* Constructor. */
#if defined(astCLASS)            /* Protected. */
#define astFitsChan astINVOKE(F,astFitsChan_)
#else
#define astFitsChan astINVOKE(F,astFitsChanId_)
#define astFitsChanFor astINVOKE(F,astFitsChanForId_)
#endif

#if defined(astCLASS)            /* Protected */

/* Initialiser. */
#define astInitFitsChan(mem,size,init,vtab,name,source,sourcewrap,sink,sinkwrap) \
astINVOKE(O,astInitFitsChan_(mem,size,init,vtab,name,source,sourcewrap,sink,sinkwrap))

/* Vtab Initialiser. */
#define astInitFitsChanVtab(vtab,name) astINVOKE(V,astInitFitsChanVtab_(vtab,name))
/* Loader. */
#define astLoadFitsChan(mem,size,vtab,name,channel) \
astINVOKE(O,astLoadFitsChan_(mem,size,vtab,name,astCheckChannel(channel)))
#endif

/* Interfaces to public member functions. */
/* -------------------------------------- */
/* Here we make use of astCheckFitsChan to validate FitsChan pointers
   before use.  This provides a contextual error report if a pointer
   to the wrong sort of Object is supplied. */

#define astPutFits(this,card,overwrite) \
astINVOKE(V,astPutFits_(astCheckFitsChan(this),card,overwrite))

#define astPutCards(this,cards) \
astINVOKE(V,astPutCards_(astCheckFitsChan(this),cards))

#define astDelFits(this) \
astINVOKE(V,astDelFits_(astCheckFitsChan(this)))

#define astFindFits( this, name, card, inc ) \
astINVOKE(V,astFindFits_(astCheckFitsChan(this),name,card,inc))


#if defined(astCLASS)            /* Protected */

#define astFitsEof(this) \
astINVOKE(V,astFitsEof_(astCheckFitsChan(this)))

#define astFitsGetCF(this,name,value) \
astINVOKE(V,astFitsGetCF_(astCheckFitsChan(this),name,value))

#define astFitsGetCI(this,name,value) \
astINVOKE(V,astFitsGetCI_(astCheckFitsChan(this),name,value))

#define astFitsGetF(this,name,value) \
astINVOKE(V,astFitsGetF_(astCheckFitsChan(this),name,value))

#define astFitsGetI(this,name,value) \
astINVOKE(V,astFitsGetI_(astCheckFitsChan(this),name,value))

#define astFitsGetL(this,name,value) \
astINVOKE(V,astFitsGetL_(astCheckFitsChan(this),name,value))

#define astFitsGetS(this,name,value) \
astINVOKE(V,astFitsGetS_(astCheckFitsChan(this),name,value))

#define astFitsGetCN(this,name,value) \
astINVOKE(V,astFitsGetCN_(astCheckFitsChan(this),name,value))

#define astFitsGetCom(this,name,comment) \
astINVOKE(V,astFitsGetCom_(astCheckFitsChan(this),name,comment))

#define astFitsSetCom(this,name,comment,overwrite) \
astINVOKE(V,astFitsSetCom_(astCheckFitsChan(this),name,comment,overwrite))

#define astFitsSetI(this,name,value,comment,overwrite) \
astINVOKE(V,astFitsSetI_(astCheckFitsChan(this),name,value,comment,overwrite))

#define astFitsSetF(this,name,value,comment,overwrite) \
astINVOKE(V,astFitsSetF_(astCheckFitsChan(this),name,value,comment,overwrite))

#define astFitsSetS(this,name,value,comment,overwrite) \
astINVOKE(V,astFitsSetS_(astCheckFitsChan(this),name,value,comment,overwrite))

#define astFitsSetCN(this,name,value,comment,overwrite) \
astINVOKE(V,astFitsSetCN_(astCheckFitsChan(this),name,value,comment,overwrite))

#define astFitsSetCI(this,name,value,comment,overwrite) \
astINVOKE(V,astFitsSetCI_(astCheckFitsChan(this),name,value,comment,overwrite))

#define astFitsSetCF(this,name,value,comment,overwrite) \
astINVOKE(V,astFitsSetCF_(astCheckFitsChan(this),name,value,comment,overwrite))

#define astFitsSetL(this,name,value,comment,overwrite) \
astINVOKE(V,astFitsSetL_(astCheckFitsChan(this),name,value,comment,overwrite))

#define astSplit astSplit_

#define astEmpty(this) \
astINVOKE(V,astEmpty_(astCheckFitsChan(this)))

#define astKeyFields(this,filter,maxfld,ubnd,lbnd) \
astINVOKE(V,astKeyFields_(astCheckFitsChan(this),filter,maxfld,ubnd,lbnd))

#define astClearCard(this) \
astINVOKE(V,astClearCard_(astCheckFitsChan(this)))
#define astGetCard(this) \
astINVOKE(V,astGetCard_(astCheckFitsChan(this)))
#define astSetCard(this,card) \
astINVOKE(V,astSetCard_(astCheckFitsChan(this),card))
#define astTestCard(this) \
astINVOKE(V,astTestCard_(astCheckFitsChan(this)))

#define astClearDefB1950(this) \
astINVOKE(V,astClearDefB1950_(astCheckFitsChan(this)))
#define astGetDefB1950(this) \
astINVOKE(V,astGetDefB1950_(astCheckFitsChan(this)))
#define astSetDefB1950(this,defb950) \
astINVOKE(V,astSetDefB1950_(astCheckFitsChan(this),defb950))
#define astTestDefB1950(this) \
astINVOKE(V,astTestDefB1950_(astCheckFitsChan(this)))

#define astClearCDMatrix(this) \
astINVOKE(V,astClearCDMatrix_(astCheckFitsChan(this)))
#define astGetCDMatrix(this) \
astINVOKE(V,astGetCDMatrix_(astCheckFitsChan(this)))
#define astSetCDMatrix(this,cdmatrix) \
astINVOKE(V,astSetCDMatrix_(astCheckFitsChan(this),cdmatrix))
#define astTestCDMatrix(this) \
astINVOKE(V,astTestCDMatrix_(astCheckFitsChan(this)))

#define astClearCarLin(this) \
astINVOKE(V,astClearCarLin_(astCheckFitsChan(this)))
#define astGetCarLin(this) \
astINVOKE(V,astGetCarLin_(astCheckFitsChan(this)))
#define astSetCarLin(this,carln) \
astINVOKE(V,astSetCarLin_(astCheckFitsChan(this),carln))
#define astTestCarLin(this) \
astINVOKE(V,astTestCarLin_(astCheckFitsChan(this)))

#define astClearClean(this) \
astINVOKE(V,astClearClean_(astCheckFitsChan(this)))
#define astGetClean(this) \
astINVOKE(V,astGetClean_(astCheckFitsChan(this)))
#define astSetClean(this,value) \
astINVOKE(V,astSetClean_(astCheckFitsChan(this),value))
#define astTestClean(this) \
astINVOKE(V,astTestClean_(astCheckFitsChan(this)))

#define astClearFitsDigits(this) \
astINVOKE(V,astClearFitsDigits_(astCheckFitsChan(this)))
#define astGetFitsDigits(this) \
astINVOKE(V,astGetFitsDigits_(astCheckFitsChan(this)))
#define astSetFitsDigits(this,fitsdigits) \
astINVOKE(V,astSetFitsDigits_(astCheckFitsChan(this),fitsdigits))
#define astTestFitsDigits(this) \
astINVOKE(V,astTestFitsDigits_(astCheckFitsChan(this)))

#define astClearWarnings(this) \
astINVOKE(V,astClearWarnings_(astCheckFitsChan(this)))
#define astGetWarnings(this) \
astINVOKE(V,astGetWarnings_(astCheckFitsChan(this)))
#define astSetWarnings(this,warnings) \
astINVOKE(V,astSetWarnings_(astCheckFitsChan(this),warnings))
#define astTestWarnings(this) \
astINVOKE(V,astTestWarnings_(astCheckFitsChan(this)))

#define astGetAllWarnings(this) \
astINVOKE(V,astGetAllWarnings_(astCheckFitsChan(this)))

#define astGetNcard(this) \
astINVOKE(V,astGetNcard_(astCheckFitsChan(this)))

#define astClearEncoding(this) \
astINVOKE(V,astClearEncoding_(astCheckFitsChan(this)))
#define astGetEncoding(this) \
astINVOKE(V,astGetEncoding_(astCheckFitsChan(this)))
#define astSetEncoding(this,encoding) \
astINVOKE(V,astSetEncoding_(astCheckFitsChan(this),encoding))
#define astTestEncoding(this) \
astINVOKE(V,astTestEncoding_(astCheckFitsChan(this)))

#define astClearIwc(this) \
astINVOKE(V,astClearIwc_(astCheckFitsChan(this)))
#define astGetIwc(this) \
astINVOKE(V,astGetIwc_(astCheckFitsChan(this)))
#define astSetIwc(this,iwc) \
astINVOKE(V,astSetIwc_(astCheckFitsChan(this),iwc))
#define astTestIwc(this) \
astINVOKE(V,astTestIwc_(astCheckFitsChan(this)))

#endif

#endif
