#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc ButtonMarker {which x y} {
    global marker
    global current

    # if nothing is loaded, abort

    if {![$current(frame) has fits]} {
	return
    }

    # see if we are on a handle

    set h [$which get marker handle $x $y]
    set id [lindex $h 0]
    set marker(handle) [lindex $h 1]

    if {$marker(handle)} {
	$which marker $id edit begin $marker(handle)
	set marker(motion) beginEdit
	return
    }

    # else, see if we are on a segment of a polygon

    set h [$which get marker polygon segment $x $y]
    set id [lindex $h 0]
    set segment [lindex $h 1]
    if {$segment} {
	$which marker $id create polygon vertex $segment $x $y
	$which marker $id edit begin $marker(handle)
	set marker(handle) [expr 4+$segment+1]
	set marker(motion) beginEdit
	return
    }

    # else, see if we are on a marker

    if {[$which get marker id $x $y]} {
	$which marker select $x $y
	$which marker move begin $x $y
	set marker(motion) beginMove
	UpdateMarkerMenu
	return
    }

    # see if any markers are selected

    if {[$which get marker select number]>0} {
	$which marker unselect all
	set marker(motion) none
	return
    }

    # else, create a marker

    set marker(handle) 0
    set marker(motion) none

    set cmd "$which marker create $marker(shape) $x $y"
    switch -- $marker(shape) {
	circle {append cmd " 0"}
	annulus {append cmd " .001 .002 $marker(annulus,annuli)"}
	panda {append cmd " $marker(panda,ang1) $marker(panda,ang2) $marker(panda,angnum) .001 .002 $marker(panda,annuli)"}
	ellipse {append cmd " 0 0"}
	"ellipse annulus" {append cmd " .001 .001 .002 $marker(ellipseannulus,annuli)"}
	box {append cmd " 0 0"}
	"box annulus" {append cmd " .002 .002 .004 $marker(boxannulus,annuli)"}
	polygon {append cmd " .001 .001"}
	line {append cmd " $x $y"}
	text {
	    set txt "Region"
	    set r [EntryDialog "Text Region" "Enter Text:" 40 txt]
	    if {$r == 1 && $txt != {}} {
		append cmd " 0 text = \{\"$txt\"\}"
	    } else {
		return
	    }
	}
	ruler {append cmd " $x $y"}
	compass {append cmd " 15"}
	projection {append cmd " $x $y $marker(projection,thick)"}
    }
    append cmd " color = $marker(color)"
    append cmd " width = $marker(width)"
    append cmd \
      " font = \{\"$marker(font) $marker(font,size) $marker(font,style)\"\}"
    append cmd " edit = $marker(edit) \
			 move = $marker(move) \
			 rotate = $marker(rotate) \
			 delete = $marker(delete) \
			 fixed = $marker(fixed) \
                         include = $marker(include) \
                         source = $marker(source) "

    $which marker unselect all

    set marker(id) [eval $cmd]
    set marker(motion) beginCreate
    set marker(createx) $x
    set marker(createy) $y

    switch -- $marker(shape) {
	circle -
	annulus -
	panda -
	ellipse -
	"ellipse annulus" -
	box -
	"box annulus" -
	compass -
	polygon {
	    set marker(handle) 1
	    $which marker $marker(id) edit begin $marker(handle)
	}
	line -
	ruler -
	projection {
	    set marker(handle) 2
	    $which marker $marker(id) edit begin $marker(handle)
	}
    }
}

proc ShiftMarker {which x y} {
    global current
    global marker

    # if nothing is loaded, abort

    if {![$current(frame) has fits]} {
	return
    }

    # see if we are on a handle

    set h [$which get marker handle $x $y]
    set id [lindex $h 0]
    set marker(handle) [lindex $h 1]

    if {$marker(handle)} {
	$which marker $id rotate begin
	set marker(motion) beginRotate
	return
    }

    # else, see if we are on a marker

    if {[$which marker select shift $x $y]} {
	UpdateMarkerMenu
	$which marker move begin $x $y
	set marker(motion) beginMove
	return
    }

    # else, start a region select

    $which region select begin $x $y
    set marker(motion) shiftregion
}

proc ControlMarker {which x y} {
    global current
    global marker
    
    # if nothing is loaded, abort

    if {![$current(frame) has fits]} {
	return
    }

    # we need this cause MotionMarker maybe called, 
    # and we don't want it
    set marker(motion) none
    set marker(handle) -1

    set id [$which get marker id $x $y]
    if {$id} {
	# are we on a selected annulus?
	if {[$which get marker select $x $y] == $id} {
	    switch -- [$which get marker $id type] {
		annulus {
		    set marker(handle) \
			[$which marker $id create annulus radius $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
		panda {
		    set marker(handle) \
			[$which marker $id create panda radius $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
		"ellipse annulus" {
		    set marker(handle) \
			[$which marker $id create ellipse annulus radius $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
		"box annulus" {
		    set marker(handle) \
			[$which marker $id create box annulus size $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
	    }
	}
    }
}

proc ControlShiftMarker {which x y} {
    global current
    global marker
    
    # if nothing is loaded, abort

    if {![$current(frame) has fits]} {
	return
    }

    # we need this cause MotionMarker maybe called, 
    # and we don't want it
    set marker(motion) none
    set marker(handle) -1

    set id [$which get marker id $x $y]
    if {$id} {
	# are we on a selected annulus?
	if {[$which get marker select $x $y] == $id} {
	    switch -- [$which get marker $id type] {
		panda {
		    set marker(handle) \
			[$which marker $id create panda angle $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
	    }
	}
    }
}

proc MotionMarker {which x y} {
    global current
    global marker

    # if nothing is loaded, abort

    if {![$current(frame) has fits]} {
	return
    }

    switch -- $marker(motion) {
	none {}

	beginCreate -
	create {
	    $which marker edit motion $x $y $marker(handle)
	    set marker(motion) create
	}

	beginMove -
	move {
	    $which marker move motion $x $y
	    set marker(motion) move
	}

	beginEdit -
	edit {
	    $which marker edit motion $x $y $marker(handle)
	    set marker(motion) edit
	}

	beginRotate -
	rotate {
	    $which marker rotate motion $x $y $marker(handle)
	    set marker(motion) rotate
	}

	region -
	shiftregion {$which region select motion $x $y}
    }
}

proc ReleaseMarker {which x y} {
    global current
    global marker

    # if nothing is loaded, abort

    if {![$current(frame) has fits]} {
	return
    }

    switch -- $marker(motion) {
	none {}
	beginCreate {
	    # the user has just clicked, so resize to make visible or delete
	    # assumes marker(id) from create
	    $which marker edit end
	    DefaultMarker $which

	    unset marker(createx)
	    unset marker(createy)
	    unset marker(id)
	}
	create {
	    $which marker edit end

	    # determine if this is an accident and just create the default
	    set diffx [expr $x-$marker(createx)]
	    set diffy [expr $y-$marker(createy)]
	    if {[expr sqrt($diffx*$diffx + $diffy*$diffy)]<2} {
		DefaultMarker $which
	    }

	    # special callbacks
	    switch -- [$which get marker $marker(id) type] {
		projection {SetupProjectionPlot $marker(id)}
	    }

	    unset marker(createx)
	    unset marker(createy)
	    unset marker(id)
	}
	beginMove -
	beginRotate {}
	beginEdit {}
	move {$which marker move end}
	edit {$which marker edit end}
	rotate {$which marker rotate end}
	region {$which region select end}
	shiftregion {$which region select shift end}
    }

    unset marker(motion)
    unset marker(handle)
}

proc DefaultMarker {which} {
    global marker
    global current

    # scale the default size to take into account the current
    switch -- [$which get marker $marker(id) type] {
	circle {
	    $which marker $marker(id) circle radius \
		[expr ($marker(circle,radius)/double($current(zoom)))] \
		image degrees
	}
	annulus {
	    $which marker $marker(id) annulus radius \
		[expr ($marker(annulus,inner)/double($current(zoom)))] \
		[expr ($marker(annulus,outer)/double($current(zoom)))] \
		$marker(annulus,annuli) image degrees
	}
	panda {
	    $which marker $marker(id) panda \
		$marker(panda,ang1) $marker(panda,ang2) $marker(panda,angnum) \
		[expr ($marker(panda,inner)/double($current(zoom)))] \
		[expr ($marker(panda,outer)/double($current(zoom)))] \
		$marker(panda,annuli) image degrees
	}
	ellipse {
	    $which marker $marker(id) ellipse radius \
		[expr ($marker(ellipse,radius1)/double($current(zoom)))] \
		[expr ($marker(ellipse,radius2)/double($current(zoom)))] \
		image degrees
	}
	{ellipse annulus} {
	    $which marker $marker(id) ellipse annulus radius \
	      [expr ($marker(ellipseannulus,radius1)/double($current(zoom)))] \
	      [expr ($marker(ellipseannulus,radius2)/double($current(zoom)))] \
	      [expr ($marker(ellipseannulus,radius3)/double($current(zoom)))] \
		$marker(ellipseannulus,annuli) image degrees
	}
	box {
	    $which marker $marker(id) box size \
		[expr ($marker(box,width)/double($current(zoom)))] \
		[expr ($marker(box,height)/double($current(zoom)))] \
		image degrees
	}
	{box annulus} {
	    $which marker $marker(id) box annulus size \
		[expr ($marker(boxannulus,size1)/double($current(zoom)))] \
		[expr ($marker(boxannulus,size2)/double($current(zoom)))] \
		[expr ($marker(boxannulus,size3)/double($current(zoom)))] \
		$marker(boxannulus,annuli) image degrees
	}
	compass {
	    $which marker $marker(id) compass radius $marker(compass,radius) \
		image degrees
	}
	polygon {
	    $which marker $marker(id) polygon reset \
		[expr ($marker(polygon,width)/double($current(zoom)))] \
		[expr ($marker(polygon,height)/double($current(zoom)))] \
		image degrees
	}
	line -
	ruler -
	projection {$which marker $marker(id) delete}
    }
}

proc DoubleMarker {which x y} {
    global current

    # if nothing is loaded, abort

    if {![$current(frame) has fits]} {
	return
    }

    set id [$which get marker id $x $y]

    if {$id} {
	if [$which get marker $id PROPERTY SELECT] {
	    MarkerDialog $which $id
	}
    }
}

proc DeleteMarker {which x y} {
    global current

    # if nothing is loaded, abort
    if {![$current(frame) has fits]} {
	return
    }

    # see if we are on a polygon
    set h [$which get marker handle $x $y]
    set id [lindex $h 0]
    set handle [lindex $h 1]

    set t [$which get marker $id type]
    if {($t == "polygon") || ($t == "annulus") || ($t == "panda") || \
	($t == "ellipse annulus") || ($t == "box annulus")} {
	if {$handle > 4} {
	    switch -- $t {
		polygon {$which marker $id delete polygon vertex $handle}
		annulus {$which marker $id delete annulus radius $handle}
		panda {$which marker $id delete panda $handle}
		"ellipse annulus" {$which marker $id delete \
				       ellipse annulus radius $handle}
		"box annulus" {$which marker $id delete \
				   box annulus size $handle}
	    }
	} else {
	    # delete polygon
	    $which marker delete
	    UpdateGroupDialog
	}
    } else {
	# delete marker
	$which marker delete
	UpdateGroupDialog
    }
}

proc CursorMarker {which x y handleCursor overCursor} {
    global ds9
    global current

    # if nothing is loaded, abort

    if {![$current(frame) has fits]} {
	return
    }

    # are we over any selected marker handles?
    # remember, handles are outside of a marker

    set h [$which get marker handle $x $y]
    set id [lindex $h 0]
    set handle [lindex $h 1]
    if {$handle} {
	if {$handle < 5} {
	    # edit/rotate handle
	    SetCursor $handleCursor
	} else { 
	    # polygon/annulus vertex
	    SetCursor dotbox
	}
	return
    }

    # else, see if we are on a segement of a polygon

    set h [$which get marker polygon segment $x $y]
    if {[lindex $h 0]} {
	SetCursor draped_box
	return
    }

    # are we over a marker?

    set id [$which get marker select $x $y]
    if {$id} {
	# are we on a selected annulus and control key down?
	if {([$which get marker $id type] == "annulus") ||
	    ([$which get marker $id type] == "panda") ||
	    ([$which get marker $id type] == "ellipse annulus") ||
	    ([$which get marker $id type] == "box annulus")} {
	    SetCursor $overCursor
	} else {
	    SetCursor fleur
	}
	return
    }

    # else, set no cursor

    SetCursor {}
}

proc ShowMarker {} {
    global current
    global marker

    if {$current(frame) != {}} {
	$current(frame) marker fg show $marker(show)
	$current(frame) marker bg show $marker(show)
    }
}

proc PreserveMarker {} {
    global current
    global marker

    if {$current(frame) != {}} {
	$current(frame) marker preserve $marker(preserve)
    }
}

proc FrontMarker {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker move front
    }
}

proc BackMarker {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker move back
	$current(frame) marker unselect all
    }
}

proc SelectAllMarker {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker select all
    }
}

proc UnselectAllMarker {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker unselect all
    }
}

proc DeleteSelectMarker {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker delete
	UpdateGroupDialog
    }
}

proc DeleteAllMarker {} {
    global current
    
    if {$current(frame) != ""} {
	if {[tk_messageBox -type okcancel -default cancel \
		 -message "Delete All Regions?" -icon question] == "ok"} {
	    $current(frame) marker delete all
	    UpdateGroupDialog
	}
    }
}

proc ChangeMarkerColor {} {
    global current
    global marker
    
    if {$current(frame) != ""} {
	$current(frame) marker color $marker(color)
    }
}

proc ChangeMarkerWidth {} {
    global current
    global marker
    
    if {$current(frame) != ""} {
	$current(frame) marker width $marker(width)
    }
}

proc ChangeMarkerProp {prop} {
    global current
    global marker

    if {$current(frame) != ""} {
	$current(frame) marker property $prop $marker($prop)
    }
}

proc ChangeMarkerFont {} {
    global current
    global marker

    if {$current(frame) != ""} {
	$current(frame) marker font \
	    \"$marker(font) $marker(font,size) $marker(font,style)\"
    }
}

proc UnselectAllMarkers {} {
    global ds9

    foreach f $ds9(frames) {
	$f marker unselect all
    }
}

proc UpdateMarkerMenu {} {
    global ds9
    global current
    global marker
    global buttons
    global menu

    if {$current(frame) != {}} {
	set marker(show) [$current(frame) get marker fg show]
	set marker(preserve) [$current(frame) get marker preserve]

	switch -- $ds9(mode) {
	    pointer {
		if {[$current(frame) get marker select number] == 1} {
		    set marker(color) [$current(frame) get marker color]
		    set marker(edit) [$current(frame) get marker property edit]
		    set marker(move) [$current(frame) get marker property move]
		    set marker(rotate) \
			[$current(frame) get marker property rotate]
		    set marker(delete) \
			[$current(frame) get marker property delete]
		    set marker(fixed) \
			[$current(frame) get marker property fixed]
		    set marker(include) \
			[$current(frame) get marker property include]
		    set marker(source) \
			[$current(frame) get marker property source]

		    set f [$current(frame) get marker font]

		    set marker(font) [lindex $f 0]
		    set marker(font,size) [lindex $f 1]
		    set marker(font,style) [lindex $f 2]
		}
	    }
	}
    }

    UpdateMarkerFormatMenu
}

proc UpdateMarkerFormatMenu {} {
    global marker
    global current
    global ds9
    global menu

    set mm $ds9(mb).region

    switch -- $marker(format) {
	ds9 {
	    UpdateCoordMenu "$current(frame)" $mm.coord
	    AdjustCoord "$current(frame)" marker(system)
	}

	ciao {
	    UpdateCoordMenu "$current(frame)" $mm.coord
	    switch -- $marker(system) {
		detector -
		amplifier -
		physical -
		image {set marker(system) physical}
		wcs -
		default {set marker(system) wcs}
	    }
	    set marker(sky) fk5
	    set marker(skyformat) sexagesimal
	    AdjustCoord "$current(frame)" marker(system)

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "Image" -state disabled
	    $mm.coord entryconfig "Physical" -state normal
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Amplifier" -state disabled
		$mm.coord entryconfig "Detector" -state disabled
	    }

	    $mm.coord entryconfig "Equatorial B1950" -state disabled
	    $mm.coord entryconfig "Equatorial J2000" -state normal
	    $mm.coord entryconfig "ICRS" -state disabled
	    $mm.coord entryconfig "Galactic" -state disabled
	    $mm.coord entryconfig "Ecliptic" -state disabled

	    $mm.coord entryconfig "Degrees" -state disabled
	    $mm.coord entryconfig "Sexagesimal" -state normal
	}

	saotng {
	    UpdateCoordMenu "$current(frame)" $mm.coord
	    switch -- $marker(system) {
		detector -
		amplifier -
		image -
		physical {set marker(system) image}
		wcs -
		default {set marker(system) wcs}
	    }
	    AdjustCoord "$current(frame)" marker(system)

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "Physical" -state disabled
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Amplifier" -state disabled
		$mm.coord entryconfig "Detector" -state disabled
	    }
	}

	saoimage {
	    set marker(system) image

	    SetCoordMenu $mm.coord disabled
	    $mm.coord entryconfig "Image" -state normal
	}

	pros {
	    UpdateCoordMenu "$current(frame)" $mm.coord
	    switch -- $marker(system) {
		detector -
		physical -
		amplifier {set marker(system) physical}
		image {}
		wcs -
		default {set marker(system) wcs}
	    }

	    if {$marker(sky) == "icrs"} {
		set marker(sky) fk5
	    }
	    AdjustCoord "$current(frame)" marker(system)

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "ICRS" -state disabled
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Detector" -state disabled
		$mm.coord entryconfig "Amplifier" -state disabled
	    }
	}
	
	xy {
	    UpdateCoordMenu "$current(frame)" $mm.coord
	    switch -- $marker(system) {
		detector -
		physical -
		amplifier -
		image -
		wcs {}
		default {set marker(system) wcs}
	    }
	    AdjustCoord "$current(frame)" marker(system)

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	}
    }
}

proc UpdateMarkerPrefsFormatMenu {} {
    global prefs
    global marker
    global current
    global ds9

    set mm $ds9(mb).prefs.region
    SetCoordMenu $mm.coord normal

    switch -- $prefs(marker,skyformat) {
	ds9 {}

	ciao {
	    switch -- $prefs(marker,system) {
		detector -
		amplifier -
		physical -
		image {set prefs(marker,system) physical}
		wcs -
		default {set prefs(marker,system) wcs}
	    }
	    set prefs(marker,sky) fk5
	    set prefs(marker,skyformat) sexagesimal

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "Image" -state disabled
	    $mm.coord entryconfig "Physical" -state normal
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Amplifier" -state disabled
		$mm.coord entryconfig "Detector" -state disabled
	    }

	    $mm.coord entryconfig "Equatorial B1950" -state disabled
	    $mm.coord entryconfig "Equatorial J2000" -state normal
	    $mm.coord entryconfig "ICRS" -state disabled
	    $mm.coord entryconfig "Galactic" -state disabled
	    $mm.coord entryconfig "Ecliptic" -state disabled

	    $mm.coord entryconfig "Degrees" -state disabled
	    $mm.coord entryconfig "Sexagesimal" -state normal
	}

	saotng {
	    switch -- $prefs(marker,system) {
		detector -
		amplifier -
		image -
		physical {set prefs(marker,system) image}
		wcs -
		default {set prefs(marker,system) wcs}
	    }

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "Physical" -state disabled
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Amplifier" -state disabled
		$mm.coord entryconfig "Detector" -state disabled
	    }
	}

	saoimage {
	    SetCoordMenu $mm.coord disabled
	    set prefs(marker,system) image
	    $mm.coord entryconfig "Image" -state normal
	}

	pros {
	    switch -- $prefs(marker,system) {
		detector -
		physical -
		amplifier {set prefs(marker,system) physical}
		image {}
		wcs -
		default {set prefs(marker,system) wcs}
	    }
	    if {$prefs(marker,sky) == "icrs"} {
		set prefs(marker,sky) fk5
	    }

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "Detector" -state disabled
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Amplifier" -state disabled
		$mm.coord entryconfig "ICRS" -state disabled
	    }
	}
	
	xy {
	    switch -- $prefs(marker,system) {
		detector -
		physical -
		amplifier -
		image -
		wcs {}
		default {set prefs(marker,system) wcs}
	    }
	    $mm.coord entryconfig "Multiple WCS" -state disabled
	}
    }
}

proc OpenMarkerFile {} {
    LoadMarker fg [OpenFileDialog markerfbox]
}

proc LoadMarker {layer filename} {
    global current
    global marker
    global message

    if {$current(frame) != ""} {
	if {$filename != {}} {
	    SetWatchCursor

	    # determine if its a fits file
	    # first, strip the filename
	    if {![regexp -nocase {(.*)(\[.*\])} $filename foo base ext]} {
		set base $filename
		set ext {}
	    }
	    set fd [open $base]
	    if {$fd != ""} {
		set l [read $fd 9]
		close $fd
	    }

	    if {$l == "SIMPLE  ="} {
		# its a fits file

		# see if we need to add an extension
		if {$ext == ""} {
		    set filename "$base\[REGION\]"
		}

		# open it
		if [catch {$current(frame) marker $layer load fits \
			       "\{$filename\}" $marker(color) $marker(width) \
			       "\{$marker(font) $marker(font,size) $marker(font,style)\}"}] {
		    Error "$message(error,marker,file)"
		}
	    } else {
		# no, its ascii
		switch -- $marker(format) {
		    xy {
			if [catch {$current(frame) marker $layer load \
				       $marker(format) "\{$filename\}" \
				       $marker(system) $marker(sky)}] {
			    Error "$message(error,marker,file)"
			}
		    }
		    default {
			if [catch {$current(frame) marker $layer load \
				       $marker(format) "\{$filename\}"}] {
			    Error "$message(error,marker,file)"
			}
		    }
		}
	    }
	    UnsetWatchCursor
	    UpdateGroupDialog
	}
    }
}

proc SaveMarker {} {
    global current
    global marker

    if {$current(frame) != {}} {
	SetWatchCursor
	set filename [SaveFileDialog markerfbox]

	if {$filename != {}} {
	    $current(frame) marker save "\{$filename\}" \
		$marker(format) $marker(system) $marker(sky) \
		$marker(skyformat) $marker(wcs)
	}
	UnsetWatchCursor
    }
}

proc DisplayMarker {} {
    global current
    global marker

    if {$current(frame) != {}} {
	SimpleTextDialog markerlist "Region List" 80 20 insert top \
	    [$current(frame) marker list $marker(format) \
		 $marker(system) $marker(sky) \
		 $marker(skyformat) $marker(strip) $marker(wcs)]
    }
}

proc InfoMarker {} {
    global current
    global marker

    if {$current(frame) != {}} {
	set l [$current(frame) get marker select]
	set i 0
	foreach d $l {
	    incr i
	    if {$i > $marker(maxdialog)} {
		return
	    }
	    MarkerDialog $current(frame) $d
	}
    }
}

proc MarkerDialog {frame id} {

    set type [$frame get marker $id type]

    switch -- $type {
	circle {CircleDialog $frame $id}
	annulus {AnnulusDialog $frame $id}
	panda {PandaDialog $frame $id}
	ellipse {EllipseDialog $frame $id}
	"ellipse annulus" {EllipseAnnulusDialog $frame $id}
	box {BoxDialog $frame $id}
	"box annulus" {BoxAnnulusDialog $frame $id}
	polygon {PolygonDialog $frame $id}
	line {LineDialog $frame $id}
	text {TextDialog $frame $id}
	ruler {RulerDialog $frame $id}
	compass {CompassDialog $frame $id}
	projection {ProjectionDialog $frame $id}
	{circle point} -
	{box point} -
	{diamond point} -
	{cross point} -
	{x point} -
	{arrow point} -
	{boxcircle point} {PointDialog $frame $id}
    }
}

proc InitMarkerDialog {frame id} {
    global marker

    set w ".marker$id"
    set mb ".mb$id"

    # init marker variables

    set marker($frame,$id,system) $marker(dialog,system)
    set marker($frame,$id,clabel) $marker(dialog,system)
    set marker($frame,$id,sky) $marker(dialog,sky)
    set marker($frame,$id,skyformat) $marker(dialog,skyformat)

    MoveMarkerCB $frame $id
    TextMarkerCB $frame $id
    ColorMarkerCB $frame $id
    LineWidthMarkerCB $frame $id
    PropertyMarkerCB $frame $id
    FontMarkerCB $frame $id

    # register callbacks

    $frame marker $id callback move MoveMarkerCB $frame
    $frame marker $id callback text TextMarkerCB $frame
    $frame marker $id callback color ColorMarkerCB $frame
    $frame marker $id callback width LineWidthMarkerCB $frame
    $frame marker $id callback property PropertyMarkerCB $frame
    $frame marker $id callback font FontMarkerCB $frame

    # define menus

    MBMarkerDialog $frame $mb
    FileMenuMarkerDialog $frame $mb $id
    ColorMenuMarkerDialog $frame $mb $id
    WidthMenuMarkerDialog $frame $mb $id
#    PropMenuMarkerDialog $frame $mb $id  let each type define
    FontMenuMarkerDialog $frame $mb $id
#    CoordMenuMarkerDialog $frame $mb $id MoveMarkerCB let each type define
}

proc MBMarkerDialog {frame mb} {
    global menu

    menu $mb -tearoff 0 -selectcolor $menu(selectcolor)
    $mb add cascade -label File -menu $mb.file
    $mb add cascade -label Color -menu $mb.color
    $mb add cascade -label Width -menu $mb.width
    $mb add cascade -label Property -menu $mb.properties
    $mb add cascade -label Font -menu $mb.font
    $mb add cascade -label Coord -menu $mb.coord
}

proc FileMenuMarkerDialog {frame mb id} {
    global menu
    global marker

    menu $mb.file -tearoff 0 -selectcolor $menu(selectcolor)
}

proc ColorMenuMarkerDialog {frame mb id} {
    global menu
    global marker

    menu $mb.color -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.color add radiobutton -label "Black" \
	-variable marker($frame,$id,color) -value black \
	-command "ColorMarkerDialog $frame $id"
    $mb.color add radiobutton -label "White" \
	-variable marker($frame,$id,color) -value white \
	-command "ColorMarkerDialog $frame $id"
    $mb.color add radiobutton -label "Red" \
	-variable marker($frame,$id,color) -value red \
	-command "ColorMarkerDialog $frame $id"
    $mb.color add radiobutton -label "Green" \
	-variable marker($frame,$id,color) -value green \
	-command "ColorMarkerDialog $frame $id"
    $mb.color add radiobutton -label "Blue" \
	-variable marker($frame,$id,color) -value blue \
	-command "ColorMarkerDialog $frame $id"
    $mb.color add radiobutton -label "Cyan" \
	-variable marker($frame,$id,color) -value cyan \
	-command "ColorMarkerDialog $frame $id"
    $mb.color add radiobutton -label "Magenta" \
	-variable marker($frame,$id,color) -value magenta \
	-command "ColorMarkerDialog $frame $id"
    $mb.color add radiobutton -label "Yellow" \
	-variable marker($frame,$id,color) -value yellow \
	-command "ColorMarkerDialog $frame $id"
}

proc WidthMenuMarkerDialog {frame mb id} {
    global menu
    global marker

    menu $mb.width -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.width add radiobutton -label "1" \
	-variable marker($frame,$id,linewidth) -value 1 \
	-command "LineWidthMarkerDialog $frame $id"
    $mb.width add radiobutton -label "2" \
	-variable marker($frame,$id,linewidth) -value 2 \
	-command "LineWidthMarkerDialog $frame $id"
    $mb.width add radiobutton -label "3" \
	-variable marker($frame,$id,linewidth) -value 3 \
	-command "LineWidthMarkerDialog $frame $id"
    $mb.width add radiobutton -label "4" \
	-variable marker($frame,$id,linewidth) -value 4 \
	-command "LineWidthMarkerDialog $frame $id"
}

proc PropMenuMarkerDialog {frame mb id} {
    global menu
    global marker

    menu $mb.properties -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.properties add checkbutton -label "Can Edit" \
	-variable marker($frame,$id,edit) \
	-command "PropertyMarkerDialog $frame $id edit"
    $mb.properties add checkbutton -label "Can Move" \
	-variable marker($frame,$id,move) \
	-command "PropertyMarkerDialog $frame $id move"
    $mb.properties add checkbutton -label "Can Rotate" \
	-variable marker($frame,$id,rotate) \
	-command "PropertyMarkerDialog $frame $id rotate"
    $mb.properties add checkbutton -label "Can Delete" \
	-variable marker($frame,$id,delete) \
	-command "PropertyMarkerDialog $frame $id delete"
    $mb.properties add checkbutton -label "Fixed in Size" \
	-variable marker($frame,$id,fixed) \
	-command "PropertyMarkerDialog $frame $id fixed"
    $mb.properties add separator
    $mb.properties add radiobutton -label Include \
	-variable marker($frame,$id,include) -value 1 \
	-command "PropertyMarkerDialog $frame $id include"
    $mb.properties add radiobutton -label Exclude \
	-variable marker($frame,$id,include) -value 0 \
	-command "PropertyMarkerDialog $frame $id include"
    $mb.properties add separator
    $mb.properties add radiobutton -label Source \
	-variable marker($frame,$id,source) -value 1 \
	-command "PropertyMarkerDialog $frame $id source"
    $mb.properties add radiobutton -label Background \
	-variable marker($frame,$id,source) -value 0 \
	-command "PropertyMarkerDialog $frame $id source"
}

proc CoordMenuMarkerDialog {frame mb id cb} {
    global menu
    global marker
    global ds9

    menu $mb.coord -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.coord add radiobutton -label "WCS" \
	-variable marker($frame,$id,system) -value wcs \
	-command "$cb $frame $id"
    $mb.coord add cascade -label "Multiple WCS" -menu $mb.coord.wcs
    $mb.coord add separator
    $mb.coord add radiobutton -label "Image" \
	-variable marker($frame,$id,system) -value image \
	-command "$cb $frame $id"
    $mb.coord add radiobutton -label "Physical" \
	-variable marker($frame,$id,system) -value physical \
	-command "$cb $frame $id"
    if {$ds9(amp,det)} {
	$mb.coord add radiobutton -label "Amplifier" \
	    -variable marker($frame,$id,system) -value amplifier \
	    -command "$cb $frame $id"
	$mb.coord add radiobutton -label "Detector" \
	    -variable marker($frame,$id,system) -value detector \
	    -command "$cb $frame $id"
    }
    $mb.coord add separator
    $mb.coord add radiobutton -label "Equatorial B1950" \
	-variable marker($frame,$id,sky) -value fk4 \
	-command "$cb $frame $id"
    $mb.coord add radiobutton -label "Equatorial J2000" \
	-variable marker($frame,$id,sky) -value fk5 \
	-command "$cb $frame $id"
    $mb.coord add radiobutton -label "ICRS" \
	-variable marker($frame,$id,sky) -value icrs \
	-command "$cb $frame $id"
    $mb.coord add radiobutton -label "Galactic" \
	-variable marker($frame,$id,sky) -value galactic \
	-command "$cb $frame $id"
    $mb.coord add radiobutton -label "Ecliptic" \
	-variable marker($frame,$id,sky) -value ecliptic \
	-command "$cb $frame $id"
    $mb.coord add separator
    $mb.coord add radiobutton -label "Degrees" \
	-variable marker($frame,$id,skyformat) -value degrees \
	-command "$cb $frame $id"
    $mb.coord add radiobutton -label "Sexagesimal" \
	-variable marker($frame,$id,skyformat) -value sexagesimal \
	-command "$cb $frame $id"

    menu $mb.coord.wcs -tearoff 0 -selectcolor $menu(selectcolor)
    foreach l {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z} {
	$mb.coord.wcs add radiobutton -label "WCS $l" \
	    -variable marker($frame,$id,system) -value "wcs$l" \
	    -command "$cb $frame $id"
    }

    UpdateCoordMenu $frame $mb.coord
}

proc DistMenuMarkerDialog {frame mb id cb name coord format} {
    global menu
    global marker
    global ds9

    menu $mb.$name -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.$name add radiobutton -label "WCS" \
	-variable marker($frame,$id,$coord) -value wcs \
	-command "$cb $frame $id"
    $mb.$name add cascade -label "Multiple WCS" -menu $mb.$name.wcs
    $mb.$name add separator
    $mb.$name add radiobutton -label "Image" \
	-variable marker($frame,$id,$coord) -value image \
	-command "$cb $frame $id"
    $mb.$name add radiobutton -label "Physical" \
	-variable marker($frame,$id,$coord) -value physical \
	-command "$cb $frame $id"
    if {$ds9(amp,det)} {
	$mb.$name add radiobutton -label "Amplifier" \
	    -variable marker($frame,$id,$coord) -value amplifier \
	    -command "$cb $frame $id"
	$mb.$name add radiobutton -label "Detector" \
	    -variable marker($frame,$id,$coord) -value detector \
	    -command "$cb $frame $id"
    }
    $mb.$name add separator

    $mb.$name add radiobutton -label "Degrees" \
	-variable marker($frame,$id,$format) -value degrees \
	-command "$cb $frame $id"
    $mb.$name add radiobutton -label "ArcMin" \
	-variable marker($frame,$id,$format) -value arcmin \
	-command "$cb $frame $id"
    $mb.$name add radiobutton -label "ArcSec" \
	-variable marker($frame,$id,$format) -value arcsec \
	-command "$cb $frame $id"

    menu $mb.$name.wcs -tearoff 0 -selectcolor $menu(selectcolor)
    foreach l {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z} {
	$mb.$name.wcs add radiobutton -label "WCS $l" \
	    -variable marker($frame,$id,$coord) -value "wcs$l" \
	    -command "$cb $frame $id"
    }

    UpdateDistMenu $frame $mb.$name 0 0
}

proc AnnuliMenuMarkerDialog {frame mb id name} {
    global menu
    global marker

    menu $mb.$name -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.$name add radiobutton -label "Equal Distance" \
	-variable marker($frame,$id,$name) -value dist
    $mb.$name add radiobutton -label "Equal Area" \
	-variable marker($frame,$id,$name) -value area
}

proc FontMenuMarkerDialog {frame mb id} {
    global menu
    global marker

    menu $mb.font -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.font add radiobutton -label "Times" \
	-variable marker($frame,$id,font) -value times \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "Helvetica" \
	-variable marker($frame,$id,font) -value helvetica \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "Symbol" \
	-variable marker($frame,$id,font) -value symbol \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "Courier" \
	-variable marker($frame,$id,font) -value courier \
	-command "FontMarkerDialog $frame $id"
    $mb.font add separator
    $mb.font add radiobutton -label "9" \
	-variable marker($frame,$id,font,size) -value 9 \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "10" \
	-variable marker($frame,$id,font,size) -value 10 \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "12" \
	-variable marker($frame,$id,font,size) -value 12 \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "14" \
	-variable marker($frame,$id,font,size) -value 14 \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "18" \
	-variable marker($frame,$id,font,size) -value 18 \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "24" \
	-variable marker($frame,$id,font,size) -value 24 \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "30" \
	-variable marker($frame,$id,font,size) -value 30 \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "36" \
	-variable marker($frame,$id,font,size) -value 36 \
	-command "FontMarkerDialog $frame $id"
    $mb.font add separator
    $mb.font add radiobutton -label "Plain" \
	-variable marker($frame,$id,font,style) -value normal \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "Bold" \
	-variable marker($frame,$id,font,style) -value bold \
	-command "FontMarkerDialog $frame $id"
    $mb.font add radiobutton -label "Italic" \
	-variable marker($frame,$id,font,style) -value italic \
	-command "FontMarkerDialog $frame $id"
}

proc CommonMarkerDialog {frame id} {
    global ds9
    global marker

    set w ".marker$id"
    set mb ".mb$id"

    # create window

    set type [string toupper "[$frame get marker $id type]"]

    toplevel $w -colormap $ds9(main)
    wm title $w $type
    wm iconname $w $type
    wm group $w $ds9(top)

    $w configure -menu $mb

    # Dialog

    frame $w.basic -relief groove -borderwidth 2
    frame $w.basic.f
    frame $w.ref -relief groove -borderwidth 2
    frame $w.ref.f
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.basic.f $w.ref.f -anchor w -padx 4 -pady 4
    pack $w.basic $w.ref -fill x 
    pack $w.buttons -fill x -ipadx 4 -ipady 4

    # ID

    label $w.basic.f.idTitle -text "Id"
    label $w.basic.f.idValue -text "$id"

    # Text

    label $w.basic.f.textTitle -text "Text"
    entry $w.basic.f.textValue -textvariable marker($frame,$id,text) -width 45

    grid $w.basic.f.idTitle $w.basic.f.idValue -padx 4 -sticky w
    grid $w.basic.f.textTitle $w.basic.f.textValue -padx 4 -sticky w

    # Center

    label $w.ref.f.centerTitle -text "Center"
    entry $w.ref.f.centerX -textvariable marker($frame,$id,x) -width 13
    entry $w.ref.f.centerY -textvariable marker($frame,$id,y) -width 13
    label $w.ref.f.centerCoord -relief groove -width 9 -padx 4 \
	-textvariable marker($frame,$id,clabel)

    grid $w.ref.f.centerTitle $w.ref.f.centerX $w.ref.f.centerY \
	$w.ref.f.centerCoord -padx 4 -sticky w
}

proc ApplyMarkerDialog {frame id} {
    global marker

    $frame marker $id move to \
	$marker($frame,$id,system) $marker($frame,$id,sky) \
	$marker($frame,$id,x) $marker($frame,$id,y) 
    $frame marker $id text \{$marker($frame,$id,text)\}

    UpdateMarkerMenu
}

proc DeleteMarkerCBs {frame id} {
    $frame marker $id delete callback move MoveMarkerCB
    $frame marker $id delete callback text TextMarkerCB
    $frame marker $id delete callback color ColorMarkerCB
    $frame marker $id delete callback width LineWidthMarkerCB
    $frame marker $id delete callback property PropertyMarkerCB
    $frame marker $id delete callback font FontMarkerCB
}

proc DeleteMarkerDialog {frame id} {
    global marker

    set w ".marker$id"
    set mb ".mb$id"

    destroy $w
    destroy $mb

    unset marker($frame,$id,x)
    unset marker($frame,$id,y)
    unset marker($frame,$id,system)
    unset marker($frame,$id,clabel)
    unset marker($frame,$id,sky)
    unset marker($frame,$id,skyformat)

    unset marker($frame,$id,text)
    unset marker($frame,$id,font)
    unset marker($frame,$id,font,size)
    unset marker($frame,$id,font,style)
    unset marker($frame,$id,linewidth)

    unset marker($frame,$id,color)
    unset marker($frame,$id,edit)
    unset marker($frame,$id,move)
    unset marker($frame,$id,rotate)
    unset marker($frame,$id,delete)
    unset marker($frame,$id,fixed)
    unset marker($frame,$id,include)
    unset marker($frame,$id,source)
}

proc RotateMarkerDialog {frame id} {
    global marker

    $frame marker $id angle $marker($frame,$id,angle) $marker($frame,$id,system)
}

proc ColorMarkerDialog {frame id} {
    global marker

    $frame marker $id color $marker($frame,$id,color)
}

proc LineWidthMarkerDialog {frame id} {
    global marker

    $frame marker $id width $marker($frame,$id,linewidth)
}

proc PropertyMarkerDialog {frame id prop} {
    global marker

    $frame marker $id property $prop $marker($frame,$id,$prop)
}

proc FontMarkerDialog {frame id} {
    global marker

    $frame marker $id font \
	\"$marker($frame,$id,font) $marker($frame,$id,font,size) \
	$marker($frame,$id,font,style)\"
}

proc MoveMarkerCB {frame id} {
    global marker

    set center [$frame get marker $id center \
		    $marker($frame,$id,system) $marker($frame,$id,sky) \
		    $marker($frame,$id,skyformat)]
    set marker($frame,$id,x) [lindex $center 0]
    set marker($frame,$id,y) [lindex $center 1]
}

proc RotateMarkerCB {frame id} {
    global marker

    set marker($frame,$id,angle) \
	[$frame get marker $id angle $marker($frame,$id,system)]
}

proc ColorMarkerCB {frame id} {
    global marker

    set marker($frame,$id,color) [$frame get marker $id color]
}

proc LineWidthMarkerCB {frame id} {
    global marker

    set marker($frame,$id,linewidth) [$frame get marker $id width]
}

proc TextMarkerCB {frame id} {
    global marker

    set marker($frame,$id,text) [$frame get marker $id text]
}

proc FontMarkerCB {frame id} {
    global marker

    set f [$frame get marker $id font]

    set marker($frame,$id,font) [lindex $f 0]
    set marker($frame,$id,font,size) [lindex $f 1]
    set marker($frame,$id,font,style) [lindex $f 2]
}

proc PropertyMarkerCB {frame id} {
    global marker

    set marker($frame,$id,edit) [$frame get marker $id property edit]
    set marker($frame,$id,move) [$frame get marker $id property move]
    set marker($frame,$id,rotate) [$frame get marker $id property rotate]
    set marker($frame,$id,delete) [$frame get marker $id property delete]
    set marker($frame,$id,fixed) [$frame get marker $id property fixed]
    set marker($frame,$id,include) [$frame get marker $id property include]
    set marker($frame,$id,source) [$frame get marker $id property source]
}

proc CoordMarkerCB {frame id} {
    global marker
    set mb ".mb$id"

    AdjustCoord $frame marker($frame,$id,system)

    set marker($frame,$id,clabel) $marker($frame,$id,system)
    switch -- $marker($frame,$id,system) {
	image -
	physical -
	amplifier -
	detector {}
	default {
	    if [$frame has wcs $marker($frame,$id,system)] {
		if [$frame has wcs equatorial $marker($frame,$id,system)] {
		    set marker($frame,$id,clabel) $marker($frame,$id,sky)
		} else {
		    set name [$frame get wcs name $marker($frame,$id,system)]
		    if {$name != ""} {
			set marker($frame,$id,clabel) $name
		    }
		}
	    }
	}
    }
}

