/*
 * Handling of USB devices - currently this primary purpose
 * is to be able to do a "list usb" command.
 *
 * Copyright (C) 2002, Olaf Kirch <okir@lst.de>
 */

#include <sys/ioctl.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <ctype.h>
#include <linux/usbdevice_fs.h>
#include <linux/major.h>
#include "resmgrd.h"

#include <linux/usbdevice_fs.h>

typedef struct {
	unsigned int	bus;
	unsigned int	dev;
} usb_id_t;

typedef struct res_usb_name res_usb_name_t;
struct res_usb_name {
	res_name_t	base;
	usb_id_t	usb_id;
};

static res_name_t *	res_usb_parse_name(const char *);
static const char * 	res_usb_print_name(res_name_t *);
static void	 	res_usb_free_name(res_name_t *);
static int		res_usb_match(res_name_t *, res_device_t *);
static int		res_usb_open(res_name_t *, int);
static int		get_usb_id(const char *, usb_id_t *);
static const char *	get_usb_name(usb_id_t *);

res_family_t		res_family_usb = {
	"usb",
	0,			/* no special access flag for USB devs */
	res_usb_parse_name,
	res_usb_print_name,
	res_usb_free_name,
	res_usb_match,
	res_usb_open
};

/*
 * USB devices can be named
 * 	usb:/proc/bus/usb/xxx/yyy
 * 	usb:bus,dev
 * 	usb:bus,dev:/proc/bus/usb/xxx/yyy
 */
res_name_t *
res_usb_parse_name(const char *name)
{
	res_usb_name_t	*sp;
	usb_id_t	usb_id;

	if (name[0] == '/') {
		if (get_usb_id(name, &usb_id) < 0)
			return 0;
	} else {
		usb_id.bus = strtoul(name, (char **) &name, 10);
		if (*name != ',')
			return NULL;
		usb_id.dev = strtoul(name+1, (char **) &name, 10);
		if (*name != ':' && *name != '\0')
			return NULL;
	}

	sp = (res_usb_name_t *) calloc(1, sizeof(*sp));
	sp->base.ops = &res_family_usb;
	sp->usb_id = usb_id;

	return (res_name_t *) sp;
}

void
res_usb_free_name(res_name_t *np)
{
	free(np);
}

const char *
res_usb_print_name(res_name_t *np)
{
	static char	namebuf[256];
	res_usb_name_t	*sp = (res_usb_name_t *) np;

	snprintf(namebuf, sizeof(namebuf), "usb:%u,%u:%s",
		sp->usb_id.bus, sp->usb_id.dev,
		get_usb_name(&sp->usb_id));
	return namebuf;
}

int
res_usb_match(res_name_t *np, res_device_t *dev)
{
	res_usb_name_t	*sp = (res_usb_name_t *) np;
	usb_id_t	dev_id;

	if (get_usb_id(dev->name, &dev_id) < 0)
		return 0;

	return !memcmp(&sp->usb_id, &dev_id, sizeof(usb_id_t));
}

int
res_usb_open(res_name_t *np, int flags)
{
	res_usb_name_t	*sp = (res_usb_name_t *) np;
	int		fd = -1;

	if ((np = res_name_parse(get_usb_name(&sp->usb_id))) != NULL)
		fd = res_name_open(np, flags);
	res_name_free(np);
	return fd;
}

int
get_usb_id(const char *name, usb_id_t *id)
{
	const unsigned int	n = sizeof(_PATH_PROC_BUS_USB)-1;
	struct stat		stb;

	debug("get_usb_id(%s)", name);
	if (stat(name, &stb) >= 0 && S_ISCHR(stb.st_mode)) {
		struct usbdevfs_connectinfo ci;
		int fd;

		switch (major(stb.st_rdev)) {
		case USB_ACM_MAJOR:
		case USB_ACM_AUX_MAJOR:
		case USB_CHAR_MAJOR:
			fd = open(name, O_RDWR|O_NDELAY);
			if (fd >= 0
			 && ioctl(fd, USBDEVFS_CONNECTINFO, &ci) >= 0) {
				log("usb dev %x\n", ci.devnum);
			} else log("ioctl: %m");
			close(fd);
			break;
		default:
			break;
		}
	}

	if (strncmp(name, _PATH_PROC_BUS_USB, n) || name[n] != '/')
		return -1;
	name += n;
	while (*name == '/')
		name++;
	id->bus = strtoul(name, (char **) &name, 10);
	while (*name == '/')
		name++;
	id->dev = strtoul(name, (char **) &name, 10);
	if (*name != '\0')
		return -1;
	return 0;
}

const char *
get_usb_name(usb_id_t *id)
{
	static char	namebuf[256];

	snprintf(namebuf, sizeof(namebuf), "%s/%03u/%03u",
		_PATH_PROC_BUS_USB, id->bus, id->dev);
	return namebuf;
}
