/*
 * openpgp.h - OpenPGP base
 * Copyright (C) 2003  Justin Karneges
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef CS_OPENPGP_H
#define CS_OPENPGP_H

#include<qobject.h>
#include<qstring.h>
#include<qvaluelist.h>
#include<qdatetime.h>
#include<qpair.h>

namespace OpenPGP
{
	enum { None, Encrypt, Decrypt, Sign, Verify };
	enum { VerifyGood, VerifyBad, VerifyNoKey, VerifyError };
	class Engine;

	class Key
	{
	public:
		Key();
		~Key();

		const QString & keyID() const;
		const QString & userID() const;
		void setKeyID(const QString &);
		void setUserID(const QString &);

	private:
		QString v_keyID;
		QString v_userID;
	};

	typedef QValueList<Key> KeyList;

	class Request : public QObject
	{
		Q_OBJECT
	public:
		Request(Engine *);
		~Request();

		int type() const;

		void encrypt(const QByteArray &in, const QStringList &keys);
		void decrypt(const QString &in);
		void sign(const QByteArray &in, const QString &keyID);
		void verify(const QByteArray &in, const QString &sig);

		void submitPassphrase(const QString &);

		QString encrypted() const;
		QByteArray decrypted() const;
		QString signature() const;
		QString keyID() const;
		QDateTime timestamp() const;
		int verifyResult() const;

		bool badPassphrase() const;

	signals:
		void finished(bool);
		void needPassphrase();

	private:
		class Private;
		Private *d;

		friend class Engine;
		void op_encryptFinished(bool, const QString &);
		void op_decryptFinished(bool, const QByteArray &);
		void op_signFinished(bool, const QString &);
		void op_verifyFinished(int, const QString &, const QDateTime &);
		void op_needPassphrase();
		void op_setBadPassphrase(bool);
	};

	class Engine : public QObject
	{
		Q_OBJECT
	public:
		Engine(QObject *parent=0);
		virtual ~Engine()=0;

		virtual bool checkAvailability()=0;
		virtual QString id() const=0;
		virtual QString name() const=0;

		virtual void init()=0;
		virtual KeyList secretKeys() const=0;
		virtual KeyList publicKeys() const=0;

	signals:
		void initFinished(bool, const QString &);
		void keysUpdated();

	protected:
		friend class Request;
		virtual void encrypt(Request *, const QByteArray &, const QStringList &keys)=0;
		virtual void decrypt(Request *, const QString &)=0;
		virtual void sign(Request *, const QByteArray &, const QString &keyID)=0;
		virtual void verify(Request *, const QByteArray &, const QString &sig)=0;
		virtual void submitPassphrase(Request *, const QString &)=0;
		void encryptFinished(Request *, bool, const QString &);
		void decryptFinished(Request *, bool, const QByteArray &);
		void signFinished(Request *, bool, const QString &);
		void verifyFinished(Request *, int, const QString &, const QDateTime &);
		void needPassphrase(Request *);
		void setBadPassphrase(Request *, bool);
	};

	typedef QPair<QString,QString> EngineItem;

	QValueList<EngineItem> getAllEngines();
	QValueList<EngineItem> getAvailableEngines();
	Engine * createEngine(const QString &);

	QString stripHeaderFooter(const QString &);
	QString addHeaderFooter(const QString &, int);
}

#endif
