/*
** Copyright (C) 2000 Alan McIvor <alan@mcivor.gen.nz>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include "project.h"
#include "reporter.h"


void Reporter::HTMLCostReport(Project *project, FILE *f)
{
    fprintf(f,"<html>\n<body>\n\n");

    fprintf(f, "<h1>Project Costing</h1>\n");
    fprintf(f, "<table BORDER=1 CELLPADDING=2>\n");

    fprintf(f,
	    "<tr><th>ID</th><th>Task</th><th>Personnel</th><th>Equipment</th><th>Total</th></tr>\n");

    double grand_total = 0.0;

    for ( Project::TPLCI pt = project->beginTaskList() ; 
	  pt != project->endTaskList() ; 
	  pt++ )
    {
	TASK * t = *pt;
	if ( t->isVacation() )
	    continue;			/* vacation */

	double time_cost = t->timeCost();
	double item_cost = t->itemCost();
	double total = time_cost + item_cost;

	fprintf(f,
		"<tr><td>%s</td><td><b>%s</bf></td><td align=right>%6.2f</td>\n",
		t->id(),
		t->name(),
		time_cost);

	if ( t->numItems() == 0 )
	    fprintf(f, "<td>&nbsp;</td>\n");
	else
	{
	    fprintf(f, "<td><table>\n");
	    for ( Project::IPLCI pi = t->begin_items() ; pi != t->end_items() ; pi++ )
		fprintf(f,
			"<tr><td>%s</td><td align=right>%6.2f</td></tr>\n",
			(*pi)->description().c_str(),
			(*pi)->cost());
	    fprintf(f,
		    "<tr><td><em>Equipment Total</em></td><td align=right><em>%6.2f</em></td></tr>\n",
		    item_cost);
	    fprintf(f, "</table></td>\n");
	}
	
	fprintf(f,
		"<td align=right><b>%6.2f</b></td></tr>\n",
		total);
	
	grand_total += total;
    }

    fprintf(f,
	    "<tr><td>&nbsp;</td><td>&nbsp;</td><td>&nbsp;</td><td>&nbsp;</td><td>&nbsp;</td></tr>\n");

    for ( Project::IPLCI pi = project->begin_items() ; pi != project->end_items() ; pi++ )
    {
	fprintf(f, 
		"<tr><td>&nbsp;</td><td colspan=3>%s</td><td align=right>%6.2f</td></tr>\n",
		(*pi)->description().c_str(),
		(*pi)->cost());

	grand_total += (*pi)->cost();
    }
    
    fprintf(f,
	    "<tr><td>&nbsp;</td><td colspan=3><b>Total</b></td><td align=right><b>%6.2f</b></td></tr>\n",
	    grand_total);
	
    fprintf(f, "</table>\n");

    fprintf(f,"\n\n</body>\n</html>\n");
}


void Reporter::TeXCostReport(Project *project, FILE *f)
{
    fprintf(f, "\\section{Project Costing}\n\n");
    fprintf(f,"\\begin{supertabular}{|l|l|l|r|}\n");

    fprintf(f, "\\hline\n");
    fprintf(f, "\\textbf{ID} & \\textbf{Task} & \\textbf{Item} & \\textbf{Cost} \\\\\n");
    fprintf(f, "\\hline\n");

    double grand_total = 0.0;

    for ( Project::TPLCI pt = project->beginTaskList() ; 
	  pt != project->endTaskList() ; 
	  pt++ )
    {
	TASK * t = *pt;
	if ( t->isVacation() )
	    continue;			/* vacation */

	double time_cost = t->timeCost();
	double item_cost = t->itemCost();
	double total = time_cost + item_cost;

	fprintf(f,
		"%s & \\textbf{%s} & Personnel & %6.2f \\\\\n",
		t->id(),
		t->name(),
		time_cost);

	if ( t->numItems() != 0 )
	{
	    for ( Project::IPLCI pi = t->begin_items() ; pi != t->end_items() ; pi++ )
		fprintf(f,
			" & & %s & %6.2f \\\\\n",
			(*pi)->description().c_str(),
			(*pi)->cost());
	}
	
	grand_total += total;
    }

    if ( project->numItems() != 0 )
    {
	fprintf(f, "\\hline\n");
	for ( Project::IPLCI pi = project->begin_items() ; 
	      pi != project->end_items() ; pi++ )
	{
	    fprintf(f, 
		    " & & %s & %6.2f \\\\ \n",
		    (*pi)->description().c_str(),
		    (*pi)->cost());
	    
	    grand_total += (*pi)->cost();
	}
    }
    
    fprintf(f, "\\hline\n");
    fprintf(f,
	    " & \\multicolumn{2}{l|}{\\textbf{Total}} & \\textbf{%6.2f} \\\\ \n",
	    grand_total);
	
    fprintf(f, "\\hline\n");
    fprintf(f, "\\end{supertabular}\n");
}


void Reporter::CostReport(Project *project, char *filename, REPORTTYPE rep_type)
{
    FILE *f;

    project->SortTasks(tg_sortbyresource);

    f = OpenOutputFile(filename);

    switch ( rep_type )
    {
	case HTML_REPORT:
	    HTMLCostReport(project, f);
	    break;

	case TEX_REPORT:
	    TeXCostReport(project, f);
	    break;

	case TXT_REPORT:
	default:
	    Error("Unhandled rep_type in CostReport()\n");
    }
    
    if ( f != stdout )
	fclose(f);
}

