#import "MACatalogs.h"
#import <MySQLToolsCommon/MSchemaDataSource.h>
#import <MySQLToolsCommon/NSString_extras.h>
#import <MySQLToolsCommon/MTreeDataSource.h>
#import <MySQLToolsCommon/MMySQLDispatcher.h>
#import <MySQLToolsCommon/MTableEditor.h>
#import <MySQLToolsCommon/MDialogs.h>
#import <MySQLToolsCommon/myxutil.h>
#import <MySQLToolsCommon/mxUtils.h>
#import "MATableMaintenanceController.h"

@interface MACatalogIndexItem : MTreeItem
{
  @public
  MYX_TABLE_INDEX *index;
  MYX_TABLE_INDEX_COLUMN *column;
}
- (id)initRoot:(MYX_SCHEMA_TABLE_STATUS*)item;
@end

@implementation MACatalogIndexItem
- (id)initRoot:(MYX_SCHEMA_TABLE_STATUS*)item
{
  self= [super initWithTag:'C' repr:nil];
  if (self)
  {
    unsigned int i, j, k;
    for (i= 0; i < item->schema_tables_num; i++)
    {
      MYX_TABLE_STATUS *table= item->schema_tables+i;
      MACatalogIndexItem *tit=nil, *cit;
      for (j= 0; j < table->indexes_num; j++)
      {
        MYX_TABLE_INDEX *idx= table->indexes+j;
        if (strcmp((char*)idx->table_name, (char*)table->table_name)==0)
        {
          tit= [[MACatalogIndexItem alloc] initWithTag:'I' repr:[NSString stringWithUTF8String:table->table_name]];
          tit->index= idx;
          [self addChild:tit];
          [tit release];
          
          for (k= 0; k < idx->index_columns_num; k++)
          {
            cit= [[MACatalogIndexItem alloc] initWithTag:'C' repr:[NSString stringWithUTF8String:idx->key_name]];
            cit->column= idx->index_columns+k;
            [tit addChild:cit];
            [cit release];
          }
        }
      }
    }
  }
  return self;
}

- (id)valueForIdentifier:(NSString*)ident
{
  if ([self tag] == 'I')
  {
    if ([ident isEqualToString:@"column"])
    {
      return [NSString stringWithUTF8String:index->table_name];
    }
    else if ([ident isEqualToString:@"name"])
    {
      return [NSString stringWithUTF8String:index->key_name];
    }
    else if ([ident isEqualToString:@"type"])
    {
      return [NSString stringWithUTF8String:index->index_type];
    }
    else if ([ident isEqualToString:@"unique"])
    {
      return index->unique?@"UNIQUE":@"-";
    }
    else if ([ident isEqualToString:@"null"])
    {
      return index->not_null?@"NOT NULL":@"NULL";
    }
  }
  else
  {
    if ([ident isEqualToString:@"column"])
    {
      return [NSString stringWithUTF8String:column->column_name];
    }
    else if ([ident isEqualToString:@"seq"])
    {
      return [NSString stringWithUTF8String:column->seq_in_index];
    }
    else if ([ident isEqualToString:@"collation"])
    {
      return *column->collation=='A'?@"Ascending":@"";
    }
  }
  return @"";
}

@end


@interface MACatalogs(Private)
- (void)tableStatusArrived:(id)arg result:(void*)res;

- (MYX_SCHEMA*)selectedSchema;
@end


@implementation MACatalogs(Private)

- (void)tableStatusArrived:(id)arg result:(void*)res
{
  long long index_length, data_length, row_count;
  unsigned int i;

  if (_tableStatus)
    myx_free_schema_table_status(_tableStatus);
  _tableStatus= (MYX_SCHEMA_TABLE_STATUS*)res;
  [tablesTable reloadData];

  if (_tableStatus)
    [[indexOutline dataSource] setRoot:[[[MACatalogIndexItem alloc] initRoot:_tableStatus] autorelease]];
  else
    [[indexOutline dataSource] setRoot:nil];
  [indexOutline reloadData];

  // calculate summaries
  index_length= 0;
  data_length= 0;
  row_count= 0;
  if (_tableStatus)
  {
    for (i= 0; i < _tableStatus->schema_tables_num; i++)
    {
      data_length+= strtoll((char*)_tableStatus->schema_tables[i].data_length?:"0", NULL, 0);
      index_length+= strtoll((char*)_tableStatus->schema_tables[i].index_length?:"0", NULL, 0);
      row_count+= strtoll((char*)_tableStatus->schema_tables[i].rows?:"0", NULL, 0);
    }
  }
  [[tableStatsMatrix cellWithTag:111] setStringValue:[NSString stringWithFormat:@"No. of Tables: %i", _tableStatus?_tableStatus->schema_tables_num:0]];
  [[tableStatsMatrix cellWithTag:112] setStringValue:[NSString stringWithFormat:@"No. of Rows: %lli", row_count]];
  [[tableStatsMatrix cellWithTag:113] setStringValue:[NSString stringWithFormat:@"Data Length: %@B", [NSString stringWithMultNumber:data_length]]];
  [[tableStatsMatrix cellWithTag:114] setStringValue:[NSString stringWithFormat:@"Index Length: %@B", [NSString stringWithMultNumber:index_length]]];
}

- (MYX_SCHEMA*)selectedSchema
{
  int row= [schemataOutline selectedRow];
  MSchemaItem *item= row >= 0 ? [schemataOutline itemAtRow:row] : nil;
  if (!item || item->type != MSchemaItemType) 
    return nil;
  return [item schema];
}

@end


@implementation MACatalogs

- (IBAction)toggleTableDetails:(id)sender
{
  NSRect trect= (sender != nil ? [[topBox window] frame] : [topBox frame]);
  float bheight= [tableDetailsTab frame].size.height;
  
  // change autoresizing, so that the layout stays the way we want as we resize the window
  [tableStatsMatrix setAutoresizingMask:NSViewWidthSizable|NSViewMinYMargin];
  [tableDetailToggleButton setAutoresizingMask:NSViewWidthSizable|NSViewMinYMargin];
  [[topBox viewWithTag:14] setAutoresizingMask:NSViewWidthSizable|NSViewMinYMargin];
  [tableScrollView setAutoresizingMask:NSViewWidthSizable|NSViewMinYMargin];
  [[tableDetailsTab superview] setAutoresizingMask:NSViewWidthSizable|NSViewHeightSizable];
  
  if ([tableDetailToggleButton state] == NSOnState)
  {    
    if (sender == nil)
    {
      trect.size.height+= bheight;
      [topBox setFrame:trect];
    }
    else
    {      
      trect= [tableScrollView frame];
      trect.size.height -= bheight;
      trect.origin.y += bheight;
      [tableScrollView setFrame:trect];
      
      trect= [tableStatsMatrix frame];
      trect.origin.y += bheight;
      [tableStatsMatrix setFrame:trect];
      
      trect= [[topBox viewWithTag:14] frame];
      trect.origin.y += bheight;
      [[topBox viewWithTag:14] setFrame:trect];

      trect= [[tableDetailsTab superview] frame];
      trect.size.height += bheight;
      [[tableDetailsTab superview] setFrame:trect];
      
      trect= [tableDetailToggleButton frame];
      trect.origin.y += bheight;
      [tableDetailToggleButton setFrame:trect];
      
      [topBox setNeedsDisplay:YES];
    }
  }
  else
  {
    if (sender == nil)
    {
      trect.size.height-= bheight;
      [topBox setFrame:trect];
    }
    else
    {
      trect= [tableScrollView frame];
      trect.size.height += bheight;
      trect.origin.y -= bheight;
      [tableScrollView setFrame:trect];
      
      trect= [tableStatsMatrix frame];
      trect.origin.y -= bheight;
      [tableStatsMatrix setFrame:trect];
      
      trect= [[topBox viewWithTag:14] frame];
      trect.origin.y -= bheight;
      [[topBox viewWithTag:14] setFrame:trect];
      
      trect= [[tableDetailsTab superview] frame];
      trect.size.height -= bheight;
      [[tableDetailsTab superview] setFrame:trect];
      
      trect= [tableDetailToggleButton frame];
      trect.origin.y -= bheight;
      [tableDetailToggleButton setFrame:trect];
      
      [topBox setNeedsDisplay:YES];
    }
  }  
  
  [[tableDetailsTab superview] setAutoresizingMask:NSViewWidthSizable];
  [tableDetailToggleButton setAutoresizingMask:NSViewWidthSizable|NSViewMaxYMargin|NSViewMaxXMargin];
  [[topBox viewWithTag:14] setAutoresizingMask:NSViewWidthSizable|NSViewMaxYMargin|NSViewMaxXMargin];
  [tableScrollView setAutoresizingMask:NSViewWidthSizable|NSViewHeightSizable];
  [tableStatsMatrix setAutoresizingMask:NSViewWidthSizable|NSViewMaxYMargin];
}


- (void)openTableMaintenance:(int)operation
{
  MATableMaintenanceController *mnt;
  NSMutableArray *tables= [NSMutableArray arrayWithCapacity:1];
  NSIndexSet *rows= [tablesTable selectedRowIndexes];
  unsigned int i;
  MYX_SCHEMA *schema= [self selectedSchema];
  if (!schema)
    return;

  for (i= [rows firstIndex]; i <= [rows lastIndex]; i= [rows indexGreaterThanIndex:i])
  {
    [tables addObject:[NSString stringWithFormat:@"%s.%s",schema->schema_name,_tableStatus->schema_tables[i].table_name]];
  }
  
  if ([tables count] == 0)
    return;

  mnt= [[MATableMaintenanceController alloc] initWithWindowNibName:@"TableMaintenance"];
  [mnt loadWindow];
  [mnt runWithTables:tables dispatcher:[_owner dispatcher] operation:operation];

  [mnt close];

  [mnt release];
}

- (IBAction)checkTable:(id)sender
{
  [self openTableMaintenance:'C'];
}

- (IBAction)repairTable:(id)sender
{
  [self openTableMaintenance:'R'];
}

- (IBAction)optimizeTable:(id)sender
{
  [self openTableMaintenance:'O'];
}

- (IBAction)createTable:(id)sender
{
  MYX_SCHEMA *schema= [self selectedSchema];
  MTableEditor *editor;
  
  if (!schema)
    return;
  
  editor= [[MTableEditor alloc] init];
  
  [editor setConnection:[_owner mysql]];
  [editor setCatalogs:[_owner catalogList]];
  [editor showNewTableForCatalog:NSStr(schema->catalog_name)
                          schema:NSStr(schema->schema_name)];
  
  [editor setReleaseOnClose:YES];
}

- (IBAction)editTable:(id)sender
{
  MYX_SCHEMA *schema= [self selectedSchema];
  int row= [tablesTable selectedRow];
  
  if (!schema)
    return;

  if (row >= 0)
  {
    MTableEditor *editor= [[MTableEditor alloc] init];
    MYX_TABLE_STATUS *ts= _tableStatus->schema_tables+row;

    [editor setConnection:[_owner mysql]];
    [editor setCatalogs:[_owner catalogList]];
    [editor showTable:NSStr((char*)ts->table_name) catalog:NSStr(schema->catalog_name)
               schema:NSStr(schema->schema_name)];
    
    [editor setReleaseOnClose:YES];
  }
}


static void *dropTables(MYSQL *mysql, void *schema, void *tables)
{
  unsigned int i, c= [(NSArray*)tables count];
  
  for (i= 0; i < c; i++)
  {
    MYX_LIB_ERROR error;
    NSString *query= [[NSString alloc] initWithFormat: @"DROP TABLE `%s`.`%@`",
      (char*)schema, [(NSArray*)tables objectAtIndex:i]];
    
    myx_query_execute_direct(mysql, [query UTF8String], &error);
    [query release];

    if (error != MYX_NO_ERROR)
    {
      return [[NSString alloc] initWithFormat: @"Error dropping table %@: %@", 
        [(NSArray*)tables objectAtIndex:i], MXGetErrorString(error)];
    }
  }
  return nil;
}


- (void)dropTablesAlertDidEnd:(NSAlert *)sheet 
                   returnCode:(int)returnCode
                  contextInfo:(void *)contextInfo
{
  MYX_SCHEMA *schema= [self selectedSchema];
  NSString *error;
  
  [[sheet window] orderOut:self];
  
  [(id)contextInfo autorelease];
  
  if (!schema || returnCode != NSAlertDefaultReturn)
    return;
  
  error= [[_owner dispatcher] performCallback: dropTables
                                     argument: schema->schema_name
                                     argument: (id)contextInfo 
                                waitForWindow: [topBox window]
                                      message: @"Dropping tables..."];
  if (!error)
  {
    NSRunAlertPanel(@"Error", error, nil, nil, nil);
  }
  
  [tablesTable reloadData];
}


- (IBAction)dropTable:(id)sender
{
  NSMutableArray *tables= [[NSMutableArray array] retain];
  NSIndexSet *indexes= [tablesTable selectedRowIndexes];
  unsigned int i;
  
  for (i= [indexes firstIndex]; i != NSNotFound; i= [indexes indexGreaterThanIndex: i])
  {
    [tables addObject: [NSString stringWithUTF8String:_tableStatus->schema_tables[i].table_name]];
  }
  
  
  if ([tables count] == 0)
    return;
  
  {  
    NSAlert *alert = [NSAlert alertWithMessageText:@"Drop Selected Tables"
                                     defaultButton:@"Drop Tables" 
                                   alternateButton:@"Cancel" 
                                       otherButton:nil
                         informativeTextWithFormat:@"The following tables will be dropped and all data contained in them will be permanently lost!\n\n %@",
      [tables componentsJoinedByString:@", "]];
    
    [alert beginSheetModalForWindow:[topBox window] modalDelegate:self 
                     didEndSelector:@selector(dropTablesAlertDidEnd:returnCode:contextInfo:) 
                        contextInfo:tables];
  }
}


- (IBAction)refreshTables:(id)sender
{
  MYX_SCHEMA *schema= [self selectedSchema];
  if (schema)
  {
	void *res;
    res=[[_owner dispatcher] performCallback:(void*(*)(MYSQL*,void*,void*))myx_get_schema_table_status
									argument:schema->catalog_name
									argument:schema->schema_name
							   waitForWindow:[_owner window]
									 message:[NSString stringWithFormat:@"Fetching Tables from '%s'...",schema->schema_name]];
	
    //                    finishedSelector:@selector(tableStatusArrived:result:)
      //                          argument:nil
        //                          target:self];
	[self tableStatusArrived:nil result:res];
  }
  else
  {
    [self tableStatusArrived:nil result:NULL];
  }
}

- (IBAction)searchCatalogs:(id)sender
{
  [_schemaDS performSearch:[sender stringValue]];
  [schemataOutline reloadData];
}

+ (NSImage*)icon
{
  return [NSImage imageNamed:@"OSX-Icons_51.png"];
}

+ (NSString*)label
{
  return @"Catalogs";
}

+ (NSString*)toolTip
{
  return @"Catalogs in the connected MySQL server.";
}

- (id)initWithOwner: (id<MAdministratorProtocol>)owner
{
  self= [super initWithNibFile: @"Catalogs" panelOwner: owner];
  if (self)
  {
    if (![[NSUserDefaults standardUserDefaults] boolForKey:@"CatalogDetailsShown"])
      [self toggleTableDetails:nil];
    _defaultFrame= [[self topView] frame];
    
    _tableIcon= [NSImage imageNamed:@"16x16_Table.png"];
    _indexIcon= [NSImage imageNamed:@"16x16_Index.png"];
    _columnIcon= [NSImage imageNamed:@"16x16_Field.png"];

    [tablesTable setDoubleAction:@selector(editTable:)];
    [tablesTable setTarget:self];
    [indexOutline setDataSource:[[MTreeDataSource alloc] init]];
	
	[tablesTable sizeLastColumnToFit];
	[indexOutline sizeLastColumnToFit];
	[schemataOutline sizeLastColumnToFit];
  }
  return self;
}

- (void)dealloc
{
  [_tableIcon release];
  if (_tableStatus)
    myx_free_schema_table_status(_tableStatus);
  [tableDetailsTab release];
  [super dealloc];
}

- (void)awakeFromNib
{
  [tableDetailsTab retain];
  
  [[addSchemaButton cell] setImageDimsWhenDisabled:NO];
  [addSchemaButton setImage:[NSImage imageNamed:@"add_normal.png"]];
  [addSchemaButton setAlternateImage:[NSImage imageNamed:@"add_pressed.png"]];

  [[delSchemaButton cell] setImageDimsWhenDisabled:NO];
  [delSchemaButton setImage:[NSImage imageNamed:@"del_disabled.png"]];
  [delSchemaButton setAlternateImage:[NSImage imageNamed:@"del_pressed.png"]];
  [delSchemaButton setEnabled:NO];
}


- (void)didShow
{
  if (![schemataOutline dataSource])
  {
    id ds= [_owner sharedSchemaDS];
	_schemaDS= [[MFilteredSchemaDataSource alloc] initWithDataSource:ds];
    int i, count= [_schemaDS outlineView:schemataOutline numberOfChildrenOfItem:nil];

    [schemataOutline setDataSource:_schemaDS];

    for (i= 0; i < count; i++)
      [schemataOutline expandItem:[_schemaDS outlineView:schemataOutline child:i ofItem:nil]
                   expandChildren:YES];
  }
}


- (IBAction)addSchema:(id)sender
{
  MStringRequestSheet *sheet= [MStringRequestSheet sheetWithTitle:@"Create Schema"
														   labels:[NSArray arrayWithObject:@"New Schema Name:"]];
  NSArray *name;
	
  if ((name= [sheet runModal:[_owner window]]))
  {
	char *query;
	MYX_LIB_ERROR error;

	query= g_strdup_printf("create database `%s`", [[name objectAtIndex:0] UTF8String]);
	
	[[_owner dispatcher] performCallback: (void*(*)(MYSQL*,void*,void*))myx_query_execute_direct
								argument: query
								argument: &error
						   waitForWindow: [_owner window]
								 message: @"Creating schema..."];
	g_free(query);
	
	if (error!=MYX_NO_ERROR)
	{
	  MXRunAlertPanelWithError(@"Error",@"Error creating schema.",error);
	}
	[_owner refreshSchemata];
	[schemataOutline reloadData];
  }
}


- (void)dropSchemaAlertDidEnd:(NSAlert*)sheet
                   returnCode:(int)rc
                  contextInfo:(void*)context
{
  char *query;
  MYX_LIB_ERROR error;
  
  [[sheet window] orderOut:self];
  
  if (rc != NSAlertDefaultReturn)
    return;
  
  query= g_strdup_printf("drop database `%s`",
                         [(MSchemaItem*)context schema]->schema_name);
  
  [[_owner dispatcher] performCallback: (void*(*)(MYSQL*,void*,void*))myx_query_execute_direct
                              argument: query
                              argument: &error
                         waitForWindow: [topBox window]
                               message: @"Dropping schema..."];
  
  g_free(query);
  if (error!=MYX_NO_ERROR)
  {
    MXRunAlertPanelWithError(@"Error",@"Error dropping schema.",error);
  }

  [_owner refreshSchemata];
  [schemataOutline reloadData];  
}


- (IBAction)delSchema:(id)sender
{
  MSchemaItem *item= [schemataOutline itemAtRow:[schemataOutline selectedRow]];
  
  NSAlert *alert = [NSAlert alertWithMessageText:@"Drop Selected Schema"
                                   defaultButton:@"Drop Schema" 
                                 alternateButton:@"Cancel" 
                                     otherButton:nil
                       informativeTextWithFormat:@"The schema '%@' will be dropped and all data contained in it will be permanently lost!",
    [item repr]];
  
  [alert beginSheetModalForWindow:[topBox window] modalDelegate:self 
                   didEndSelector:@selector(dropSchemaAlertDidEnd:returnCode:contextInfo:) 
                      contextInfo:item];
}


//============================= Delegate ================================

- (BOOL)validateMenuItem:(id <NSMenuItem>)menuItem
{
  switch ([[menuItem menu] indexOfItem:menuItem])
  {
    case 1:
    case 2:
    case 3:
    case 6:
    case 8:
      if ([tablesTable selectedRow] < 0)
        return NO;
      else
        return YES;
    default:
      return YES;
  }
}


- (void)tableViewSelectionDidChange:(NSNotification *)aNotification
{
  if ([aNotification object] == tablesTable)
  {
    int row= [tablesTable selectedRow];
    MYX_TABLE_STATUS *ts;
    MYX_TABLE_STATUS zero;
    
    if (row < 0)
    {
      memset(&zero, 0, sizeof(zero));
      ts= &zero;      
    }
    else
    {
      ts= _tableStatus->schema_tables+row;
    }
    
    [tableStatusText setStringValue:[NSString stringWithFormat:@"%s\n%s\n%s\n%s\n%s",
                                        (char*)ts->table_type?:"",
                                        (char*)ts->row_format?:"",
                                    (char*)ts->auto_increment?:"",
                                    (char*)ts->create_options?:"",
                                           (char*)ts->comment?:""]];
    
    [tableStatsDataText setStringValue:[NSString stringWithFormat:@"%s\n%s\n%s\n%s\n%s\n%s",
                                                 (char*)ts->rows?:"",
                                       (char*)ts->avg_row_length?:"",
                                          (char*)ts->data_length?:"",
                                      (char*)ts->max_data_length?:"",
                                         (char*)ts->index_length?:"",
                                            (char*)ts->data_free?:""]];
    
    [tableStatsTimeText setStringValue:[NSString stringWithFormat:@"%s\n%s\n%s",
                                          (char*)ts->create_time?:"",
                                          (char*)ts->update_time?:"",
                                           (char*)ts->check_time?:""]];
  }
}


- (void)tableView:(NSTableView *)aTableView
  willDisplayCell:(id)aCell
   forTableColumn:(NSTableColumn *)aTableColumn
              row:(int)rowIndex
{
  if (aTableView == tablesTable && [[aTableColumn identifier] isEqualToString:@"name"])
  {
    [aCell setImage:_tableIcon];
  }
}

- (void)outlineView:(NSOutlineView *)outlineView 
    willDisplayCell:(id)cell 
     forTableColumn:(NSTableColumn *)tableColumn 
               item:(id)item
{
  if (outlineView == schemataOutline)
  {
	if ([item respondsToSelector:@selector(icon)])
	  [cell setImage:[item icon]];
	else
	  [cell setImage:nil];
  }
  else if (outlineView == indexOutline && [[tableColumn identifier] isEqualToString:@"column"])
  {
    if ([outlineView levelForItem:item]==0)
      [cell setImage:_indexIcon];
    else
      [cell setImage:_columnIcon];
  }
}

- (void)outlineViewSelectionDidChange:(NSNotification *)notification
{
  NSOutlineView *oview= [notification object];

  if (oview == schemataOutline)
  {
    int row= [schemataOutline selectedRow];
    if (row>=0 && [schemataOutline levelForRow:row]>0)
    {
      [actionButton setEnabled:YES];
      [delSchemaButton setEnabled:YES];
      [delSchemaButton setImage:[NSImage imageNamed:@"del_normal.png"]];
    }
    else
    {
      [actionButton setEnabled:NO];
      [delSchemaButton setEnabled:NO];
      [delSchemaButton setImage:[NSImage imageNamed:@"del_disabled.png"]];
    }
    
    [self refreshTables:nil];
  }
}


//============================= DataSource ================================

- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
  if (aTableView == tablesTable)
    return _tableStatus ? _tableStatus->schema_tables_num : 0;
  else
    return 0;
}

- (id)tableView:(NSTableView *)aTableView 
objectValueForTableColumn:(NSTableColumn *)aTableColumn 
            row:(int)rowIndex
{
  if (aTableView == tablesTable)
  {
    NSString *ident= [aTableColumn identifier];
    MYX_TABLE_STATUS *table= _tableStatus->schema_tables+rowIndex;
    
    if ([ident isEqualToString:@"name"])
    {
      return [NSString stringWithUTF8String:table->table_name];
    }
    else if ([ident isEqualToString:@"type"])
    {
      return [NSString stringWithUTF8String:(char*)table->table_type?:""];
    }
    else if ([ident isEqualToString:@"format"])
    {
      return [NSString stringWithUTF8String:(char*)table->row_format?:""];
    }
    else if ([ident isEqualToString:@"rows"])
    {
      return [NSString stringWithUTF8String:(char*)table->rows?:""];
    }
    else if ([ident isEqualToString:@"data"])
    {
      return [NSString stringWithMultNumber:strtoll((char*)table->data_length?:"0",NULL,0)];
    }
    else if ([ident isEqualToString:@"index"])
    {
      return [NSString stringWithMultNumber:strtoll((char*)table->index_length?:"0",NULL,0)];
    }
    else if ([ident isEqualToString:@"update"])
    {
      return [NSString stringWithUTF8String:(char*)table->update_time?:""];
    }
  }
  return nil;
}

@end
