#!/usr/bin/perl
#
#    sapdbc2depot - updates sapdbc depot files, if neccessary.
#
#    @(#)sapdbc2depot     2003-11-27
#
#    UliJ, SAP AG
#
#    ========== licence begin GPL
#    Copyright (C) 2002 SAP AG
#
#    This library is free software; you can redistribute it and/or
#    modify it under the terms of the GNU Lesser General Public
#    License as published by the Free Software Foundation; either
#    version 2.1 of the License, or (at your option) any later version.
#
#    This library is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    Lesser General Public License for more details.
#
#    You should have received a copy of the GNU Lesser General Public
#    License along with this library; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#    ========== licence end
#
##############################################################################

use strict;
use SAPDB::Install::Jar;
use Perforce::P4Admin;
use Perforce::P4Sync;
use Perforce::P4Info;
use Perforce::P4File;
use File::Copy;
use Mail::Automail;

print "\nsapdbc2depot v0.03 (2004-11-29)\n\n";

print "INFO: sapdbc2depot has been disabled.\n\n";
exit(0);

unless ($ENV{'OWN'})
{
    print "ERROR: OWN is not set!\nExiting...\n";
    CleanExit(1, "OWN variable is not set!");
}

unless ($ENV{'LC_STATE'})
{
    print "WARNING! Cannot determine LC_STATE! Setting to 'DEV'.\n\n";
    $ENV{'LC_STATE'} = 'DEV';
}

my $Server_Dir = ($ENV{'LC_STATE'} eq 'DEV' ? 'misc' : ($ENV{'LC_STATE'} eq 'COR' ? '' : ($ENV{'LC_STATE'} eq 'RAMP' ? 'deliver' : '')));

unless ($Server_Dir)
{
    if ($ENV{'LC_STATE'} eq 'COR')
    {
        print "Nothing to do for COR makes...\nExiting\n";
        exit (0);
    }
    elsif ($ENV{'LC_STATE'} eq 'HOT') {
        print "Nothing to do for HOT makes...\nExiting\n";
        exit(0);
    }
    else
    {
        print "ERROR! LC_STATE '$ENV{'LC_STATE'}' unknown!!\nExiting...\n";
        CleanExit(0, "LC_STATE '$ENV{'LC_STATE'}' is unknown!");
    }
}


my $SrcDir   = "$ENV{'OWN'}/usr/runtime/jar";
my $DestDir  = ($ENV{'TMP'} ? "$ENV{'TMP'}/sapdbc2depot" : "$ENV{'OWN'}/tmp/sapdbc2depot"); 
my $P4Port   = "perforce5000.wdf.sap.corp:5000";
my $P4User   = "d031096";
my $P4Client = mktemp();
my $SvrPath  = "//db/java/sapdb/$Server_Dir/...";
my $SvrFile  = "//db/java/sapdb/$Server_Dir/sapdbc.jar";
my @view     = ("$SvrPath //$P4Client/...");
my $file_open = 0;
my $client_created = 0;

mkdir ("$DestDir", 0777);

print "Checking sapdbc local version..";

unless (-e "$ENV{'OWN'}/usr/runtime/jar/sapdbc.jar")
{
    print "..NO LOCAL FILE!\n\nExiting...\n";
    CleanExit(0, "Local JAR file is missing!");
}

my $local_str = SAPDB::Install::Jar::JarBuildString("$ENV{'OWN'}/usr/runtime/jar/sapdbc.jar");

if ($local_str)
{
    print "..OK ($local_str)\n";
}
else
{
    print "..FAILED!\n\nExiting...\n";
    CleanExit(1, "Unable to extract build string from JAR!");
}

my $local_ver = "$1$2$3$4$5$6$7" if ($local_str =~ /^JAR\s+(\d+)\.(\d+)\.(\d+)\s+Build:\s+(\d+)-(\d+)-(\d+)-(\d+).*/);

print "Creating Perforce client..";

if (P4SetClient($P4Port, $P4Client, "$DestDir", $P4User, undef, undef, undef, \@view))
{
    $client_created = 1;
    print "..OK\n";
}
else
{
    print "..FAILED ($P4Client)!\n\nCan't continue without P4Client -> Exiting...\n";
    CleanExit(0, "Failed to create P4 client '$P4Client'!")
}

print "Getting change list..";

my $chlist = P4GetChangeList($P4Port, $P4User, "$SvrPath");
if ($chlist)
{
    print "..OK\n";
}
else
{
    print "..FAILED!\n";
    CleanExit(0, "Unable to retrieve change list from Perforce server!");
}

print "Retrieving sapdbc depot version..";

if (P4Sync($P4Port, $P4User, $P4Client, $SvrPath, $chlist))
{
    print "..OK\n";
}
else
{
    print "..FAILED!\n";
    CleanExit(0, "Unable to sync Perforce sapdbc version!");
}

print "Checking sapdbc depot version..";

my $depot_str = SAPDB::Install::Jar::JarBuildString("$DestDir/sapdbc.jar");

if ($depot_str)
{
    print "..OK ($depot_str)\n";
}
else
{
    print "..FAILED!\n";
    CleanExit(1, "Unable to extract build string from server-side JAR file!");
}

my $depot_ver = "$1$2$3$4$5$6$7" if ($depot_str =~ /^JAR\s+(\d+)\.(\d+)\.(\d+)\s+Build:\s+(\d+)-(\d+)-(\d+)-(\d+).*/);

if ($local_ver <= $depot_ver)
{
    print "sapdbc depot version is up-to-date!\n";
    CleanExit(0);
}

print "Checking out depot file..";

if (P4Edit($P4Port, $P4User, $P4Client, $SvrFile))
{
    print "..OK\n";
    $file_open = 1;
}
else
{
    print "..FAILED!\n";
    CleanExit(0, "Unable to check out depot version of JAR file!");
}

print "Switching files..";

unless (unlink("$DestDir/sapdbc.jar"))
{
    print "..FAILED (delete)!\n";
    CleanExit(0, "Unable to delete depot version of JAR file!");
}

unless (copy("$ENV{'OWN'}/usr/runtime/jar/sapdbc.jar","$DestDir/sapdbc.jar"))
{
    print "..FAILED (copy)!\n";
    CleanExit(0, "Unable to copy local version to depot version of JAR!");
}

print "..OK\n";
print "Submitting new file..";

if (P4Submit($P4Port, $P4User, $P4Client, $SvrFile))
{
    print "..OK\n";
    $file_open = 0;
}
else
{
    print "..FAILED!\n";
    CleanExit(0, "Unable to submit new version!");
}

CleanExit(0, "OK");

sub CleanExit
{
    my ($rc, $msgstr) = @_;
    my $msg;
    my %mail;
    
    if ($file_open)
    {
	    print "Reverting change..";
	    
	    if (P4Revert($P4Port, $P4User, $P4Client, "$DestDir/sapdbc.jar"))
	    {
	        print "..OK\n";
	    }
	    else
	    {
	        print "..FAILED!\n";
	    }
    }
    
    if ($client_created) {    
        print "Removing P4Client..";
        if (P4RemClient($P4Port, $P4Client))
        {
            print "..OK\n";
        }
        else
        {
            print "..FAILED!\n";
        }
    }
    
    if ($msgstr eq "OK") {
        $msg  = "A new sapdbc driver has been uploaded.\n\n";
        $msg .= "LC_STATE = $ENV{'LC_STATE'}\n";
        $msg .= "File     = $SvrFile\n";
        $msg .= "Version  = $local_str\n\n";
        $msg .= "Old Version was: $depot_str\n\n";
        
	    %mail = ( 'To'      => 'marco.paskamp@sap.com,ulrich.jansen@sap.com',
	         'Message' => "$msg",
	         'Category' => 'Make',
	         'Type' => 'Monitor',
	         'Infotext' => "SAPDBC New Driver ($ENV{'LC_STATE'}) - $local_str" );
    
    } elsif ($msgstr ne "") {
        $msg  = "There has been an error while uploading new sapdb version!\n\n";
        $msg .= "LC_STATE = $ENV{'LC_STATE'}\n";
        $msg .= "File     = $SvrFile\n";
        $msg .= "Version  = $local_str\n\n";
        $msg .= "Error    = $msgstr\n\n";
        
	    %mail = ( 'To'      => 'marco.paskamp@sap.com,ulrich.jansen@sap.com',
	         'Message' => "$msg",
	         'Category' => 'Make',
	         'Type' => 'Monitor',
	         'Infotext' => "ERROR uploading new SAPDBC Driver ($ENV{'LC_STATE'})" );
    }
    

    if ($msgstr) {
    
        unless (Mail::Automail::Send(%mail)) {
            print "Sendmail error: $Mail::Sendmail::error\n";
        }

    }
    
    exit($rc);
}

sub mktemp
{
    my @time = localtime(time);
    
    return "SDBC_$time[0]$time[1]$time[2]$time[3]$time[4]$time[5]";
}

