require 'test/unit'

require 'gettext.rb'

require 'testlib1.rb'
require 'testlib2.rb'
require 'testlib3.rb'
require 'test_rubyparser.rb'
require 'test_sgettext.rb'

class TestGetText < Test::Unit::TestCase
  include GetText

  def test_bindtextdomain
    return if /linux/ !~ RUBY_PLATFORM
    GetText.bindtextdomain("libc")
    GetText.locale = "ja_JP.eucJP"
    GetText.charset = "euc-jp"
    assert_equal("λޤ", GetText._("Terminated"))
    GetText.bindtextdomain("test1", "locale")
    assert_equal("japanese", GetText._("language"))

    GetText.bindtextdomain("libc", "/usr/share/locale")
    assert_equal("λޤ", GetText._("Terminated"))

    GetText.bindtextdomain("test2", "locale")
    assert_equal("JAPANESE", GetText._("LANGUAGE"))

    GetText.bindtextdomain("libc", "/usr/share/locale/")
    assert_equal("λޤ", GetText._("Terminated"))

    GetText.bindtextdomain("libc", "/usr/share/locale", "de")
    assert_equal("Beendet", GetText._("Terminated"))

    GetText.bindtextdomain("test1", "locale")
    GetText.locale = "fr"
    assert_equal("french", GetText._("language"))

    GetText.bindtextdomain("test1", "locale", "ja")
    assert_equal("japanese", GetText._("language"))
  end

  def test_bindtextdomain_include_module
    bindtextdomain("test1", "locale")
    assert_equal("japanese", _("language"))
  end

  def test_gettext
    bindtextdomain("test1", "locale")
    assert_equal("japanese", gettext("language"))
  end

  def test_setlocale
    bindtextdomain("test1", "locale")
    assert_equal("japanese", _("language"))
    setlocale("en")
    assert_equal("language", _("language"))
    setlocale("fr")
    assert_equal("french", _("language"))
  end

  def test_class
    bindtextdomain("test2", "locale")

    testlib1 = TestLib1.new
    assert_equal("japanese", testlib1.test)
    assert_equal("japanese", testlib1.test) #no influence of previous line
    assert_equal("ONE IS 1.", testlib1.test_formatted_string)
    testlib1.setlocale("fr")
    assert_equal("JAPANESE", _("LANGUAGE")) #no influence of previous line
    assert_equal("french", testlib1.test)
  end

  def test_subclass
    testlib2 = TestLib2.new
    assert_equal("JAPANESE", testlib2.test2)
    assert_equal("japanese", testlib2.test)
  end

  def test_nested_module
    testlib3 = TestLib3::Test3.new
    assert_equal("JAPANESE", testlib3.test3)
  end

  def test_no_msgstr
    bindtextdomain("test1", "locale", "ja")
    assert_equal("nomsgstr", _("nomsgstr"))
  end

  def test_complex
    testlib4 = TestLib4.new
    assert_equal("AAA", testlib4.test_1)
    assert_equal("AAA\n", testlib4.test_2)
    assert_equal("BBB\nCCC", testlib4.test_3)
    assert_equal("BBB
CCC
DDD
", testlib4.test_4)
    assert_equal("EEE", testlib4.test_5)
    assert_equal("EEEfooFFF", testlib4.test_6)
    assert_equal("GGGHHHIII", testlib4.test_7)
  end

  def test_noop
    assert_equal("test", N_("test"))
  end

  def test_sgettext
    testlib5 = TestSGetText.new
    assert_equal("MATCHED", testlib5.test_1)
    assert_equal("MATCHED", testlib5.test_2)
    assert_equal("AAA", testlib5.test_3)
    assert_equal("CCC", testlib5.test_4)
    assert_equal("CCC", testlib5.test_5)
    assert_equal("BBB", testlib5.test_6)
    assert_equal("B|BB", testlib5.test_7)
    assert_equal("MATCHED", testlib5.test_8)
    assert_equal("BBB", testlib5.test_9)
  end

  def test_plural
    bindtextdomain("plural", "locale", "ja")
    assert_equal("all", n_("one", "two", 0))
    assert_equal("all", n_("one", "two", 1))
    assert_equal("all", n_("one", "two", 2))

    setlocale("da")
    assert_equal("plural", n_("one", "two", 0))
    assert_equal("one", n_("one", "two", 1))
    assert_equal("plural", n_("one", "two", 2))

    setlocale("fr")
    assert_equal("one", ngettext("one", "two", 0))
    assert_equal("one", ngettext("one", "two", 1))
    assert_equal("plural", ngettext("one", "two", 2))

    setlocale("la")
    assert_equal("one", ngettext("one", "two", 21))
    assert_equal("one", ngettext("one", "two", 1))
    assert_equal("plural", ngettext("one", "two", 2))
    assert_equal("zero", ngettext("one", "two", 0))

    setlocale("ir")
    assert_equal("one", ngettext("one", "two", 1))
    assert_equal("two", ngettext("one", "two", 2))
    assert_equal("plural", ngettext("one", "two", 3))
    assert_equal("plural", ngettext("one", "two", 0))

    setlocale("li")
    assert_equal("one", ngettext("one", "two", 1))
    assert_equal("two", ngettext("one", "two", 22))
    assert_equal("three", ngettext("one", "two", 11))

    setlocale("cr")
    assert_equal("one", ngettext("one", "two", 1))
    assert_equal("two", ngettext("one", "two", 2))
    assert_equal("three", ngettext("one", "two", 5))

    setlocale("po")
    assert_equal("one", ngettext("one", "two", 1))
    assert_equal("two", ngettext("one", "two", 2))
    assert_equal("three", ngettext("one", "two", 5))

    setlocale("sl")
    assert_equal("one", ngettext("one", "two", 1))
    assert_equal("two", ngettext("one", "two", 2))
    assert_equal("three", ngettext("one", "two", 3))
    assert_equal("three", ngettext("one", "two", 4))
    assert_equal("four", ngettext("one", "two", 5))

    bindtextdomain("plural_error", "locale", "ja")
    assert_equal("a", n_("one", "two", 0))   
    setlocale("fr")
    assert_equal("one", n_("one", "two", 0))   
    setlocale("da")
    assert_equal("one", n_("one", "two", 0))   
  end
    
end
