/* kasbar.cpp
**
** Copyright (C) 2001-2004 Richard Moore <rich@kde.org>
** Contributor: Mosfet
**     All rights reserved.
**
** KasBar is dual-licensed: you can choose the GPL or the BSD license.
** Short forms of both licenses are included below.
*/

/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
** MA 02111-1307, USA.
*/

/*
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

/*
** Bug reports and questions can be sent to kde-devel@kde.org
*/
#include <qcursor.h>
#include <qpainter.h>
#include <qmemarray.h>

#include <kapplication.h>
#include <kdebug.h>
#include <krootpixmap.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>
#include <kpixmapio.h>

#include "kasbar.h"

#include "kasitem.h"
#include "kasbar.moc"

static const int SMALL_EXTENT = 36;
static const int MEDIUM_EXTENT = 52;
static const int LARGE_EXTENT = 68;

KasBar::KasBar( Orientation o, QWidget *parent, const char *name, WFlags f )
   : QWidget( parent, name, f ),
     items(),
     orient( o ),
     itemUnderMouse_( 0 ),
     boxesPerLine_(10), // Temp value
     maxBoxes_( 100 ), // Temp value
     itemSize_( Medium ),
     itemExtent_( MEDIUM_EXTENT ),
     transparent_( false ),
     enableTint_( false ),
     tintAmount_( 0.1 ), 
     tintColour_( colorGroup().mid() ),
     actBg( 0 ), inactBg( 0 ),
     labelPenColor_( Qt::white ), labelBgColor_( Qt::black ),
     activePenColor_( Qt::black ), activeBgColor_( Qt::white ),
     inactivePenColor_( Qt::black ), inactiveBgColor_( Qt::white ),
     progressColor_( Qt::green )
{
    setBackgroundMode( NoBackground );
    items.setAutoDelete( true );
    setMouseTracking( true );
    setMaxBoxes( 0 );
}

KasBar::~KasBar()
{
   delete actBg;
   delete inactBg;
}

void KasBar::setItemSize( int size )
{
   if ( size == itemSize_ )
      return;
   itemSize_ = size;

   switch( size ) {
   case Small:
     itemExtent_ = SMALL_EXTENT;
     break;
   case Medium:
     itemExtent_ = MEDIUM_EXTENT;
     break;
   case Large:
     itemExtent_ = LARGE_EXTENT;
     break;
   default:
     itemExtent_ = MEDIUM_EXTENT;
   }

   delete actBg;
   delete inactBg;
   actBg = 0;
   inactBg = 0;

   emit itemSizeChanged( size );
   emit configChanged();

   updateLayout();
}

void KasBar::setTransparent( bool enable )
{
   if ( transparent_ == enable )
      return;

   transparent_ = enable;

   if ( transparent_ ) {
       kdDebug(1345) << "KasBar: Enabling transparency" << endl;

      rootPix = new KRootPixmap( this );
      connect( rootPix, SIGNAL( backgroundUpdated(const QPixmap &) ),
	       this, SLOT( setBackground(const QPixmap &) ) );

      rootPix->setCustomPainting( true );

      if ( enableTint_ )
	 rootPix->setFadeEffect( tintAmount_, tintColour_ );

      rootPix->start();
   }
   else {
       kdDebug(1345) << "KasBar: Disabling transparency" << endl;

       rootPix->stop();
       delete rootPix;
       rootPix = 0;
   }

   emit configChanged();
   repaint();
}

void KasBar::setTint( bool enable )
{
   if ( enableTint_ == enable )
      return;

   enableTint_ = enable;

   if ( transparent_ && rootPix ) {
      if ( enableTint_ ) {
	 rootPix->setFadeEffect( tintAmount_, tintColour_ );
      }
      else {
	 rootPix->setFadeEffect( 0.0, tintColour_ );
      }

      emit configChanged();
      repaint( true );
   }
}

void KasBar::setTint( double amount, QColor color )
{
   tintAmount_ = amount;
   tintColour_ = color;

   if ( transparent_ && enableTint_ ) {
      rootPix->setFadeEffect( tintAmount_, tintColour_ );
      emit configChanged();

      if ( rootPix->isAvailable() )
	rootPix->repaint( true );
   }
}

void KasBar::setTintColor( const QColor &c )
{
   setTint( tintAmount_, c );
}

void KasBar::setTintAmount( int percent )
{
   double amt = (double) percent / 100.0;
   setTint( amt, tintColour_ );
}

void KasBar::setLabelPenColor( const QColor &color )
{
    labelPenColor_ = color;
    repaint();
}

void KasBar::setLabelBgColor( const QColor &color )
{
    labelBgColor_ = color;
    repaint();
}

void KasBar::setInactivePenColor( const QColor &color )
{
    inactivePenColor_ = color;
    repaint();
}

void KasBar::setInactiveBgColor( const QColor &color )
{
    inactiveBgColor_ = color;
    repaint();
}

void KasBar::setActivePenColor( const QColor &color )
{
    activePenColor_ = color;
    repaint();
}

void KasBar::setActiveBgColor( const QColor &color )
{
    activeBgColor_ = color;
    repaint();
}

void KasBar::setProgressColor( const QColor &color )
{
    progressColor_ = color;
    repaint();
}

void KasBar::setMaxBoxes( int count )
{
   if ( count == maxBoxes_ )
       return;

   if ( count == 0 )
       count = 15; // XXX Hacked

   maxBoxes_ = count;
   emit configChanged();
   setBoxesPerLine( count );
}

void KasBar::setBoxesPerLine( int count )
{
   boxesPerLine_ = QMIN( count, maxBoxes_ );
   updateLayout();
}


QSize KasBar::sizeHint( Orientation o,  QSize sz )
{
    if ( o == Horizontal )
	setBoxesPerLine( sz.width() / itemExtent() );
    else
	setBoxesPerLine( sz.height() / itemExtent() );

   unsigned int r=0, c=0;
   if( items.count() > (unsigned int) boxesPerLine_ ) {
      r = items.count()/boxesPerLine_;
      c = boxesPerLine_;
   }
   else {
      r = 1;
      c = items.count();
   }

   if( r*c < items.count() ) // remainders
      ++r;

   QSize s;
   if( o == Horizontal ) {
      s.setWidth( c*itemExtent() );
      s.setHeight( r*itemExtent() );
   }
   else {
      s.setWidth( r*itemExtent() );
      s.setHeight( c*itemExtent() );
   }

   return s;
}

void KasBar::updateLayout()
{
//    kdDebug(1345) << "KasBar: updateLayout(), count is " << items.count() << endl;
   if ( !isUpdatesEnabled() )
       return;

   // Work out the number of rows and columns
   unsigned int r=0, c=0;
   if( items.count() > (unsigned int) boxesPerLine_ ) {
      r = items.count()/boxesPerLine_;
      c = boxesPerLine_;
   }
   else{
      r = 1;
      c = items.count();
   }

   if( r*c < items.count() ) // remainders
      ++r;

   QSize sz;
   if ( orient == Horizontal )
       sz = QSize( c * itemExtent(), r * itemExtent() );
   else
       sz = QSize( r * itemExtent(), c * itemExtent() );

   if ( sz != size() ) {
       resize( sz );
   }

   update();
}

void KasBar::append( KasItem *i )
{
   if ( !i )
      return;

   items.append( i );
   updateLayout();
}

void KasBar::insert( int index, KasItem *i )
{
   if ( (!i) || (index < 0) )
      return;

   items.insert( index, i );
   updateLayout();
}

void KasBar::remove( KasItem *i )
{
   items.remove( i );

   if ( i == itemUnderMouse_ )
      itemUnderMouse_ = 0;
   updateLayout();
}

void KasBar::clear()
{
   items.clear();
   itemUnderMouse_ = 0;
   updateLayout();
}

void KasBar::mousePressEvent(QMouseEvent *ev)
{
   KasItem *i = itemAt( ev->pos() );
   if ( i )
      i->mousePressEvent( ev );
}

void KasBar::updateMouseOver()
{
    updateMouseOver( mapFromGlobal( QCursor::pos() ) );
}

void KasBar::updateMouseOver( QPoint pos )
{
   KasItem *i = itemAt(pos);

   if ( i == itemUnderMouse_ )
       return;

   if ( itemUnderMouse_ )
       itemUnderMouse_->mouseLeave();
   if ( i )
       i->mouseEnter();

   itemUnderMouse_ = i;
}

void KasBar::mouseMoveEvent(QMouseEvent *ev)
{
    updateMouseOver( ev->pos() );
}

void KasBar::dragMoveEvent ( QDragMoveEvent *ev )
{
   KasItem *i = itemAt( ev->pos() );
   if ( itemUnderMouse_ != i ) {
      if ( itemUnderMouse_ )
	 itemUnderMouse_->dragLeave();
      if ( i )
	 i->dragEnter();
      itemUnderMouse_ = i;
   }
}

void KasBar::paintEvent(QPaintEvent *ev)
{
//    kdDebug(1345) << "KasBar: paintEvent()" << endl;

   QPixmap buff( size() );
   QPainter p( &buff );

   paintBackground( &p, ev->rect() );

   QRect cr;
   KasItem *i;
   int r=0, c=0;

   if ( orientation() == Horizontal ) {
       int totalcols = width()/itemExtent();

       for ( i = items.first(); i; i = items.next() ) {
	   if ( c >= totalcols ) {
	       c = 0; ++r;
	   }
	   cr.setRect(c*itemExtent(), r*itemExtent(), itemExtent(), itemExtent());
	   if ( ev->rect().intersects(cr) )
	       i->paint( &p, c*itemExtent(), r*itemExtent() );
	   ++c;
       }
   }
   else {
       int totalrows = height()/itemExtent();

       for ( i = items.first(); i; i = items.next() ) {
	   if ( r >= totalrows ) {
	       r = 0; ++c;
	   }
	   cr.setRect(c*itemExtent(), r*itemExtent(), itemExtent(), itemExtent());
	   if ( ev->rect().intersects(cr) )
	       i->paint( &p, c*itemExtent(), r*itemExtent() );
	   ++r;
       }
   }

   QPainter q( this );
   q.drawPixmap( ev->rect().topLeft(), buff, ev->rect() );
}

void KasBar::resizeEvent(QResizeEvent *ev)
{
//    kdDebug(1345) << "KasBar: resizeEvent()" << endl;

    QWidget::resizeEvent(ev);
    emit layoutChanged();
}

void KasBar::updateItem( KasItem *i )
{
   QPoint p = itemPos( i );
   update( QRect( p, QSize( itemExtent(), itemExtent()  ) ) );
}

void KasBar::repaintItem(KasItem *i, bool erase )
{
//    kdDebug(1345) << "KasBar: repaintItem()" << endl;
   QPoint p = itemPos( i );
   repaint( QRect( p, QSize( itemExtent(), itemExtent()  ) ), transparent_ || erase );
}

KasItem* KasBar::itemAt(const QPoint &p)
{
   QRect cr;
   KasItem *i;
   int r=0, c=0;
   int totalcols = width()/itemExtent();
   int totalrows = height()/itemExtent();

   if(orient == Horizontal){
      for (i = items.first(); i; i = items.next()){
	 if(c >= totalcols){
	    c = 0;
	    ++r;
	 }
	 cr.setRect(c*itemExtent(), r*itemExtent(), itemExtent(), itemExtent());
	 if(cr.contains(p))
	    return(i);
	 ++c;
      }
   }
   else {
      for (i = items.first(); i; i = items.next()) {
	 if(r >= totalrows){
	    r = 0; ++c;
	 }
	 cr.setRect(c*itemExtent(), r*itemExtent(), itemExtent(), itemExtent());
	 if(cr.contains(p))
	    return(i);
	 ++r;
      }
   }

   return 0;
}

QPoint KasBar::itemPos( KasItem *i )
{
   int x;
   int y;
   int totalcols = width()/itemExtent();
   int totalrows = height()/itemExtent();
   int index = items.find( i );
   if ( index == -1 ) {
      x = y = -1;
      return QPoint( x, y );
   }

   int r = 0;
   int c = 0;

   if ( ( orient == Horizontal ) && totalcols ) {
      r = index / totalcols;
      c = index % totalcols;
   }
   else if ( ( orient == Vertical ) && totalrows ) {
      c = index / totalrows;
      r = index % totalrows;
   }

   x = c * itemExtent();
   y = r * itemExtent();
   return QPoint( x+1, y+1 );  // +1,+1 prevents #50654
}

KPixmap *KasBar::activeBg()
{
   if ( !actBg ) {
      actBg = new KPixmap;

      actBg->resize( itemExtent()-4, itemExtent()-13-4 );
      KPixmapEffect::gradient( *actBg,
			       colorGroup().light(), colorGroup().mid(),
			       KPixmapEffect::DiagonalGradient );
   }

   return actBg;
}

KPixmap *KasBar::inactiveBg()
{
   if ( !inactBg ) {
      inactBg = new KPixmap;

      inactBg->resize( itemExtent()-4, itemExtent()-13-4 );
      KPixmapEffect::gradient( *inactBg,
			       colorGroup().mid(), colorGroup().dark(),
			       KPixmapEffect::DiagonalGradient );
   }

   return inactBg;
}

void KasBar::setBackground( const QPixmap &newBg )
{
    bg = newBg;
    update();
}

void KasBar::paintBackground( QPainter *p, const QRect &r )
{
    // If we're transparent and know the bg
    if ( transparent_ && (!bg.isNull()) ) {
	p->drawPixmap( r.topLeft(), bg, r );
	return;
    }

    // We're opaque or we don't know the bg
    p->fillRect( rect(), colorGroup().brush(QColorGroup::Mid) );
}

