/*
 *
 * $Id: postgre.c, v 0.9 2001/04/06 23:22:45 kaiser Exp $
 *
 * postgresql backend to ipac
 * Copyright (C) 2001-2002 Al Zaharov
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * The author can be reached via email: kaiser13@mail2000.ru, or by 
 * fido: Al_Zaharov@p88.f58.n5005.z2.fidonet.org
 *
*/

#include <string.h>
#include <stdio.h>
#include <dirent.h>
#include <fnmatch.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/param.h>
#include <unistd.h>
#include <search.h>
#include <assert.h>
#include <netdb.h>
#include <stdarg.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <dlfcn.h>

/* - T.Mohan 5/7/2001
 * #include <ctype.h> for isalnum() in parse_interface()
 */
#include <ctype.h>
#include "ipac.h"
#include "libiptc.h"
#include "../../lib/libnet.h"


#ifndef PROC_SYS_MODPROBE
#define PROC_SYS_MODPROBE "/proc/sys/kernel/modprobe"
#endif

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#ifndef IPT_LIB_DIR
#define IPT_LIB_DIR "/lib/iptables"
#endif

#define FMT_NUMERIC	0x0001
#define FMT_NOCOUNTS	0x0002
#define FMT_KILOMEGAGIGA 0x0004
#define FMT_OPTIONS	0x0008
#define FMT_NOTABLE	0x0010
#define FMT_NOTARGET	0x0020
#define FMT_VIA		0x0040
#define FMT_NONEWLINE	0x0080
#define FMT_LINENUMBERS 0x0100

#define FMT_PRINT_RULE (FMT_NOCOUNTS | FMT_OPTIONS | FMT_VIA \
			| FMT_NUMERIC | FMT_NOTABLE)
#define FMT(tab,notab) ((format) & FMT_NOTABLE ? (notab) : (tab))

extern char *authhost;

/* - T.Mohan 5/7/2001
 * interface structure to pass to append rule
 */

struct iface_struct {
	char* ifaceIn;
	char* ifaceOut;
	int invflags;
};

typedef struct iface_struct s_iface;


/* Include file for additions: new matches and targets. */
struct iptables_match
{
	struct iptables_match *next;

	ipt_chainlabel name;

	const char *version;

	/* Size of match data. */
	size_t size;

	/* Size of match data relevent for userspace comparison purposes */
	size_t userspacesize;

	/* Function which prints out usage message. */
	void (*help)(void);

	/* Initialize the match. */
	void (*init)(struct ipt_entry_match *m, unsigned int *nfcache);

	/* Function which parses command options; returns true if it
           ate an option */
	int (*parse)(int c, char **argv, int invert, unsigned int *flags,
		     const struct ipt_entry *entry,
		     unsigned int *nfcache,
		     struct ipt_entry_match **match);

	/* Final check; exit if not ok. */
	void (*final_check)(unsigned int flags);

	/* Prints out the match iff non-NULL: put space at end */
	void (*print)(const struct ipt_ip *ip,
		      const struct ipt_entry_match *match, int numeric);

	/* Saves the match info in parsable form to stdout. */
	void (*save)(const struct ipt_ip *ip,
		     const struct ipt_entry_match *match);

	/* Pointer to list of extra command-line options */
	const struct option *extra_opts;

	/* Ignore these men behind the curtain: */
	unsigned int option_offset;
	struct ipt_entry_match *m;
	unsigned int mflags;
	unsigned int used;
};

struct iptables_target
{
	struct iptables_target *next;

	ipt_chainlabel name;

	const char *version;

	/* Size of target data. */
	size_t size;

	/* Size of target data relevent for userspace comparison purposes */
	size_t userspacesize;

	/* Function which prints out usage message. */
	void (*help)(void);

	/* Initialize the target. */
	void (*init)(struct ipt_entry_target *t, unsigned int *nfcache);

	/* Function which parses command options; returns true if it
           ate an option */
	int (*parse)(int c, char **argv, int invert, unsigned int *flags,
		     const struct ipt_entry *entry,
		     struct ipt_entry_target **target);

	/* Final check; exit if not ok. */
	void (*final_check)(unsigned int flags);

	/* Prints out the target iff non-NULL: put space at end */
	void (*print)(const struct ipt_ip *ip,
		      const struct ipt_entry_target *target, int numeric);

	/* Saves the targinfo in parsable form to stdout. */
	void (*save)(const struct ipt_ip *ip,
		     const struct ipt_entry_target *target);

	/* Pointer to list of extra command-line options */
	struct option *extra_opts;

	/* Ignore these men behind the curtain: */
	unsigned int option_offset;
	struct ipt_entry_target *t;
	unsigned int tflags;
	unsigned int used;
};

enum ipt_tryload {
	DONT_LOAD,
	TRY_LOAD,
	LOAD_MUST_SUCCEED
};


/** structure for run file (rule file) */
struct runfile_line_type {
	char *line;
	struct runfile_line_type *next;
};

static iptc_handle_t handle;

struct iptables_match *iptables_matches = NULL;
struct iptables_target *iptables_targets = NULL;

extern void register_match(struct iptables_match *me);
extern void register_target(struct iptables_target *me);

static int clear_runfile(char *serv_name, int greedy);

/* plain file ipac interface entries */
int iptables_ipac_init(int flag);
int iptables_ipac_set(rule_type **firstrule, int first);
int iptables_ipac_read(rule_type **firstrule);
int iptables_ipac_deny(char *login);
int iptables_ipac_accept(char *login);
int iptables_ipac_check(void);
int iptables_ipac_remove_user(char *login);
int iptables_ipac_alarm();

static const acc_agent_t interface_entry = {
	"iptables",
	iptables_ipac_init,
	iptables_ipac_set,
	iptables_ipac_read,
	iptables_ipac_deny,
	iptables_ipac_accept,
	iptables_ipac_check,
	iptables_ipac_remove_user,
	iptables_ipac_alarm,
};

const acc_agent_t *ipac_ag_interface_iptables() {
	return &interface_entry;
}

/* - T.Mohan 5/7/2001
 * exit_error().  moved here to avoid impicit declaration
 * compiler warning.
 */

void
exit_error(enum exittype status, char *msg, ...)
{
	va_list args;

	va_start(args, msg);
//	fprintf(stderr, "%s v%s: ", program_name, program_version);
	vfprintf(stderr, msg, args);
	va_end(args);
	fprintf(stderr, "\n");
//	if (status == PARAMETER_PROBLEM)
//		exit_tryhelp(status);
	if (status == VERSION_PROBLEM)
		fprintf(stderr,
	"Perhaps iptables or your "
					    "kernel needs to be upgraded.\n");
	exit(status);
}

//===================================================
static int
tcp_service_to_port(const char *name)
{
	struct servent *service;

	if ((service = getservbyname(name, "tcp")) != NULL)
		return ntohs((unsigned short) service->s_port);

	return -1;
}

static u_int16_t
parse_tcp_port(const char *port)
{
	int portnum;

	if ((portnum = string_to_number(port, 0, 65535)) != -1 ||
	    (portnum = tcp_service_to_port(port)) != -1)
		return (u_int16_t)portnum;
	
	fprintf(stderr, "invalid TCP port/service `%s' specified\n", port);
	exit(1);
}

/* - T.Mohan 5/7/2001
 * parse_tcp_ports() function source from 
 * iptables-1.2.2 file:extensions/ibip6t_tcp.c
 */

static void
parse_tcp_ports(const char *portstring, u_int16_t *ports)
{
	char *buffer;
	char *cp;

	buffer = strdup(portstring);
	if ((cp = strchr(buffer, ':')) == NULL)
		ports[0] = ports[1] = parse_tcp_port(buffer);
	else {
		*cp = '\0';
		cp++;

		ports[0] = buffer[0] ? parse_tcp_port(buffer) : 0;
		ports[1] = cp[0] ? parse_tcp_port(cp) : 0xFFFF;
	}
	free(buffer);
}

static int
udp_service_to_port(const char *name)
{
	struct servent *service;

	if ((service = getservbyname(name, "udp")) != NULL)
		return ntohs((unsigned short) service->s_port);

	return -1;
}

static u_int16_t
parse_udp_port(const char *port)
{
	int portnum;

	if ((portnum = string_to_number(port, 0, 65535)) != -1 ||
	    (portnum = udp_service_to_port(port)) != -1)
		return (u_int16_t)portnum;
	
	fprintf(stderr, "invalid UDP port/service `%s' specified\n", port);
	exit(1);
}

/* - T.Mohan 5/7/2001
 * parse_udp_ports() function source from 
 * iptables-1.2.2 file:extensions/ibip6t_udp.c
 */

static void
parse_udp_ports(const char *portstring, u_int16_t *ports)
{
	char *buffer;
	char *cp;

	buffer = strdup(portstring);
	if ((cp = strchr(buffer, ':')) == NULL)
		ports[0] = ports[1] = parse_udp_port(buffer);
	else {
		*cp = '\0';
		cp++;

		ports[0] = buffer[0] ? parse_udp_port(buffer) : 0;
		ports[1] = cp[0] ? parse_udp_port(cp) : 0xFFFF;
	}
	free(buffer);
}


/* - T.Mohan 5/7/2001
 * parse_interface() function source modified from 
 * iptables-1.2.2  file:iptables.c 
 */

void
parse_interface(const char *arg, char *vianame, unsigned char *mask)
{
	int vialen = strlen(arg);
	unsigned int i;

	memset(mask, 0, IFNAMSIZ);
	memset(vianame, 0, IFNAMSIZ);

	if (vialen + 1 > IFNAMSIZ)
		exit_error(PARAMETER_PROBLEM,
			   "interface name `%s' must be shorter than IFNAMSIZ"
			   " (%i)", arg, IFNAMSIZ-1);

	strcpy(vianame, arg);
	if (vialen == 0)
		memset(mask, 0, IFNAMSIZ);
	else if (vianame[vialen - 1] == '+') {
		memset(mask, 0xFF, vialen - 1);
		memset(mask + vialen - 1, 0, IFNAMSIZ - vialen + 1);
	} else {
		/* Include nul-terminator in match */
		memset(mask, 0xFF, vialen + 1);
		memset(mask + vialen + 1, 0, IFNAMSIZ - vialen - 1);
	}
	for (i = 0; vianame[i]; i++) {
		if (!isalnum(vianame[i])
                    && vianame[i] != '_'
		    && vianame[i] != '+'
                    && vianame[i] != '.') {
			exit_error(PARAMETER_PROBLEM, "Warning: weird character in interface"
			       " `%s' (No aliases, :, ! or *).\n", vianame);
		}
	}
}

int
check_inverse_type(char* src)
{
	if (src)
 {
		if (memcmp(src, "!", 1) == 0)
 {
			int slen = strlen(src);

			//strip the "!"
			memcpy(src, src+1, slen);

			//if all there was, was a `!' after doing the strip,
			// return no inverse and don't complain about it.
			if (slen == 1)
				return 0;

			if (memcmp(src, "!", 1) == 0)
				exit_error(PARAMETER_PROBLEM,
					   "Multiple `!' flags not allowed");

			return 1;
		}
	}
	return 0;
}

int
check_inverse(const char option[], int *invert)
{
	if (option && strcmp(option, "!") == 0) {
		if (*invert)
			exit_error(PARAMETER_PROBLEM,
				   "Multiple `!' flags not allowed");

		*invert = TRUE;
		return TRUE;
	}
	return FALSE;
}

// ---------------------------------------------------------------------
// ---------------------------------------------------------------------

static char *
proto_to_name(u_int8_t proto, int nolookup)
{
	unsigned int i;

	if (proto && !nolookup) {
		struct protoent *pent = getprotobynumber(proto);
		if (pent)
			return pent->p_name;
	}

	for (i = 0; i < sizeof(chain_protos)/sizeof(struct pprot); i++)
		if (chain_protos[i].num == proto)
			return chain_protos[i].name;

	return NULL;
}

static char *
addr_to_network(const struct in_addr *addr)
{
	struct netent *net;

	if ((net = getnetbyaddr((long) ntohl(addr->s_addr), AF_INET)) != NULL)
		return (char *) net->n_name;

	return (char *) NULL;
}

char *
addr_to_dotted(const struct in_addr *addrp)
{
	static char buf[20];
	const unsigned char *bytep;

	bytep = (const unsigned char *) &(addrp->s_addr);
	sprintf(buf, "%d.%d.%d.%d", bytep[0], bytep[1], bytep[2], bytep[3]);
	return buf;
}

static char *
addr_to_host(const struct in_addr *addr)
{
	struct hostent *host;

	if ((host = gethostbyaddr((char *) addr,
				  sizeof(struct in_addr), AF_INET)) != NULL)
		return (char *) host->h_name;

	return (char *) NULL;
}

static char *
addr_to_anyname(const struct in_addr *addr)
{
	char *name;

	if ((name = addr_to_host(addr)) != NULL ||
	    (name = addr_to_network(addr)) != NULL)
		return name;

	return addr_to_dotted(addr);
}

static char *
mask_to_dotted(const struct in_addr *mask)
{
	int i;
	static char buf[20];
	u_int32_t maskaddr, bits;

	maskaddr = ntohl(mask->s_addr);

	if (maskaddr == 0xFFFFFFFFL)
		/* we don't want to see "/32" */
		return "";

	i = 32;
	bits = 0xFFFFFFFEL;
	while (--i >= 0 && maskaddr != bits)
		bits <<= 1;
	if (i >= 0)
		sprintf(buf, "/%d", i);
	else
		/* mask was not a decent combination of 1's and 0's */
		sprintf(buf, "/%s", addr_to_dotted(mask));

	return buf;
}

static struct ipt_entry *
generate_entry(const struct ipt_entry *fw,
	       struct iptables_match *matches,
	       struct ipt_entry_target *target)
{
	unsigned int size;
	struct iptables_match *m;
	struct ipt_entry *e;

	size = sizeof(struct ipt_entry);
	for (m = matches; m; m = m->next) {
		if (!m->used)
			continue;

		size += m->m->u.match_size;
	}

	e = xmalloc(size + target->u.target_size);
	*e = *fw;
	e->target_offset = size;
	e->next_offset = size + target->u.target_size;

	size = 0;
	for (m = matches; m; m = m->next) {
		if (!m->used)
			continue;

		memcpy(e->elems + size, m->m, m->m->u.match_size);
		size += m->m->u.match_size;
	}
	memcpy(e->elems + size, target, target->u.target_size);

	return e;
}

static char *get_modprobe(void)
{
	int procfile;
	char *ret;

	procfile = open(PROC_SYS_MODPROBE, O_RDONLY);
	if (procfile < 0)
		return NULL;

	ret = malloc(1024);
	if (ret) {
		switch (read(procfile, ret, 1024)) {
		case -1: goto fail;
		case 1024: goto fail; /* Partial read.  Wierd */
		}
		if (ret[strlen(ret)-1]=='\n') 
			ret[strlen(ret)-1]=0;
		close(procfile);
		return ret;
	}
 fail:
	free(ret);
	close(procfile);
	return NULL;
}

struct iptables_target *
find_target(const char *name, enum ipt_tryload tryload)
{
	struct iptables_target *ptr;

	/* Standard target? */
	if (strcmp(name, "") == 0
	    || strcmp(name, IPTC_LABEL_ACCEPT) == 0
	    || strcmp(name, IPTC_LABEL_DROP) == 0
	    || strcmp(name, IPTC_LABEL_QUEUE) == 0
	    || strcmp(name, IPTC_LABEL_RETURN) == 0)
		name = "standard";


	for (ptr = iptables_targets; ptr; ptr = ptr->next) {
		if (strcmp(name, ptr->name) == 0)
		{
			break;
		}
	}

	if (!ptr && tryload != DONT_LOAD) {
		char path[sizeof(IPT_LIB_DIR) + sizeof("/libipt_.so")
			 + strlen(name)];
		sprintf(path, IPT_LIB_DIR "/libipt_%s.so", name);
		if (dlopen(path, RTLD_NOW)) {
			/* Found library.  If it didn't register itself,
			   maybe they specified match as a target. */
			ptr = find_target(name, DONT_LOAD);
			if (!ptr) {
				fprintf(stderr, "Couldn't load target `%s'\n",
					   name);
				exit(1);
			}
		} else if (tryload == LOAD_MUST_SUCCEED) {
			fprintf(stderr, "Couldn't load target `%s':%s\n",
				   name, dlerror());
			exit(1);
			}
	}

	if (ptr)
		ptr->used = 1;

	return ptr;
}

static int iptables_insmod(const char *modname, const char *modprobe)
{
	char *buf = NULL;
	char *argv[3];

//	 If they don't explicitly set it, read out of kernel 
	if (!modprobe) {
		buf = get_modprobe();
		if (!buf)
			return -1;
		modprobe = buf;
	}

	switch (fork()) {
	case 0:
		argv[0] = (char *)modprobe;
		argv[1] = (char *)modname;
		argv[2] = NULL;
		execv(argv[0], argv);

//		 not usually reached 
		exit(0);
	case -1:
		return -1;

	default: // parent 
		wait(NULL);
	}

	free(buf);
	return 0;
}

void
register_target(struct iptables_target *me)
{
	if (find_target(me->name, DONT_LOAD)) {
		fprintf(stderr, "%s: target `%s' already registered.\n",
			"fddfewev", me->name);
		exit(1);
	}

	if (me->size != IPT_ALIGN(me->size)) {
		fprintf(stderr, "%s: target `%s' has invalid size %u.\n",
			"fddfgdsse", me->name, me->size);
		exit(1);
	}

//	Prepend to list.
	me->next = iptables_targets;
	iptables_targets = me;
	me->t = NULL;
	me->tflags = 0;
}

unsigned char * make_delete_mask(struct ipt_entry *fw)
{
	/* Establish mask for comparison */
	unsigned int size;
	struct iptables_match *m;
	unsigned char *mask, *mptr;

	size = sizeof(struct ipt_entry);
	for (m = iptables_matches; m; m = m->next) {
		if (!m->used)
			continue;

		size += IPT_ALIGN(sizeof(struct ipt_entry_match)) + m->size;
	}

	mask = xcalloc(1, size
			 + IPT_ALIGN(sizeof(struct ipt_entry_target))
			 + iptables_targets->size);

	memset(mask, 0xFF, sizeof(struct ipt_entry));
	mptr = mask + sizeof(struct ipt_entry);

	for (m = iptables_matches; m; m = m->next) {
		if (!m->used)
			continue;

		memset(mptr, 0xFF,
		       IPT_ALIGN(sizeof(struct ipt_entry_match))
		       + m->userspacesize);
		mptr += IPT_ALIGN(sizeof(struct ipt_entry_match)) + m->size;
	}

	memset(mptr, 0xFF,
	       IPT_ALIGN(sizeof(struct ipt_entry_target))
	       + iptables_targets->userspacesize);

	return mask;
}

struct iptables_match *
find_match(const char *name, enum ipt_tryload tryload)
{
	struct iptables_match *ptr;
    
	for (ptr = iptables_matches; ptr; ptr = ptr->next) {
		if (strcmp(name, ptr->name) == 0)
			break;
	}

	if (!ptr && tryload != DONT_LOAD) {
		char path[sizeof(IPT_LIB_DIR) + sizeof("/libipt_.so")
			 + strlen(name)];
		sprintf(path, IPT_LIB_DIR "/libipt_%s.so", name);
		if (dlopen(path, RTLD_NOW)) {
			/* Found library.  If it didn't register itself,
			   maybe they specified target as match. */
			ptr = find_match(name, DONT_LOAD);

			if (!ptr) {
				fprintf(stderr, "Couldn't load match `%s'\n",
					   name);
				exit(1);
			}
		} else if (tryload == LOAD_MUST_SUCCEED) {
			fprintf(stderr, "Couldn't load match `%s':%s\n",
				   name, dlerror());
			exit(1);
		}
	}

	if (ptr)
		ptr->used = 1;

	return ptr;
}

void
register_match(struct iptables_match *me)
{
	struct iptables_match **i;

/*
	if (strcmp(me->version, program_version) != 0) {
		fprintf(stderr, "%s: match `%s' v%s (I'm v%s).\n",
			program_name, me->name, me->version, program_version);
		exit(1);
	}
*/
	if (find_match(me->name, DONT_LOAD)) {
		fprintf(stderr, "%s: match `%s' already registered.\n",
			"fetchipac??", me->name);
		exit(1);
	}

	if (me->size != IPT_ALIGN(me->size)) {
		fprintf(stderr, "%s: match `%s' has invalid size %u.\n",
			"fetchipac??", me->name, me->size);
		exit(1);
	}

	/* Append to list. */
	for (i = &iptables_matches; *i; i = &(*i)->next);
	me->next = NULL;
	*i = me;

	me->m = NULL;
	me->mflags = 0;
}


static struct iptables_match *
find_proto(const char *pname, enum ipt_tryload tryload, int nolookup)
{
	int proto;

	proto = string_to_number(pname, 0, 255);
	if (proto != -1)
		return find_match(proto_to_name(proto, nolookup), tryload);

	return find_match(pname, tryload);
}

static void
print_num(u_int64_t number, unsigned int format)
{
	if (format & FMT_KILOMEGAGIGA) {
		if (number > 99999) {
			number = (number + 500) / 1000;
			if (number > 9999) {
				number = (number + 500) / 1000;
				if (number > 9999) {
					number = (number + 500) / 1000;
					printf(FMT("%4lluG ","%lluG "),number);
				}
				else printf(FMT("%4lluM ","%lluM "), number);
			} else
				printf(FMT("%4lluK ","%lluK "), number);
		} else
			printf(FMT("%5llu ","%llu "), number);
	} else
		printf(FMT("%8llu ","%llu "), number);
}

static int
print_match(const struct ipt_entry_match *m,
	    const struct ipt_ip *ip,
	    int numeric)
{
	struct iptables_match *match = find_match(m->u.user.name, TRY_LOAD);

	if (match) {
		if (match->print)
			match->print(ip, m, numeric);
		else
			printf("%s ", match->name);
	} else {
		if (m->u.user.name[0])
			printf("UNKNOWN match `%s' ", m->u.user.name);
	}
	/* Don't stop iterating. */
	return 0;
}

/* e is called `fw' here for hysterical raisins */
static void
print_firewall(const struct ipt_entry *fw,
	       const char *targname,
	       unsigned int num,
	       unsigned int format,
	       const iptc_handle_t handle)
{
	struct iptables_target *target = NULL;
	const struct ipt_entry_target *t;
	u_int8_t flags;
	char buf[BUFSIZ];

	/* User creates a chain called "REJECT": this overrides the
	   `REJECT' target module.  Keep feeding them rope until the
	   revolution... Bwahahahahah */
	if (!iptc_is_chain(targname, handle))
		target = find_target(targname, TRY_LOAD);
	else
		target = find_target(IPT_STANDARD_TARGET, LOAD_MUST_SUCCEED);

	t = ipt_get_target((struct ipt_entry *)fw);
	flags = fw->ip.flags;

	if (format & FMT_LINENUMBERS)
		printf(FMT("%-4u ", "%u "), num+1);

	if (!(format & FMT_NOCOUNTS)) {
		print_num(fw->counters.pcnt, format);
		print_num(fw->counters.bcnt, format);
	}

	if (!(format & FMT_NOTARGET))
		printf(FMT("%-9s ", "%s "), targname);

	fputc(fw->ip.invflags & IPT_INV_PROTO ? '!' : ' ', stdout);
	{
		char *pname = proto_to_name(fw->ip.proto, format&FMT_NUMERIC);
		if (pname)
			printf(FMT("%-5s", "%s "), pname);
		else
			printf(FMT("%-5hu", "%hu "), fw->ip.proto);
	}

	if (format & FMT_OPTIONS) {
		if (format & FMT_NOTABLE)
			fputs("opt ", stdout);
		fputc(fw->ip.invflags & IPT_INV_FRAG ? '!' : '-', stdout);
		fputc(flags & IPT_F_FRAG ? 'f' : '-', stdout);
		fputc(' ', stdout);
	}

	if (format & FMT_VIA) {
		char iface[IFNAMSIZ+2];

		if (fw->ip.invflags & IPT_INV_VIA_IN) {
			iface[0] = '!';
			iface[1] = '\0';
		}
		else iface[0] = '\0';

		if (fw->ip.iniface[0] != '\0') {
			strcat(iface, fw->ip.iniface);
			/* If it doesn't compare the nul-term, it's a
			   wildcard. */
			if (fw->ip.iniface_mask[strlen(fw->ip.iniface)] == 0)
				strcat(iface, "+");
		}
		else if (format & FMT_NUMERIC) strcat(iface, "*");
		else strcat(iface, "any");
		printf(FMT(" %-6s ","in %s "), iface);

		if (fw->ip.invflags & IPT_INV_VIA_OUT) {
			iface[0] = '!';
			iface[1] = '\0';
		}
		else iface[0] = '\0';

		if (fw->ip.outiface[0] != '\0') {
			strcat(iface, fw->ip.outiface);
			/* If it doesn't compare the nul-term, it's a
			   wildcard. */
			if (fw->ip.outiface_mask[strlen(fw->ip.outiface)] == 0)
				strcat(iface, "+");
		}
		else if (format & FMT_NUMERIC) strcat(iface, "*");
		else strcat(iface, "any");
		printf(FMT("%-6s ","out %s "), iface);
	}

	fputc(fw->ip.invflags & IPT_INV_SRCIP ? '!' : ' ', stdout);
	if (fw->ip.smsk.s_addr == 0L && !(format & FMT_NUMERIC))
		printf(FMT("%-19s ","%s "), "anywhere");
	else {
		if (format & FMT_NUMERIC)
			sprintf(buf, "%s", addr_to_dotted(&(fw->ip.src)));
		else
			sprintf(buf, "%s", addr_to_anyname(&(fw->ip.src)));
		strcat(buf, mask_to_dotted(&(fw->ip.smsk)));
		printf(FMT("%-19s ","%s "), buf);
	}

	fputc(fw->ip.invflags & IPT_INV_DSTIP ? '!' : ' ', stdout);
	if (fw->ip.dmsk.s_addr == 0L && !(format & FMT_NUMERIC))
		printf(FMT("%-19s","-> %s"), "anywhere");
	else {
		if (format & FMT_NUMERIC)
			sprintf(buf, "%s", addr_to_dotted(&(fw->ip.dst)));
		else
			sprintf(buf, "%s", addr_to_anyname(&(fw->ip.dst)));
		strcat(buf, mask_to_dotted(&(fw->ip.dmsk)));
		printf(FMT("%-19s","-> %s"), buf);
	}

	if (format & FMT_NOTABLE)
		fputs("  ", stdout);

	IPT_MATCH_ITERATE(fw, print_match, &fw->ip, format & FMT_NUMERIC);

	if (target) {
		if (target->print)
			/* Print the target information. */
			target->print(&fw->ip, t, format & FMT_NUMERIC);
	} else if (t->u.target_size != sizeof(*t))
		printf("[%u bytes of unknown target data] ",
		       t->u.target_size - sizeof(*t));

	if (!(format & FMT_NONEWLINE))
		fputc('\n', stdout);
}

static void
print_firewall_line(const struct ipt_entry *fw,
		    const iptc_handle_t h)
{
	struct ipt_entry_target *t;

	t = ipt_get_target((struct ipt_entry *)fw);
	print_firewall(fw, t->u.user.name, 0, FMT_PRINT_RULE, h);
}


// ---------------------------------------------------------------------


/** delete run file (rule file) from memory, freeing dynamically
 *  allocated memory
 */
static void 
destroy_runfile_lines(struct runfile_line_type *lines)
{
	struct runfile_line_type *next;
	while(lines != NULL)
	{
		if (lines->line != NULL)
			free(lines->line);
		next = lines->next;
		free(lines);
		lines = next;
	}
}

//------------------------------------------------------------------
/** read run file (rule file) and store it in memory, using a singly 
 *  linked list of instances of struct runfile_line_type
 *  return the list read or NULL in case of error
 */
static struct runfile_line_type 
*read_runfile()
{
	FILE *frunfile;
	char runfile_line[MAX_RULE_NAME_LENGTH*2], *cp; 
	struct runfile_line_type *result, *lastline, *cur;
	
	int tmp=0;

	frunfile = fopen(RUNFILE, "r");
	if (frunfile == NULL) {
		fprintf(stderr, "%s: cant open run file \"%s\": %s "
				"(fetchipac -S not ran?)\n",
			me, RUNFILE, strerror(errno));
		return NULL;
	}

	result = NULL;
	lastline = NULL;
	while(fgets(runfile_line, MAX_RULE_NAME_LENGTH*2, frunfile) != NULL) {
		tmp++;
		cp = strchr(runfile_line, '\n');
		if (cp)
			*cp = '\0';
		if (*runfile_line == '#')
			continue;

		cur = (struct runfile_line_type *)
				xmalloc(sizeof(struct runfile_line_type));
		cur->line = xstrdup(runfile_line);
		cur->next = NULL;
		if (result == NULL)
			result = cur;
		else 
			lastline->next = cur;
		lastline = cur;
	}
	if (!feof(frunfile)) {
		fprintf(stderr, "%s: reading \"%s\": %s\n",
			me, RUNFILE, strerror(errno));
		fclose(frunfile);
		destroy_runfile_lines(result);
		result = NULL;
	}
	fclose(frunfile);
	return result;
}

/** read kernel accounting data in iptables system
 * read from stream f
 * create records with data (packet and data counters) and
 * rule names and store them into instances of rule_type (declared
 * in ipac.h) using rule names from runfile
 * if a rule name is equal to a previously used rule name, add the
 * counters and re-use the old record
 * complain if something is wrong with the data.
 * return 0 for success, 1 otherwise
 */
static int 
read_iptables(struct runfile_line_type *runfile, rule_type **firstrule)
{
	char *cp="\0";
	rule_type *rule, *lastrule;
	rule_type *chain, *lastchain, *firstchain;
	struct runfile_line_type *nextline;
	void *node;
	void *ruletree = NULL;
	void *chaintree = NULL;
	struct ipt_counters *counters = NULL;

	if (handle)
		iptc_commit(&handle);	// we need fresh snapshot of the rules

	iptables_ipac_init(0);		// init after commit is a must
	
	/* create the rule_type records in correct order as from 
	 * run file.
	 */
	lastrule = *firstrule = NULL;
	chain = lastchain = firstchain = NULL;
	for (nextline=runfile; nextline!=NULL; nextline=nextline->next) {
		cp = strchr(nextline->line, '|');
		if (cp == 0)
			continue;	/* bad entry */
		rule = new_rule();

		chain = new_rule();

		strncpy(rule->name, cp+1, MAX_RULE_NAME_LENGTH);
		strncpy(chain->name, nextline->line, cp-nextline->line);
		chain->name[cp-nextline->line]='\0';

		node = tsearch(chain, &chaintree, rule_compare);
		if (*(rule_type **)node != chain) {
			free(chain);
			chain=*(rule_type **)node;	// chain is already there
		} else {
			if (lastchain != NULL)
				lastchain->next = chain;
			lastchain = chain;
			if (firstchain == NULL)
				firstchain = chain;
		}
		
		if (rule->name[0] == '%') {
			chain->pkts++;
			continue;
		}

		counters = iptc_read_counter(chain->name, chain->pkts, &handle);
		if (counters) {
			iptc_zero_counter(chain->name, chain->pkts, &handle);
			chain->pkts++;
		}

		/* use a binary tree to find rules with same name */
		node = tsearch(rule, &ruletree, rule_compare);
		if (*(rule_type **)node != rule) {
			free(rule);
			rule=*(rule_type **)node;
		} else {
			if (lastrule != NULL)
				lastrule->next = rule;
			lastrule = rule;
			if (*firstrule == NULL)
				*firstrule = rule;
		}
		if (counters) {
			rule->pkts += counters->pcnt;
			rule->bytes += counters->bcnt;
		} else {
			rule->pkts = 0;
			rule->bytes = 0;
		}
	}
	iptc_commit(&handle);
	iptables_ipac_init(0);
	free_tree(&ruletree);
	free_tree(&chaintree);
	return 0;
}

/*
 * Prepare ipt entry for such a funcs as insert, delete, append, replace rule
 * 
 */
static int
prepare_entry (char *chain, char *saddr, char *sport, char *daddr, char *dport,
		char *proto, char *targ, char *iface, struct ipt_entry **e)
{
	struct ipt_entry fw;
	unsigned int naddrs = 0;
	struct in_addr *addrs = NULL;
	struct iptables_match *m;
	struct iptables_target *target = NULL;
	struct iptables_target *t;
	size_t size;
	int inverse;

	memset(&fw, 0, sizeof(fw));

	if (!strcmp(proto, "all"))
		proto[0]='\0';

	if (iface && strlen(iface)>1) {
		if ((!strncmp(chain+strlen(chain)-2, "~o", 2)) ||
				(!strncmp(chain+strlen(chain)-3, "~fi", 3)) ||
				(!strncmp(chain+strlen(chain)-4, "~c_o", 4)) ||
			        (!strncmp(chain+strlen(chain)-5, "~c_fi", 5))) {
			inverse = check_inverse_type(iface);
			parse_interface(iface, fw.ip.iniface, fw.ip.iniface_mask);
	                fw.ip.invflags |= (inverse ? IPT_INV_VIA_IN : 0);
			fw.nfcache = NFC_IP_IF_IN;
		} else {
			inverse = check_inverse_type(iface);
			parse_interface(iface, fw.ip.outiface, fw.ip.outiface_mask);
	                fw.ip.invflags |= (inverse ? IPT_INV_VIA_OUT : 0);
			fw.nfcache = NFC_IP_IF_OUT;
		}
	} else
		fw.ip.invflags = 0;

	for (m = iptables_matches; m; m = m->next) {
		m->mflags = 0;
		m->used = 0;
	}

	for (t = iptables_targets; t; t = t->next) {
		t->tflags = 0;
		t->used = 0;
	}
	
	if (!iptc_is_chain(chain, handle)) {
		fprintf(stderr, "%s is not a chain\n", chain);
		return (1);
	}
	
	if (strlen(saddr)>2) {
		if (check_inverse_type(saddr))
			fw.ip.invflags |= IPT_INV_SRCIP;
		parse_hostnetworkmask(saddr, &addrs, &(fw.ip.smsk), &naddrs);
		if (naddrs>1)
			exit_error(PARAMETER_PROBLEM, 
				"Incorrect rule: more than 1 source address\n");
		fw.ip.src.s_addr = addrs[0].s_addr;
		fw.nfcache |= NFC_IP_SRC;
	}

	if (strlen(daddr)>2) {
		if (check_inverse_type(daddr))
			fw.ip.invflags |= IPT_INV_DSTIP;
		parse_hostnetworkmask(daddr, &addrs, &(fw.ip.dmsk), &naddrs);
		if (naddrs>1)
			exit_error(PARAMETER_PROBLEM, "Incorrect rule: more than 1 "
					    "destination address\n");
		fw.ip.dst.s_addr = addrs[0].s_addr;
		fw.nfcache |= NFC_IP_DST;
	}

	if ((sport[0]!='\0' || dport[0]!='\0') && proto[0]=='\0')
		exit_error(PARAMETER_PROBLEM, "Incorrect rule: source or "
		    "destination port specified while protocol is not\n");
	
	/* Loading target /if any/ */
	target = find_target(IPT_STANDARD_TARGET, LOAD_MUST_SUCCEED);
	size = sizeof(struct ipt_entry_target) + target->size;
	target->t = xcalloc(1, size);
	target->t->u.target_size = size;
	strcpy(target->t->u.user.name, targ);
	target->init(target->t, &fw.nfcache);

	if(check_inverse_type(proto))
		fw.ip.invflags |= IPT_INV_PROTO;

	if (proto[0] != '\0') {
		fw.ip.proto = parse_protocol(proto);
		fw.nfcache |= NFC_IP_PROTO;
	}

	if (proto[0] != '\0' && proto[0] != 'i') {
		m = find_proto(proto, LOAD_MUST_SUCCEED, 0);
		size = IPT_ALIGN(sizeof(struct ipt_entry_match)) + m->size;
		m->m = xcalloc(size, 1);
		m->m->u.match_size = size;
		strcpy(m->m->u.user.name, m->name);
		m->init(m->m, &fw.nfcache);

		if (sport[0]!='\0' || dport[0]!='\0') {
			if (!strcmp(proto, "tcp")) 
{
				struct ipt_tcp *tcpinfo = (struct ipt_tcp *)m->m->data;

				if (sport[0]!='\0')
					/* - T.Mohan 5/7/2001 */
					parse_tcp_ports(sport, tcpinfo->spts);

				if (dport[0]!='\0')
					/* - T.Mohan 5/7/2001 */
					parse_tcp_ports(dport, tcpinfo->dpts);
			}

			if (!strcmp(proto, "udp")) 
{
				struct ipt_udp *udpinfo = 
						(struct ipt_udp *)m->m->data;
				if (sport[0]!='\0')
					/* - T.Mohan 5/7/2001 */
					parse_udp_ports(sport, udpinfo->spts);

				if (dport[0]!='\0')
					/* - T.Mohan 5/7/2001 */
					parse_udp_ports(dport, udpinfo->dpts);
			}
		}
	}
	
	target->final_check(target->tflags);
	*e = generate_entry(&fw, iptables_matches, target->t);

	if (!handle) if (!(handle = iptc_init("filter")))
			    exit_error(PARAMETER_PROBLEM, 
				"iptables: %s\n", iptc_strerror(errno));
			
	return 0;
}

/*
 * Try to insert rule into kernel return 0 in case all right, 1 otherwise
 */
static int
insert_rule (char *chain, char *saddr, char *sport, char *daddr, char *dport,
		char *proto, char *targ, int rule_num, char *iface)
{
	struct ipt_entry *e = NULL;
	int ret=1;

	if (prepare_entry(chain, saddr, sport, daddr, dport, 
					    proto, targ, iface, &e)!=0)
		return (1);
	if (verbose>1) {
		printf("Inserting rule\n");
		print_firewall_line(e, handle);
	}
	ret &= iptc_insert_entry(chain, e, rule_num, &handle);
	free(e);
	return ret;
}

/*
 * Try to atomically replace rule in kernel return 0 in case all right, 1 otherwice
 */
static int
replace_rule (char *chain, char *saddr, char *sport, char *daddr, char *dport,
		char *proto, char *targ, int rule_num, char *iface)
{
	struct ipt_entry *e = NULL;
	int ret=1;

	if (prepare_entry(chain, saddr, sport, daddr, dport, proto, 
			targ, iface, &e)!=0)
		return (1);

	if (verbose>1) {
		printf("Replacing rule %d in '%s'\n", rule_num, chain);
		print_firewall_line(e, handle);
	}
	ret &= iptc_replace_entry(chain, e, rule_num, &handle);
	free(e);
	return ret;
}

/*
 * Try to append rule into kernel return 0 in case all right, 1 otherwice
 */

/* - T.Mohan 5/7/2001
 * append_rule () modified to pass struct iface_struct
 * parse_tcp_ports() & parse_udp_ports, correctly parses xxxx:xxxx style port rules
 */

static int
append_rule (char *chain, char *saddr, char *sport, char *daddr, char *dport, 
		char *proto, char *targ, char *iface)
{
	struct ipt_entry *e = NULL;
	
	if (prepare_entry(chain, saddr, sport, daddr, dport, proto, 
			targ, iface, &e)!=0)
		return (1);

	if (verbose>1) {
		printf("Appending rule to chain '%s'\n", chain);
		print_firewall_line(e, handle);
	}
	if (!iptc_append_entry(chain, e, &handle)) {
		fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
		return (1);
	}
	free(e);
	return 0;
}

static int
delete_rule (char *chain, char *saddr, char *sport, char *daddr, char *dport,
		char *proto, char *targ, char *iface)
{
	struct ipt_entry *e = NULL;
	unsigned char *mask = NULL;
	int ret=1;

	if (prepare_entry(chain, saddr, sport, daddr, dport, proto, 
			targ, iface, &e)!=0)
		return (1);

	if (verbose>1) {
		printf("Deleting rule\n");
		print_firewall_line(e, handle);
	}

	mask = make_delete_mask(e);
	ret &= iptc_delete_entry(chain, e, mask, &handle);
	free(e);
	return ret;
}

static int
delete_num_rule (char *chain, int num)
{
	struct ipt_entry *e = NULL;
	unsigned char *mask = NULL;
	int ret = 1;

	mask = make_delete_mask(e);
//	if (!iptc_delete_num_entry(chain, num, &handle)) {
//		fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
//		return (1);
//	}
	ret &= iptc_delete_num_entry(chain, num, &handle);
	free(e);
	return ret;
}



/** Setup chains if they doesn't exist 
 *
 */
static int 
setup_tables(void)
{
	if (verbose)
		fprintf(stderr, "Setup tables..\n");
	if (!iptc_is_chain("ipac~fi", handle))
		if (!iptc_create_chain("ipac~fi", &handle)) {
			fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
			return (1);
		}
	if (!iptc_is_chain("ipac~fo", handle))
		if (!iptc_create_chain("ipac~fo", &handle)) {
			fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
			return (1);
		}
	if (!iptc_is_chain("ipac~i", handle))
		if (!iptc_create_chain("ipac~i", &handle)) {
			fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
			return (1);
		}
	if (!iptc_is_chain("ipac~o", handle))
		if (!iptc_create_chain("ipac~o", &handle)) {
			fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
			return (1);
		}

	if ((!is_classic)) {
		if (!iptc_is_chain("ipac~c_fi", handle))
			if (!iptc_create_chain("ipac~c_fi", &handle)) {
				fprintf(stderr, "iptables: %s\n", 
							iptc_strerror(errno));
				return (1);
			}
		if (!iptc_is_chain("ipac~c_fo", handle))
			if (!iptc_create_chain("ipac~c_fo", &handle)) {
				fprintf(stderr, "iptables: %s\n", 
							iptc_strerror(errno));
				return (1);
			}
		if (!iptc_is_chain("ipac~c_i", handle))
			if (!iptc_create_chain("ipac~c_i", &handle)) {
				fprintf(stderr, "iptables: %s\n", 
							iptc_strerror(errno));
				return (1);
			}
		if (!iptc_is_chain("ipac~c_o", handle))
			if (!iptc_create_chain("ipac~c_o", &handle)) {
				fprintf(stderr, "iptables: %s\n", 
							iptc_strerror(errno));
				return (1);
			}
	}
	return 0;
};

static int
flush_acc_tables(void)
{
	raw_rule_type *d, *d1;

	if (iptc_is_chain("ipac~fi", handle))
		if (!iptc_flush_entries("ipac~fi", &handle)) {
			fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
			return (1);
		}
	if (iptc_is_chain("ipac~fo", handle))
		if (!iptc_flush_entries("ipac~fo", &handle)) {
			fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
			return (1);
		}
	if (iptc_is_chain("ipac~i", handle))
		if (!iptc_flush_entries("ipac~i", &handle)) {
			fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
			return (1);
		}
	if (iptc_is_chain("ipac~o", handle))
		if (!iptc_flush_entries("ipac~o", &handle)) {
			fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
			return (1);
		}

	if (!is_classic) {
		if (iptc_is_chain("ipac~c_fi", handle))
			if (!iptc_flush_entries("ipac~c_fi", &handle)) {
				fprintf(stderr, "iptables: %s\n", 
							iptc_strerror(errno));
				return (1);
			}
		if (iptc_is_chain("ipac~c_fo", handle))
			if (!iptc_flush_entries("ipac~c_fo", &handle)) {
				fprintf(stderr, "iptables: %s\n", 
							iptc_strerror(errno));
				return (1);
			}
		if (iptc_is_chain("ipac~c_i", handle))
			if (!iptc_flush_entries("ipac~c_i", &handle)) {
				fprintf(stderr, "iptables: %s\n", 
							iptc_strerror(errno));
				return (1);
			}
		if (iptc_is_chain("ipac~c_o", handle))
			if (!iptc_flush_entries("ipac~c_o", &handle)) {
				fprintf(stderr, "iptables: %s\n", 
							iptc_strerror(errno));
				return (1);
			}
	}
// Try to flush our old chains
	if (access_agent->get_raw_list("iptables", "", &d)) {
		fprintf(stderr, "access error\n");
		return 1;
	}
	d1=d;
	while(d) {
		if (strlen(d->name)>7)
			if (!memcmp(d->name, "%chain%", 7))
				if (iptc_is_chain(d->name+8, handle))
					iptc_flush_entries(d->name+8, &handle);
		d=d->next;
	}
	free_raw_list(d1);
	return 0;
}


int iptables_ipac_init(int flag)
{
	char *modprobe = NULL;

	if (!handle)
		handle = iptc_init("filter");

	if (!handle) {
//		try to insmod the module if iptc_init failed
		iptables_insmod("ip_tables", modprobe);
		handle = iptc_init("filter");
	}

	if (!handle) {
		fprintf(stderr, "ipac-ng: can't initialize iptables table `filter'\n"
		"\tis \"Network packet filtering (replaces ipchains)\"\n"
		"\tin Networking options of your kernel enabled?\n"
		"\tif so then you *have* to enable \"IP tables support\" in\n"
		"\t\"IP: Netfilter Configuration\" *and* \"Packet filtering\"\n"
		"\tsome lines below in your kernel configuration.\n"
		"\tPlease don't send bug reports\n"
		"\tif you failed to enable these features! \n"
		"\t You have to check that /usr/include/linux points to the\n"
		"\tright kernel's headers (somewhere in /usr/src/linux/include/linux?).\n"
		"\tIf it's not then you have to correct this and recompile ipac-ng\n"
		"---\nwbw, kaiser.\n\n"
		"\tiptables reported that %s\n", iptc_strerror(errno));
		exit (1);
	}
	return 0;
};

/* 
 * setup auth host - open access to AUTH_SERVER for all clients
 */
static int
setup_auth_host()
{
	user_list *list;

	access_agent->get_user_list(&list);
	while(list) {
		if (strncmp(list->login, "admin", 5)) {
			if (insert_rule("ipac~c_fo", authhost, "", list->ip, "", 
							"", "RETURN", 0, ""))
					return (1);
			if (insert_rule("ipac~c_fi",  list->ip, "", authhost, "", 
							"", "RETURN", 0, ""))
					return (1);
		}
		list=list->next;
	}
	return 0;
}

static int 
setup_rules(void)
{
	raw_rule_type *d, *d1;
	char targ[MAX_RULE_NAME_LENGTH+2];
	char chain[MAX_RULE_NAME_LENGTH+2];
	FILE *frunfile;

	if(access_agent->get_raw_list("iptables", "", &d)) {
		fprintf(stderr, "access error\n");
		return 1;
	}

	frunfile = fopen(RUNFILE, "w");	//needs to make some error handling later
	if (!frunfile) {
		fprintf(stderr, "%s: opening runfile \"%s\": %s\n",
					    me, RUNFILE, strerror(errno));
		return 1;
	}
	d1=d;
	while(d) 
	{
		/* Trying to implement hierarchic rules */
		if (is_classic)
			strcpy(targ, "");
		else 
			strcpy(targ, d->policy);
		
		strcpy(chain, d->dest); // %-)

		/* Are we dealing with new chain? if so create it */
		if (d->name[0]=='%') {
			if (!strncmp(d->name, "%chain%", 7)) {
				if (strlen(d->name)<8) {
					fprintf(stderr, "error: new "
							"chain name missing\n");
					return 1;
				}
				strcpy(targ, d->name+8);
				iptc_create_chain(targ, &handle);
				if (is_classic)
					fprintf(frunfile, "%s|%%\n", chain);
			} else {
				fprintf(stderr, "error: incorrect symbol %% "
						"in rule name\n");
				return 1;
			}
		} else
			if (!(((strlen(chain)>4) && 
				    (!memcmp(chain+strlen(chain)-4, "~c", 2))) ||
			    ((strlen(chain)>5) &&
				    (!memcmp(chain+strlen(chain)-5, "~c", 2)))))
				if (is_classic || 
					!strncmp(chain, "ipac", 4) ||
					!strncmp(chain, "admin", 5) ||
					((strlen(d->name)>3) && 
					    !memcmp(d->name+strlen(d->name)-3, 
								"~st", 3)))
					fprintf(frunfile,"%s|%s\n", chain, d->name);
				
		if (is_classic || 
				!strncmp(chain, "ipac", 4) ||
				!strncmp(chain, "admin", 5) ||
				((strlen(d->name)>3) && 
					!memcmp(d->name+strlen(d->name)-3, "~st", 3)))
			append_rule(chain, d->snet, d->sport, d->dnet, d->dport,
						    d->protocol, targ, d->iface);
		d=d->next;
	}
	fclose(frunfile);
	free_raw_list(d1);
	return 0;
}

/** 
 Setup all possible rules in iptables 
 */
int iptables_ipac_set(rule_type **firstrule, int first)
{
	unsigned int ref=0;

	if (verbose)
		fprintf(stderr, "Flushing accounting chains..\n");
	flush_acc_tables();
	if (verbose)
		fprintf(stderr, "Setting up acc chains..\n");
	if (first==1)
		setup_tables();
	if (verbose)
		fprintf(stderr, "Setting up accounting rules..\n");
    	setup_rules();
	
	if (first==1) {
		iptc_get_references(&ref, "ipac~fi", &handle);
		if (ref!=0) {
			delete_rule("OUTPUT", "0/0", "", "0/0", "", "", "ipac~i", "");
			delete_rule("FORWARD", "0/0", "", "0/0", "", "", "ipac~fi", "");
		}
		iptc_get_references(&ref, "ipac~fo", &handle);
		if ((ref!=0) && (first==1)) {
			delete_rule("INPUT", "0/0", "", "0/0", "", "", "ipac~o", "");
			delete_rule("FORWARD", "0/0", "", "0/0", "", "", "ipac~fo", "");
		}
		insert_rule("OUTPUT", "0/0", "", "0/0", "", "", "ipac~i", 0, "");
		insert_rule("INPUT", "0/0", "", "0/0", "", "", "ipac~o", 0, "");
		insert_rule("FORWARD", "0/0", "", "0/0", "", "", "ipac~fo", 0, "");
		insert_rule("FORWARD", "0/0", "", "0/0", "", "", "ipac~fi", 0, "");
		if (!is_classic) {
			iptc_get_references(&ref, "ipac~c_fi", &handle);
			if (ref!=0) {
				delete_rule("OUTPUT", "", "", "", "", "", "ipac~c_i", "");
				delete_rule("FORWARD", "", "", "", "", "", "ipac~c_fi", "");
			}
			iptc_get_references(&ref, "ipac~c_fo", &handle);
			if (ref!=0) {
				delete_rule("INPUT", "", "", "", "", "", "ipac~c_o", "");
				delete_rule("FORWARD", "", "", "", "", "", "ipac~c_fo", "");
			}
			insert_rule("OUTPUT", "", "", "", "", "", "ipac~c_i", 0, "");
			insert_rule("INPUT", "", "", "", "", "", "ipac~c_o", 0, "");
			insert_rule("FORWARD", "", "", "", "", "", "ipac~c_fi", 0, "");
			insert_rule("FORWARD", "", "", "", "", "", "ipac~c_fo", 0, "");
		}
	}
	if (!is_classic && authhost!=NULL && strlen(authhost)>2)
		setup_auth_host();
	iptc_commit(&handle);
	iptables_ipac_init(0);
	return 0;
}

int 
iptables_ipac_read(rule_type **firstrule)
{
	struct runfile_line_type *runfile;

	runfile = read_runfile();
	if (runfile == NULL)
		return 1;

	return read_iptables(runfile, firstrule);
}

/*
 * Accept packets to/from host with login name
 */ 
int 
iptables_ipac_accept(char *login)
{
	raw_rule_type *d;
	int logged_in=0;
	unsigned int tci=-1,tco=-1,tcfi=-1,tcfo=-1;
	char *name;
	FILE *frunfile;
	char n[MAX_RULE_NAME_LENGTH+10];
	char n1[MAX_RULE_NAME_LENGTH+10];
	char serv_name[MAX_RULE_NAME_LENGTH+10];

	strcpy(serv_name, login);
	name = strsep(&login, " ");
	while(strsep(&login, " "));

	iptables_ipac_init(0);
	access_agent->get_raw_list("iptables", name, &d);


	strcpy(n, name); strcat(n, "~c_i");

	if (insert_rule(n, "", "", "", "", "", "", 2, "")) {
		logged_in = 1;
		delete_num_rule(n, 2);
	}

	while(d && !memcmp(d->name, "%chain%", 7)) d = d->next; // skip chains

	strcpy(n, name); strcat(n, " ");
	strcpy(n1, name); strcat(n1, "~st");

	if (!logged_in) {
		while(lock(LOCKFILE));
		frunfile = fopen(RUNFILE, "a");
    		if (!frunfile) {
            		fprintf(stderr, "%s: opening runfile \"%s\": %s\n",
                                        me, RUNFILE, strerror(errno));
	                return 1;
    		}
		clear_runfile(name, 1);
		while (d) {
			if ((!strncmp(d->name, n, strlen(n))) ||
				    (!strcmp(d->name, n1))) {
				if (!(((strlen(d->dest)>4) && 
					    (!memcmp(d->dest+strlen(d->dest)-4, "~c", 2))) ||
					    ((strlen(d->dest)>5) &&
					    (!memcmp(d->dest+strlen(d->dest)-5, "~c", 2))))) {
					append_rule(d->dest, d->snet, d->sport, d->dnet, 
				    	    d->dport, d->protocol, "RETURN", d->iface);
				} else {
				        append_rule(d->dest, d->snet, d->sport, d->dnet, 
					    d->dport, d->protocol, "DROP", d->iface);
				}
				if (((!strncmp(d->dest+strlen(d->dest)-2, "~i", 2)) ||
				    (!strncmp(d->dest+strlen(d->dest)-2, "~o", 2)) ||
				    (!strncmp(d->dest+strlen(d->dest)-3, "~fi", 3)) ||
				    (!strncmp(d->dest+strlen(d->dest)-3, "~fo", 3))) &&
				    (strncmp(d->name+strlen(d->name)-3, "~st", 3))) {
					fprintf(frunfile, "%s|%s\n", d->dest, d->name);
				}
			}
	    		d=d->next;
		}
		// delete old stopper
		strcpy(n1, name); strcat(n1, "~c_fi"); delete_num_rule(n1, 0);
		strcpy(n1, name); strcat(n1, "~c_fo"); delete_num_rule(n1, 0);
		strcpy(n1, name); strcat(n1, "~c_i"); delete_num_rule(n1, 0);
		strcpy(n1, name); strcat(n1, "~c_o"); delete_num_rule(n1, 0);
		access_agent->get_raw_list("iptables", name, &d);
		while(d && !memcmp(d->name, "%chain%", 7)) d = d->next;
		fclose(frunfile);
		unlock(LOCKFILE);
	}	

	while(d) {
		if (!strncmp(d->dest+strlen(d->dest)-4, "~c_i", 4))
			tci++;
		if (!strncmp(d->dest+strlen(d->dest)-4, "~c_o", 4))
			tco++;
		if (!strncmp(d->dest+strlen(d->dest)-5, "~c_fi", 5))
			tcfi++;
		if (!strncmp(d->dest+strlen(d->dest)-5, "~c_fo", 5))
			tcfo++;
		if (!strcmp(d->name, serv_name)) {
			if (!strncmp(d->dest+strlen(d->dest)-4, "~c_i", 4)) {
				replace_rule(d->dest, d->snet, d->sport, d->dnet,
				    d->dport, d->protocol, "RETURN", tci, d->iface);
			} else if (!strncmp(d->dest+strlen(d->dest)-4, "~c_o", 4)) {
				replace_rule(d->dest, d->snet, d->sport, d->dnet,
				    d->dport, d->protocol, "RETURN", tco, d->iface);
			} else if (!strncmp(d->dest+strlen(d->dest)-5, "~c_fi", 5)) {
				replace_rule(d->dest, d->snet, d->sport, d->dnet,
				    d->dport, d->protocol, "RETURN", tcfi, d->iface);
			} else if (!strncmp(d->dest+strlen(d->dest)-5, "~c_fo", 5)) {
				replace_rule(d->dest, d->snet, d->sport, d->dnet,
				    d->dport, d->protocol, "RETURN", tcfo, d->iface);
			}
		}
	    	d=d->next;
	}
	iptc_commit(&handle);
	iptables_ipac_init(0);
	return 0;
}

static int
clear_runfile(char *serv_name, int greedy)
{
	FILE *frunfile;
        char runfile_line[MAX_RULE_NAME_LENGTH + 50], *cp;
        struct runfile_line_type *result, *lastline, *cur;
	int tmp;
	
	frunfile=fopen(RUNFILE, "r");
	if (!frunfile) {
		fprintf(stderr, "%s: opening runfile \"%s\": %s\n",
					    me, RUNFILE, strerror(errno));
		return 1;
	}
	result = NULL; lastline = NULL;

        while(fgets(runfile_line, MAX_RULE_NAME_LENGTH+50, frunfile) != NULL) {
    		tmp++;
                cp = strchr(runfile_line, '\n');
                if (cp)
                        *cp = '\0';
                        if (*runfile_line == '#')
	                        continue;
                        cur = (struct runfile_line_type *)
                                    xmalloc(sizeof(struct runfile_line_type));
	                cur->line = xstrdup(runfile_line);
                        cur->next = NULL;
                        if (result == NULL)
	                        result = cur;
                        else
	                        lastline->next = cur;
                        lastline = cur;
	}																																						    
        if (!feof(frunfile)) {
	        fprintf(stderr, "%s: reading \"%s\": %s\n",
					me, RUNFILE, strerror(errno));
		fclose(frunfile);
		destroy_runfile_lines(result);
		result = NULL;
		return 1;
	}																																																			    
	fclose(frunfile);
	frunfile=fopen(RUNFILE, "w");
	if (!frunfile) {
		fprintf(stderr, "%s: opening runfile \"%s\": %s\n",
					    me, RUNFILE, strerror(errno));
		return 1;
	}
	for (cur=result; cur!=NULL; cur=cur->next) {
		cp = strchr(cur->line, '|');
		if (cp == 0)
			continue;
		if (strcmp(cp+1, serv_name)) {
			if (greedy) {
				if (strncmp(cp+1, serv_name, strlen(serv_name)))
					fprintf(frunfile, "%s\n", cur->line);
			} else
				fprintf(frunfile, "%s\n", cur->line);
		}
	}
	fclose(frunfile);
	destroy_runfile_lines(result);
	return 0;
}

/* 
 * Deny packet crossing to/from user login
 * Warning: there are 2 possible cases:
 *	a) login=user_name
 *		in this case we have to close all services
 *	b) login="user_name - service_name"
 *		in such case we only need to close that service
 * there may be some traffic on the counters for this user. drop this info to spool
 */
int 
iptables_ipac_deny(char *login)
{
	raw_rule_type *d, *d1;
	int tci=-1,tco=-1,tcfi=-1,tcfo=-1, tmp=0;
	char *name;
	char n[MAX_RULE_NAME_LENGTH+10];
	char n1[MAX_RULE_NAME_LENGTH+10];
	char serv_name[MAX_RULE_NAME_LENGTH+10];
	timestamp_t t;
	FILE *spool;
	rule_type *rule = NULL, *lastrule = NULL, *firstrule = NULL, *rule1;
	void *ruletree = NULL;
	void *node;
        struct ipt_counters *counters = NULL;

	strcpy(serv_name, login);
	name = strsep(&login, " ");
	while(strsep(&login, " "));

	iptables_ipac_init(0);
	access_agent->get_raw_list("iptables", name, &d);

	while(d && !memcmp(d->name, "%chain%", 7)) d = d->next; // skip chains

	d1 = d;

	// entirely delete
	if (!strcmp(serv_name, name)) {
		while(lock(LOCKFILE));
		while(d) {
			if (!(((!strncmp(d->dest+strlen(d->dest)-2, "~i", 2)) ||
			    (!strncmp(d->dest+strlen(d->dest)-2, "~o", 2)) ||
			    (!strncmp(d->dest+strlen(d->dest)-3, "~fi", 3)) ||
			    (!strncmp(d->dest+strlen(d->dest)-3, "~fo", 3))) &&
			    (strncmp(d->name+strlen(d->name)-3, "~st", 3)))) {
				d = d->next;
				continue;
			}

			if (!strncmp(d->dest+strlen(d->dest)-2, "~i", 2)) {
				tci++;
				counters = iptc_read_counter(d->dest, tci, &handle);
				if (!iptc_zero_counter(d->dest, tci, &handle))
				    	fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
			} 
			if (!strncmp(d->dest+strlen(d->dest)-2, "~o", 2)) {
				tco++;
				counters = iptc_read_counter(d->dest, tco, &handle);
				if (!iptc_zero_counter(d->dest, tco, &handle))
				    	fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
			}
			if (!strncmp(d->dest+strlen(d->dest)-3, "~fi", 3)) {
				tcfi++;
				counters = iptc_read_counter(d->dest, tcfi, &handle);
				if (!iptc_zero_counter(d->dest, tcfi, &handle))
				    	fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
	        	}
			if (!strncmp(d->dest+strlen(d->dest)-3, "~fo", 3)) {
				tcfo++;
				counters = iptc_read_counter(d->dest, tcfo, &handle);
				if (!iptc_zero_counter(d->dest, tcfo, &handle))
				    	fprintf(stderr, "iptables: %s\n", iptc_strerror(errno));
			}
			rule = new_rule();
			strcpy(rule->name, d->name);
			/* use a binary tree to find rules with same name */
			node = tsearch(rule, &ruletree, rule_compare);
			if (*(rule_type **)node != rule) {
				free(rule);
				rule=*(rule_type **)node;
			} else {
				if (lastrule != NULL)
					lastrule->next = rule;
				lastrule = rule;
				if (firstrule == NULL)
					firstrule = rule;
			}
			rule->pkts += counters->pcnt;
			rule->bytes += counters->bcnt;
			d=d->next;
		}
		rule = firstrule;
		rule1 = firstrule;
		while(rule) {
			if (rule->pkts == 0) {
				if (rule == firstrule) {
					firstrule = rule->next;
					rule1 = rule->next;
				} else
					rule1->next = rule->next;
				free(rule);
				rule = rule->next;
			} else {
				rule1 = rule;
				rule = rule->next;
				tmp++;
			}
		}
		
		if (tmp>0) {
			spool=fopen(spoolfile, "a");
			if (spool == NULL) {
				fprintf(stderr, "cant open spool file '%s' for append "
					"some traffic may be lost.\n%s\n", spoolfile, strerror(errno));
	    			exit (1);
			}

			time(&t);
			fprintf(spool, "BILL\n%lu 1\n", t);
			fprintf(spool, "( %s\n", hostname);

			rule = firstrule;
			while(rule) {
				fprintf(spool, "%llu %llu |%s|\n", rule->bytes, rule->pkts, rule->name);
				rule = rule->next;
			}
			fprintf(spool, ")\n\n");
			fclose(spool);
		}
	        free_tree(&ruletree);
		d = d1;

		strcpy(n1, name); strcat(n1, "~i"); iptc_flush_entries(n1, &handle);
		strcpy(n1, name); strcat(n1, "~o"); iptc_flush_entries(n1, &handle);		
		strcpy(n1, name); strcat(n1, "~fi"); iptc_flush_entries(n1, &handle);		
		strcpy(n1, name); strcat(n1, "~fo"); iptc_flush_entries(n1, &handle);
		strcpy(n1, name); strcat(n1, "~c_i"); iptc_flush_entries(n1, &handle);
		strcpy(n1, name); strcat(n1, "~c_o"); iptc_flush_entries(n1, &handle);
		strcpy(n1, name); strcat(n1, "~c_fi"); iptc_flush_entries(n1, &handle);
		strcpy(n1, name); strcat(n1, "~c_fo"); iptc_flush_entries(n1, &handle);
		strcpy(n1, name); strcat(n1, "~st");
		// find and insert stoppers
		while(d) {
			if (!strcmp(d->name, n1))
				append_rule(d->dest, d->snet, d->sport, d->dnet, 
				        d->dport, d->protocol, "DROP", NULL);
			d=d->next;
		}
		strcpy(n1, name); strcat(n1, " ");
		// greedy deletion
		clear_runfile(n1, 1);
		unlock(LOCKFILE);
	}

	d = d1;

	tci=-1; tco=-1; tcfi=-1; tcfo=-1;

	strcpy(n, name); strcat(n, " ");
	strcpy(n1, name); strcat(n1, "~st");

	while(d) {
		if (!strncmp(d->dest+strlen(d->dest)-4, "~c_i", 4))
			tci++;
		else if (!strncmp(d->dest+strlen(d->dest)-4, "~c_o", 4))
			tco++;
		else if (!strncmp(d->dest+strlen(d->dest)-5, "~c_fi", 5))
			tcfi++;
        	else if (!strncmp(d->dest+strlen(d->dest)-5, "~c_fo", 5))
			tcfo++;
		if (!strcmp(d->name, serv_name)) {
			if (!strncmp(d->dest+strlen(d->dest)-4, "~c_i", 4)) {
				replace_rule(d->dest, d->snet, d->sport, d->dnet,
				    d->dport, d->protocol, "DROP", tci, NULL);
			} else if (!strncmp(d->dest+strlen(d->dest)-4, "~c_o", 4)) {
				replace_rule(d->dest, d->snet, d->sport, d->dnet,
				    d->dport, d->protocol, "DROP", tco, NULL);
			} else if (!strncmp(d->dest+strlen(d->dest)-5, "~c_fi", 5)) {
				replace_rule(d->dest, d->snet, d->sport, d->dnet,
				    d->dport, d->protocol, "DROP", tcfi, NULL);
			} else if (!strncmp(d->dest+strlen(d->dest)-5, "~c_fo", 5)) {
				replace_rule(d->dest, d->snet, d->sport, d->dnet,
				    d->dport, d->protocol, "DROP", tcfo, NULL);
			}
		}
	    	d=d->next;
	}
	iptc_commit(&handle);
	iptables_ipac_init(0);
	return 0;
}

int
iptables_ipac_check(void){
	int tmp=0;	

	if ((iptc_is_chain("ipac~fi", handle) +
		iptc_is_chain("ipac~fo", handle))!=2)
		return 1;
	iptc_get_references(&tmp, "ipac~fo", &handle);
	if (tmp==0)
		return 1;
	iptc_get_references(&tmp, "ipac~fi", &handle);
	if (tmp==0)
		return 1;
	return 0;
}

void
flush_remove_chain(char *ch_name)
{
	if (!handle)
		iptables_ipac_init(0);
	if (iptc_is_chain(ch_name, handle)) {
		iptc_flush_entries(ch_name, &handle);
		iptc_delete_chain(ch_name, &handle);
		iptc_commit(&handle);
		iptables_ipac_init(0);	// always do init after commit!
	}
}


int
iptables_ipac_remove_user(char *login)
{
	char tmp[MAX_RULE_NAME_LENGTH];
	
	strcpy(tmp, login);
	strcpy(tmp+strlen(tmp), "~fo");
	flush_remove_chain(tmp);
	strcpy(tmp, login);
	strcpy(tmp+strlen(tmp), "~fi");
	flush_remove_chain(tmp);
	strcpy(tmp, login);
	strcpy(tmp+strlen(tmp), "~c_fi");
	flush_remove_chain(tmp);
	strcpy(tmp, login);
	strcpy(tmp+strlen(tmp), "~c_fo");
	flush_remove_chain(tmp);
	strcpy(tmp, login);
	strcpy(tmp+strlen(tmp), "~o");
	flush_remove_chain(tmp);
	strcpy(tmp, login);
	strcpy(tmp+strlen(tmp), "~i");
	flush_remove_chain(tmp);
	strcpy(tmp, login);
	strcpy(tmp+strlen(tmp), "~c_i");
	flush_remove_chain(tmp);
	strcpy(tmp, login);
	strcpy(tmp+strlen(tmp), "~c_o");
	flush_remove_chain(tmp);
	return 0;
}

/*
 * Deny packets from any to any.
 * Make by inserting drop rule to the forward chain
 */
int 
iptables_ipac_alarm(void)
{
	int ret;
        if (!handle)
                iptables_ipac_init(0);
	ret = insert_rule("FORWARD", "", "", "", "", "", "DROP", 0, "");
	iptc_commit(&handle);
	return ret;
}
