#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# UserAction.py
#
# DESCRIPTION:
# A generic UIdriver base for common message-based GUI toolkits.
#
# NOTES:
#

from gnue.common.apps import GConfig
from gnue.forms import GFKeyMapper
import os

_iconpath = GConfig.getInstalledBase('form_images', 'common_images')
_iconset = gConfigForms('IconSet')

######################################################################
#
#
#

class UserAction:

  def __init__(self, event, description, help, canToggle=0):
    self.event = event
    self.canToggle = canToggle
    self.__iconloc = {}

    # Description is short; appropriate for mouse-over tips
    self.description = description

    # Help is more detailed; suitable for a manual or online help
    self.help = help

  def getIconLocation(self, format="png",size="32x32"):
    try:
      return self.__iconloc[size + format]
    except:
      iconset = _iconset == 'auto' and forms_ui.default_iconset or _iconset
      loc = os.path.join(_iconpath, 'forms', iconset,
          '%s-%s.%s' % (self.event.lower(),size, format))
      if not os.path.isfile(loc):
        loc = os.path.join(_iconpath, 'forms', 'default',
            '%s-%s.%s' % (self.event.lower(),size, format))
        if not os.path.isfile(loc):
          loc = None

      self.__iconloc[size + format] = loc
      return loc

  def getHotKeyText(self, metamappings={}, separator='+'):
    if metamappings:
      m = copy.copy(_baseKeyMap)
      m.update(metamappings)
    else:
      m = _baseKeyMap
    return GFKeyMapper.KeyMapper.getEventKeystrokeRepr(self.event,
        metamappings=m, separator=separator)

  def getHotKeyTuple (self):
    return GFKeyMapper.KeyMapper.getUIEventKeyStroke (self.event)


######################################################################
#
# Build a map of the standard user actions
#

userActionMap = {}

for action in [

   UserAction(
       event="COMMIT",
       description=u_("Save all changes to the database."),
       help=u_("""Save all changes to the database.""") ),

    UserAction(
       event="ROLLBACK",
       description=u_("Clear form."),
       help=u_("Clear form and revert back to your original data.") ),

    UserAction(
       event="PRINTOUT",
       description=u_("Perform print routine for this form."),
       help=u_("Perform print routine for this form.") ),

    UserAction(
       event="EXIT",
       description=u_("Leave the application."),
       help=u_("Leave the application.") ),

    UserAction(
       event="CUT",
       description=u_("Cut the selected text"),
       help=u_("Cut the selected text") ),

    UserAction(
       event="COPY",
       description=u_("Copy the selected text"),
       help=u_("Copy the selected text") ),

    UserAction(
       event="COPYRECORD",
       description=u_("Copy the current record to the clipboard."),
       help=u_("Copy the current record to the clipboard. The record is copied with tabs separating values.") ),

    UserAction(
       event="COPYSCREEN",
       description=u_("Copy the current screen to the clipboard as plain text."),
       help=u_("Copy the current screen to the clipboard as plain text.") ),

    UserAction(
       event="PASTE",
       description=u_("Paste text into the current field."),
       help=u_("Paste text into the current field.") ),

    UserAction(
       event="SELECTALL",
       description=u_("Select all text."),
       help=u_("Select all text.") ),

    UserAction(
       event="PREVENTRY",
       description=u_("Move to the previous logical field."),
       help=u_("Move to the previous logical field.") ),

    UserAction(
       event="NEXTENTRY",
       description=u_("Move to the next logical field."),
       help=u_("Move to the next logical field.") ),

    UserAction(
       event="FIRSTRECORD",
       description=u_("Jump to the first record in the current block."),
       help=u_("Jump to the first record in the current block.") ),

    UserAction(
       event="PREVRECORD",
       description=u_("Jump to the previous record in the current block."),
       help=u_("Jump to the previous record in the current block.") ),

    UserAction(
       event="NEXTRECORD",
       description=u_("Jump to the next record in the current block."),
       help=u_("Jump to the next record in the current block.") ),

    UserAction(
       event="LASTRECORD",
       description=u_("Jump to the last record in the current block."),
       help=u_("Jump to the last record in the current block.") ),

    UserAction(
       event="JUMPPROMPT",
       description= u_("Prompts for a record number to which the system "
                      "should jump."),
       help=u_('Prompts for a record number to which the system should jump.') ),

    UserAction(
       event="NEWRECORD",
       description=u_("Insert a new record into the current block."),
       help=u_("Insert a new record into the current block.") ),

    UserAction(
       event="MARKFORDELETE",
       description=u_('Mark record for removal at next commit.'),
       help=u_('Mark record for removal at next commit.') ),

    UserAction(
       event="NEXTBLOCK",
       description=u_('Navigate to the next data block.'),
       help=u_('Navigate to the next data block.') ),

    UserAction(
       event="PREVBLOCK",
       description=u_('Navigate to the previous data block.'),
       help=u_('Navigate to the previous data block.') ),

    UserAction(
       event="NEXTPAGE",
       description=u_('Navigate to the next page.'),
       help=u_('Navigate to the next page.') ),

    UserAction(
       event="PREVPAGE",
       description=u_('Navigate to the previous page.'),
       help=u_('Navigate to the previous page.') ),

    UserAction(
       event="ENTERQUERY",
       description=u_('Switch to query mode. Select Execute Query once your criteria has been entered.'),
       help=u_('Switch to query mode. Select Execute Query once your criteria has been entered.'),
       canToggle='CANCELQUERY' ),

    UserAction(
       event="ENTEREDIT",
       description=u_('Switch to edit mode.  This will allow you to edit the current entries.'),
       help=u_('Switch to edit mode.'),
       canToggle='CANCELEDIT' ),

    UserAction(
       event="COPYQUERY",
       description=u_('Switch to query mode (if not already) and retrieve the last query parameters. Select Execute Query once your criteria has been entered.'),
       help=u_('Switch to query mode (if not already) and retrieve the last query parameters. Select Execute Query once your criteria has been entered.') ),

    UserAction(
       event="CANCELQUERY",
       description=u_('Cancel query mode.'),
       help=u_('Cancel query mode.') ),

    UserAction(
       event="EXECQUERY",
       description=u_('Perform a query and show the results.'),
       help=u_('Perform a query and show the results.') ),

    UserAction(
       event="ABOUT",
       description=u_('Display info about GNUe Forms.'),
       help=u_('Display info about GNUe Forms.') ),

    UserAction(
       event="HELPKEYS",
       description=u_('Display the keystroke bindings currently in use.'),
       help=u_('Display the keystroke bindings currently in use.') ),

  ]:

  userActionMap[action.event] = action


_baseKeyMap = {
  'META': 'Alt',
  'CONTROL': 'Ctrl',
  'SHIFT': 'Shift',
  'INSERT': 'Ins',
  'DELETE': 'Del',
  'HOME': 'Home',
  'END': 'End',
  'PAGEUP': 'PgUp',
  'PAGEDOWN': 'PgDn',
  'UP': 'Up',
  'DOWN': 'Down',
  'LEFT': 'Left',
  'RIGHT': 'Right',
  'TAB': 'Tab',
  'ENTER': 'Enter',
  'RETURN': 'Enter',
  'BACKSPACE': 'Back' }
