/***************************************************************************
                          FLManager.h  -  description
                             -------------------
    begin                : Sun Jul 15 2001
    copyright            : (C) 2001,2002 by Federico Albujer Zornoza
    email                : mail@infosial.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef FLMANAGER_H
#define FLMANAGER_H

#include <qvariant.h>
#include <qstring.h>
#include <qwidget.h>
#include <qapplication.h>
#include <qsqldatabase.h>
#include <qsqlquery.h>
#include <qmessagebox.h>
#include <qsqlcursor.h>
#include <qsqlrecord.h>
#include <qsqldriver.h>
#include <qsqlfield.h>
#include <qdom.h>
#include <qfile.h>
#include <qdatetime.h>
#include <qprogressdialog.h>

#include "FLTableMetaData.h"

class FLFieldMetaData;
class FLRelationMetaData;
class FLCompoundKey;
class FLSqlQuery;
class FLParameterQuery;
class FLGroupByQuery;
class FLAction;

 /**  Esta clase sirve como administrador de la base de datos.

      Encargada de abrir los formularios u obtener sus definiciones (ficheros .ui).
      Tambien mantiene los metadatos de todas la tablas de la base de
      datos, ofreciendo la posibilidad de recuperar los metadatos
      mediante objetos FLTableMetaData de una tabla dada.

      @author Federico Albujer Zornoza
      @version 0.4 */
class FLManager
{
	public:

	/** constructor */
  	FLManager ();

	/** destructor */
  	~FLManager ();

	/** Para obtener definicion de una tabla de la base de datos, a partir de un fichero XML.

            El nombre de la tabla corresponde con el nombre del fichero mas la extensin ".mtd"
            que contiene en XML la descripcin de la tablas. Este mtodo escanea el fichero
            y construye/devuelve el objeto FLTableMetaData correspondiente, adems
            realiza una copia de estos metadatos en una tabla de la misma base de datos
            para poder determinar cuando ha sido modificados y as, si es necesario, reconstruir
            la tabla para que se adapte a la nuevos metadatos. NO SE HACEN
            CHEQUEOS DE ERRORES SINTCTICOS EN EL XML.

            IMPORTANTE : Para que estos mtodos funcionen correctamente, es estrictamente
                                       necesario haber creado la base de datos con codificacin UNICODE;
                                       "createdb -E UNICODE facturalux".
      	@param n	Nombre de la tabla de la base de datos de la que obtener los metadatos
      	@return	Un objeto FLTableMetaData con los metadatos de la tabla solicitada */
  	static FLTableMetaData *metadata (const QString & n);

        /** Para obtener una consulta de la base de datos, a partir de un fichero XML.

            El nombre de la consulta corresponde con el nombre del fichero mas la extensin ".qry"
            que contiene en XML la descripcin de la consulta. Este mtodo escanea el fichero
            y construye/devuelve el objeto FLSqlQuery. NO SE HACEN
            CHEQUEOS DE ERRORES SINTCTICOS EN EL XML.

      	@param n	Nombre de la consulta de la base de datos que se quiere obtener
      	@return	Un objeto FLSqlQuery que representa a la consulta que se quiere obtener */
  	static FLSqlQuery * query (const QString & n);

        /** Obtiene la definicin de una accin a partir de su nombre.

            Este mtodo busca en el fichero actions.xml la accin que se le pasa
            como nombre y construye y devuelve el objeto FLAction correspondiente.
            NO SE HACEN CHEQUEOS DE ERRORES SINTCTICOS EN EL XML.

            @param  n   Nombre de la accion
            @return Un objeto FLAction con la descripcion de la accion */
       static FLAction * action( const QString &n );

	/** Comprueba si existe la tabla especificada en la base de datos.

      	@param n	Nombre de la tabla que se quiere comprobar si existe
      	@return	TRUE si existe la tabla, FALSE en caso contrario **/
  	static bool existsTable (const QString & n);

	/** Crea una nueva secuencia en la tabla de secuencias(contadores)

	  @author Ral Alexis Betancort Santana
	  @param name Nombre de la secuencia a crear
	  @return Siempre devuelve TRUE, hay que rectificarlo

	  */
	static bool createSequence(const QString &name);
	/** Comprueba si los metadatos definidos actualmente para una tabla en un fichero .mtd,
            difieren de los que tiene actualmente esa tabla en la base de datos (difieren con
            la copia que se tiene actualmente de ellos).

	     Cuando se realizan modificaciones en la definicin
	     de una tabla, es probable que esa tabla ya exista en la base de datos. Con este
	     mtodo podemos determinar posteriormente si lo metadatos definidos actualmente
	     difieren de los de la tabla existente en la base de datos.

	     @param   n Nombre de la tabla a comprobar.
	     @return	TRUE si los metadatos que hemos definido en FLManager::metadata()
	     			para una tabla (nuevos) son iguales a los metadatos de la tabla que tenemos
	     			en la base de datos (viejos), y FALSE en caso contrario. */
	static bool checkMetaData(const QString &n);

        /** Esta funcin es esencialmente igual a la anterior, se proporciona por conveniencia.

              Compara los metadatos de dos tablas,  la definicin en XML de esas dos tablas se
              pasan como dos cadenas de caracteres.

	     @param   mtd1 Cadena de caracteres con XML que describe la primera tabla.
            @param   mtd2 Cadena de caracteres con XML que describe la primera tabla.
	     @return	TRUE si las dos descripciones son iguales, y FALSE en caso contrario. */
	static bool checkMetaData(const QString &mtd1, const QString &mtd2)
        {
            return ( mtd1 == mtd2 );
        }

	/** Modifica la estructura o metadatos de una tabla, preservando los posibles datos
	    que pueda contener.
	
	    Segn la definicin existente en un momento dado de los metadatos en el fichero .mtd, este
	    mtodo reconstruye la tabla con esos metadatos sin la prdida de informacin o datos,
	    que pudieran existir en ese momento en la tabla.
	
	    @param n Nombre de la tabla a reconstruir.
           @return TRUE si la modificacin tuvo xito.*/
	static bool alterTable(const QString &n);

        /** Esta funcin es esencialmente igual a la anterior, se proporciona por conveniencia.

              Modifica la estructura de una tabla dada, preservando los datos. La nueva
              estructura y la vieja se pasan en cadenas de caracteres con la descripcion XML.

              @param n Nombre de la tabla a reconstruir.
              @param mtd1 Descripcion en XML de la vieja estructura.
              @param mtd2 Descripcion en XML de la nueva estructura.
              @return TRUE si la modificacin tuvo xito. */
        static bool alterTable(const QString &mtd1, const QString &mtd2);

	/** Crea una tabla en la base de datos.

      	@param n	Nombre de la tabla que se quiere crear
      	@return	Un objeto FLTableMetaData con los metadatos de la tabla que se ha creado, o
      			0 si no se pudo crear la tabla */
  	static FLTableMetaData *createTable (const QString & n);

        /** Crea una tabla en la base de datos.

      	@param tmd  Metadatos de la tabla
      	@return	Un objeto FLTableMetaData con los metadatos de la tabla que se ha creado, o
      			0 si no se pudo crear la tabla */
  	static FLTableMetaData *createTable (FLTableMetaData * tmd);

	/** Devuelve el contenido del valor de de un campo formateado para ser reconocido
		por la base de datos PostgreSQL en condiciones LIKE, dentro de la clausura WHERE de SQL.

		Este mtodo toma como parametros los metadatos del campo definidos con
		FLFieldMetaData. Adems de TRUE y FALSE como posibles valores de un campo
		lgico tambin acepta los valores S y No (o su traduccin al idioma correspondiente).
		Las fechas son adaptadas al forma AAAA-MM-DD, que es el formato reconocido por PostgreSQL .

		@param  fMD Objeto FLFieldMetaData que describre los metadatos para el campo
		@param  v   Valor que se quiere formatear para el campo indicado */
	static QString formatValueLike( FLFieldMetaData *fMD, const QVariant & v );

	/** Este mtodo hace lo mismo que el anterior, y se suministra por conveniencia.

		@param  t   Tipo de datos del valor
		@param  v   Valor que se quiere formatear para el campo indicado */
	static QString formatValueLike( int t, const QVariant & v );

	/** Devuelve el contenido del valor de de un campo formateado para ser reconocido
		por la base de datos PostgreSQL, dentro de la clausura WHERE de SQL.

		Este mtodo toma como parametros los metadatos del campo definidos con
		FLFieldMetaData. Adems de TRUE y FALSE como posibles valores de un campo
		lgico tambin acepta los valores S y No (o su traduccin al idioma correspondiente).
		Las fechas son adaptadas al forma AAAA-MM-DD, que es el formato reconocido por PostgreSQL .

		@param  fMD Objeto FLFieldMetaData que describre los metadatos para el campo
		@param  v   Valor que se quiere formatear para el campo indicado */
	static QString formatValue( FLFieldMetaData *fMD, const QVariant & v );

	/** Este mtodo hace lo mismo que el anterior, y se suministra por conveniencia.

		@param  t   Tipo de datos del valor
		@param  v   Valor que se quiere formatear para el campo indicado */
	static QString formatValue( int t, const QVariant & v );

	/** Crea un objeto FLTableMetaData a partir de un elemento XML.

		Dado un elemento XML, que contiene la descripcin de una
		tablas, construye y devuelve el objeto FLTableMetaData
		correspondiente.
		NO SE HACEN CHEQUEOS DE ERRORES SINTCTICOS EN EL XML.

		@param mtd    Elemento XML con la descripcin de la tabla.
		@return Objeto FLTableMetaData que contiene la descricin de la relacin. */
  	static FLTableMetaData *metadata(QDomElement *mtd);

  	/** Crea un objeto FLFieldMetaData a partir de un elemento XML.

        Dado un elemento XML, que contiene la descripcin de un
        campo de una tabla construye y agrega a una lista de descripciones
        de campos el objeto FLFieldMetaData correspondiente, que contiene
        dicha definicin del campo. Tambien lo agrega a una lista de claves
        compuesta, si el campo construido pertenece a una clave compuesta.
        NO SE HACEN CHEQUEOS DE ERRORES SINTCTICOS EN EL XML.

        @param  field   Elemento XML con la descripcin del campo.
        @param  fl      Lista de descripciones de campos, a la que
                        agregar el campo construido.
        @param  ckl     Lista de claves compuestas. */  	
  	static void metadataField(QDomElement *field,FLTableMetaData::FLFieldMetaDataList *fl,FLCompoundKey *ckl);

  	/** Crea un objeto FLRelationMetaData a partir de un elemento XML.

        Dado un elemento XML, que contiene la descripcin de una
        relacin entre tablas, construye y devuelve el objeto FLRelationMetaData
        correspondiente, que contiene dicha definicin de la relacin.
        NO SE HACEN CHEQUEOS DE ERRORES SINTCTICOS EN EL XML.

        @param  relation    Elemento XML con la descripcin de la relacin.
        @return Objeto      FLRelationMetaData que contiene la descricin de la relacin. */
  	static FLRelationMetaData *metadataRelation(QDomElement *relation);

        /** Crea un objeto FLParameterQuery a partir de un elemento XML.

        Dado un elemento XML, que contiene la descripcin de una
        parmetro de una consulta, construye y devuelve el objeto FLParameterQuery
        correspondiente.
        NO SE HACEN CHEQUEOS DE ERRORES SINTCTICOS EN EL XML.

        @param  parameter   Elemento XML con la descripcin del parmetro de una consulta.
        @return Objeto      FLParameterQuery que contiene la descricin del parmetro. */
  	static FLParameterQuery * queryParameter( QDomElement * parameter );

        /** Crea un objeto FLGroupByQuery a partir de un elemento XML.

        Dado un elemento XML, que contiene la descripcin de un nivel de agrupamiento
        de una consulta, construye y devuelve el objeto FLGroupByQuery correspondiente.
        NO SE HACEN CHEQUEOS DE ERRORES SINTCTICOS EN EL XML.

        @param  group   Elemento XML con la descripcin del nivel de agrupamiento de una
                                    consulta.
        @return Objeto  FLGroupByQuery que contiene la descricin del nivel de agrupamiento. */
  	static FLGroupByQuery * queryGroup( QDomElement * group );
};

#endif
