/*
 *  ---------
 * |.**> <**.|  CardContact
 * |*       *|  Software & System Consulting
 * |*       *|  Minden, Germany
 * |**> <**|  Copyright (c) 1999. All rights reserved
 *  --------- 
 *
 * See file LICENSE for details on licensing
 *
 * Abstract :       Defines tools/types for basic memory card handling
 *
 * Author :         Frank Thater (FTH)
 *
 * Last modified:   04/25/2000
 *
 *****************************************************************************/

#include <string.h>

#include "memorycards.h"


/************************************************************
 
 Processes basic memorycard functions.
 
 * Read main memory (MF, EF, EF_DIR, EF_ATR, AID) from buffer
 * Select file (MF, EF, EF_DIR, EF_ATR, AID) 
 * Update main memory (EF, EF_DIR, EF_ATR)

 return values:

 COMMAND_UNKNOWN  -  command not supported by basic funtions
 < 0              -  error reading/writing...
 OK               -  command successfully completed
 ***********************************************************/
   
int MemoryCardProcess(struct eco5000_t *ctx,
                      unsigned int lc,
                      unsigned char *cmd,
                      unsigned int *lr,
                      unsigned char *rsp)
{
    int rc;

    /* Interindustry Commands */
    
    switch(cmd[1]) {
        case 0xA4:
            rc = MemoryCard_Select_File(ctx, lc, cmd, lr, rsp);
            break;
        case 0xB0:
            rc = MemoryCard_Read_Binary(ctx, lc, cmd, lr, rsp);
            break;
        case 0xD6:
            rc = MemoryCard_Update_Binary(ctx, lc, cmd, lr, rsp);
            break;
        default:
            return COMMAND_UNKNOWN;
    }
 
    return rc;
}



/* 
 * Select FILE/AID  
 * - set the values in the ctx-structure for file- and application-IDs
 *
 */

int MemoryCard_Select_File(struct eco5000_t *ctx,
                           unsigned int lc,
                           unsigned char *cmd,
                           unsigned int *lr,
                           unsigned char *rsp)
{

    struct memorycard_t *pt = ctx->Data.card;
    unsigned int temp;
    unsigned char buffer[256], aid[16];
  
    if (cmd[2] == 0x00)  {  /* FID selected */
    
        temp = (cmd[5] << 8) | cmd[6];
    
        switch(temp) {
            case MF:
                pt->SelectedFile = MF;
                pt->Path = 0x00;
                break;
            case DIR_DATA_SECTION:
                pt->SelectedFile = DIR_DATA_SECTION;
                pt->Path = pt->DIR_data_reference;
                break;
            case ATR_DATA_SECTION:
                pt->SelectedFile = ATR_DATA_SECTION;
                if (pt->DIR_data_reference == 0x04)
                    pt->Path = 0x00;
                else
                    pt->Path = 0x04;
                break;
            default:
                return COMMAND_UNKNOWN;
        }

	rsp[0] = HIGH(SMARTCARD_SUCCESS);
        rsp[1] = LOW(SMARTCARD_SUCCESS);
        *lr = 2;      
    
    
    } else if (cmd[2] == 0x04) {  /* AID selected */
    
        memcpy(&aid, cmd + 5, cmd[4]);
        memcpy(&buffer, pt->Memory + pt->DIR_data_reference, pt->Length_of_DIR_section);
    
        if (Find_AID (ctx, buffer, aid) != OK) {
            rsp[0] = HIGH(FILE_NOT_FOUND);
            rsp[1] = LOW(FILE_NOT_FOUND);
            *lr = 2;
            return OK;
        }
    
        pt->SelectedFile = AID;
	rsp[0] = HIGH(SMARTCARD_SUCCESS);
        rsp[1] = LOW(SMARTCARD_SUCCESS);
        *lr = 2;   
    
    } else {  /* ID not found */

        return COMMAND_UNKNOWN;
    
    }
  
    return OK;

}


/*
 * MemoryCard_Read_Binary
 * - read the content of the memory image and return response
 *
 */

int MemoryCard_Read_Binary(struct eco5000_t *ctx,
                           unsigned int lc,
                           unsigned char *cmd,
                           unsigned int *lr,
                           unsigned char *rsp)
{

    struct memorycard_t *pt = ctx->Data.card;

    unsigned int offset;
    unsigned int expected_bytes;
    unsigned int buffer_to_short = 0;

   
     /* No file selected */
    if (pt->SelectedFile == NONE) {
        rsp[0] = HIGH(FILE_NOT_FOUND);
	rsp[1] = LOW(FILE_NOT_FOUND);
	*lr = 2;
        return OK;
    }

    offset = (cmd[2] << 8) | cmd[3];
    DecodeAPDU(lc, cmd, NULL, NULL, &expected_bytes);
    
    switch(pt->SelectedFile) {
        case AID:
            offset += pt->Path;
            break;
        case DIR_DATA_SECTION:
            offset += pt->Path;
            expected_bytes = pt->Length_of_DIR_section;
            break;
        case ATR_DATA_SECTION:
            offset += pt->Path;
            expected_bytes = pt->DIR_data_reference - 0x04;
            break;
        case MF:
            offset += pt->Path;
            break;
        default:
            return COMMAND_UNKNOWN;
    }

#ifdef DEBUG
    printf("\nREAD BINARY() - trying to get %u bytes\n", expected_bytes);
#endif

    if (expected_bytes == 0x00)
        expected_bytes = pt->Number_of_Data_Units - offset;

    /* to avoid index out of bounce errors ... */
    if ((offset + expected_bytes) > pt->Number_of_Data_Units)
        expected_bytes = pt->Number_of_Data_Units - offset;

    if ((expected_bytes + 2)  > *lr) {
        expected_bytes = *lr - 2;
	buffer_to_short = 1;
    }

    memcpy(rsp, pt->Memory + offset, expected_bytes); 
    
    if(buffer_to_short == 1) {
      *lr = 2;
      rsp[expected_bytes] = HIGH(DATA_CORRUPTED);
      rsp[expected_bytes + 1] = LOW(DATA_CORRUPTED);
      return ERR_MEMORY;
    }
    else {
      *lr = expected_bytes + 2;
      rsp[expected_bytes] = HIGH(SMARTCARD_SUCCESS);
      rsp[expected_bytes + 1] = LOW(SMARTCARD_SUCCESS);
      return OK;
    }
}



/*
 * MemoryCard_Update_Binary
 * - map the update of EF_DIR, ATR_DIR, FID to an update of MF
 * - call the CT_MOD_function to provide update of MF
 *
 */

int MemoryCard_Update_Binary(struct eco5000_t *ctx,
                             unsigned int lc,
                             unsigned char *cmd,
                             unsigned int *lr,
                             unsigned char *rsp)
{
    struct memorycard_t *pt = ctx->Data.card;
    unsigned char address = 0x00;
    unsigned int offset;
    int rc;
    unsigned int oldSelectedFile = NONE;

    /* No file selected */
    if (pt->SelectedFile == NONE) {
        rsp[0] = 0x6A;
        rsp[1] = 0x82;
        *lr = 2;
        return OK;
    }

    offset = (cmd[2] << 8) | cmd[3];
  
    /* calculate the offsets, so only MF muste be read ... */
    switch (pt->SelectedFile) {
        case DIR_DATA_SECTION:
            address = pt->DIR_data_reference + offset;
            break;
        case ATR_DATA_SECTION:
            address = pt->ATR_data_reference + offset;
            break;
        case AID:
            address = pt->Path + offset;
            break;
        default:
            return COMMAND_UNKNOWN;
    }
  
    oldSelectedFile = pt->SelectedFile;
    pt->SelectedFile = MF;

    /* now set the new offsets according to MF */
    cmd[2] = (unsigned char) address >> 8;
    cmd[3] = (unsigned char) address & 0xFF;
 
    rc = (*ctx->CTModFunc)(ctx, lc, cmd, lr, rsp); /* now call for MF again */

    pt->SelectedFile = oldSelectedFile;

    return rc;
}



/* Find_AID()  -  Searches for AID in DIR DATA area and sets PATH if found.
   
   in:  ctx    -> for handling ECO reader
        buffer -> content of DIR DATA area
        aid    -> Application ID to search for

        out: < 0    -> AID not found
             OK     -> AID found                      */

int Find_AID (struct eco5000_t *ctx,
              unsigned char *buffer,
              unsigned char *aid)

{
    unsigned char length_tag = 0x00;
    unsigned char tbuffer[16];
    struct memorycard_t *pt = NULL;
    unsigned int value = 0x00;
    int index = 0;
    unsigned char found = FALSE;

    pt = ctx->Data.card;
  
    while (index < pt->Length_of_DIR_section) {
   
        value = buffer[index];
        if (value == APPLICATION_IDENTIFIER) {
            length_tag = buffer[index + 1];
            memcpy(&tbuffer, buffer + index + 2, length_tag);
            if (memcmp(aid, tbuffer, length_tag)) {
                pt->Path = pt->DIR_data_reference + pt->Length_of_DIR_section;
                found = TRUE;
            }
        }  

        if ((value == PATH) && (found == TRUE)) {
            length_tag = buffer[index + 1];   
            if (length_tag == 0x02) {
                pt->Path = (buffer[index+2] << 8) + buffer[index+3];
            }
            else {
                pt->Path = buffer[index + 2];
            }
        }
        index++;
    }

    if (found == FALSE) 
        return -1;

    return OK;
}


/*
 * DetermineVariables
 * - set values in the memorycard_t structure (Number_of_Data_Units, ... )
 * - determine start of ATR_area, DIR_area, ...
 * - set up new baud rate
 * 
 */

int DetermineVariables(struct eco5000_t *ctx)

{
    struct memorycard_t *pt = ctx->Data.card;
    unsigned char help = 0x00;
    unsigned char i = 0x00;
    unsigned char buffer[2];
    int rc;

    help = (ctx->ATR[1] & 0x78) >> 3;

    if(ctx->ATR[1] == 0xff)
        help = 0x00;
 
    if (help)
        pt->Number_of_Data_Units = 64 << help;
    else
        pt->Number_of_Data_Units = 256;

#ifdef DEBUG 
    printf("\nAllocating %lu bytes!\n", pt->Number_of_Data_Units);
    printf("\nHelp = %02x", help);
#endif


    help = ctx->ATR[1] & 0x07;
    pt->Length_Data_Units = 1 << help; 
 

    if((pt->Memory = malloc(pt->Number_of_Data_Units)) == NULL)
        return ERR_CT;

    if (ctx->ATR[2] == 0x10) {
        if ((ctx->ATR[3] & 0x80) != 0x84) {
            /* starts after ATR data */
            pt->DIR_data_reference = (ctx->ATR[3] & 0x7F);
            pt->ATR_data_reference = 0x04;
        }   
        else {
            /* starts after ATR header */
            pt->DIR_data_reference = 0x04;
            pt->ATR_data_reference = NONE;   /* not available */
            i = pt->DIR_data_reference;  
            pt->Length_of_DIR_section = pt->Memory[i+1] + 2; 
        }
    } else 
        pt->DIR_data_reference = NONE;


    pt->Path = 0x00;


    /* Set up the baud rate */
  
    buffer[0] = 0xFC;

    rc = ecoCommand(ctx, SET_BAUDRATE, 1, buffer, 0, NULL);

    if (rc < 0)
        return rc;

    /* now set up the serial port */
    if (rs232Mode(ctx->fh, 115200, -1, -1, -1, -1) < 0) 
        return ERR_CT;  

    ctx->Baud = 115200;

    return OK;
}















