/* 
 * tile.c
 *
 * crafted - a pud editor for the freecraft project.
 * 
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>

#include "crafted.h"
#include "tileset.h"
#include "tile.h"

gint get_solid_index(gint style)
{
  gint index = 0, idx = 0;
  gint max_index = 0, i;
  
  for (i=0 ; i<16 ; i++)
    if (solid_tiles[style][i] == 0)
    {
      max_index = i;
      break;
    }

  while (index == 0)
  {
    idx = rand()%max_index;
    index = solid_tiles[style][idx];
  }
  return idx;
}

static gint get_mixed_index(gint style, gint direction)
{
  gint index = 0, idx;

  while (index == 0)
  {
    idx = rand() % 16;
    index = mixed_tiles[style][(direction << 4) + idx];
  }
  return idx;
}

gboolean affect_tile(gint x, gint y, guint16 tile)
{
  gint index;
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if ((Pud.fields[y*Pud.width+x] & 0xFFF0) != tile)
  {
    modified = TRUE;
    if (tile & 0xFF00)
    { /* mixed tiles */
      tile += get_mixed_index(tile >> 8, (tile & 0xF0) >> 4);
      index = mixed_tiles[(tile & 0xFF00) >> 8][tile & 0xFF];
    } else
    { /* solid tiles */
      tile += get_solid_index(tile >> 4);
      index = solid_tiles[(tile & 0x00F0) >> 4][tile & 0xF];
    }
    Pud.fields[y*Pud.width+x] = tile;
    gdk_window_copy_area(drawing_area->window,
                         drawing_area->style->white_gc,
                         x*32-x_offset, y*32-y_offset, tileset_pixmap,
                         (index % 16)*32,
                         (index / 16)*32, 32,32);
  }
  return modified;
}

void affect_exact_tile(gint x, gint y, guint16 tile)
{
  gint index;

  g_return_if_fail((0<=x) && (x<Pud.width));
  g_return_if_fail((0<=y) && (y<Pud.height));
  if (tile & 0xFF00)
  { /* mixed tiles */
    index = mixed_tiles[(tile & 0xFF00) >> 8][tile & 0xFF];
  } else
  { /* solid tiles */
    index = solid_tiles[(tile & 0x00F0) >> 4][tile & 0xF];
  }
  Pud.fields[y*Pud.width+x] = tile;
  gdk_window_copy_area(drawing_area->window,
                       drawing_area->style->white_gc,
                       x*32-x_offset, y*32-y_offset, tileset_pixmap,
                       (index % 16)*32,
                       (index / 16)*32, 32,32);
}

/*************************************************************************/

static gboolean tile_is_light_water(guint16 tile)
{
  if ((tile & 0xFFF0) == 0x0010)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_dark_water(guint16 tile)
{
  if ((tile & 0xFFF0) == 0x0020)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_light_coast(guint16 tile)
{
  if ((tile & 0xFFF0) == 0x0030)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_dark_coast(guint16 tile)
{
  if ((tile & 0xFFF0) == 0x0040)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_light_grass(guint16 tile)
{
  if ((tile & 0xFFF0) == 0x0050)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_dark_grass(guint16 tile)
{
  if ((tile & 0xFFF0) == 0x0060)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_forest(guint16 tile)
{
  if ((tile & 0xFFF0) == 0x0070)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_rocks(guint16 tile)
{
  if ((tile & 0xFFF0) == ROCKS<<4)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_bridge(guint16 tile)
{
  if ((tile & 0xFFF0) == BRIDGE<<4 )
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_human_closed_wall(guint16 tile)
{
  if ((tile & 0xFFF0) == 0x0090)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_orc_closed_wall(guint16 tile)
{
  if ((tile & 0xFFF0) == 0x00A0)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_human_open_wall(guint16 tile)
{
  if ((tile & 0xFFF0) == 0x00B0)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_orc_open_wall(guint16 tile)
{
  if ((tile & 0xFFF0) == 0x00C0)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_dark_water_light_water(guint16 tile)
{
  if ((tile & 0xFF00) == 0x0100)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_light_water_light_coast(guint16 tile)
{
  if ((tile & 0xFF00) == 0x0200)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_dark_coast_light_coast(guint16 tile)
{
  if ((tile & 0xFF00) == 0x0300)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_rocks_light_coast(guint16 tile)
{
  if ((tile & 0xFF00) == 0x0400)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_light_coast_light_grass(guint16 tile)
{
  if ((tile & 0xFF00) == 0x0500)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_dark_grass_light_grass(guint16 tile)
{
  if ((tile & 0xFF00) == 0x0600)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_forest_light_grass(guint16 tile)
{
  if ((tile & 0xFF00) == 0x0700)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_human_wall_light_grass(guint16 tile)
{
  if ((tile & 0xFF00) == 0x0800)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_orc_wall_light_grass(guint16 tile)
{
  if ((tile & 0xFF00) == 0x0900)
    return TRUE;
  else
    return FALSE;
}

static gboolean tile_is_bridge_light_coast(guint16 tile)
{
  if ((tile & 0xFF00) == BRIDGE_LIGHT_COAST<<8 )
    return TRUE;
  else
    return FALSE;
}

/*************************************************************************/

static gboolean check_north_with_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    /* XXX: is there a smarter way of dealing with this? */
    affect_tile(x, 0, 0x0010);
    return FALSE;
  }

  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0120);
  } else if (tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0010);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0010);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0100);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0110);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0120);
        break;
    }
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x02B0);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x02C0);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x02D0);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0010);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x02B0);
  }
  return modified;
}

static gboolean check_south_with_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0010);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x01B0);
  } else if (tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x0010);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0010);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0130);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0170);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x01B0);
        break;
    }
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0220);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0260);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x02A0);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x0010);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0220);
  }
  return modified;
}

static gboolean check_east_with_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0010);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0190);
  } else if (tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0010);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0010);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0110);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0170);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0190);
        break;
    }
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0240);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0260);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x02C0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0010);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0240);
  }
  return modified;
}

static gboolean check_west_with_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0010);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0140);
  } else if (tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0010);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0010);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0100);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0130);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0140);
        break;
    }
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0290);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x02A0);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x02D0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0010);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0290);
  }
  return modified;
}

/*************************************************************************/

static gboolean check_north_with_dark_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0020); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0020);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x01B0);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x01C0);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x01D0);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0020);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x01B0);
  }
  return modified;
}

static gboolean check_south_with_dark_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0020);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x0020);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0120);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0160);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x01A0);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x0020);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0120);
  }
  return modified;
}

static gboolean check_east_with_dark_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0020);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0020);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0140);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0160);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x01C0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0020);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0140);
  }
  return modified;
}

static gboolean check_west_with_dark_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0020);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0020);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0190);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x01A0);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x01D0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0020);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0190);
  }
  return modified;
}

/*************************************************************************/

static gboolean check_north_with_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0030); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile) ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0220);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0030);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0200);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0210);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0220);
        break;
    }
  } else if (tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0030);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0030);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0300);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0310);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0320);
        break;
    }
  } else if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0320);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0030);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0400);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0410);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0420);
        break;
    }
  } else if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0420);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x05B0);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x05C0);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x05D0);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0030);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x05B0);
  }
  return modified;
}

static gboolean check_south_with_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0030);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile) ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x02B0);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0030);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0230);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0270);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x02B0);
        break;
    }
  } else if (tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x0030);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0030);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0330);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0370);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x03B0);
        break;
    }
  } else if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x03B0);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0030);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0430);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0470);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x04B0);
        break;
    }
  } else if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x04B0);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0520);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0560);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x05A0);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x0030);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0520);
  }
  return modified;
}

static gboolean check_east_with_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0030);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_water(checked_tile) ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0290);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0030);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0210);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0270);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0290);
        break;
    }
  } else if (tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0030);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0030);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0310);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0370);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0390);
        break;
    }
  } else if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0390);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0030);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0410);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0470);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0490);
        break;
    }
  } else if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0490);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0540);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0560);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x05C0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0030);
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0540);
  }
  return modified;
}

static gboolean check_west_with_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0030);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_water(checked_tile) ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0240);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0030);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0200);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0230);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0240);
        break;
    }
  } else if (tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0030);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0030);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0300);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0330);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0340);
        break;
    }
  } else if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0340);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0030);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0400);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0430);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0440);
        break;
    }
  } else if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0440);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0590);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x05A0);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x05D0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0030);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0590);
  }
  return modified;
}

/*************************************************************************/

static gboolean check_north_with_dark_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0040); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0040);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x03B0);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x03C0);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x03D0);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0040);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x03B0);
  }
  return modified;
}

static gboolean check_south_with_dark_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0040);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x0040);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0320);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0360);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x03A0);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x0040);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0320);
  }
  return modified;
}

static gboolean check_east_with_dark_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0040);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0040);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0340);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0360);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x03C0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0040);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0340);
  }
  return modified;
}

static gboolean check_west_with_dark_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0040);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0040);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0390);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x03A0);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x03D0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0040);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0390);
  }
  return modified;
}

/*************************************************************************/

static gboolean check_north_with_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0050); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0620);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0050);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0600);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0610);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0620);
        break;
    }
  } else if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0720);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0050);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0700);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0710);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0720);
        break;
    }
  } else if (tile_is_light_grass(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0050);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0050);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0500);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0510);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0520);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0520);
  }
  return modified;
}

static gboolean check_south_with_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0050);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x06B0);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0050);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0630);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0670);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x06B0);
    }
  } else if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x07B0);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0050);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0730);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0770);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x07B0);
        break;
    }
  } else if (tile_is_light_grass(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x0050);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0050);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0530);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0570);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x05B0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x05B0);
  }
  return modified;
}

static gboolean check_east_with_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0050);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0690);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0050);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0610);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0670);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0690);
        break;
    }
  } else if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0790);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0050);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0710);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0770);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0790);
        break;
    }
  } else if (tile_is_light_grass(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0050);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0050);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0510);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0570);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0590);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0590);
  }
  return modified;
}

static gboolean check_west_with_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0050);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0640);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0050);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0600);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0630);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0640);
        break;
    }
  } else if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0740);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0050);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0700);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0730);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0740);
        break;
    }
  } else if (tile_is_light_grass(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0050);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0050);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0500);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0530);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0540);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0540);
  }
  return modified;
}

/*************************************************************************/

static gboolean check_north_with_dark_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0060); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0060);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x06B0);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x06C0);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x06D0);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0060);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x06B0);
  }
  return modified;
}

static gboolean check_south_with_dark_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0060);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x0060);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0620);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0660);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x06A0);
        break;
      case  2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x0060);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0620);
  }
  return modified;
}

static gboolean check_east_with_dark_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0060);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0060);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0640);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0660);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x06C0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0060);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0640);
  }
  return modified;
}

static gboolean check_west_with_dark_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0060);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0060);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0690);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x06A0);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x06D0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0060);
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0690);
  }
  return modified;
}

/*************************************************************************/

static gboolean check_north_with_forest(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0070); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0070);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x07B0);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x07C0);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x07D0);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0070);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x07B0);
  }
  return modified;
}

static gboolean check_south_with_forest(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0070);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x0070);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0720);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0760);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x07A0);
        break;
      case  2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x0070);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0720);
  }
  return modified;
}

static gboolean check_east_with_forest(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0070);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0070);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0740);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0760);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x07C0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0070);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0740);
  }
  return modified;
}

static gboolean check_west_with_forest(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0070);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0070);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0790);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x07A0);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x07D0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0070);
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0790);
  }
  return modified;
}

/*************************************************************************/

static gboolean check_north_with_rocks(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0080); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0080);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x04B0);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x04C0);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x04D0);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0080);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x04B0);
  }
  return modified;
}

static gboolean check_south_with_rocks(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0080);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x0080);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0420);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0460);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x04A0);
        break;
      case  2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x0080);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0420);
  }
  return modified;
}

static gboolean check_east_with_rocks(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0080);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x0080);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0440);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0460);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x04C0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x0080);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0440);
  }
  return modified;
}

static gboolean check_west_with_rocks(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0080);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0080);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0490);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x04A0);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x04D0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0080);
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0490);
  }
  return modified;
}

/*************************************************************************/

static gboolean check_north_with_bridge(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, BRIDGE<<4); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_bridge(checked_tile))
  {
    modified = affect_tile(x, y-1, BRIDGE<<4);
  } else if (tile_is_bridge_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, (BRIDGE_LIGHT_COAST<<8)|0xB0);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, (BRIDGE_LIGHT_COAST<<8)|0xC0);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, (BRIDGE_LIGHT_COAST<<8)|0xD0);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, (BRIDGE_LIGHT_COAST<<8)|0x80);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, (BRIDGE_LIGHT_COAST<<8)|0xB0);
  }
  return modified;
}

static gboolean check_south_with_bridge(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, BRIDGE<<4);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_bridge(checked_tile))
  {
    modified = affect_tile(x, y+1, BRIDGE<<4);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, (BRIDGE_LIGHT_COAST<<8)|0x20);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, (BRIDGE_LIGHT_COAST<<8)|0x60);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, (BRIDGE_LIGHT_COAST<<8)|0xA0);
        break;
      case  2: case 6: case 10:
        modified = affect_tile(x, y+1, BRIDGE<<4);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, (BRIDGE_LIGHT_COAST<<8)|0x20);
  }
  return modified;
}

static gboolean check_east_with_bridge(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, BRIDGE<<4);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_bridge(checked_tile))
  {
    modified = affect_tile(x+1, y, BRIDGE<<4);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, (BRIDGE_LIGHT_COAST<<8)|0x40);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, (BRIDGE_LIGHT_COAST<<8)|0x60);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, (BRIDGE_LIGHT_COAST<<8)|0xC0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, BRIDGE<<4);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, (BRIDGE_LIGHT_COAST<<8)|0x40);
  }
  return modified;
}

static gboolean check_west_with_bridge(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, BRIDGE<<4);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_bridge(checked_tile))
  {
    modified = affect_tile(x-1, y, BRIDGE<<4);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, (BRIDGE_LIGHT_COAST<<8)|0x90);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, (BRIDGE_LIGHT_COAST<<8)|0xA0);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, (BRIDGE_LIGHT_COAST<<8)|0xD0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, BRIDGE<<4);
    }
  } else
  {
    modified = affect_tile(x-1, y, (BRIDGE_LIGHT_COAST<<8)|0x90);
  }
  return modified;
}

/*************************************************************************/
static gboolean check_north_with_dark_water_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0140); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0160);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0130);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0140);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0150);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0160);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0130);
  }
  return modified;
}

static gboolean check_south_with_dark_water_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0140);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x01C0);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0100);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0140);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0180);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x01C0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0100);
  }
  return modified;
}

static gboolean check_east_with_dark_water_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0120);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x01A0);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0100);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0120);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0180);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x01A0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0100);
  }
  return modified;
}

static gboolean check_west_with_dark_water_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0120);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0160);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0110);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0120);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0150);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0160);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0110);
  }
  return modified;
}

static gboolean check_north_with_light_water_dark_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0190); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x01A0);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0170);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0180);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0190);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x01A0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0170);
  }
  return modified;
}

static gboolean check_south_with_light_water_dark_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0190);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x01D0);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0110);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0150);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0190);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x01D0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0110);
  }
  return modified;
}

static gboolean check_east_with_light_water_dark_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x01B0);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x01D0);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0130);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0150);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x01B0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x01D0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0130);
  }
  return modified;
}

static gboolean check_west_with_light_water_dark_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x01B0);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_water(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x01C0);
  } else if (tile_is_dark_water_light_water(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0170);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0180);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x01B0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x01C0);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0170);
  }
  return modified;
}

static gboolean check_north_with_light_water_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0240); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile)             ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0260);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0230);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0240);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0250);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0260);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0230);
  }
  return modified;
}

static gboolean check_south_with_light_water_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0240);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile)             ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x02C0);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0200);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0240);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0280);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x02C0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0200);
  }
  return modified;
}

static gboolean check_east_with_light_water_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0220);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_water(checked_tile)             ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x02A0);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0200);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0220);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0280);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x02A0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0200);
  }
  return modified;
}

static gboolean check_west_with_light_water_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0220);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_water(checked_tile)             ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0260);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0210);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0220);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0250);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0260);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0210);
  }
  return modified;
}

static gboolean check_north_with_light_coast_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0290); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile)             ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x02A0);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0270);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0280);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0290);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x02A0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0270);
  }
  return modified;
}

static gboolean check_south_with_light_coast_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0290);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile)             ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x02D0);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0210);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0250);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0290);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x02D0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0210);
  }
  return modified;
}

static gboolean check_east_with_light_coast_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x02B0);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_water(checked_tile)             ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x02D0);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0230);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0250);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x02B0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x02D0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0230);
  }
  return modified;
}

static gboolean check_west_with_light_coast_light_water(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x02B0);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_water(checked_tile)             ||
      tile_is_dark_water_light_water(checked_tile) ||
      tile_is_light_water(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x02C0);
  } else if (tile_is_light_water_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0270);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0280);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x02B0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x02C0);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0270);
  }
  return modified;
}

static gboolean check_north_with_dark_coast_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0340); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0360);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0330);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0340);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0350);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0360);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0330);
  }
  return modified;
}

static gboolean check_south_with_dark_coast_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0340);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x03C0);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0300);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0340);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0380);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x03C0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0300);
  }
  return modified;
}

static gboolean check_east_with_dark_coast_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0320);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x03A0);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0300);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0320);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0380);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x03A0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0300);
  }
  return modified;
}

static gboolean check_west_with_dark_coast_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0320);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0360);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0310);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0320);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0350);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0360);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0310);
  }
  return modified;
}

static gboolean check_north_with_light_coast_dark_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0390); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x03A0);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0370);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0380);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0390);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x03A0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0370);
  }
  return modified;
}

static gboolean check_south_with_light_coast_dark_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0390);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x03D0);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0310);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0350);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0390);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x03D0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0310);
  }
  return modified;
}

static gboolean check_east_with_light_coast_dark_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x03B0);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x03D0);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0330);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0350);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x03B0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x03D0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0330);
  }
  return modified;
}

static gboolean check_west_with_light_coast_dark_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x03B0);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_coast(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x03C0);
  } else if (tile_is_dark_coast_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0370);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0380);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x03B0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x03C0);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0370);
  }
  return modified;
}

static gboolean check_north_with_rocks_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0440); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0460);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0430);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0440);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0450);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0460);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0430);
  }
  return modified;
}

static gboolean check_south_with_rocks_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0440);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x04C0);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0400);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0440);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0480);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x04C0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0400);
  }
  return modified;
}

static gboolean check_east_with_rocks_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0420);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x04A0);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0400);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0420);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0480);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x04A0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0400);
  }
  return modified;
}

static gboolean check_west_with_rocks_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0420);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0460);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0410);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0420);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0450);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0460);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0410);
  }
  return modified;
}

static gboolean check_north_with_light_coast_rocks(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0490); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x04A0);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0470);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0480);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0490);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x04A0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0470);
  }
  return modified;
}

static gboolean check_south_with_light_coast_rocks(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0490);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x04D0);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0410);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0450);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0490);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x4D0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0410);
  }
  return modified;
}

static gboolean check_east_with_light_coast_rocks(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x04B0);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x04D0);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0430);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0450);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x04B0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x04D0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0430);
  }
  return modified;
}

static gboolean check_west_with_light_coast_rocks(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x04B0);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_rocks(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x04C0);
  } else if (tile_is_rocks_light_coast(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0470);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0480);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x04B0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x04D0);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0470);
  }
  return modified;
}

static gboolean check_north_with_light_coast_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0540); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile)              ||
      tile_is_dark_water_light_water(checked_tile)  ||
      tile_is_light_water(checked_tile)             ||
      tile_is_light_water_light_coast(checked_tile) ||
      tile_is_dark_coast(checked_tile)              ||
      tile_is_dark_coast_light_coast(checked_tile)  ||
      tile_is_rocks(checked_tile)                   ||
      tile_is_rocks_light_coast(checked_tile)       ||
      tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0560);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0530);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0540);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0550);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0560);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0530);
  }
  return modified;
}

static gboolean check_south_with_light_coast_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0540);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile)              ||
      tile_is_dark_water_light_water(checked_tile)  ||
      tile_is_light_water(checked_tile)             ||
      tile_is_light_water_light_coast(checked_tile) ||
      tile_is_dark_coast(checked_tile)              ||
      tile_is_dark_coast_light_coast(checked_tile)  ||
      tile_is_rocks(checked_tile)                   ||
      tile_is_rocks_light_coast(checked_tile)       ||
      tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x05C0);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0500);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0540);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0580);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x05C0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0500);
  }
  return modified;
}

static gboolean check_east_with_light_coast_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0520);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_water(checked_tile)              ||
      tile_is_dark_water_light_water(checked_tile)  ||
      tile_is_light_water(checked_tile)             ||
      tile_is_light_water_light_coast(checked_tile) ||
      tile_is_dark_coast(checked_tile)              ||
      tile_is_dark_coast_light_coast(checked_tile)  ||
      tile_is_rocks(checked_tile)                   ||
      tile_is_rocks_light_coast(checked_tile)       ||
      tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x05A0);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0500);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0520);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0580);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x05A0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0500);
  }
  return modified;
}

static gboolean check_west_with_light_coast_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0520);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_water(checked_tile)              ||
      tile_is_dark_water_light_water(checked_tile)  ||
      tile_is_light_water(checked_tile)             ||
      tile_is_light_water_light_coast(checked_tile) ||
      tile_is_dark_coast(checked_tile)              ||
      tile_is_dark_coast_light_coast(checked_tile)  ||
      tile_is_rocks(checked_tile)                   ||
      tile_is_rocks_light_coast(checked_tile)       ||
      tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0560);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0510);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0520);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0550);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0560);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0510);
  }
  return modified;
}

static gboolean check_north_with_light_grass_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0590); /* XXX */
    return FALSE;
  }

  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile)              ||
      tile_is_dark_water_light_water(checked_tile)  ||
      tile_is_light_water(checked_tile)             ||
      tile_is_light_water_light_coast(checked_tile) ||
      tile_is_dark_coast(checked_tile)              ||
      tile_is_dark_coast_light_coast(checked_tile)  ||
      tile_is_rocks(checked_tile)                   ||
      tile_is_rocks_light_coast(checked_tile)       ||
      tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x05A0);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0570);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0580);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0590);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x05A0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0570);
  }
  return modified;
}

static gboolean check_south_with_light_grass_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0590);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_water(checked_tile)              ||
      tile_is_dark_water_light_water(checked_tile)  ||
      tile_is_light_water(checked_tile)             ||
      tile_is_light_water_light_coast(checked_tile) ||
      tile_is_dark_coast(checked_tile)              ||
      tile_is_dark_coast_light_coast(checked_tile)  ||
      tile_is_rocks(checked_tile)                   ||
      tile_is_rocks_light_coast(checked_tile)       ||
      tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x05D0);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0510);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0550);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0590);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x05D0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0510);
  }
  return modified;
}

static gboolean check_east_with_light_grass_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x05B0);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_water(checked_tile)              ||
      tile_is_dark_water_light_water(checked_tile)  ||
      tile_is_light_water(checked_tile)             ||
      tile_is_light_water_light_coast(checked_tile) ||
      tile_is_dark_coast(checked_tile)              ||
      tile_is_dark_coast_light_coast(checked_tile)  ||
      tile_is_rocks(checked_tile)                   ||
      tile_is_rocks_light_coast(checked_tile)       ||
      tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x05D0);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0530);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0550);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x05B0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x05D0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0530);
  }
  return modified;
}

static gboolean check_west_with_light_grass_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x05B0);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_water(checked_tile)              ||
      tile_is_dark_water_light_water(checked_tile)  ||
      tile_is_light_water(checked_tile)             ||
      tile_is_light_water_light_coast(checked_tile) ||
      tile_is_dark_coast(checked_tile)              ||
      tile_is_dark_coast_light_coast(checked_tile)  ||
      tile_is_rocks(checked_tile)                   ||
      tile_is_rocks_light_coast(checked_tile)       ||
      tile_is_light_coast(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x05C0);
  } else if (tile_is_light_coast_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0570);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0580);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x05B0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x05C0);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0570);
  }
  return modified;
}

static gboolean check_north_with_dark_grass_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0640); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0660);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0630);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0640);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0650);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0660);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0630);
  }
  return modified;
}

static gboolean check_south_with_dark_grass_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0640);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x06C0);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0600);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0640);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0680);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x06C0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0600);
  }
  return modified;
}

static gboolean check_east_with_dark_grass_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0620);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x06A0);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0600);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0620);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0680);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x06A0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0600);
  }
  return modified;
}

static gboolean check_west_with_dark_grass_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0620);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0660);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0610);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0620);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0650);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0660);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0610);
  }
  return modified;
}

static gboolean check_north_with_light_grass_dark_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0690); /* XXX */
    return FALSE;
  }

  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x06A0);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0670);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0680);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0690);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x06A0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0670);
  }
  return modified;
}

static gboolean check_south_with_light_grass_dark_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0690);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x06D0);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0610);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0650);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0690);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x06D0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0610);
  }
  return modified;
}

static gboolean check_east_with_light_grass_dark_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x06B0);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x06D0);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0630);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0650);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x06B0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x06D0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0630);
  }
  return modified;
}

static gboolean check_west_with_light_grass_dark_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x06B0);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_dark_grass(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x06C0);
  } else if (tile_is_dark_grass_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0670);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0680);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x06B0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x06C0);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0670);
  }
  return modified;
}

static gboolean check_north_with_forest_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0740); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x0760);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0730);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0740);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0750);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x0760);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0730);
  }
  return modified;
}

static gboolean check_south_with_forest_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0740);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x07C0);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0700);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0740);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0780);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x07C0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0700);
  }
  return modified;
}

static gboolean check_east_with_forest_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x0720);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x07A0);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0700);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0720);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x0780);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x07A0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0700);
  }
  return modified;
}

static gboolean check_west_with_forest_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x0720);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x0760);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0710);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0720);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x0750);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x0760);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0710);
  }
  return modified;
}

static gboolean check_north_with_light_grass_forest(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y<1)
    return FALSE;

  if (y==1)
  {
    affect_tile(x, 0, 0x0790); /* XXX */
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y-2)*Pud.width+x];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x, y-1, 0x07A0);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 1: case 2:
        modified = affect_tile(x, y-1, 0x0770);
        break;
      case 3: case 4: case 5: case 6:
        modified = affect_tile(x, y-1, 0x0780);
        break;
      case 7: case 8: case 9: case 10:
        modified = affect_tile(x, y-1, 0x0790);
        break;
      case 11: case 12: case 13:
        modified = affect_tile(x, y-1, 0x07A0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y-1, 0x0770);
  }
  return modified;
}

static gboolean check_south_with_light_grass_forest(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (y>Pud.height-2)
    return FALSE;

  if (y == Pud.height - 2)
  {
    affect_tile(x, Pud.height-1, 0x0790);
    return FALSE;
  }
  
  checked_tile = Pud.fields[(y+2)*Pud.width+x];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x, y+1, 0x07D0);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 3: case 7: case 11:
        modified = affect_tile(x, y+1, 0x0710);
        break;
      case 0: case 4: case 8: case 12:
        modified = affect_tile(x, y+1, 0x0750);
        break;
      case 1: case 5: case 9: case 13:
        modified = affect_tile(x, y+1, 0x0790);
        break;
      case 2: case 6: case 10:
        modified = affect_tile(x, y+1, 0x07D0);
        break;
    }
  } else
  {
    modified = affect_tile(x, y+1, 0x0710);
  }
  return modified;
}

static gboolean check_east_with_light_grass_forest(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x>Pud.width-2)
    return FALSE;

  if (x == Pud.width - 2)
  {
    affect_tile(Pud.width-1, y, 0x07B0);
    return FALSE;
  }
  
  checked_tile = Pud.fields[y*Pud.width+x+2];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x+1, y, 0x07D0);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 1: case 7: case 9:
        modified = affect_tile(x+1, y, 0x0730);
        break;
      case 0: case 2: case 8: case 10:
        modified = affect_tile(x+1, y, 0x0750);
        break;
      case 3: case 5: case 11: case 13:
        modified = affect_tile(x+1, y, 0x07B0);
        break;
      case 4: case 6: case 12:
        modified = affect_tile(x+1, y, 0x07D0);
        break;
    }
  } else
  {
    modified = affect_tile(x+1, y, 0x0730);
  }
  return modified;
}

static gboolean check_west_with_light_grass_forest(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (x<1)
    return FALSE;

  if (x==1)
  {
    affect_tile(0, y, 0x07B0);
    return FALSE;
  }

  checked_tile = Pud.fields[y*Pud.width+x-2];
  if (tile_is_forest(checked_tile))
  {
    modified = affect_tile(x-1, y, 0x07C0);
  } else if (tile_is_forest_light_grass(checked_tile))
  {
    switch ((checked_tile & 0xF0) >> 4)
    {
      case 0: case 3: case 4:
        modified = affect_tile(x-1, y, 0x0770);
        break;
      case 1: case 2: case 5: case 6:
        modified = affect_tile(x-1, y, 0x0780);
        break;
      case 7: case 8: case 11: case 12:
        modified = affect_tile(x-1, y, 0x07B0);
        break;
      case 9: case 10: case 13:
        modified = affect_tile(x-1, y, 0x07C0);
        break;
    }
  } else
  {
    modified = affect_tile(x-1, y, 0x0770);
  }
  return modified;
}

/*************************************************************************/

static gboolean check_north(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 tile;

  if ((0>x) || (x>=Pud.width))
    return FALSE;
  if ((0>y) || (y>=Pud.height))
    return FALSE;
  tile = Pud.fields[y*Pud.width+x];

  switch (tile >> 4)
  {
    case 0x001:
    case 0x013: case 0x017: case 0x01B:
    case 0x022: case 0x026: case 0x02A:
      modified = check_north_with_light_water(x, y);
      break;
    case 0x002:
    case 0x012: case 0x016: case 0x01A:
      modified = check_north_with_dark_water(x, y);
      break;
    case 0x003:
    case 0x023: case 0x027: case 0x02B:
    case 0x033: case 0x037: case 0x03B:
    case 0x043: case 0x047: case 0x04B:
    case 0x052: case 0x056: case 0x05A:
      modified = check_north_with_light_coast(x, y);
      break;
    case 0x004:
    case 0x032: case 0x036: case 0x03A:
      modified = check_north_with_dark_coast(x, y);
      break;
    case 0x005:
    case 0x053: case 0x057: case 0x05B:
    case 0x063: case 0x067: case 0x06B:
    case 0x073: case 0x077: case 0x07B:
      modified = check_north_with_light_grass(x, y);
      break;
    case 0x006:
    case 0x062: case 0x066: case 0x06A:
      modified = check_north_with_dark_grass(x, y);
      break;
    case 0x007:
    case 0x072: case 0x076: case 0x07A:
      modified = check_north_with_forest(x, y);
      break;
    case 0x008:
    case 0x042: case 0x046: case 0x04A:
      modified = check_north_with_rocks(x, y);
      break;
    case 0x010: case 0x014: case 0x018: case 0x01C:
      modified = check_north_with_dark_water_light_water(x, y);
      break;
    case 0x011: case 0x015: case 0x019: case 0x01D:
      modified = check_north_with_light_water_dark_water(x, y);
      break;
    case 0x020: case 0x024: case 0x028: case 0x02C:
      modified = check_north_with_light_water_light_coast(x, y);
      break;
    case 0x021: case 0x025: case 0x029: case 0x02D:
      modified = check_north_with_light_coast_light_water(x, y);
      break;
    case 0x030: case 0x034: case 0x038: case 0x03C:
      modified = check_north_with_dark_coast_light_coast(x, y);
      break;
    case 0x031: case 0x035: case 0x039: case 0x03D:
      modified = check_north_with_light_coast_dark_coast(x, y);
      break;
    case 0x040: case 0x044: case 0x048: case 0x04C:
      modified = check_north_with_rocks_light_coast(x, y);
      break;
    case 0x041: case 0x045: case 0x049: case 0x04D:
      modified = check_north_with_light_coast_rocks(x, y);
      break;
    case 0x050: case 0x054: case 0x058: case 0x05C:
      modified = check_north_with_light_coast_light_grass(x, y);
      break;
    case 0x051: case 0x055: case 0x059: case 0x05D:
      modified = check_north_with_light_grass_light_coast(x, y);
      break;
    case 0x060: case 0x064: case 0x068: case 0x06C:
      modified = check_north_with_dark_grass_light_grass(x, y);
      break;
    case 0x061: case 0x065: case 0x069: case 0x06D:
      modified = check_north_with_light_grass_dark_grass(x, y);
      break;
    case 0x070: case 0x074: case 0x078: case 0x07C:
      modified = check_north_with_forest_light_grass(x, y);
      break;
    case 0x071: case 0x075: case 0x079: case 0x07D:
      modified = check_north_with_light_grass_forest(x, y);
      break;
    default:
      g_print("Tile %0x not handled\n",tile);
      break;
  }

  return modified;
}

static gboolean check_south(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 tile;

  if ((0>x) || (x>=Pud.width))
    return FALSE;
  if ((0>y) || (y>=Pud.height))
    return FALSE;
  tile = Pud.fields[y*Pud.width+x];

  switch (tile >> 4)
  {
    case 0x001:
    case 0x010: case 0x011: case 0x012:
    case 0x02B: case 0x02C: case 0x02D:
      modified = check_south_with_light_water(x, y);
      break;
    case 0x002:
    case 0x01B: case 0x01C: case 0x01D:
      modified = check_south_with_dark_water(x, y);
      break;
    case 0x003:
    case 0x020: case 0x021: case 0x022:
    case 0x030: case 0x031: case 0x032:
    case 0x040: case 0x041: case 0x042:
    case 0x05B: case 0x05C: case 0x05D:
      modified = check_south_with_light_coast(x, y);
      break;
    case 0x004:
    case 0x03B: case 0x03C: case 0x03D:
      modified = check_south_with_dark_coast(x, y);
      break;
    case 0x005:
    case 0x050: case 0x051: case 0x052:
    case 0x060: case 0x061: case 0x062:
    case 0x070: case 0x071: case 0x072:
      modified = check_south_with_light_grass(x, y);
      break;
    case 0x006:
    case 0x06B: case 0x06C: case 0x06D:
      modified = check_south_with_dark_grass(x, y);
      break;
    case 0x007:
    case 0x07B: case 0x07C: case 0x07D:
      modified = check_south_with_forest(x, y);
      break;
    case 0x008:
    case 0x04B: case 0x04C: case 0x04D:
      modified = check_south_with_rocks(x, y);
      break;
    case 0x013: case 0x014: case 0x015: case 0x016:
      modified = check_south_with_dark_water_light_water(x, y);
      break;
    case 0x017: case 0x018: case 0x019: case 0x01A:
      modified = check_south_with_light_water_dark_water(x, y);
      break;
    case 0x023: case 0x024: case 0x025: case 0x026:
      modified = check_south_with_light_water_light_coast(x, y);
      break;
    case 0x027: case 0x028: case 0x029: case 0x02A:
      modified = check_south_with_light_coast_light_water(x, y);
      break;
    case 0x033: case 0x034: case 0x035: case 0x036:
      modified = check_south_with_dark_coast_light_coast(x, y);
      break;
    case 0x037: case 0x038: case 0x039: case 0x03A:
      modified = check_south_with_light_coast_dark_coast(x, y);
      break;
    case 0x043: case 0x044: case 0x045: case 0x046:
      modified = check_south_with_rocks_light_coast(x, y);
      break;
    case 0x047: case 0x048: case 0x049: case 0x04A:
      modified = check_south_with_light_coast_rocks(x, y);
      break;
    case 0x053: case 0x054: case 0x055: case 0x056:
      modified = check_south_with_light_coast_light_grass(x, y);
      break;
    case 0x057: case 0x058: case 0x059: case 0x05A:
      modified = check_south_with_light_grass_light_coast(x, y);
      break;
    case 0x063: case 0x064: case 0x065: case 0x066:
      modified = check_south_with_dark_grass_light_grass(x, y);
      break;
    case 0x067: case 0x068: case 0x069: case 0x06A:
      modified = check_south_with_light_grass_dark_grass(x, y);
      break;
    case 0x073: case 0x074: case 0x075: case 0x076:
      modified = check_south_with_forest_light_grass(x, y);
      break;
    case 0x077: case 0x078: case 0x079: case 0x07A:
      modified = check_south_with_light_grass_forest(x, y);
      break;
    default:
      g_print("Tile %0x not handled\n",tile);
      break;
  }

  return modified;
}

static gboolean check_east(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 tile;

  if ((0>x) || (x>=Pud.width))
    return FALSE;
  if ((0>y) || (y>=Pud.height))
    return FALSE;
  tile = Pud.fields[y*Pud.width+x];

  switch (tile >> 4)
  {
    case 0x001:
    case 0x010: case 0x013: case 0x014:
    case 0x029: case 0x02A: case 0x02D:
      modified = check_east_with_light_water(x, y);
      break;
    case 0x002:
    case 0x019: case 0x01A: case 0x01D:
      modified = check_east_with_dark_water(x, y);
      break;
    case 0x003:
    case 0x020: case 0x023: case 0x024:
    case 0x030: case 0x033: case 0x034:
    case 0x040: case 0x043: case 0x044:
    case 0x059: case 0x05A: case 0x05D:
      modified = check_east_with_light_coast(x, y);
      break;
    case 0x004:
    case 0x039: case 0x03A: case 0x03D:
      modified = check_east_with_dark_coast(x, y);
      break;
    case 0x005:
    case 0x050: case 0x053: case 0x054:
    case 0x060: case 0x063: case 0x064:
    case 0x070: case 0x073: case 0x074:
      modified = check_east_with_light_grass(x, y);
      break;
    case 0x006:
    case 0x069: case 0x06A: case 0x06D:
      modified = check_east_with_dark_grass(x, y);
      break;
    case 0x007:
    case 0x079: case 0x07A: case 0x07D:
      modified = check_east_with_forest(x, y);
      break;
    case 0x008:
    case 0x049: case 0x04A: case 0x04D:
      modified = check_east_with_rocks(x, y);
      break;
    case 0x011: case 0x012: case 0x015: case 0x016:
      modified = check_east_with_dark_water_light_water(x, y);
      break;
    case 0x017: case 0x018: case 0x01B: case 0x01C:
      modified = check_east_with_light_water_dark_water(x, y);
      break;
    case 0x021: case 0x022: case 0x025: case 0x026:
      modified = check_east_with_light_water_light_coast(x, y);
      break;
    case 0x027: case 0x028: case 0x02B: case 0x02C:
      modified = check_east_with_light_coast_light_water(x, y);
      break;
    case 0x031: case 0x032: case 0x035: case 0x036:
      modified = check_east_with_dark_coast_light_coast(x, y);
      break;
    case 0x037: case 0x038: case 0x03B: case 0x03C:
      modified = check_east_with_light_coast_dark_coast(x, y);
      break;
    case 0x041: case 0x042: case 0x045: case 0x046:
      modified = check_east_with_rocks_light_coast(x, y);
      break;
    case 0x047: case 0x048: case 0x04B: case 0x04C:
      modified = check_east_with_light_coast_rocks(x, y);
      break;
    case 0x051: case 0x052: case 0x055: case 0x056:
      modified = check_east_with_light_coast_light_grass(x, y);
      break;
    case 0x057: case 0x058: case 0x05B: case 0x05C:
      modified = check_east_with_light_grass_light_coast(x, y);
      break;
    case 0x061: case 0x062: case 0x065: case 0x066:
      modified = check_east_with_dark_grass_light_grass(x, y);
      break;
    case 0x067: case 0x068: case 0x06B: case 0x06C:
      modified = check_east_with_light_grass_dark_grass(x, y);
      break;
    case 0x071: case 0x072: case 0x075: case 0x076:
      modified = check_east_with_forest_light_grass(x, y);
      break;
    case 0x077: case 0x078: case 0x07B: case 0x07C:
      modified = check_east_with_light_grass_forest(x, y);
      break;
    default:
      g_print("Tile %0x not handled\n",tile);
      break;
  }
  return modified;
}

static gboolean check_west(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 tile;

  if ((0>x) || (x>=Pud.width))
    return FALSE;
  if ((0>y) || (y>=Pud.height))
    return FALSE;
  tile = Pud.fields[y*Pud.width+x];

  switch (tile >> 4)
  {
    case 0x001:
    case 0x011: case 0x017: case 0x019:
    case 0x024: case 0x026: case 0x02C:
      modified = check_west_with_light_water(x, y);
      break;
    case 0x002:
    case 0x014: case 0x016: case 0x01C:
      modified = check_west_with_dark_water(x, y);
      break;
    case 0x003:
    case 0x021: case 0x027: case 0x029:
    case 0x031: case 0x037: case 0x039:
    case 0x041: case 0x047: case 0x049:
    case 0x054: case 0x056: case 0x05C:
      modified = check_west_with_light_coast(x, y);
      break;
    case 0x004:
    case 0x034: case 0x036: case 0x03C:
      modified = check_west_with_dark_coast(x, y);
      break;
    case 0x005:
    case 0x051: case 0x057: case 0x059:
    case 0x061: case 0x067: case 0x069:
    case 0x071: case 0x077: case 0x079:
      modified = check_west_with_light_grass(x, y);
      break;
    case 0x006:
    case 0x064: case 0x066: case 0x06C:
      modified = check_west_with_dark_grass(x, y);
      break;
    case 0x007:
    case 0x074: case 0x076: case 0x07C:
      modified = check_west_with_forest(x, y);
      break;
    case 0x008:
    case 0x044: case 0x046: case 0x04C:
      modified = check_west_with_rocks(x, y);
      break;
    case 0x010: case 0x012: case 0x018: case 0x01A:
      modified = check_west_with_dark_water_light_water(x, y);
      break;
    case 0x013: case 0x015: case 0x01B: case 0x01D:
      modified = check_west_with_light_water_dark_water(x, y);
      break;
    case 0x020: case 0x022: case 0x028: case 0x02A:
      modified = check_west_with_light_water_light_coast(x, y);
      break;
    case 0x023: case 0x025: case 0x02B: case 0x02D:
      modified = check_west_with_light_coast_light_water(x, y);
      break;
    case 0x030: case 0x032: case 0x038: case 0x03A:
      modified = check_west_with_dark_coast_light_coast(x, y);
      break;
    case 0x033: case 0x035: case 0x03B: case 0x03D:
      modified = check_west_with_light_coast_dark_coast(x, y);
      break;
    case 0x040: case 0x042: case 0x048: case 0x04A:
      modified = check_west_with_rocks_light_coast(x, y);
      break;
    case 0x043: case 0x045: case 0x04B: case 0x04D:
      modified = check_west_with_light_coast_rocks(x, y);
      break;
    case 0x050: case 0x052: case 0x058: case 0x05A:
      modified = check_west_with_light_coast_light_grass(x, y);
      break;
    case 0x053: case 0x055: case 0x05B: case 0x05D:
      modified = check_west_with_light_grass_light_coast(x, y);
      break;
    case 0x060: case 0x062: case 0x068: case 0x06A:
      modified = check_west_with_dark_grass_light_grass(x, y);
      break;
    case 0x063: case 0x065: case 0x06B: case 0x06D:
      modified = check_west_with_light_grass_dark_grass(x, y);
      break;
    case 0x070: case 0x072: case 0x078: case 0x07A:
      modified = check_west_with_forest_light_grass(x, y);
      break;
    case 0x073: case 0x075: case 0x07B: case 0x07D:
      modified = check_west_with_light_grass_forest(x, y);
      break;
    default:
      g_print("Tile %0x not handled\n",tile);
      break;
  }
  return modified;
}

/*************************************************************************/

static gboolean first_pass_light_water(gint x, gint y)
{
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  /* check the North tile */
  if (check_north_with_light_water(x, y))
    modified = TRUE;
  /* check South */
  if (check_south_with_light_water(x, y))
    modified = TRUE;
  /* check East */
  if (check_east_with_light_water(x, y))
    modified = TRUE;
  /* check West */
  if (check_west_with_light_water(x, y))
    modified = TRUE;

  if (check_north(x-1, y))
    modified = TRUE;
  if (check_north(x+1, y))
    modified = TRUE;

  if (check_south(x-1, y))
    modified = TRUE;
  if (check_south(x+1, y))
    modified = TRUE;

  return modified;
}

static gboolean first_pass_dark_water(gint x, gint y)
{
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  /* check the North tile */
  if (check_north_with_dark_water(x, y))
    modified = TRUE;
  /* check the South tile */
  if (check_south_with_dark_water(x, y))
    modified = TRUE;
  /* check the East tile */
  if (check_east_with_dark_water(x, y))
    modified = TRUE;
  /* check the West tile */
  if (check_west_with_dark_water(x, y))
    modified = TRUE;

  if (check_north(x-1, y))
    modified = TRUE;
  if (check_north(x+1, y))
    modified = TRUE;

  if (check_south(x-1, y))
    modified = TRUE;
  if (check_south(x+1, y))
    modified = TRUE;

  return modified;
}

static gboolean first_pass_light_coast(gint x, gint y)
{
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  /* check the North tile */
  if (check_north_with_light_coast(x, y))
    modified = TRUE;
  /* check the South tile */
  if (check_south_with_light_coast(x, y))
    modified = TRUE;
  /* check the East tile */
  if (check_east_with_light_coast(x, y))
    modified = TRUE;
  /* check the West tile */
  if (check_west_with_light_coast(x, y))
    modified = TRUE;

  if (check_north(x-1, y))
    modified = TRUE;
  if (check_north(x+1, y))
    modified = TRUE;

  if (check_south(x-1, y))
    modified = TRUE;
  if (check_south(x+1, y))
    modified = TRUE;

  return modified;
}

static gboolean first_pass_dark_coast(gint x, gint y)
{
  gboolean modified = FALSE;
  guint16 old_tile, checked_tile;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  /* check the North tile */
  if (check_north_with_dark_coast(x, y))
    modified = TRUE;
  /* check the South tile */
  if (check_south_with_dark_coast(x, y))
    modified = TRUE;
  /* check the East tile */
  if (check_east_with_dark_coast(x, y))
    modified = TRUE;
  /* check the West tile */
  if (check_west_with_dark_coast(x, y))
    modified = TRUE;

  if (check_north(x-1, y))
    modified = TRUE;
  if (check_north(x+1, y))
    modified = TRUE;

  if (check_south(x-1, y))
    modified = TRUE;
  if (check_south(x+1, y))
    modified = TRUE;

  return modified;
}

static gboolean first_pass_light_grass(gint x, gint y)
{
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  /* check the North tile */
  if (check_north_with_light_grass(x, y))
    modified = TRUE;
  /* check the South tile */
  if (check_south_with_light_grass(x, y))
    modified = TRUE;
  /* check the East tile */
  if (check_east_with_light_grass(x, y))
    modified = TRUE;
  /* check the West tile */
  if (check_west_with_light_grass(x, y))
    modified = TRUE;

  if (check_north(x-1, y))
    modified = TRUE;
  if (check_north(x+1, y))
    modified = TRUE;

  if (check_south(x-1, y))
    modified = TRUE;
  if (check_south(x+1, y))
    modified = TRUE;

  return modified;
}

static gboolean first_pass_dark_grass(gint x, gint y)
{
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  /* check the North tile */
  if (check_north_with_dark_grass(x, y))
    modified = TRUE;
  /* check the South tile */
  if (check_south_with_dark_grass(x, y))
    modified = TRUE;
  /* check the East tile */
  if (check_east_with_dark_grass(x, y))
    modified = TRUE;
  /* check the West tile */
  if (check_west_with_dark_grass(x, y))
    modified = TRUE;

  if (check_north(x-1, y))
    modified = TRUE;
  if (check_north(x+1, y))
    modified = TRUE;

  if (check_south(x-1, y))
    modified = TRUE;
  if (check_south(x+1, y))
    modified = TRUE;

  return modified;
}

static gboolean first_pass_forest(gint x, gint y)
{
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  /* check the North tile */
  if (check_north_with_forest(x, y))
    modified = TRUE;
  /* check the South tile */
  if (check_south_with_forest(x, y))
    modified = TRUE;
  /* check the East tile */
  if (check_east_with_forest(x, y))
    modified = TRUE;
  /* check the West tile */
  if (check_west_with_forest(x, y))
    modified = TRUE;

  if (check_north(x-1, y))
    modified = TRUE;
  if (check_north(x+1, y))
    modified = TRUE;

  if (check_south(x-1, y))
    modified = TRUE;
  if (check_south(x+1, y))
    modified = TRUE;

  return modified;
}

static gboolean first_pass_rocks(gint x, gint y)
{
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  /* check the North tile */
  if (check_north_with_rocks(x, y))
    modified = TRUE;
  /* check the South tile */
  if (check_south_with_rocks(x, y))
    modified = TRUE;
  /* check the East tile */
  if (check_east_with_rocks(x, y))
    modified = TRUE;
  /* check the West tile */
  if (check_west_with_rocks(x, y))
    modified = TRUE;

  if (check_north(x-1, y))
    modified = TRUE;
  if (check_north(x+1, y))
    modified = TRUE;

  if (check_south(x-1, y))
    modified = TRUE;
  if (check_south(x+1, y))
    modified = TRUE;

  return modified;
}

static gboolean first_pass_bridge(gint x, gint y)
{
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  /* check the North tile */
  if (check_north_with_bridge(x, y))
    modified = TRUE;
  /* check the South tile */
  if (check_south_with_bridge(x, y))
    modified = TRUE;
  /* check the East tile */
  if (check_east_with_bridge(x, y))
    modified = TRUE;
  /* check the West tile */
  if (check_west_with_bridge(x, y))
    modified = TRUE;

  if (check_north(x-1, y))
    modified = TRUE;
  if (check_north(x+1, y))
    modified = TRUE;

  if (check_south(x-1, y))
    modified = TRUE;
  if (check_south(x+1, y))
    modified = TRUE;

  return modified;
}

static gboolean second_pass(gint x, gint y)
{
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (check_north(x-1, y-1))
    modified = TRUE;
  if (check_north(x,   y-1))
    modified = TRUE;
  if (check_north(x+1, y-1))
    modified = TRUE;

  if (check_south(x-1, y+1))
    modified = TRUE;
  if (check_south(x,   y+1))
    modified = TRUE;
  if (check_south(x+1, y+1))
    modified = TRUE;

  if (check_east(x+1, y-1))
    modified = TRUE;
  if (check_east(x+1, y))
    modified = TRUE;
  if (check_east(x+1, y+1))
    modified = TRUE;

  if (check_west(x-1, y-1))
    modified = TRUE;
  if (check_west(x-1, y))
    modified = TRUE;
  if (check_west(x-1, y+1))
    modified = TRUE;

  if (check_north(x-2, y-1))
    modified = TRUE;
  if (check_north(x+2, y-1))
    modified = TRUE;

  if (check_south(x-2, y+1))
    modified = TRUE;
  if (check_south(x+2, y+1))
    modified = TRUE;

  return modified;
}

static gboolean third_pass(gint x, gint y)
{
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (check_north(x-2, y-2))
    modified = TRUE;
  if (check_north(x-1, y-2))
    modified = TRUE;
  if (check_north(x,   y-2))
    modified = TRUE;
  if (check_north(x+1, y-2))
    modified = TRUE;
  if (check_north(x+2, y-2))
    modified = TRUE;

  if (check_south(x-2, y+2))
    modified = TRUE;
  if (check_south(x-1, y+2))
    modified = TRUE;
  if (check_south(x,   y+2))
    modified = TRUE;
  if (check_south(x+1, y+2))
    modified = TRUE;
  if (check_south(x+2, y+2))
    modified = TRUE;

  if (check_east(x+2, y-2))
    modified = TRUE;
  if (check_east(x+2, y-1))
    modified = TRUE;
  if (check_east(x+2, y))
    modified = TRUE;
  if (check_east(x+2, y+1))
    modified = TRUE;
  if (check_east(x+2, y+2))
    modified = TRUE;

  if (check_west(x-2, y-2))
    modified = TRUE;
  if (check_west(x-2, y-1))
    modified = TRUE;
  if (check_west(x-2, y))
    modified = TRUE;
  if (check_west(x-2, y+1))
    modified = TRUE;
  if (check_west(x-2, y+2))
    modified = TRUE;

  if (check_north(x-3, y-2))
    modified = TRUE;
  if (check_north(x+3, y-2))
    modified = TRUE;

  if (check_south(x-3, y+2))
    modified = TRUE;
  if (check_south(x+3, y+2))
    modified = TRUE;
  return modified;
}

static gboolean fourth_pass(gint x, gint y)
{
  gboolean modified = FALSE;

  g_return_val_if_fail((0<=x) && (x<Pud.width), FALSE);
  g_return_val_if_fail((0<=y) && (y<Pud.height), FALSE);
  if (check_north(x-3, y-3))
    modified = TRUE;
  if (check_north(x-2, y-3))
    modified = TRUE;
  if (check_north(x-1, y-3))
    modified = TRUE;
  if (check_north(x,   y-3))
    modified = TRUE;
  if (check_north(x+1, y-3))
    modified = TRUE;
  if (check_north(x+2, y-3))
    modified = TRUE;
  if (check_north(x+3, y-3))
    modified = TRUE;

  if (check_south(x-3, y+3))
    modified = TRUE;
  if (check_south(x-2, y+3))
    modified = TRUE;
  if (check_south(x-1, y+3))
    modified = TRUE;
  if (check_south(x,   y+3))
    modified = TRUE;
  if (check_south(x+1, y+3))
    modified = TRUE;
  if (check_south(x+2, y+3))
    modified = TRUE;
  if (check_south(x+3, y+3))
    modified = TRUE;

  if (check_east(x+3, y-3))
    modified = TRUE;
  if (check_east(x+3, y-2))
    modified = TRUE;
  if (check_east(x+3, y-1))
    modified = TRUE;
  if (check_east(x+3, y))
    modified = TRUE;
  if (check_east(x+3, y+1))
    modified = TRUE;
  if (check_east(x+3, y+2))
    modified = TRUE;
  if (check_east(x+3, y+3))
    modified = TRUE;

  if (check_west(x-3, y-3))
    modified = TRUE;
  if (check_west(x-3, y-2))
    modified = TRUE;
  if (check_west(x-3, y-1))
    modified = TRUE;
  if (check_west(x-3, y))
    modified = TRUE;
  if (check_west(x-3, y+1))
    modified = TRUE;
  if (check_west(x-3, y+2))
    modified = TRUE;
  if (check_west(x-3, y+3))
    modified = TRUE;

  if (check_north(x-4, y-3))
    modified = TRUE;
  if (check_north(x+4, y-3))
    modified = TRUE;

  if (check_south(x-4, y+3))
    modified = TRUE;
  if (check_south(x+4, y-3))
    modified = TRUE;
  return modified;
}

void check_around(gint x, gint y, gint tool)
{
  gint idx;
  gboolean modified = FALSE;

  g_return_if_fail((0<=x) && (x<Pud.width));
  g_return_if_fail((0<=y) && (y<Pud.height));
  switch (tool)
  {
    case LIGHT_WATER:
      modified = first_pass_light_water(x, y);
      break;
    case DARK_WATER:
      modified = first_pass_dark_water(x, y);
      break;
    case LIGHT_COAST:
      modified = first_pass_light_coast(x, y);
      break;
    case DARK_COAST:
      modified = first_pass_dark_coast(x, y);
      break;
    case LIGHT_GRASS:
      modified = first_pass_light_grass(x, y);
      break;
    case DARK_GRASS:
      modified = first_pass_dark_grass(x, y);
      break;
    case FOREST:
      modified = first_pass_forest(x, y);
      break;
    case ROCKS:
      modified = first_pass_rocks(x, y);
      break;
    case BRIDGE:
      modified = first_pass_bridge(x, y);
      break;
  }
  if (modified)
  { 
    modified = second_pass(x, y);
  }
  if (modified)
  {
    modified = third_pass(x, y);
  }
  if (modified)
  {
    modified = fourth_pass(x, y);
  }
}

