/*--------------------------------------------------------------------
  	
  Public domain 2002, Rene Herman.
  
  Background load for the contest benchmark. This load creates a bunch
  of processes shipping a record around a "circular pipe".
  
  Concept from Bob Matthews' process_load as shipped with the irman
  (version 0.5) benchmark.
  
--------------------------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <limits.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include "sysinfo.h"
#include "trivial.h"
#include "programs.h"
#include "process_load.h"

/* user settable */
#define N 4		   /* number of processes to fork (per CPU) */
#define R (2 * PIPE_BUF)   /* record size: 0 < R <= SSIZE_MAX       */	

extern int opt_pl_nr_procs;

char buf[R];

static int child(int readfd, int writefd)
{
	ssize_t r;

	while ((r = read(readfd, buf, R)) > 0)
		if (write(writefd, buf, r) != r) {
			perror("Write error");
			return EXIT_FAILURE;
		}
	if (r) {
		perror("Read error 1");
		return EXIT_FAILURE;
	}
	return EXIT_SUCCESS;
}

static volatile sig_atomic_t term;

static void sigterm(int signum gcc_unused)
{
	term = 1;
}

int do_process_load(void)
{
	int 		 i, n;
	struct sigaction action;
	int		 prevfd[2], currfd[2];
	ssize_t		 r;

	n = N;
	if(!(n=opt_pl_nr_procs)) {
		if(get_cpus(&n)){
			perror("Could not get info on number of processors\n");
			return EXIT_FAILURE;
		}
		n *= N;
	}

	sigemptyset(&action.sa_mask);

	/* children ignore SIGTERM, we handle it globally  */
	action.sa_handler = SIG_IGN;
	action.sa_flags = 0;
	if (sigaction(SIGTERM, &action, NULL) == -1) {
		perror("Could not install signal handler");
		return EXIT_FAILURE;
	}	

	/* create first pipe */
	if (pipe(prevfd) == -1) {
		perror("Could not create pipe");
		return EXIT_FAILURE;
	}

	/* fork n - 1 children for a total of n processes */
	for (i = n - 1; i; i--) {
		if (pipe(currfd) == -1) {
			perror("Could not create pipe");
			return EXIT_FAILURE;
		}	
		switch (fork()) {
		case -1:
			perror("Could not fork");
			return EXIT_FAILURE;
		case 0:
			close(prevfd[1]);
			close(currfd[0]);
			_exit(child(prevfd[0], currfd[1]));
		default:	
			close(prevfd[0]);
			close(currfd[1]);
			prevfd[0] = currfd[0];
		}
	}

	/* establish termination handler */
	action.sa_handler = sigterm;
	action.sa_flags = SA_RESTART;
	if (sigaction(SIGTERM, &action, NULL) == -1) {
		perror("Could not install signal handler");
		return EXIT_FAILURE;
	}	

	/* kick the ball until sigterm */
	while (!term) {
		/* SA_RESTART ensures no EINTR; partial is okay */
		if ((r = write(prevfd[1], buf, R)) == -1) {
			perror("Write error");
			return EXIT_FAILURE;
		}
		if (read(currfd[0], buf, r) == -1) {
			perror("Read error 2");
			return EXIT_FAILURE;
		}
		if(++i==10000){
			report_progress();
			i=0;
		}
	}

	/* close initiates cascade of cleanly exiting children */
	close(prevfd[1]);

	/* if interrupted read(), last child still in write() */
	while ((r = read(currfd[0], buf, R)) > 0);
	if (r) {
		perror("Read error 3");
		return EXIT_FAILURE;
	}

	/* reap 'em and weep */
	while (--n)
		if (wait(NULL) == -1) {
			perror("Could not collect children");
			return EXIT_FAILURE;
		}

	return EXIT_SUCCESS;
}

